///////////////////////////////////////////////////////////////////////////////
// $Id: mapview.cpp,v 1.11 2004/07/03 21:52:47 krake Exp $
//
// Package:   MOAGG Edit - Level Editor for MOAGG
// Copyright: Kevin Krammer, 2003
//
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
///////////////////////////////////////////////////////////////////////////////

/*! \file    mapview.cpp
    \author  Kevin Krammer, kevin.krammer@gmx.at
    \brief   Implementation of the view area for level maps
*/

// Qt includes
#include <qlabel.h>
#include <qpainter.h>

// local includes
#include "mapdocument.h"
#include "maplayer.h"
#include "mapview.h"

// tool includes
#include "maptoolline.h"
#include "maptoolpoint.h"
#include "maptoolrectangle.h"
#include "maptoolselect.h"

///////////////////////////////////////////////////////////////////////////////

MapView::MapView(MapDocument* doc, QLabel* poslabel, QWidget* parent, const char* name)
	: QScrollView(parent, name, Qt::WResizeNoErase | Qt::WRepaintNoErase |
	                            Qt::WNorthWestGravity | Qt::WPaintClever |
	                            Qt::WStaticContents),
	m_doc(doc),
	m_posLabel(poslabel),
	m_numCols(30),
	m_numRows(30),
	m_cellSize(16, 16),
	m_drawGrid(true),
	m_currentTool(0)
{
	resizeContents(m_numCols * m_cellSize.width(), m_numRows * m_cellSize.height());

	viewport()->setMouseTracking(true);

	setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

	// init tools
	m_toolLine      = new MapToolLine(m_doc);
	m_toolPoint     = new MapToolPoint(m_doc);
	m_toolRectangle = new MapToolRectangle(m_doc);
	m_toolSelect    = new MapToolSelect(m_doc);
}

///////////////////////////////////////////////////////////////////////////////

MapView::~MapView()
{
	delete m_toolLine;
	delete m_toolPoint;
	delete m_toolRectangle;
	delete m_toolSelect;
}

///////////////////////////////////////////////////////////////////////////////

void MapView::activateTool(Tool tool)
{
	switch (tool)
	{
		case Select:
			m_currentTool = m_toolSelect;
			break;

		case Point:
			m_currentTool = m_toolPoint;
			break;

		case Line:
			m_currentTool = m_toolLine;
			break;

		case Rectangle:
			m_currentTool = m_toolRectangle;
			break;

		default:
			m_currentTool = 0;
			break;
	}

	if (m_currentTool != 0)
	{
		m_currentTool->activate();
		m_doc->setSelectionArea(QRect());
		updateContents();
	}
}

///////////////////////////////////////////////////////////////////////////////

void MapView::setNumCols(int cols)
{
	if (cols <= 0) return;

	m_numCols = cols;

	updateContents();
}

///////////////////////////////////////////////////////////////////////////////

void MapView::setNumRows(int rows)
{
	if (rows <= 0) return;

	m_numRows = rows;

	updateContents();
}

///////////////////////////////////////////////////////////////////////////////

void MapView::setDrawGrid(bool on)
{
	m_drawGrid = on;

	updateContents();
}

///////////////////////////////////////////////////////////////////////////////

void MapView::contentsMousePressEvent(QMouseEvent* e)
{
	int col = columnAt(e->pos().x());
	int row = rowAt(e->pos().y());

	if (m_currentTool != 0)
	{
		QRect coverArea = m_currentTool->coverArea();

		if (m_currentTool->usedMousePress(static_cast<uint>(row),
		                                  static_cast<int>(col),
	                                      e->button()))
		{
			QRect r = m_currentTool->coverArea().unite(coverArea);
			int w = m_cellSize.width();
			int h = m_cellSize.height();

			updateContents(r.x() * w, r.y() * h,
			               r.width() * h, r.height() * h);
			return;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

void MapView::contentsMouseReleaseEvent(QMouseEvent* e)
{
	int col = columnAt(e->pos().x());
	int row = rowAt(e->pos().y());

	if (m_currentTool != 0)
	{
		QRect coverArea = m_currentTool->coverArea();

		if (m_currentTool->usedMouseRelease(static_cast<uint>(row),
		                                    static_cast<int>(col),
	                                        e->button()))
		{
			QRect r = m_currentTool->coverArea().unite(coverArea);
			int w = m_cellSize.width();
			int h = m_cellSize.height();

			updateContents(r.x() * w, r.y() * h,
			               r.width() * h, r.height() * h);
			return;
		}
	}

	m_doc->setSelectionArea(QRect());
	updateContents();
}

///////////////////////////////////////////////////////////////////////////////

void MapView::contentsMouseMoveEvent(QMouseEvent* e)
{
	int col = columnAt(e->pos().x());
	int row = rowAt(e->pos().y());

	if (col < 0 || row < 0 || col >= m_numCols || row >= m_numRows)
	{
		m_posLabel->setText(tr("X: - , Y: - "));
		return;
	}
	QString posText;
	posText.sprintf(tr("X:%2d, Y:%2d"), col, row);
	m_posLabel->setText(posText);


	if (m_currentTool != 0)
	{
		QRect coverArea = m_currentTool->coverArea();

		if (m_currentTool->usedMouseMove(static_cast<uint>(row),
		                                 static_cast<int>(col),
	                                     e->state()))
		{
			QRect r = m_currentTool->coverArea().unite(coverArea);
			int w = m_cellSize.width();
			int h = m_cellSize.height();

			updateContents(r.x() * w, r.y() * h,
			               r.width() * h, r.height() * h);
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

void MapView::contentsContextMenuEvent(QContextMenuEvent* e)
{
	switch (e->reason())
	{
		case QContextMenuEvent::Mouse:
		{
			// check if on selecttion
			int col = columnAt(e->pos().x());
			int row = rowAt(e->pos().y());
			if (!m_doc->selectionArea().contains(col, row)) return;
			break;
		}

		default:
			if (!m_doc->selectionArea().isValid()) return;
			break;
	}

	e->accept();

	emit contextMenu(e->globalPos());
}

///////////////////////////////////////////////////////////////////////////////

void MapView::drawContents(QPainter* p, int clipx, int clipy, int clipw, int cliph)
{
	//qDebug("drawContents(%d, %d, %d, %d)", clipx, clipy, clipw, cliph);

	int maxCol = columnAt(clipx+clipw);
	int maxRow = rowAt(clipy+cliph);

	maxCol = maxCol >= m_numCols ? m_numCols-1 : maxCol;
	maxRow = maxRow >= m_numRows ? m_numRows-1 : maxRow;

	for (int r = rowAt(clipy); r <= maxRow; ++r)
	{
		for (int c = columnAt(clipx); c <= maxCol; ++c)
		{
			paintCell(p, r, c);
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

void MapView::paintCell (QPainter * p, int row, int col)
{
	int x = col * m_cellSize.width();
	int y = row * m_cellSize.height();

	QPixmap* pix = m_doc->pixmap(row, col);
	if (pix != 0)
	{
		p->drawPixmap(x, y, *pix);
	}
	if (m_drawGrid)
	{
		int x2 = x+m_cellSize.width() - 1;
		int y2 = y+m_cellSize.height() - 1;
		p->setPen(Qt::white);
		p->drawLine(x, y, x2, y);
		p->drawLine(x, y, x, y2);
	}

	if (m_currentTool != 0 && m_currentTool->coversTileAt(row, col))
	{
		p->fillRect(x, y, m_cellSize.width(), m_cellSize.height(),
					QBrush(Qt::white, QBrush::Dense4Pattern));
	}
	else if (m_doc->selectionArea().contains(row, col))
	{
		p->fillRect(x, y, m_cellSize.width(), m_cellSize.height(),
					QBrush(Qt::white, QBrush::Dense4Pattern));
	}
}

///////////////////////////////////////////////////////////////////////////////

void MapView::slotDocumentChanged()
{
	if (m_currentTool != 0)
		m_currentTool->activate();
	
	setNumRows(m_doc->numRows());
	setNumCols(m_doc->numCols());
	int w = m_numCols * m_cellSize.width();
	w = QMAX(w, visibleWidth());
	int h = m_numRows * m_cellSize.height();
	h = QMAX(h, visibleHeight());

	resizeContents(w, h);
	repaintContents(true);
}

// End of file

