/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * The goal of a border router is to take advantage of "collateral
 * flow" that occurs "external" to a Mobile Mesh Routing Protocol
 * cloud. This is accomplished by border routers tunneling packets
 * between peer border routers. A tunnel is needed between each pair
 * of members. The IP addresses on the ends of the tunnels do not need
 * to be globally unique across the Internet, they only need to be
 * unique within the Mobile Mesh Routing Protocol cloud. This enables
 * us to use the private IP addresses of 10.X.X.X and 192.168.X.X for
 * the tunnels.
 *
 * Tunnels are set up automatically when a border router discovers a
 * peer. The discovery process requires each border router to
 * periodically send a BorderAdvertisement message to a multicast
 * group address. When a border hears from a new peer, it sets up a
 * point to point tunnel with it. When a border has not heard from a
 * peer within a DEAD_INTERVAL, it assumes the peer has departed and
 * tears down the tunnel. 
 *
 * Under Linux, each tunnel results in a unique IP interface. When a
 * tunnel is created, the border program must launch the link
 * discovery program on the tunnel interface. It then dynamically adds
 * the tunnel interface to the mobile mesh router. Similary, when the
 * tunel is destroyed, the border must dynamically remove the tunnel
 * interface from the mobile mesh router and then kill the link
 * discovery prorgam running on the interface.
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *         
 *
 * $Id$
 *       
 */
#ifndef __BoBorder_h
#define __BoBorder_h

#include <fstream.h>
#include <list.h>
#include <map.h>
#include <set.h>

#include <BoMsg.h>
#include <UtChild.h>
#include <UtDebug.h>
#include <UtInterface.h>
#include <UtUdpSocket.h>
#include <UtUnixSocket.h>
#include <UtReport.h>
#include <UtString.h>
#include <UtTime.h>


class Border {
protected:
  UnixSocket *cMmrpSocket;    // Where we send dynamic add/delete interface cmds to router 
  UdpSocket  *cIfSocket;      // Where we send BorderAdvertisement messages
  Interface  cInterface;      // The interface we are bound to

  fd_set cFds;
  int cFdLimit;
  InetAddr   cGroupAddr;       // Multicast group address where we discover borders
  String     cTunnelAddr;      // Unique private IP address for our tunnels

  unsigned short cPort;
  unsigned short cMmrpPort;
  unsigned int cAdPeriod;     // Seconds between sending BorderAdvertisement messages
  unsigned int cDeadInterval; // Seconds after hearing a BorderAdvertisement 
                              // that we declare a link down
  unsigned int cPrunePeriod;  // How frequently we look for expired BorderAdvertisement's

  // Keys are source IP addr's of BorderAdvertisement's that we've heard
  // values are the BorderAdvertisement's
  typedef map<unsigned int, BorderAdvertisement, less<unsigned int> > AdMap;
  AdMap cAdMap;

  // Keys are source IP addr's of BorderAdvertisement's that we've heard, 
  // values are when we heard the BorderAdvertisement
  typedef map<unsigned int, Time,  less<unsigned int> > AdTimeMap;
  AdTimeMap cAdTimeMap;

  // Keys are source IP addr's of BorderAdvertisement's that we've heard
  // values are the names of the corresponding tunnels
  typedef map<unsigned int, String, less<unsigned int> > TunnelMap;
  TunnelMap cTunnelMap;

  // Keys are source IP addr's of BorderAdvertisement's that we've heard
  // values are pointer to Child objects that are running the Link Discovery prorgram
  typedef map<unsigned int, Child*, less<unsigned int> > ChildMap;
  ChildMap cChildMap;


  bool cExpectRouterReply;

  int cTunnelCount;
  String cDiscoverConfigFile;

  /* 1) Builds a new IP interface for tunneling to the given addr
   * 2) Starts the Link Discovery program on the tunnel interface
   * 3) Then dynamically adds the tunnel interface to Mobile Mesh Router
   */
  void AddTunnel(unsigned int addr) {
    String error;
    String name("bo");
    name += String((int)cTunnelCount);
    String cmd = String("mmtunnel add ") + name + " " + 
      DumpAddr(cInterface.Addr()) + " " + DumpAddr(addr) + " " + cTunnelAddr;
    if(!Child::System(cmd,0,0,&error)) {
      String err = String("Border::AddTunnel() - '") + cmd + "' failed: " + error;
      Report::Error(err);
    }
    cTunnelMap[addr] = name;
    cTunnelCount++;

    Child* child = new Child(String("mmdiscover -i ") + name + " -f " + cDiscoverConfigFile);
    child->Open();
    
    // The discover program must have a chance to initialize itself
    // before we tell the Mobile Mesh Router since mmrp tries to ping it.
    // Wait until the discover program responds to a ping before telling mmrp.
    String q("q\n");
    String discoverSocketName = "ln-" + name;
    UnixDatagram ping(q,discoverSocketName);
    UnixSocket tmpSocket;
    while(!tmpSocket.Send(ping)) {
      sleep(1);
    }
    cChildMap[addr] = child;
    
    String s = String("a ") + name + "\n";
    String socketName = "mmrp-c" + String((int)cMmrpPort);
    UnixDatagram msg(s,socketName);
    if(!cMmrpSocket->Send(msg)) {
      String err = String("Border::AddTunnel() - send to mmrp failed!");
      Report::Error(err);
    }
  }

  /* 1) Dynamically removes the tunnel interfaces associated with the given remote addr
   * from the Mobile Mesh Router. 
   * 2) Stops the Link Discovery program on the tunnel interface.
   * 3) Deletes the tunnel interface.
   */
  void DelTunnel(unsigned int addr) {

    String error;
    TunnelMap::iterator j = cTunnelMap.find(addr);
    String name = (*j).second;

    String s = String("d ") + name + "\n";
    String socketName = "mmrp-c" + String((int)cMmrpPort);
    UnixDatagram msg(s,socketName);
    if(!cMmrpSocket->Send(msg)) {
      String err = String("Border::DelTunnel() - send to mmrp failed!");
      Report::Warn(err);
    }

    ChildMap::iterator k = cChildMap.find(addr);
    Child* child = (*k).second;
    child->Close();
    delete(child);
    cChildMap.erase(k);

    String cmd = String("mmtunnel del ") + name;
    cTunnelMap.erase(j);
    if(!Child::System(cmd,0,0,&error)) {
      String err = String("Border::DelTunnel() - '") + cmd + "' failed: " + error;
      Report::Warn(err);
    }
  }
  
  /* A crude test to determine if the Mobile Mesh router is still operational
   * Send a ping request...we harvest ping replies asynchronously
   * later. If we didn't hear a ping reply since last time we checked, we exit.
   */
  void CheckRouter() {
    String err = String("mmrp on port ") + String((int)cMmrpPort)  + " is not responding. " +
      "Verify mmrp is running on this port and restart it before running border.";
    
    if(cExpectRouterReply) {
      Report::Error(err);
    }
    cExpectRouterReply = true;
    
    // Send a ping request to LinkDiscover program on this interface
    String s = "q\n";
    String name = "mmrp-c" + String((int)cMmrpPort);
    UnixDatagram reply(s,name);
    if(!cMmrpSocket->Send(reply)) {
      Report::Error(err);
    }
  }
  
  /* Iterates across the Ad Time map and looks for Ads that were heard longer
   * than a Dead period ago. Any of these are removed from the Ad map and
   * the Ad time map and the corresponding tunnel is torn down.
   */
  void PruneOldAds() {
    Time now = Time::Now();
    AdTimeMap::iterator it;
    AdTimeMap& atm = cAdTimeMap;
    for(it = atm.begin(); it != atm.end(); ) {
      unsigned int addr = (*it).first;
      Time&   t = (*it).second;
      if(double(now) > double(t + cDeadInterval)) {
	Report::Warn(String("PruneOldAds() - Ad from ") + DumpAddr(addr) +
		     " expired...deleting.");
	//#### Dynamically remove interface and delete tunnel
	DelTunnel(addr);
	AdMap& am = cAdMap;
	am.erase(addr);
	atm.erase(it++);
      }
      else {
	it++;
	if(Debug("PruneOldAds")) {
	  Report::Debug(String("PruneOldAds() - Ad from ") + DumpAddr(addr) +
			" arrived at " + t.Str());
	}
      }
    }
  }

  /* Multicasts a BorderAdvertisement out the fixed network interface
   */
  void SendAd() {
    BorderAdvertisement a;
    if(Debug("SendAd"))
      Report::Debug(String("SendAd() - sending ") + a.Dump());
    UdpDatagram dg(a.Encode(), cGroupAddr);
    cIfSocket->Send(dg);
  }

  /* Stores the Ad...  If we didn't have an Ad from this neighbor,
   * create a tunnel and dynamically add the interface 
   */
  void HandleAd(const BorderAdvertisement& a, const InetAddr& src) {

    if(Debug("HandleAd")) {
      String s = String("HandleAd() - From ") +  
	String(src) + " on " + DumpAddr(cInterface.Addr()) + " : " + a.Dump();
      Report::Debug(s);
    }

    bool haveIt = (cAdMap.find(src.Addr()) != cAdMap.end());

    if(!haveIt) {
      //### Create tunnel and add interface
      AddTunnel(src.Addr());
    }
    cAdMap[src.Addr()]; // If class BorderAdvertisement had any data then
                        // this should be an assignment: cAdMap[src.Addr()] = a;
    cAdTimeMap[src.Addr()] = Time::Now();
  }

  /* Handle replies from the Mobile Mesh Router
   *
   * Replies are in the following format:
   *
   *  Listing of a current interfaces:
   *     "l <addr> "
   *
   *  Ping reply:
   *     "r"
   *
   */
  void HandleRouter(UnixDatagram& dg) {
    String d = dg.GetData();
    UnixAddr src = dg.GetAddr();

    if(Debug("HandleRouter")) {
      Report::Debug(String("HandleRouter() - From: ") + String(dg.GetAddr()) + "  '" + dg.GetData() + "' ");
    }

    if(d.length() == 0) {
      char buf[1024];
      sprintf(buf,"Border::HandleRouter() - Invalid message from %s, discarding.",
	      String(src).c_str());
      Report::Warn(buf);
      return;
    }

    unsigned char rep = d[0];
    
    String err("Border::HandleRouter() - Invalid command format.");

    // Consume report
    d.Token(&d);
    switch(rep) {
    case 'l':
      {
	break;
      }
    case 'r':
      {
	// Ping reply
	cExpectRouterReply = false;
	break;
      }
    default:
      {
	Report::Warn(err);
      }
    }
  }

  /* Determine what type of message this is, decode it, and dispatch it */
  void HandleReceive(UdpDatagram& dg) {
    String d = dg.GetData();
    InetAddr src = dg.GetAddr();
    
    if(d.length() < 2) {
      char buf[1024];
      sprintf(buf,"Border::HandleReceive() - Invalid message from %s, discarding.",
	      String(src).c_str());
      Report::Warn(buf);
      return;
    }

    // Make sure we're not hearing one of our own!
    if(cInterface.Addr() == src.Addr()) {
      return;
    }

    unsigned char version = d[0];
    if(version != MsgTypes::Version) {
      char buf[1024];
      sprintf(buf,"Border::HandleReceive() - Invalid version from %s, discarding.",
	      String(src).c_str());
      Report::Warn(buf);
      return;
    }

    unsigned char type = d[1];
    switch(type) {
    case MsgTypes::tBorderAdvertisement:
      {
	BorderAdvertisement ad;
	if(ad.Decode(d))
	  HandleAd(ad,src);
	break;
      }
    default:
      {
	char buf[1024];
	sprintf(buf,"Border::HandleReceive() - Invalid message type from %s, discarding.",
		String(src).c_str());
	Report::Warn(buf);
      }
    }
  }


  /* Reads configuration information from the stream
   */
  virtual void ReadConfiguration(istream& is) {
 
    String err("Invalid line in config file: ");
    String s;

    while(s.Getline(is)) {
      String line = s;
      String type     = s.Token(&s);
      if(!type.length()) {
	continue; // Ignore blank lines
      }
      if(type[0] == '#') {
	continue; // Ignore lines that begin with a '#' comment
      }
      else if(type == "groupAddr") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	cGroupAddr = InetAddr(v);
      }
      else if(type == "port") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	cPort = v.Int();
      }
      else if(type == "mmrpPort") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	cMmrpPort = v.Int();
      }
      else if(type == "adPeriod") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	cAdPeriod = v.Int();
      }
      else if(type == "deadInterval") {
	String v = s.Token(&s);
	if(!v.length()) {
	  Report::Error(err + line);
	}
	cDeadInterval = v.Int();
      }
      else if(type != "") {
	Report::Error(err + line);
      }
    }
  }

public:
  Border(const String& tunnelAddr, const String& ifname, 
         const String& configFile, const String& discoverConfigFile, const String& debugFile) :
    cGroupAddr("224.1.2.3"), cTunnelAddr(tunnelAddr), 
    cPort(17556), cMmrpPort(20470), cAdPeriod(5), 
    cDeadInterval(30), cPrunePeriod(1), cExpectRouterReply(false), cTunnelCount(0),
    cDiscoverConfigFile(discoverConfigFile)
  {
    
    // Make sure we were given a valid interface name!
    list<Interface> sysList = Interface::Discover();
    bool found = false;
    list<Interface>::iterator it;
    for(it = sysList.begin(); it != sysList.end(); it++) {
      if((*it).Name() == ifname) {
	found = true;
	cInterface = (*it);
	break;
      }
    }
    if(!found) {
      Report::Error(String("Invalid interface name: ") + 
		    ifname.c_str() + 
		    ",  make sure interface is up with 'ifconfig'. ");
    }

    if(debugFile.length())
      Debug::Load(debugFile);

    ifstream is(configFile.c_str());
    if(!is) {
      Report::Error(String("Missing configuration file: ") + configFile);
    }
    ReadConfiguration(is);

    ifstream iss(discoverConfigFile.c_str());
    if(!iss) {
      Report::Error(String("Missing configuration file: ") + discoverConfigFile);
    }

    cGroupAddr = InetAddr(cGroupAddr.Addr(), cPort);

    // Here if the interface name is valid
    FD_ZERO(&cFds);
    cFdLimit = 0;
    
    // Build a Unix socket to send cmds to the Mobile Mesh Router
    cMmrpSocket = new UnixSocket();
    int fd = cMmrpSocket->Fd();
    FD_SET(fd,&cFds);
    if(fd > cFdLimit) cFdLimit = fd;
    
    // Build a Udp socket for sending BorderAdvertisement messages to our peers
    cIfSocket = new UdpSocket(cGroupAddr,0);
    cIfSocket->BindDevice(ifname);
    fd = cIfSocket->Fd();
    FD_SET(fd,&cFds);
    if(fd > cFdLimit) cFdLimit = fd;
    
    cFdLimit += 1; // Sheesh, vagaries of select()
  }
  
  virtual ~Border() {
    // Destroy any active tunnels
    TunnelMap::iterator it;
    for(it = cTunnelMap.begin(); it != cTunnelMap.end(); it++) {
      DelTunnel((*it).first);
    }
    cTunnelMap.clear();

    delete(cMmrpSocket);
    cMmrpSocket = 0;
    delete(cIfSocket);
    cIfSocket = 0;
  }
  

  /* Main event loop for the Link Discover...
   *
   */
  void Run() {

    Time now = Time::Now();
    Time adTime = now + cAdPeriod;
    Time pruneTime = now + cPrunePeriod + Time(0.5);  // prune when we are not busy...

    while(1) {
      Time earliest = Min(adTime,pruneTime);
      Time timeleft = earliest - Time::Now();


      fd_set fds = cFds;      
      timeval tv = timeval(timeleft);
      int n;

      if(Debug("Run")) {
	Report::Debug(String("Run() - timeleft=") + timeleft.Str());
      }

      if((n=select(cFdLimit, &fds, 0, 0, &tv)) == -1) {
	char buf[1024];
	sprintf(buf,"Border::Run() - select failed: %s",strerror(errno));
	Report::Error(buf);
      }

      if(n > 0) {
	if(FD_ISSET(cIfSocket->Fd(),&fds)) {
	  // Handle receive
	  UdpDatagram dg = cIfSocket->Recv();
	  if(Debug("Run")) {
	    Report::Debug(String("Run() - From ") + String(dg.GetAddr()) + 
			  " on " + cInterface.Name());
	  }
	  HandleReceive(dg);
	}
	if(FD_ISSET(cMmrpSocket->Fd(),&fds)) {
	  UnixDatagram dg = cMmrpSocket->Recv();
	  HandleRouter(dg);
	}
      }

      Time now = Time::Now();
      if(double(earliest) <= double(now)) {
	// Time to do something!

	// Should we prune oldies?
	if(double(pruneTime) <= double(now)) {
	  pruneTime = pruneTime + cPrunePeriod;
	  PruneOldAds();
	}

	// Should we send a BorderAdvertisement?
	if(double(adTime) <= double(now)) {
	  adTime = adTime + cAdPeriod;
	  PruneOldAds();
	  SendAd();
	  Debug::Update();  // Convenient place to check if debug flags have changed
	  CheckRouter();
	}
      }
    }
  }
  
  
};


#endif
