#include <config.h>

#include <string.h>
#include <hildon-cp-plugin/hildon-cp-plugin-interface.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>
#include <glade/glade.h>
#include <gdk/gdkx.h>

#include <X11/extensions/Xrandr.h>

#include "capplet-util.h"

enum {
	COL_NAME,
	COL_ID,
	N_COLUMNS
};

static int resolution_count = 0;
static XRRScreenSize *resolutions_supported = NULL;

#define REVERT_COUNT 20

static struct {
	Rotation  rotation;
	gchar const    * name;
} const rotations[] = {
	{RR_Rotate_0,   N_("normal")},
	{RR_Rotate_180, N_("inverted")}
};

static void
get_default_resolutions() {
    GConfClient *client;
    char *res, *tmp;
    int w, h;

	if(resolutions_supported) free(resolutions_supported);
	resolutions_supported = NULL;
	resolution_count = 0;
	client = gconf_client_get_default ();
	res = gconf_client_get_string (client, "/desktop/moblin/screen/resolutions", NULL);
	g_object_unref (client);
	if(!res) return;

	g_debug("gconf resolutions are %s\n", res);
	for(tmp = strtok(res, ":;, \t\n"); tmp != NULL; tmp = strtok(NULL, ":;, \t\n"))
	{
	    if(sscanf(tmp, "%dx%d", &w, &h) == 2)
	    {
		resolution_count++;
		if(resolutions_supported) {
		    resolutions_supported = 
			(XRRScreenSize *)realloc(resolutions_supported, 
			sizeof(XRRScreenSize)*resolution_count);
		} else {
		    resolutions_supported = 
			(XRRScreenSize *)malloc(sizeof(XRRScreenSize));
		}
		resolutions_supported[resolution_count-1].width = w;
		resolutions_supported[resolution_count-1].height = h;
		g_debug("adding resolution %dx%d\n", w, h);
	    }
	}
	g_free (res);
}

static Rotation
display_rotation_from_text(gchar const* text) {
	int i = 0;
	g_return_val_if_fail(text, RR_Rotate_0);

	for(; i < G_N_ELEMENTS(rotations); i++) {
		if(!strcmp(text, _(rotations[i].name))) {
			break;
		}
	}

	g_return_val_if_fail(i < G_N_ELEMENTS(rotations), RR_Rotate_0);

	return rotations[i].rotation;
}

struct ScreenInfo
{
  int current_width;
  int current_height;
  SizeID current_size;

  int minWidth;
  int minHeight;
  int maxWidth;
  int maxHeight;
  
  Rotation current_rotation;
  Rotation old_rotation;
  Rotation rotations;

  SizeID old_size;
  
  XRRScreenConfiguration *config;
  XRRScreenSize *sizes;
  int n_sizes;
  
  GtkWidget *resolution_widget;
  GtkWidget *rotate_widget;
};

struct DisplayInfo {
  int n_screens;
  struct ScreenInfo *screens;
};


static void generate_resolution_menu(struct ScreenInfo* screen_info);

static struct DisplayInfo *
read_display_info (GdkDisplay *display)
{
    struct DisplayInfo *info;
    struct ScreenInfo *screen_info;
    GdkScreen *screen;
    GdkWindow *root_window;
    int i, j, k, sz;
    Display *dpy;
    Window window;
    XRRScreenSize *res;

	dpy = gdk_x11_display_get_xdisplay (display);
	window = gdk_x11_get_default_root_xwindow();
	info = g_new (struct DisplayInfo, 1);
	info->n_screens = gdk_display_get_n_screens (display);
	info->screens = g_new (struct ScreenInfo, info->n_screens);

	for (i = 0; i < info->n_screens; i++)
	{
	    screen = gdk_display_get_screen (display, i);
	    screen_info = &info->screens[i];
	    screen_info->current_width = gdk_screen_get_width (screen);
	    screen_info->current_height = gdk_screen_get_height (screen);
	    root_window = gdk_screen_get_root_window (screen);
	    screen_info->config = 
		XRRGetScreenInfo (gdk_x11_display_get_xdisplay (display),
		gdk_x11_drawable_get_xid (GDK_DRAWABLE (root_window)));
	    screen_info->current_size = 
		XRRConfigCurrentConfiguration(screen_info->config, 
		&screen_info->current_rotation);

	    if(resolutions_supported) {
		res = XRRConfigSizes (screen_info->config, &sz);
		screen_info->n_sizes = 0;
		screen_info->sizes = NULL;
		for(j = 0; j < resolution_count; j++) {
		    for(k = 0; k < sz; k++) {
			if((res[k].width == resolutions_supported[j].width)&&
			  (res[k].height == resolutions_supported[j].height))
			{
			    screen_info->n_sizes++;
			    if(screen_info->sizes) {
				screen_info->sizes = 
				    (XRRScreenSize *)realloc(screen_info->sizes, 
				    sizeof(XRRScreenSize)*(screen_info->n_sizes));
			    } else {
				screen_info->sizes = 
				    (XRRScreenSize *)malloc(sizeof(XRRScreenSize));
			    }
			    screen_info->sizes[screen_info->n_sizes-1] = res[k];
			}
		    }
		}
		/* if we've found no compatible sizes, add the current size */
		if((screen_info->n_sizes < 1)&&(screen_info->current_size < sz))
		{
		    screen_info->n_sizes = 1;
		    screen_info->sizes = 
			(XRRScreenSize *)malloc(sizeof(XRRScreenSize));
		    screen_info->sizes[0] = res[screen_info->current_size];
		}
	    } else {
		screen_info->sizes = 
		    XRRConfigSizes (screen_info->config, &screen_info->n_sizes);
	    }
	    screen_info->rotations = 
		XRRConfigRotations (screen_info->config, 
		&screen_info->current_rotation);
	    XRRGetScreenSizeRange (dpy, window, 
		&screen_info->minWidth, &screen_info->minHeight, 
		&screen_info->maxWidth, &screen_info->maxHeight);
	}
	return info;
}

static void
update_display_info (struct DisplayInfo *info, GdkDisplay *display)
{
    struct ScreenInfo *screen_info;
    GdkScreen *screen;
    GdkWindow *root_window;
    Display *dpy;
    Window window;
    XRRScreenSize *res;
    int i, j, k, sz;

	dpy = gdk_x11_display_get_xdisplay (display);
	window = gdk_x11_get_default_root_xwindow();

	g_assert (info->n_screens == gdk_display_get_n_screens (display));
  
	for (i = 0; i < info->n_screens; i++)
	{
	    screen = gdk_display_get_screen (display, i);
	    screen_info = &info->screens[i];
	    screen_info->old_size = screen_info->current_size;
	    screen_info->old_rotation = screen_info->current_rotation;
	    screen_info->current_width = gdk_screen_get_width (screen);
	    screen_info->current_height = gdk_screen_get_height (screen);
	    root_window = gdk_screen_get_root_window (screen);
	    XRRFreeScreenConfigInfo (screen_info->config);
	    screen_info->config = 
		XRRGetScreenInfo (gdk_x11_display_get_xdisplay (display),
		gdk_x11_drawable_get_xid (GDK_DRAWABLE (root_window)));
	    screen_info->current_size = 
		XRRConfigCurrentConfiguration (screen_info->config, 
		&screen_info->current_rotation);

	    if(resolutions_supported) {
		res = XRRConfigSizes (screen_info->config, &sz);
		screen_info->n_sizes = 0;
		if(screen_info->sizes) {
		    free(screen_info->sizes);
		    screen_info->sizes = NULL;
		}
		for(j = 0; j < resolution_count; j++) {
		    for(k = 0; k < sz; k++) {
			if((res[k].width == resolutions_supported[j].width)&&
			   (res[k].height == resolutions_supported[j].height))
			{
			    screen_info->n_sizes++;
			    if(screen_info->sizes) {
				screen_info->sizes = 
				    (XRRScreenSize *)realloc(screen_info->sizes, 
				    sizeof(XRRScreenSize)*(screen_info->n_sizes));
			    } else {
				screen_info->sizes = 
				    (XRRScreenSize *)malloc(sizeof(XRRScreenSize));
			    }
			    screen_info->sizes[screen_info->n_sizes-1] = res[k];
			}
		    }
		}
		/* if we've found no compatible sizes, add the current size */
		if((screen_info->n_sizes < 1)&&(screen_info->current_size < sz))
		{
		    screen_info->n_sizes = 1;
		    screen_info->sizes = 
			(XRRScreenSize *)malloc(sizeof(XRRScreenSize));
		    screen_info->sizes[0] = res[screen_info->current_size];
		}
	    } else {
		screen_info->sizes = 
		    XRRConfigSizes (screen_info->config, 
		    &screen_info->n_sizes);
	    }
	    screen_info->rotations    = XRRConfigRotations (screen_info->config, &screen_info->current_rotation);
	    XRRGetScreenSizeRange (dpy, window, 
		&screen_info->minWidth, &screen_info->minHeight, 
		&screen_info->maxWidth, &screen_info->maxHeight);
	}
}

static int
get_current_resolution (struct ScreenInfo *screen_info)
{
  GtkComboBox* combo = GTK_COMBO_BOX (screen_info->resolution_widget);
  GtkTreeIter iter;
  int i = 0;

  gtk_combo_box_get_active_iter (combo, &iter);
  gtk_tree_model_get (gtk_combo_box_get_model (combo), &iter,
		      COL_ID, &i,
		      -1);
  return i;
}

static Rotation
get_current_rotation(struct ScreenInfo* screen_info) {
	gchar* text;
	Rotation rot;
	text = gtk_combo_box_get_active_text (GTK_COMBO_BOX (screen_info->rotate_widget));
	rot = display_rotation_from_text (text);
	g_free (text);
	return rot;
}

static gchar *
get_current_rotation_string(struct ScreenInfo* screen_info) {
	return gtk_combo_box_get_active_text (GTK_COMBO_BOX (screen_info->rotate_widget));
}

static gboolean
apply_config (struct DisplayInfo *info)
{
  int i;
  GdkDisplay *display;
  Display *xdisplay;
  GdkScreen *screen;
  gboolean changed;
  Window window;

  window = gdk_x11_get_default_root_xwindow();

  display = gdk_display_get_default ();
  xdisplay = gdk_x11_display_get_xdisplay (display);

  changed = FALSE;
  for (i = 0; i < info->n_screens; i++)
    {
      struct ScreenInfo *screen_info = &info->screens[i];
      Status status;
      GdkWindow *root_window;
      int new_res;
      Rotation new_rot;

      screen = gdk_display_get_screen (display, i);
      root_window = gdk_screen_get_root_window (screen);

      new_res  = get_current_resolution (screen_info);
      new_rot  = get_current_rotation (screen_info);
      
      g_debug("new_res = %d, new_rot = %d\n", new_res, new_rot);
      if (new_res  != screen_info->current_size ||
	  new_rot  != screen_info->current_rotation)
	{
	  changed = TRUE; 

          g_debug("CHANGING THE SETTINGS\n");
	  status = XRRSetScreenConfig (xdisplay, 
					      screen_info->config,
					      gdk_x11_drawable_get_xid (GDK_DRAWABLE (root_window)),
					      new_res,
					      new_rot,
					      GDK_CURRENT_TIME);
	}
    }

  update_display_info (info, display);

  return changed;
}

static int
revert_config (struct DisplayInfo *info)
{
  int i;
  GdkDisplay *display;
  Display *xdisplay;
  GdkScreen *screen;
  char *cmd;
  Window window;

  window = gdk_x11_get_default_root_xwindow();
  display = gdk_display_get_default ();
  xdisplay = gdk_x11_display_get_xdisplay (display);
  
  for (i = 0; i < info->n_screens; i++)
    {
      struct ScreenInfo *screen_info = &info->screens[i];
      Status status;
      GdkWindow *root_window;

      screen = gdk_display_get_screen (display, i);
      root_window = gdk_screen_get_root_window (screen);

      status = XRRSetScreenConfig (xdisplay, 
					  screen_info->config,
					  gdk_x11_drawable_get_xid (GDK_DRAWABLE (root_window)),
					  screen_info->old_size,
					  screen_info->old_rotation,
					  GDK_CURRENT_TIME);
    }

  update_display_info (info, display);

  /* Need to update the menus to the new settings */
  for (i = 0; i < info->n_screens; i++)
    {
      struct ScreenInfo *screen_info = &info->screens[i];
      
      generate_resolution_menu (screen_info);
    }

  if ((cmd = g_find_program_in_path ("gnome-screensaver-command")))
    g_free (cmd);
  else {
    /* xscreensaver should handle this itself, but does not currently so we hack
     * it.  Ignore failures in case xscreensaver is not installed */
    g_spawn_command_line_async ("xscreensaver-command -restart", NULL);
  }

  return 0;
}

static GtkWidget *
wrap_in_label (GtkWidget *child, char *text)
{
  GtkWidget *vbox, *hbox;
  GtkWidget *label;
  char *str;

  vbox = gtk_vbox_new (FALSE, 6);
  label = NULL;

  label = gtk_label_new (NULL);

  str = g_strdup_printf ("<b>%s</b>", text);
  gtk_label_set_markup (GTK_LABEL (label), str);
  g_free (str);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (vbox),
		      label,
		      FALSE, FALSE, 0);

  hbox = gtk_hbox_new (FALSE, 0);

  label = gtk_label_new ("    ");
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (hbox),
		      label,
		      FALSE, FALSE, 0);

  gtk_box_pack_start (GTK_BOX (hbox),
		      child,
		      TRUE, TRUE, 0);

  gtk_widget_show (hbox);
  
  gtk_box_pack_start (GTK_BOX (vbox),
		      hbox,
		      FALSE, FALSE, 0);

  gtk_widget_show (vbox);

  return vbox;
}

static void
resolution_changed_callback (GtkWidget *optionmenu, struct ScreenInfo *screen_info)
{
}

static void
generate_resolution_menu(struct ScreenInfo* screen_info)
{
  GtkComboBox *combo;
  GtkListStore* store;
  GtkTreeIter iter;
  int i, item, current_item;
  XRRScreenSize *sizes;
  char *str;
  SizeID current_size;
  Rotation rot;

  combo = GTK_COMBO_BOX (screen_info->resolution_widget);
  store = gtk_list_store_new(N_COLUMNS, G_TYPE_STRING, G_TYPE_INT);

  current_size = XRRConfigCurrentConfiguration (screen_info->config, &rot);

  gtk_combo_box_set_model (combo, GTK_TREE_MODEL (store));

  current_item = 0;
  item = 0;
  sizes = screen_info->sizes;
  for (i = 0; i < screen_info->n_sizes; i++)
    {
	  str = g_strdup_printf ("%dx%d", sizes[i].width, sizes[i].height);

	  if (i == current_size)
	    current_item = item;

	  gtk_list_store_append(store, &iter);
	  gtk_list_store_set(store, &iter,
			     COL_NAME, str,
			     COL_ID, i,
			     -1);

	  g_free (str);
	  item++;
    }
  
  gtk_combo_box_set_active (combo, current_item);
  if(screen_info->n_sizes < 2) {
	GTK_WIDGET_UNSET_FLAGS(combo, GTK_VISIBLE | GTK_SENSITIVE);
  } else {
	g_signal_connect (screen_info->resolution_widget, "changed",
		G_CALLBACK (resolution_changed_callback), screen_info);
	GTK_WIDGET_SET_FLAGS(combo, GTK_VISIBLE | GTK_SENSITIVE);
  }
  gtk_widget_show (screen_info->resolution_widget);
  g_object_unref (store);
}

static void
initialize_combo_layout (GtkCellLayout *layout) {
  GtkCellRenderer *cell = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start (layout, cell, TRUE);
  gtk_cell_layout_add_attribute (layout, cell, "text", COL_NAME);
}

static GtkWidget *
create_resolution_menu (struct ScreenInfo *screen_info)
{
  screen_info->resolution_widget = gtk_combo_box_new ();
  generate_resolution_menu (screen_info);

  initialize_combo_layout (GTK_CELL_LAYOUT (screen_info->resolution_widget));
  return screen_info->resolution_widget;
}

static GtkWidget *
create_rotate_menu (struct ScreenInfo *screen_info)
{
  GtkComboBox* combo = NULL;
  int i, item = 0, current_item = -1;

  screen_info->rotate_widget = gtk_combo_box_new_text ();
  combo = GTK_COMBO_BOX(screen_info->rotate_widget);

  for (i = 0; i < G_N_ELEMENTS (rotations); i++)
  {
    if ((screen_info->rotations & rotations[i].rotation) != 0)
    {
      gtk_combo_box_append_text (combo, _(rotations[i].name));
      if (screen_info->current_rotation == rotations[i].rotation) {
	current_item = item;
      }
      item++;
    }
  }

  /* it makes no sense to support only one selection element */
  gtk_widget_set_sensitive (screen_info->rotate_widget,
		  gtk_tree_model_iter_n_children (gtk_combo_box_get_model (combo), NULL) > 1);

  gtk_combo_box_set_active (combo, current_item);

  gtk_widget_show (screen_info->rotate_widget);
  return screen_info->rotate_widget;
}

static GtkWidget *
create_screen_widgets (struct ScreenInfo *screen_info, int nr, gboolean no_header)
{
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *option_menu;
  GtkWidget *ret;
  char *str;

  table = gtk_table_new (2, 2, FALSE);

  gtk_table_set_row_spacings ( GTK_TABLE (table), 6);
  gtk_table_set_col_spacings ( GTK_TABLE (table), 12);
  
  label = gtk_label_new_with_mnemonic (_("_Resolution:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table),
		    label,
		    0, 1,
		    0, 1,
		    GTK_FILL, 0,
		    0, 0);

  option_menu = create_resolution_menu (screen_info);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), option_menu);
  gtk_table_attach (GTK_TABLE (table),
		    option_menu,
		    1, 2,
		    0, 1,
		    GTK_FILL | GTK_EXPAND, 0,
		    0, 0);

  label = gtk_label_new_with_mnemonic (_("R_otation:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table),
		    label,
		    0, 1,
		    1, 2,
		    GTK_FILL, 0,
		    0, 0);
  gtk_widget_show (table);
  
  option_menu = create_rotate_menu (screen_info);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), option_menu);
  gtk_table_attach (GTK_TABLE (table),
		    option_menu,
		    1, 2,
		    1, 2,
		    GTK_FILL | GTK_EXPAND, 0,
		    0, 0);
  
  if (nr == 0)
    str = g_strdup (_("Default Settings"));
  else
    str = g_strdup_printf (_("Screen %d Settings\n"), nr+1);
  ret = wrap_in_label (table, str);
  g_free (str);
  return ret;
}


static GtkWidget *
create_dialog (struct DisplayInfo *info)
{
  GtkWidget *dialog;
  GtkWidget *screen_widget;
  int i;
  GtkWidget *vbox;
  
  dialog = gtk_dialog_new_with_buttons (_("Screen Settings"),
					NULL,
					GTK_DIALOG_NO_SEPARATOR,
 					"gtk-close",
 					GTK_RESPONSE_CLOSE,
					"gtk-apply",
					GTK_RESPONSE_APPLY,
					NULL);
					
  gtk_window_set_resizable(GTK_WINDOW (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);  
  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 2);
  
  vbox = gtk_vbox_new (FALSE, 18);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
                      vbox, FALSE, FALSE, 0);
  gtk_widget_show (vbox);
  
  for (i = 0; i < info->n_screens; i++)
    {
      screen_widget = create_screen_widgets (&info->screens[i], i, info->n_screens == 1);
      gtk_box_pack_start (GTK_BOX (vbox),
			  screen_widget, FALSE, FALSE, 0);
      gtk_widget_show (screen_widget);
    }
  return dialog;
}

struct TimeoutData {
  int time;
  GtkLabel *label;
  GtkDialog *dialog;
  gboolean timed_out;
};

static char *
timeout_string (int time)
{
  return g_strdup_printf (ngettext ("Testing the new settings. If you don't respond in %d second the previous settings will be restored.", "Testing the new settings. If you don't respond in %d seconds the previous settings will be restored.", time), time);
}

static gboolean
save_timeout_callback (gpointer _data)
{
  struct TimeoutData *data = _data;
  char *str;
  
  data->time--;

  if (data->time == 0)
    {
      gtk_dialog_response (data->dialog, GTK_RESPONSE_NO);
      data->timed_out = TRUE;
      return FALSE;
    }

  str = timeout_string (data->time);
  gtk_label_set_text (data->label, str);
  g_free (str);
  
  return TRUE;
}

static int
run_revert_dialog (struct DisplayInfo *info,
		   GtkWidget *parent)
{
  GtkWidget *dialog;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *label;
  GtkWidget *label_sec;
  GtkWidget *image;
  int res;
  struct TimeoutData timeout_data;
  guint timeout;
  char *str;

  dialog = gtk_dialog_new ();
  gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (parent));
  gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);
  gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 12);
  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Keep Resolution"));
  gtk_window_set_position(GTK_WINDOW(dialog),GTK_WIN_POS_CENTER_ALWAYS);
  
  label = gtk_label_new (NULL);
  str = g_strdup_printf ("<b>%s</b>", _("Do you want to keep this resolution?"));
  gtk_label_set_markup (GTK_LABEL (label), str);
  g_free (str);
  image = gtk_image_new_from_stock (GTK_STOCK_DIALOG_QUESTION, GTK_ICON_SIZE_DIALOG);
  gtk_misc_set_alignment (GTK_MISC (image), 0.5, 0.0);
  
  gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
  gtk_label_set_selectable (GTK_LABEL (label), TRUE);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  str = timeout_string (REVERT_COUNT);
  label_sec = gtk_label_new (str);
  g_free (str);
  gtk_label_set_line_wrap (GTK_LABEL (label_sec), TRUE);
  gtk_label_set_selectable (GTK_LABEL (label_sec), TRUE);
  gtk_misc_set_alignment (GTK_MISC (label_sec), 0.0, 0.5);

  hbox = gtk_hbox_new (FALSE, 6);
  vbox = gtk_vbox_new (FALSE, 6);

  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), label_sec, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, FALSE, FALSE, 0);
  gtk_dialog_add_buttons (GTK_DIALOG (dialog),_("Use _previous resolution"), GTK_RESPONSE_NO, _("_Keep resolution"), GTK_RESPONSE_YES, NULL);
  
  gtk_widget_show_all (hbox);

  timeout_data.time = REVERT_COUNT;
  timeout_data.label = GTK_LABEL (label_sec);
  timeout_data.dialog = GTK_DIALOG (dialog);
  timeout_data.timed_out = FALSE;
  
  timeout = g_timeout_add (1000, save_timeout_callback, &timeout_data);
  res = gtk_dialog_run (GTK_DIALOG (dialog));

  if (!timeout_data.timed_out)
    g_source_remove (timeout);

  gtk_widget_destroy (dialog);
  
  return res == GTK_RESPONSE_YES;
}

static void
save_to_gconf (struct DisplayInfo *info)
{
  GConfClient    *client;
  gboolean res;
  char *path, *key, *str;
  int i;
  
  client = gconf_client_get_default ();

  path = g_strdup ("/desktop/moblin/screen/default/");

  for (i = 0; i < info->n_screens; i++)
    {
      struct ScreenInfo *screen_info = &info->screens[i];
      int new_res;

      new_res = get_current_resolution (screen_info);

      key = g_strdup_printf ("%s%d/resolution", path, i);
      str = g_strdup_printf ("%dx%d",
			     screen_info->sizes[new_res].width,
			     screen_info->sizes[new_res].height);
      res = gconf_client_set_string  (client, key, str, NULL);
      g_free (str);
      g_free (key);

      str = get_current_rotation_string (screen_info);

      key = g_strdup_printf ("%s%d/rotation", path, i);
      res = gconf_client_set_string  (client, key, str, NULL);
      g_free (str);
      g_free (key);
    }

  g_free (path);
  g_object_unref (client);
}

static void
cb_dialog_response (GtkDialog *dialog, gint response_id, struct DisplayInfo *info)
{
  switch (response_id)
    {
    case GTK_RESPONSE_APPLY:
	if (apply_config (info))
	{
	  gtk_widget_hide(GTK_WIDGET(dialog));
	  if (!run_revert_dialog (info, GTK_WIDGET (dialog)))
	    {
	      gtk_widget_show(GTK_WIDGET(dialog));
	      revert_config (info);
	      return;
	    }
	}
	save_to_gconf (info);
    case GTK_RESPONSE_CLOSE:
    case GTK_RESPONSE_CANCEL:
    case GTK_RESPONSE_DELETE_EVENT:
	gtk_widget_destroy(GTK_WIDGET(dialog));
	gtk_main_quit();
	break;
    }
}

int main(int argc, char *argv[], char *env[])
{
  int major, minor;
  int event_base, error_base;
  GdkDisplay *display;
  GtkWidget *dialog;
  struct DisplayInfo *info;
  Display *xdisplay;

  gtk_init(&argc, &argv);

  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  get_default_resolutions();
  display = gdk_display_get_default ();
  xdisplay = gdk_x11_display_get_xdisplay (display);
  
  if (!XRRQueryExtension (xdisplay, &event_base, &error_base) ||
      XRRQueryVersion (xdisplay, &major, &minor) == 0)
    {
      GtkWidget *msg_dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
						   _("The X Server does not support the XRandR extension.  Runtime resolution changes to the display size are not available."));
      gtk_dialog_run (GTK_DIALOG (msg_dialog));
      gtk_widget_destroy (msg_dialog);
      return -1;
    }
  else if (major != 1 || minor < 1)
    {
      GtkWidget *msg_dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
						      _("The version of the XRandR extension is incompatible with this program. Runtime changes to the display size are not available."));
      gtk_dialog_run (GTK_DIALOG (msg_dialog));
      gtk_widget_destroy (msg_dialog);
      return -1;
    }
  
  info = read_display_info (display);
  dialog = create_dialog (info);

  g_signal_connect (G_OBJECT (dialog), "response", G_CALLBACK (cb_dialog_response), info);
  gtk_widget_show (dialog);

  gtk_main();
  return 0;
}
