/* network-proxy.c: network preferences capplet
 *
 * Copyright (C) 2002 Sun Microsystems Inc.
 *
 * Written by: Mark McLoughlin <mark@skynet.ie>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <glib/gprintf.h>
#include <glade/glade.h>
#include <hildon-cp-plugin/hildon-cp-plugin-interface.h>

#include <unistd.h>
#include <string.h>
#include <sys/vfs.h>
#include "capplet-util.h"
#include "gconf-property-editor.h"
#include <moblin.h>

#include <libgnome/libgnome.h>
#include <gconf/gconf-client.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <moko-finger-scroll.h>

#include <gdk/gdkx.h>
#include <X11/extensions/Xrandr.h>

enum ProxyMode
{
	PROXYMODE_NONE,
	PROXYMODE_MANUAL,
	PROXYMODE_AUTO
};

static GEnumValue proxytype_values[] = {
	{ PROXYMODE_NONE, "PROXYMODE_NONE", "none"},
	{ PROXYMODE_MANUAL, "PROXYMODE_MANUAL", "manual"},
	{ PROXYMODE_AUTO, "PROXYMODE_AUTO", "auto"},
	{ 0, NULL, NULL }
};

static GtkWidget *details_dialog = NULL;
static GSList *ignore_hosts = NULL;
static GtkTreeModel *model = NULL;

static GtkTreeModel *
create_listmodel(void)
{
	GtkListStore *store;

	store = gtk_list_store_new(1, G_TYPE_STRING);
	
	return GTK_TREE_MODEL(store);
}

static GtkTreeModel *
populate_listmodel(GtkListStore *store, GSList *list)
{
	GtkTreeIter iter;
	GSList *pointer;

	gtk_list_store_clear(store);

	pointer = list;
	while(pointer)
	{
		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter, 0, (char *) pointer->data, -1);
		pointer = g_slist_next(pointer);
	}

	return GTK_TREE_MODEL(store);
}

static GtkWidget *
config_treeview(GtkTreeView *tree, GtkTreeModel *model)
{
	GtkCellRenderer *renderer;

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(tree),
												-1, "Hosts", renderer,
												"text", 0, NULL);

	gtk_tree_view_set_model(GTK_TREE_VIEW(tree), model);

	return GTK_WIDGET(tree);
}

static void
cb_add_url (GtkButton *button, gpointer data)
{
	GladeXML *dialog = (GladeXML *) data;
	gchar *new_url = NULL;
	GConfClient *client;

	new_url = g_strdup(gtk_entry_get_text(GTK_ENTRY(WID("entry_url"))));
	if (strlen (new_url) == 0)
		return;
	ignore_hosts = g_slist_append(ignore_hosts, new_url);
	populate_listmodel(GTK_LIST_STORE(model), ignore_hosts);
	gtk_entry_set_text(GTK_ENTRY(WID("entry_url")), "");
		
	client = gconf_client_get_default ();
	gconf_client_set_list (client, (gchar *)NETPROXYKEY(IGNORE_HOSTS), GCONF_VALUE_STRING, ignore_hosts, NULL);
	g_object_unref (client);
}

static void
cb_remove_url (GtkButton *button, gpointer data)
{
	GladeXML *dialog = (GladeXML *) data;
	GtkTreeSelection *selection;
	GtkTreeIter       iter;
	GConfClient *client;

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(WID("treeview_ignore_host")));
	if (gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		gchar *url;
		GSList *pointer;

		gtk_tree_model_get (model, &iter, 0, &url, -1);

		pointer = ignore_hosts;
		while(pointer)
		{
			if(strcmp(url, (char *) pointer->data) == 0)
			{
				g_free (pointer->data);
				ignore_hosts = g_slist_delete_link(ignore_hosts, pointer);
				break;
			}
			pointer = g_slist_next(pointer);
		}

		g_free(url);
		populate_listmodel(GTK_LIST_STORE(model), ignore_hosts);
		
		client = gconf_client_get_default ();
		gconf_client_set_list(client, (gchar *)NETPROXYKEY(IGNORE_HOSTS), GCONF_VALUE_STRING, ignore_hosts, NULL);
		g_object_unref (client);
	}
}

static void
cb_dialog_response (GtkDialog *dialog, gint response_id)
{
	switch (response_id) {
	case GTK_RESPONSE_CLOSE:
	case GTK_RESPONSE_CANCEL:
	case GTK_RESPONSE_DELETE_EVENT:
		gtk_widget_destroy(GTK_WIDGET(dialog));
		/* reset globals for another run */
		details_dialog = NULL;
		if (ignore_hosts) {
		    g_slist_foreach (ignore_hosts, (GFunc) g_free, NULL);
		    g_slist_free (ignore_hosts);
		    ignore_hosts = NULL;
		}
		model = NULL;
		g_object_unref (dialog);
		gtk_main_quit();
		break;
	}
}

static void
cb_details_dialog_response (GtkDialog *dialog, gint response_id)
{
	switch (response_id) {
	case GTK_RESPONSE_CLOSE:
	case GTK_RESPONSE_CANCEL:
	case GTK_RESPONSE_DELETE_EVENT:
		gtk_widget_destroy (GTK_WIDGET (dialog));
		details_dialog = NULL;
		break;
	}
}

static void
cb_use_auth_toggled (GtkToggleButton *toggle,
		     GtkWidget *table)
{
	gtk_widget_set_sensitive (table, toggle->active);
}

static void
cb_http_details_button_clicked (GtkWidget *button,
			        GtkWidget *parent)
{
	GladeXML  *dialog;
	GtkWidget *widget;
	GConfPropertyEditor *peditor;

	if (details_dialog != NULL) {
		gtk_window_present (GTK_WINDOW (details_dialog));
		gtk_widget_grab_focus (details_dialog);
		return;
	}

	dialog = glade_xml_new (MA_GLADE_DIR "/network-proxy.glade",
				"details_dialog", NULL);

	details_dialog = widget = WID ("details_dialog");

	gtk_window_set_transient_for (GTK_WINDOW (widget), GTK_WINDOW (parent));

	gtk_label_set_use_markup (GTK_LABEL (GTK_BIN (WID ("use_auth_checkbutton"))->child), TRUE);

	g_signal_connect (G_OBJECT (WID ("use_auth_checkbutton")),
			  "toggled",
			  G_CALLBACK (cb_use_auth_toggled),
			  WID ("auth_table"));

	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_boolean (
			NULL, (gchar *)NETPROXYKEY(HTTP_USE_AUTH), WID ("use_auth_checkbutton"),
			NULL));
	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_string (
			NULL, (gchar *)NETPROXYKEY(HTTP_AUTH_USER), WID ("username_entry"),
			NULL));
	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_string (
			NULL, (gchar *)NETPROXYKEY(HTTP_AUTH_PASSWD), WID ("password_entry"), 
			NULL));

	g_signal_connect (widget, "response",
			  G_CALLBACK (cb_details_dialog_response), NULL);
	
	gtk_widget_show_all (widget);
}

static void
cb_use_same_proxy_checkbutton_clicked (GtkWidget *checkbutton,
					GladeXML *dialog)
{
	GConfClient *client;
	gboolean same_proxy;
	gchar *http_proxy;
	gint http_port;
	gchar *host;
	
	client = gconf_client_get_default ();
	same_proxy = gconf_client_get_bool (client, (gchar *)NETPROXYKEY(USE_SAME_PROXY), NULL);
	
	http_proxy = gconf_client_get_string (client, (gchar *)NETPROXYKEY(HTTP_PROXY_HOST), NULL);
	http_port = gconf_client_get_int (client, (gchar *)NETPROXYKEY(HTTP_PROXY_PORT), NULL);
	
	if (same_proxy) 
	{
		/* Save the old values */
		host = gconf_client_get_string (client, (gchar *)NETPROXYKEY(SECURE_PROXY_HOST), NULL);
		gconf_client_set_string (client, (gchar *)NETPROXYKEY(OLD_SECURE_PROXY_HOST), host, NULL);
		gconf_client_set_int (client, (gchar *)NETPROXYKEY(OLD_SECURE_PROXY_PORT), 
			gconf_client_get_int (client, (gchar *)NETPROXYKEY(SECURE_PROXY_PORT), NULL), NULL);
		g_free (host);
			
		host = gconf_client_get_string (client, (gchar *)NETPROXYKEY(FTP_PROXY_HOST), NULL);
		gconf_client_set_string (client, (gchar *)NETPROXYKEY(OLD_FTP_PROXY_HOST), host, NULL);
		gconf_client_set_int (client, (gchar *)NETPROXYKEY(OLD_FTP_PROXY_PORT), 
			gconf_client_get_int (client, (gchar *)NETPROXYKEY(FTP_PROXY_PORT), NULL), NULL);
		g_free (host);

		host = gconf_client_get_string (client, (gchar *)NETPROXYKEY(SOCKS_PROXY_HOST), NULL);
		gconf_client_set_string (client, (gchar *)NETPROXYKEY(OLD_SOCKS_PROXY_HOST), host, NULL);
		gconf_client_set_int (client, (gchar *)NETPROXYKEY(OLD_SOCKS_PROXY_PORT), 
			gconf_client_get_int (client, (gchar *)NETPROXYKEY(SOCKS_PROXY_PORT), NULL), NULL);
		g_free (host);

		/* Set the new values */
		gconf_client_set_string (client, (gchar *)NETPROXYKEY(SECURE_PROXY_HOST), http_proxy, NULL);
		gconf_client_set_int (client, (gchar *)NETPROXYKEY(SECURE_PROXY_PORT), http_port, NULL);
		
		gconf_client_set_string (client, (gchar *)NETPROXYKEY(FTP_PROXY_HOST), http_proxy, NULL);
		gconf_client_set_int (client, (gchar *)NETPROXYKEY(FTP_PROXY_PORT), http_port, NULL);
		
		gconf_client_set_string (client, (gchar *)NETPROXYKEY(SOCKS_PROXY_HOST), http_proxy, NULL);
		gconf_client_set_int (client, (gchar *)NETPROXYKEY(SOCKS_PROXY_PORT), http_port, NULL);
	}
	else
	{
		host = gconf_client_get_string (client, (gchar *)NETPROXYKEY(OLD_SECURE_PROXY_HOST), NULL);
		gconf_client_set_string (client, (gchar *)NETPROXYKEY(SECURE_PROXY_HOST), host, NULL);
		gconf_client_set_int (client, (gchar *)NETPROXYKEY(SECURE_PROXY_PORT), 
			gconf_client_get_int (client, (gchar *)NETPROXYKEY(OLD_SECURE_PROXY_PORT), NULL), NULL);
		g_free (host);
			
		host = gconf_client_get_string (client, (gchar *)NETPROXYKEY(OLD_FTP_PROXY_HOST), NULL);
		gconf_client_set_string (client, (gchar *)NETPROXYKEY(FTP_PROXY_HOST), host, NULL);
		gconf_client_set_int (client, (gchar *)NETPROXYKEY(FTP_PROXY_PORT), 
			gconf_client_get_int (client, (gchar *)NETPROXYKEY(OLD_FTP_PROXY_PORT), NULL), NULL);
		g_free (host);

		host = gconf_client_get_string (client, (gchar *)NETPROXYKEY(OLD_SOCKS_PROXY_HOST), NULL);
		gconf_client_set_string (client, (gchar *)NETPROXYKEY(SOCKS_PROXY_HOST), host, NULL);
		gconf_client_set_int (client, (gchar *)NETPROXYKEY(SOCKS_PROXY_PORT), 
			gconf_client_get_int (client, (gchar *)NETPROXYKEY(OLD_SOCKS_PROXY_PORT), NULL), NULL);
		g_free (host);
	}
	
	/* Set the proxy entries insensitive if we are using the same proxy for all */
	gtk_widget_set_sensitive (WID ("secure_host_entry"), !same_proxy);
	gtk_widget_set_sensitive (WID ("secure_port_entry"), !same_proxy);
	gtk_widget_set_sensitive (WID ("ftp_host_entry"), !same_proxy);
	gtk_widget_set_sensitive (WID ("ftp_port_entry"), !same_proxy);
	gtk_widget_set_sensitive (WID ("socks_host_entry"), !same_proxy);
	gtk_widget_set_sensitive (WID ("socks_port_entry"), !same_proxy);

	g_object_unref (client);
}

static void
cb_http_proxy_changed (GtkEditable *editable, GladeXML *dialog)
{
	GConfClient *client;
	gboolean same_proxy;
	gchar *http_proxy;
	gint http_port;

	client = gconf_client_get_default ();
	same_proxy = gconf_client_get_bool (client, (gchar *)NETPROXYKEY(USE_SAME_PROXY), NULL);

	if(same_proxy) 
	{
		http_proxy = gconf_client_get_string (client, (gchar *)NETPROXYKEY(HTTP_PROXY_HOST), NULL);
		http_port = gconf_client_get_int (client, (gchar *)NETPROXYKEY(HTTP_PROXY_PORT), NULL);

		/* Set the new values */
		gconf_client_set_string (client, (gchar *)NETPROXYKEY(SECURE_PROXY_HOST), http_proxy, NULL);
		gconf_client_set_int (client, (gchar *)NETPROXYKEY(SECURE_PROXY_PORT), http_port, NULL);
		
		gconf_client_set_string (client, (gchar *)NETPROXYKEY(FTP_PROXY_HOST), http_proxy, NULL);
		gconf_client_set_int (client, (gchar *)NETPROXYKEY(FTP_PROXY_PORT), http_port, NULL);
		
		gconf_client_set_string (client, (gchar *)NETPROXYKEY(SOCKS_PROXY_HOST), http_proxy, NULL);
		gconf_client_set_int (client, (gchar *)NETPROXYKEY(SOCKS_PROXY_PORT), http_port, NULL);
	}
}

static GConfValue *
extract_proxy_host (GConfPropertyEditor *peditor, const GConfValue *orig)
{
	char const  *entered_text = gconf_value_get_string (orig);
	GConfValue  *res = NULL;

	if (entered_text != NULL) {
		GnomeVFSURI *uri = gnome_vfs_uri_new (entered_text);
		if (uri != NULL) {
			char const  *host	  = gnome_vfs_uri_get_host_name (uri);
			if (host != NULL) {
				res = gconf_value_new (GCONF_VALUE_STRING);
				gconf_value_set_string (res, host);
			}
			gnome_vfs_uri_unref (uri);
		}
	}

	if (res != NULL)
		return res;
	return gconf_value_copy (orig);
}

static void
proxy_mode_radiobutton_clicked_cb (GtkWidget *widget,
				   GladeXML *dialog)
{
	GSList *mode_group;
	int mode;
	GConfClient *client;
	
	if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(widget)))
		return;
		
	mode_group = g_slist_copy (gtk_radio_button_get_group 
		(GTK_RADIO_BUTTON (WID ("none_radiobutton"))));
	mode_group = g_slist_reverse (mode_group);
	mode = g_slist_index (mode_group, widget);
	g_slist_free (mode_group);
	
	gtk_widget_set_sensitive (WID ("manual_box"), 
				  mode == PROXYMODE_MANUAL);
	gtk_widget_set_sensitive (WID ("same_proxy_checkbutton"), 
				  mode == PROXYMODE_MANUAL);
	gtk_widget_set_sensitive (WID ("auto_box"),
				  mode == PROXYMODE_AUTO);
	client = gconf_client_get_default ();
	gconf_client_set_bool (client, (gchar *)NETPROXYKEY(USE_PROXY),
				  mode == PROXYMODE_AUTO || mode == PROXYMODE_MANUAL, NULL);
	g_object_unref (client);
}

static void
connect_sensitivity_signals (GladeXML *dialog, GSList *mode_group)
{
	for (; mode_group != NULL; mode_group = mode_group->next)
	{
		g_signal_connect (G_OBJECT (mode_group->data), "clicked",
				  G_CALLBACK(proxy_mode_radiobutton_clicked_cb),
				  dialog);
	}
}

static void
setup_dialog (GladeXML *dialog1, GladeXML *dialog2)
{
	GConfPropertyEditor *peditor;
	GSList *mode_group;
	static GType mode_type = 0;
	GConfClient *client;
	gint port_value;
	gchar portstr[10];
	GladeXML *dialog = dialog2;

	if(!mode_type)
	{
		/* need this as static since the dialog may be open multiple */
		/* times in a single controlpanel process run */
		mode_type = g_enum_register_static ("NetworkPreferencesProxyType",
			            proxytype_values);
	}

	/* There's a bug in peditors that cause them to not initialize the entry
	 * correctly. */
	client = gconf_client_get_default ();

	/* Hackety hack */
	gtk_label_set_use_markup (GTK_LABEL (GTK_BIN (WID ("none_radiobutton"))->child), TRUE);
	gtk_label_set_use_markup (GTK_LABEL (GTK_BIN (WID ("manual_radiobutton"))->child), TRUE);
	gtk_label_set_use_markup (GTK_LABEL (GTK_BIN (WID ("auto_radiobutton"))->child), TRUE);
	
	/* Mode */
	mode_group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (WID ("none_radiobutton")));
	connect_sensitivity_signals (dialog, mode_group);

	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_select_radio_with_enum (NULL, 
			(gchar *)NETPROXYKEY(PROXY_MODE), mode_group, mode_type, 
			TRUE, NULL));
	
	/* Use same proxy for all protocols */
	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_boolean (NULL, 
			(gchar *)NETPROXYKEY(USE_SAME_PROXY), WID ("same_proxy_checkbutton"), NULL));
			
	g_signal_connect (G_OBJECT (WID ("same_proxy_checkbutton")),
			"toggled",
			G_CALLBACK (cb_use_same_proxy_checkbutton_clicked),
			dialog);
	
	/* Http */
	port_value = gconf_client_get_int (client, (gchar *)NETPROXYKEY(HTTP_PROXY_PORT), NULL);
	g_sprintf(portstr, "%d", (int)port_value);
	gtk_entry_set_text(GTK_ENTRY (WID ("http_port_entry")), portstr);
	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_string (
			NULL, (gchar *)NETPROXYKEY(HTTP_PROXY_HOST), WID ("http_host_entry"),
			"conv-from-widget-cb", extract_proxy_host,
			NULL));
	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_integer (
			NULL, (gchar *)NETPROXYKEY(HTTP_PROXY_PORT), WID ("http_port_entry"), 
			NULL));
	g_signal_connect (G_OBJECT (WID ("http_host_entry")),
			  "changed",
			  G_CALLBACK (cb_http_proxy_changed),
			  WID ("network_dialog"));
	g_signal_connect (G_OBJECT (WID ("http_port_entry")),
			  "changed",
			  G_CALLBACK (cb_http_proxy_changed),
			  WID ("network_dialog"));
	g_signal_connect (G_OBJECT (WID ("details_button")),
			  "clicked",
			  G_CALLBACK (cb_http_details_button_clicked),
			  WID ("network_dialog"));

	/* Secure */
 	port_value = gconf_client_get_int (client, (gchar *)NETPROXYKEY(SECURE_PROXY_PORT), NULL);
	g_sprintf(portstr, "%d", (int)port_value);
	gtk_entry_set_text(GTK_ENTRY (WID ("secure_port_entry")), portstr);
	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_string (
			NULL, (gchar *)NETPROXYKEY(SECURE_PROXY_HOST), WID ("secure_host_entry"),
			"conv-from-widget-cb", extract_proxy_host,
			NULL));
	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_integer (
			NULL, (gchar *)NETPROXYKEY(SECURE_PROXY_PORT), WID ("secure_port_entry"), 
			NULL));

	/* Ftp */
 	port_value = gconf_client_get_int (client, (gchar *)NETPROXYKEY(FTP_PROXY_PORT), NULL);
	g_sprintf(portstr, "%d", (int)port_value);
	gtk_entry_set_text(GTK_ENTRY (WID ("ftp_port_entry")), portstr);
	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_string (
			NULL, (gchar *)NETPROXYKEY(FTP_PROXY_HOST), WID ("ftp_host_entry"),
			"conv-from-widget-cb", extract_proxy_host,
			NULL));
	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_integer (
			NULL, (gchar *)NETPROXYKEY(FTP_PROXY_PORT), WID ("ftp_port_entry"), 
			NULL));

	/* Socks */
 	port_value = gconf_client_get_int (client, (gchar *)NETPROXYKEY(SOCKS_PROXY_PORT), NULL);
	g_sprintf(portstr, "%d", (int)port_value);
	gtk_entry_set_text(GTK_ENTRY (WID ("socks_port_entry")), portstr);
	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_string (
			NULL, (gchar *)NETPROXYKEY(SOCKS_PROXY_HOST), WID ("socks_host_entry"),
			"conv-from-widget-cb", extract_proxy_host,
			NULL));
	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_integer (
			NULL, (gchar *)NETPROXYKEY(SOCKS_PROXY_PORT), WID ("socks_port_entry"), 
			NULL));

	/* Set the proxy entries insensitive if we are using the same proxy for all */
	if (gconf_client_get_bool (client, (gchar *)NETPROXYKEY(USE_SAME_PROXY), NULL))
	{
		gtk_widget_set_sensitive (WID ("secure_host_entry"), FALSE);
		gtk_widget_set_sensitive (WID ("secure_port_entry"), FALSE);
		gtk_widget_set_sensitive (WID ("ftp_host_entry"), FALSE);
		gtk_widget_set_sensitive (WID ("ftp_port_entry"), FALSE);
		gtk_widget_set_sensitive (WID ("socks_host_entry"), FALSE);
		gtk_widget_set_sensitive (WID ("socks_port_entry"), FALSE);
	}

	/* Autoconfiguration */
	peditor = GCONF_PROPERTY_EDITOR (gconf_peditor_new_string (
			NULL, (gchar *)NETPROXYKEY(PROXY_AUTOCONFIG_URL), WID ("autoconfig_entry"),
			NULL));

	dialog = dialog1;

	g_signal_connect (WID ("network_dialog"), "response",
			  G_CALLBACK (cb_dialog_response), NULL);


	gtk_label_set_use_markup (GTK_LABEL (WID ("label_ignore_host")), TRUE);
	ignore_hosts = gconf_client_get_list(client, (gchar *)NETPROXYKEY(IGNORE_HOSTS), GCONF_VALUE_STRING, NULL);
	g_object_unref (client);

	model = create_listmodel();
	populate_listmodel(GTK_LIST_STORE(model), ignore_hosts);
	config_treeview(GTK_TREE_VIEW(WID("treeview_ignore_host")), model);

	g_signal_connect (WID ("button_add_url"), "clicked", 
						G_CALLBACK (cb_add_url), dialog);
	g_signal_connect (WID ("entry_url"), "activate", 
						G_CALLBACK (cb_add_url), dialog);
	g_signal_connect (WID ("button_remove_url"), "clicked", 
						G_CALLBACK (cb_remove_url), dialog);
}

int main(int argc, char *argv[], char *env[])
{
	GtkWidget   *widget, *pbox, *cbox, *moko;
	GladeXML    *dialog, *dialog1, *dialog2;
	GConfClient *client;
	GdkScreen *screen;
	GdkDisplay *display;
	int height, width;

	gtk_init(&argc, &argv);

	bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	client = gconf_client_get_default ();
	gconf_client_add_dir (client, "/system/gnome-vfs",
			      GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
	gconf_client_add_dir (client, "/system/http_proxy",
			      GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
	gconf_client_add_dir (client, "/system/proxy",
			      GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);


	dialog1 = glade_xml_new (MA_GLADE_DIR "/network-proxy.glade",
				"network_dialog", NULL);
	dialog2 = glade_xml_new (MA_GLADE_DIR "/network-proxy.glade",
				"vbox_child", NULL);

	dialog = dialog1;

	/* code to swap out scrolled window with moko finger scroll */
	moko = moko_finger_scroll_new();
	pbox = WID ("hbox2");
	cbox = WID ("treeview_ignore_host");
	gtk_widget_reparent(cbox, moko);
	cbox = WID("scrolledwindow1");
	gtk_container_remove(GTK_CONTAINER(pbox), cbox);
	gtk_widget_destroy(cbox);
	gtk_box_pack_start(GTK_BOX(pbox), moko, TRUE, TRUE, 0);
	gtk_widget_show (moko);

	/* code to add a moko around the contents of the Proxy Config tab */
	moko = moko_finger_scroll_new();
	pbox = WID ("hbox_parent");
	cbox = WID ("vbox_child");
	gtk_container_remove(GTK_CONTAINER(pbox), cbox);
	gtk_widget_destroy(cbox);

	dialog = dialog2;
	cbox = WID ("vbox_child");

	moko_finger_scroll_add_with_viewport(MOKO_FINGER_SCROLL(moko), cbox);
	gtk_box_pack_start(GTK_BOX(pbox), moko, TRUE, TRUE, 0);
	gtk_widget_show (moko);

	setup_dialog (dialog1, dialog2);

	display = gdk_display_get_default ();
	screen = gdk_display_get_screen (display, 0);
	width = gdk_screen_get_width (screen);
	height = gdk_screen_get_height (screen);

	dialog = dialog1;
	widget = WID ("network_dialog");
	gtk_widget_set_size_request(widget, (width*7)/8, (height*7)/8);
	gtk_widget_show_all (widget);

	g_object_unref (client);
	gtk_main();
	return 0;
}
