/***************************************************************************
 *   Copyright (C) 2007 by Kevron Rees                                     *
 *   tripzero@nextabyte.com                                                *
 *
 *   Todd Brandt <todd.e.brandt@intel.com>
 *   Danny Varner <danny.varner@intel.com>
 *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 **************************************************************************/

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

#include <stdio.h>
#include <signal.h>
#include <termios.h>
#include <stdlib.h>
#include <sys/time.h>
#include <glib.h>
#include <fstream>
#include <iostream>
#include <linux/input.h>

#ifdef LOGDEBUG
extern FILE* logfp;
#define PRINTF(args...) if(logfp) fprintf(logfp, args)
#else
#define PRINTF(args...)
#endif

using namespace std;

#define FLAG_SWAPX		0x1
#define TST_SWAPX(x)		((x)&FLAG_SWAPX)
#define SET_SWAPX(x)		(x) |= FLAG_SWAPX
#define CLR_SWAPX(x)		(x) &= ~FLAG_SWAPX
#define FLAG_SWAPY		0x2
#define TST_SWAPY(x)		((x)&FLAG_SWAPY)
#define SET_SWAPY(x)		(x) |= FLAG_SWAPY
#define CLR_SWAPY(x)		(x) &= ~FLAG_SWAPY
#define FLAG_ROTCW		0x4
#define TST_ROTCW(x)		((x)&FLAG_ROTCW)
#define SET_ROTCW(x)		(x) |= FLAG_ROTCW
#define CLR_ROTCW(x)		(x) &= ~FLAG_ROTCW
#define FLAG_ROTCCW		0x8
#define TST_ROTCCW(x)		((x)&FLAG_ROTCCW)
#define SET_ROTCCW(x)		(x) |= FLAG_ROTCCW
#define CLR_ROTCCW(x)		(x) &= ~FLAG_ROTCCW
#define CLR_ROT(x)		(x) &= ~(FLAG_ROTCW | FLAG_ROTCCW)

#define DEVICE_FILE "/dev/input/touchscreen"
#define SESSION_FILE "/etc/event.d/session"

static int evfd = -1;

static gboolean enable_touch_capture();
static gboolean disable_touch_capture();
extern "C" void capture_touch_reply(gint, gint);

string xorgsection[16] = {
	"\n",
	"Section \"InputDevice\"\n",
	"\tIdentifier \"touchscreen\"\n",
	"\tDriver  \"evtouch\"\n",
	"\tOption  \"Device\" \"/dev/input/touchscreen\"\n",
	"\tOption  \"SendCoreEvents\"\n",
	"\tOption  \"ReportingMode\" \"Raw\"\n",
	"\tOption  \"Emulate3Buttons\"\n",
	"\tOption  \"Emulate3Timeout\" \"50\"\n",
	"\tOption  \"MinX\"  \"0\"\n",
	"\tOption  \"MinY\"  \"0\"\n",
	"\tOption  \"MaxX\"  \"4096\"\n",
	"\tOption  \"MaxY\"  \"4096\"\n",
	"\tOption  \"TapTimer\" \"2\"\n",
	"\tOption  \"LongTouchTimer\" \"4\"\n",
	"EndSection\n"
};

static char*
getxorgconf()
{
FILE *fp, i;
char buf[1000];
char *name = NULL, *tmp;

	if((fp = fopen(SESSION_FILE, "r")) == NULL)
	{
	    PRINTF("%s failed to open\n", SESSION_FILE);
	    return strdup("/etc/X11/xorg.conf");
	}

	while(fgets(buf, 1000, fp)) {
	    if(strstr(buf, "startx")) {
		for(tmp = strtok(buf, " \t\n\"\'"); tmp != NULL;
			tmp = strtok(NULL, " \t\n\"\'"))
		{
		    if(!strcmp(tmp, "-config"))
		    {
			tmp = strtok(NULL, " \t\n\"\'");
			name = tmp;
			break;
		    }
		}
		if(!name)
		{
		    fclose(fp);
		    return strdup("/etc/X11/xorg.conf");
		}
		strcpy(buf, "/etc/X11/");
		strcat(buf, name);
		fclose(fp);
		return strdup(buf);
	    }
	}
	return strdup("/etc/X11/xorg.conf");
}

static gboolean
addxorgsection(char *filename)
{
FILE *fp;
char buf[1000];
int i;
int start = -1;
int touchscreen = -1;

	if((fp = fopen(filename, "r")) == NULL)
	    return FALSE;

	for(i = 0; fgets(buf, 1000, fp); i++) {
	    if(strstr(buf, "Section")&&
		strstr(buf, "InputDevice"))
	    {
		start = i;
	    } else if(strstr(buf, "Identifier")&&
		strstr(buf, "touchscreen")&&
		(start >= 0)&&(i > start)) 
	    {
		touchscreen = i;
	    } else if(strstr(buf, "EndSection")&&
		(start >= 0)&&(i > start)&&
		(touchscreen >= 0)&&(i > touchscreen))
	    {
		fclose(fp);
		return FALSE;
	    }
	}
	fclose(fp);

	if((fp = fopen(filename, "a")) == NULL)
	    return FALSE;

	for(i = 0; i < 16; i++)
		fputs(xorgsection[i].c_str(), fp);

	fclose(fp);
	return TRUE;
}

static gboolean
editxorgconf(gint x_min, gint x_max, gint y_min, gint y_max, guint flags)
{
	fstream xorg;
	ofstream newxorg;
	int inside = 0;
	int ident = 0;
	int pos=0;
	char *name, cmd[1000];

	name = getxorgconf();
	PRINTF("xorg file = %s\n", name);
	addxorgsection(name);
	string line="";
	xorg.open(name);
	if(!xorg)
	{
		PRINTF("%s failed to open\n", name);
		free(name);
		return FALSE;
	}
	newxorg.open("newxorg.conf");
	if(!newxorg)
	{
		PRINTF("newxorg.conf failed to open!\n");;
		free(name);
		return FALSE;
	}
	do
	{
		getline(xorg,line);

		if((line.find("Section",0)!=string::npos)&&
		   (line.find("InputDevice",0)!=string::npos)) {
		    inside = 1;
		    newxorg<<line<<endl;
		    continue;
		} else if(inside&&(line.find("Identifier",0)!=string::npos)&&
	                  (line.find("touchscreen",0)!=string::npos)) {
		    ident = 1;
		    newxorg<<line<<endl;
		    continue;
		} else if(line.find("EndSection",0)!=string::npos) {
		    inside = 0;
		    ident = 0;
		    newxorg<<line<<endl;
		    continue;
		}

		if((!inside)||(!ident))
		{
		    newxorg<<line<<endl;
		    continue;
		}

		pos=line.find("Option",0);
		if(pos==string::npos)
		{
		    newxorg<<line<<endl;
		    continue;
		}

		pos=line.find("MinX",0);
		if(pos!=string::npos)
		{
		    newxorg<<"\tOption\t\"MinX\"\t\""<<x_min<<"\""<<endl;
		    continue;
		}

		pos=line.find("MinY",0);
		if(pos!=string::npos)
		{
		    newxorg<<"\tOption\t\"MinY\"\t\""<<y_min<<"\""<<endl;
		    continue;
		}

		pos=line.find("MaxX",0);
		if(pos!=string::npos)
		{
		    newxorg<<"\tOption\t\"MaxX\"\t\""<<x_max<<"\""<<endl;
		    continue;
		}

		pos=line.find("MaxY",0);
		if(pos!=string::npos)
		{
		    newxorg<<"\tOption\t\"MaxY\"\t\""<<y_max<<"\""<<endl;
		    if(TST_ROTCCW(flags))
			newxorg<<"\tOption\t\"Rotate\"\t\"ccw\""<<endl;
		    else if(TST_ROTCW(flags))
			newxorg<<"\tOption\t\"Rotate\"\t\"cw\""<<endl;
		    if(TST_SWAPX(flags))
			newxorg<<"\tOption\t\"SwapX\"\t\"1\""<<endl;
		    if(TST_SWAPY(flags))
			newxorg<<"\tOption\t\"SwapY\"\t\"1\""<<endl;
		    continue;
		}

		pos=line.find("Rotate",0);
		if(pos!=string::npos) continue;

		pos=line.find("SwapX",0);
		if(pos!=string::npos) continue;

		pos=line.find("SwapY",0);
		if(pos!=string::npos) continue;

		newxorg<<line<<endl;

	}while (!xorg.eof());

	newxorg.close();
	sprintf(cmd, "cp %s %s.backup", name, name);
	system(cmd);
	sprintf(cmd, "mv newxorg.conf %s", name);
	system(cmd);
	free(name);
	return TRUE;
}

extern "C" gboolean
read_calibration(gint *x_min, gint *x_max, gint *y_min, gint *y_max, guint *flags)
{
	fstream xorg;
	int pos=0, swap;
	char *name, rotation[100];
	char found[7] = {0, 0, 0, 0, 0, 0, 0};
	int inside = 0;
	int ident = 0;
	string line="";

	*flags = 0;
	name = getxorgconf();
	PRINTF("xorg file = %s\n", name);
	xorg.open(name);
	if(!xorg)
	{
		PRINTF("%s failed to open\n", name);
		free(name);
		return FALSE;
	}
	do
	{
	    getline(xorg,line);

	    if((line.find("Section",0)!=string::npos)&&
	       (line.find("InputDevice",0)!=string::npos)) {
		inside = 1;
		continue;
	    } else if(inside&&(line.find("Identifier",0)!=string::npos)&&
	       (line.find("touchscreen",0)!=string::npos)) {
		ident = 1;
		continue;
	    } else if(line.find("EndSection",0)!=string::npos) {
		inside = 0;
		ident = 0;
		continue;
	    }

	    if((!inside)||(!ident)) continue;

	    pos=line.find("Option",0);
	    if(pos==string::npos)
		continue;

	    pos=line.find("MinX",0);
	    if(!found[0]&&(pos!=string::npos))
	    {
		found[0] = 1;
		if(!sscanf(line.c_str(), 
		    "\tOption\t\"MinX\"\t\"%d\"", x_min))
		    *x_min = NULL;
	    }

	    pos=line.find("MinY",0);
	    if(!found[1]&&(pos!=string::npos))
	    {
		found[1] = 1;
		if(!sscanf(line.c_str(), 
		    "\tOption\t\"MinY\"\t\"%d\"", y_min))
		    *y_min = NULL;
	    }

	    pos=line.find("MaxX",0);
	    if(!found[2]&&(pos!=string::npos))
	    {
		found[2] = 1;
		if(!sscanf(line.c_str(), 
		    "\tOption\t\"MaxX\"\t\"%d\"", x_max))
		    *x_max = NULL;
	    }

	    pos=line.find("MaxY",0);
	    if(!found[3]&&(pos!=string::npos))
	    {
		found[3] = 1;
		if(!sscanf(line.c_str(), 
		    "\tOption\t\"MaxY\"\t\"%d\"", y_max))
		    *y_max = NULL;
	    }

	    pos=line.find("SwapX",0);
	    if(!found[4]&&(pos!=string::npos))
	    {
		found[4] = 1;
		if(sscanf(line.c_str(), 
		    "\tOption\t\"SwapX\"\t\"%d\"", 
		    &swap) == 1)
		{
		    if(swap)
			SET_SWAPX(*flags);
		    else
			CLR_SWAPX(*flags);
		}
	    }

	    pos=line.find("SwapY",0);
	    if(!found[5]&&(pos!=string::npos))
	    {
		found[5] = 1;
		if(sscanf(line.c_str(), 
		    "\tOption\t\"SwapY\"\t\"%d\"",
		    &swap) == 1)
		{
		    if(swap)
			SET_SWAPY(*flags);
		    else
			CLR_SWAPY(*flags);
		}
	    }

	    pos=line.find("\"Rotate\"",0);
	    if(!found[6]&&(pos!=string::npos))
	    {
		found[6] = 1;
		if(sscanf(line.c_str(), 
			"\tOption\t\"Rotate\"\t\"%s\"", 
			rotation) == 1)
		{
		    if(!strncasecmp(rotation, "ccw", 3)) {
			SET_ROTCCW(*flags);
			CLR_ROTCW(*flags);
		    } else if(!strncasecmp(rotation, "cw", 2)) {
			SET_ROTCW(*flags);
			CLR_ROTCCW(*flags);
		    } else {
			CLR_ROT(*flags);
		    }
		}
	    }
	}while (!xorg.eof());

	free(name);
	return TRUE;
}

extern "C" gboolean
apply_calibration(gint x_min, gint x_max, gint y_min, gint y_max, guint flags)
{
	PRINTF("apply_calibration: xmin=%d, xmax=%d, ymin=%d, ymax=%d, flags=%x\n",
		x_min, x_max, y_min, y_max, flags);
	return editxorgconf(x_min, x_max, y_min, y_max, flags);
}

/* these are reset on each capturetouch */
/* we need all three to capture a point */
static int xcap = -1;
static int ycap = -1;
static int touch = 0;

static gboolean
get_events()
{
	struct input_event ev;

	if (touch&&(xcap != -1)&&(ycap != -1))
		return TRUE;

	/* read till sync event */
	while (read(evfd, &ev, sizeof(ev)) != -1) {
		switch (ev.type) {
		case EV_ABS:
			/* grab x and y values whenever */
			switch (ev.code) {
			case ABS_X:
				if (xcap == -1) 
				    xcap = ev.value;
				break;
			case ABS_Y:
				if (ycap == -1) 
				    ycap = ev.value;
				break;
			case ABS_Z:
				if(xcap == -1) 
				    xcap = ev.value;
				break;
			case ABS_RX:
				if(ycap == -1) 
				    ycap = ev.value;
				break;
			default:
				break;
			}
			break;
		case EV_KEY:
			/* any touch event counts */
			switch (ev.code) {
			case BTN_LEFT:
			case BTN_TOUCH:
				touch = 1;
			default:
			       break;
			}
			break;
		case EV_SYN:
			/* SYN_REPORT is where we tally up */
			/* the data captured thusfar */
			switch(ev.code) {
			case SYN_REPORT:
				/* only accept x, y if they come */
				/* together in a single packet */
				if(((xcap == -1)&&(ycap != -1))||
				   ((xcap != -1)&&(ycap == -1)))
				{
					xcap = -1;
					ycap = -1;
				}
				/* if all three data values are */
				/* ready, we can leave now */
				if (touch&&(xcap != -1)&&(ycap != -1))
					return TRUE;
				break;
			default:
				break;
			}
			break;
		default:
			break;
		}
	}
	return FALSE;
}

static void 
sig_handler(int num)
{
	static int is_busy = 0;

	if(num == SIGALRM) {
		if(disable_touch_capture())
			PRINTF("CaptureTouch Timeout\n");
		return;
	}
	if((num != SIGIO)||is_busy||(evfd == -1)) 
		return;

	is_busy = 1;
	/* we wait as long as it takes to */
	/* capture the touch data */
	if(get_events())
	{
		disable_touch_capture();
		capture_touch_reply(xcap, ycap);
	}
	is_busy = 0;
}

extern "C" gboolean
capture_touch()
{
	if(!enable_touch_capture()) return FALSE;
	PRINTF("CaptureTouch Started\n");
	return TRUE;
}

extern "C" gboolean
cancel_touch()
{
	if(disable_touch_capture())
		PRINTF("CaptureTouch Cancelled\n");
	return TRUE;
}

static gboolean
enable_touch_capture()
{
	/* if already enabled, leave */
	if(evfd != -1) return TRUE;

	/* reset the data values */
	xcap = -1;
	ycap = -1;
	touch = 0;

	/* open up the touchscreen driver */
	evfd = open(DEVICE_FILE, O_RDONLY | O_NONBLOCK);
	if (evfd == -1) {
		PRINTF("Cannot open device file!\n");
		return FALSE;
	}

	/* setup our handler for notification */
	signal(SIGIO, sig_handler);
	fcntl(evfd, F_SETOWN, getpid());
	fcntl(evfd, F_SETFL, fcntl(evfd, F_GETFL) | FASYNC);

	signal(SIGALRM, sig_handler);
	alarm(20);
	return TRUE;
}

static gboolean
disable_touch_capture()
{
	/* if already disabled, leave */
	if(evfd == -1) return FALSE;

	alarm(0);
	signal(SIGIO, SIG_DFL);
	signal(SIGALRM, SIG_DFL);
	close(evfd);
	evfd = -1;

	return TRUE;
}
