
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <termios.h>
#include <stdlib.h>
#include <sys/time.h>
#include <moblin-system-daemon/moblin-system-client.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus-glib-lowlevel.h>

#define FLAG_SWAPX              0x1
#define TST_SWAPX(x)            ((x)&FLAG_SWAPX)
#define FLAG_SWAPY              0x2
#define TST_SWAPY(x)            ((x)&FLAG_SWAPY)
#define FLAG_ROTCW              0x4
#define TST_ROTCW(x)            ((x)&FLAG_ROTCW)
#define FLAG_ROTCCW             0x8
#define TST_ROTCCW(x)           ((x)&FLAG_ROTCCW)

static DBusHandlerResult dbus_filter_callback(DBusConnection*, DBusMessage*, void*);
static DBusGConnection *connection = NULL;
static DBusGProxy *proxy = NULL;

gboolean 
activate_system_daemon (void)
{
    GError *error = NULL;

    connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
    if (connection == NULL)
    {
	g_warning("Failed to get dbus connection\n");
	g_error_free (error);
	return FALSE;
    }
    proxy = dbus_g_proxy_new_for_name (connection,
                                     "org.moblin.SystemDaemon",
                                     "/org/moblin/SystemDaemon",
                                     "org.moblin.SystemDaemon");
    if (proxy == NULL)
    {
	g_warning("Failed to establish DBUS connection\n");
	return FALSE;
    }
    return TRUE;
}

static DBusHandlerResult
dbus_filter_callback (DBusConnection * connection, DBusMessage * message, void *user_data)
{
GError *gerror = NULL;
DBusError error;
gint x, y;

	if (dbus_message_is_signal (message, 
		"org.moblin.SystemDaemon", "TouchScreenData"))
	{
		dbus_error_init(&error);
		if (dbus_message_get_args (message, &error,
			DBUS_TYPE_INT32, &x, DBUS_TYPE_INT32, &y,
			DBUS_TYPE_INVALID)) {
			printf("Touch Captured: x=%d, y=%d\n", x, y);
			if(!org_moblin_SystemDaemon_cancel_touch(proxy, &gerror))
			{
				g_warning ("cancel_touch error: %s", gerror->message);
				g_error_free (gerror);
			}
			exit(0);
		} else if(dbus_error_is_set(&error)) {
			dbus_error_free(&error);
		}
	}

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static void
sig_handler(int num)
{
GError *gerror = NULL;

	if(num == SIGINT)
	{
		printf("Touch Captured Cancelled\n");
		if(!org_moblin_SystemDaemon_cancel_touch(proxy, &gerror))
		{
			g_warning ("cancel_touch error: %s", gerror->message);
			g_error_free (gerror);
		}
		exit(0);
	}
}

int main(int argc, char *argv[])
{
	GError *gerror = NULL;
	DBusError error;
	DBusConnection *cnct;
	gchar *tz = NULL;
	gint year, month, day, hour, minute, second;
	gint xmin, xmax, ymin, ymax;
	guint flags;

	if(argc < 2)
	{
		printf("Usage: %s <cmd> <arg>\n", argv[0]);
		printf("Commands:\n");
		printf("\tget_time\n");
		printf("\tget_timezone\n");
		printf("\tset_time MM/DD/YYYY_hh:mm:ss\n");
		printf("\tcapture_touch\n");
		printf("\tread_calibration\n");
		return 0;
	}

	g_type_init();

	if(!activate_system_daemon()) exit(-1);

	cnct = dbus_g_connection_get_connection(connection);
	if (!dbus_connection_add_filter (cnct, dbus_filter_callback, NULL, NULL)) {
		g_warning("dbus_connection_add_filter failed\n");
		return -1;
	}
	dbus_error_init(&error);
	dbus_bus_add_match (cnct,
			"type='signal',"
			"interface='org.moblin.SystemDaemon',"
			"sender='org.moblin.SystemDaemon',"
			"path='/org/moblin/SystemDaemon'", &error);
	if(dbus_error_is_set(&error)) 
	{
		g_warning("dbus_bus_add_match failed\n");
		dbus_error_free(&error);
		return -1;
	}

	if(!strcmp(argv[1], "get_time")) {
		if(!org_moblin_SystemDaemon_get_time(proxy,
			&year, &month, &day, &hour, &minute, &second, &gerror))
		{
		    g_warning ("get_time error: %s", gerror->message);
		    g_error_free (gerror);
		    return -1;
		}
		month++;
		printf("Date&Time = %02d/%02d/%04d_%02d:%02d:%02d\n",
			month, day, year, hour, minute, second);
	} else if(!strcmp(argv[1], "get_timezone")) {
		if(!org_moblin_SystemDaemon_get_timezone(proxy, &tz, &gerror))
		{
		    g_warning ("get_timezone error: %s", gerror->message);
		    g_error_free (gerror);
		    return -1;
		}
		printf("Timezone = %s\n", tz);
	} else if(!strcmp(argv[1], "set_time")) {
		if((argc < 3)||(sscanf(argv[2], "%02d/%02d/%04d_%02d:%02d:%02d",
			&month, &day, &year, &hour, &minute, &second) != 6))
		{
			fprintf(stderr, "Usage: %s set_time MM/DD/YYYY_hh:mm:ss\n",
				argv[0]);
			return -1;
		}
		if(month > 0) month--;
		if(!org_moblin_SystemDaemon_set_time(proxy, year, month, day, 
			hour, minute, second, &gerror))
		{
		    g_warning ("set_time error: %s", gerror->message);
		    g_error_free (gerror);
		    return -1;
		}
	} else if(!strcmp(argv[1], "capture_touch")) {
		signal(SIGINT, sig_handler);
		if(!org_moblin_SystemDaemon_capture_touch(proxy, &gerror))
		{
		    g_warning ("capture_touch error: %s", gerror->message);
		    g_error_free (gerror);
		    return -1;
		}
		while(1);
	} else if(!strcmp(argv[1], "read_calibration")) {
		if(!org_moblin_SystemDaemon_read_calibration(proxy, 
			&xmin, &xmax, &ymin, &ymax, &flags, &gerror))
		{
		    g_warning ("read_calibration error: %s", gerror->message);
		    g_error_free (gerror);
		    return -1;
		}
		printf("XORG CONF: MinX=%d, MaxX=%d, MinY=%d, MaxY=%d\n",
			xmin, xmax, ymin, ymax);
		printf("           Rotate=%s, SwapX=%d, SwapY=%d\n",
			TST_ROTCW(flags)?"CW":TST_ROTCCW(flags)?"CCW":"NORMAL",
			TST_SWAPX(flags)?1:0, TST_SWAPY(flags)?1:0);
	}

	return 0;
}
