/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*-
 *
 * GNOME Power Manager Brightness Applet
 * Copyright (C) 2006 Benjamin Canou <bookeldor@gmail.com>
 * Copyright (C) 2007 Richard Hughes <richard@hughsie.com>
 *
 * Licensed under the GNU General Public License Version 2
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtk/gtkbox.h>
#include <gdk/gdkkeysyms.h>
#include <glib-object.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <dbus/dbus-glib.h>
#include <libhildondesktop/hildon-status-bar-item.h>

#define GPM_TYPE_BRIGHTNESS_APPLET		(gpm_brightness_applet_get_type ())
#define GPM_BRIGHTNESS_APPLET(o)		(G_TYPE_CHECK_INSTANCE_CAST ((o), GPM_TYPE_BRIGHTNESS_APPLET, GpmBrightnessApplet))
#define GPM_BRIGHTNESS_APPLET_CLASS(k)		(G_TYPE_CHECK_CLASS_CAST((k), GPM_TYPE_BRIGHTNESS_APPLET, GpmBrightnessAppletClass))
#define GPM_IS_BRIGHTNESS_APPLET(o)		(G_TYPE_CHECK_INSTANCE_TYPE ((o), GPM_TYPE_BRIGHTNESS_APPLET))
#define GPM_IS_BRIGHTNESS_APPLET_CLASS(k)	(G_TYPE_CHECK_CLASS_TYPE ((k), GPM_TYPE_BRIGHTNESS_APPLET))
#define GPM_BRIGHTNESS_APPLET_GET_CLASS(o)	(G_TYPE_INSTANCE_GET_CLASS ((o), GPM_TYPE_BRIGHTNESS_APPLET, GpmBrightnessAppletClass))

#define HAL_DBUS_SERVICE                        "org.freedesktop.Hal"
#define HAL_DBUS_PATH_MANAGER                   "/org/freedesktop/Hal/Manager"
#define HAL_DBUS_INTERFACE_MANAGER              "org.freedesktop.Hal.Manager"
#define HAL_DBUS_INTERFACE_DEVICE               "org.freedesktop.Hal.Device"
#define HAL_DBUS_PATH_LAPTOP_PANEL              "/org/freedesktop/Hal/Device/LaptopPanel"
#define HAL_DBUS_INTERFACE_LAPTOP_PANEL         "org.freedesktop.Hal.Device.LaptopPanel"
#define HAL_DBUS_INTERFACE_KBD_BACKLIGHT        "org.freedesktop.Hal.Device.KeyboardBacklight"
#define HAL_DBUS_INTERFACE_LIGHT_SENSOR         "org.freedesktop.Hal.Device.LightSensor"

const int bval[101] = {
	10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
	10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
	30, 30, 30, 30, 30, 30, 30, 30, 30, 30,
	40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
	50, 50, 50, 50, 50, 50, 50, 50, 50, 50,
	60, 60, 60, 60, 60, 60, 60, 60, 60, 60,
	70, 70, 70, 70, 70, 70, 70, 70, 70, 70,
	80, 80, 80, 80, 80, 80, 80, 80, 80, 80,
	80, 80, 80, 80, 80, 80, 80, 80, 80, 80,
	100, 100, 100, 100, 100, 100, 100, 100, 100, 100,
	100
};

typedef struct{
	GtkEventBox parent;
	/* applet state */
	gboolean call_worked; /* g-p-m refusing action */
	gboolean popped; /* the popup is shown */
	/* the popup and its widgets */
	GtkWidget *popup, *slider, *btn_plus, *btn_minus;
	/* the icon and a cache for size*/
	GdkPixbuf *icon;
	gint icon_width, icon_height;
        DBusGProxy *proxy;
        DBusGConnection *connection;
	gchar *udi;
	guint level;
	/* a cache for panel size */
	gint size;
} GpmBrightnessApplet;

typedef struct{
	GtkEventBoxClass	parent_class;
} GpmBrightnessAppletClass;

typedef struct _MoblinBrightnessApplet {
  HildonStatusBarItem* item;
  GtkWidget* button;
  GpmBrightnessApplet* applet;
} MoblinBrightnessApplet;

GType                gpm_brightness_applet_get_type  (void);


static void      gpm_brightness_applet_class_init (GpmBrightnessAppletClass *klass);
static void      gpm_brightness_applet_init       (GpmBrightnessApplet *applet);

G_DEFINE_TYPE (GpmBrightnessApplet, gpm_brightness_applet, GTK_TYPE_EVENT_BOX)

static void      gpm_applet_get_icon              (GpmBrightnessApplet *applet);
static void      gpm_applet_check_size            (GpmBrightnessApplet *applet);
static gboolean  gpm_applet_draw_cb               (GpmBrightnessApplet *applet);
static void      gpm_applet_theme_change_cb (GtkIconTheme *icon_theme, gpointer data);
static void      gpm_applet_stop_scroll_events_cb (GtkWidget *widget, GdkEvent  *event);
static gboolean  gpm_applet_destroy_popup_cb      (GpmBrightnessApplet *applet);
static void      gpm_applet_update_tooltip        (GpmBrightnessApplet *applet);
static void      gpm_applet_update_popup_level    (GpmBrightnessApplet *applet);
static gboolean  gpm_applet_plus_cb               (GtkWidget *w, GpmBrightnessApplet *applet);
static gboolean  gpm_applet_minus_cb              (GtkWidget *w, GpmBrightnessApplet *applet);
static gboolean  gpm_applet_key_press_cb          (GpmBrightnessApplet *applet, GdkEventKey *event);
static gboolean  gpm_applet_scroll_cb             (GpmBrightnessApplet *applet, GdkEventScroll *event);
static gboolean  gpm_applet_slide_cb              (GtkWidget *w, GpmBrightnessApplet *applet);
static void      gpm_applet_create_popup          (GpmBrightnessApplet *applet);
static gboolean  gpm_applet_popup_cb              (GpmBrightnessApplet *applet, GdkEventButton *event);
static void      gpm_applet_destroy_cb            (GtkObject *object);

#define GPM_BRIGHTNESS_APPLET_ICON		"moblin-brightness-applet"
#define GPM_BRIGHTNESS_APPLET_NAME		_("Power Manager Brightness Applet")
#define GPM_BRIGHTNESS_APPLET_DESC		_("Adjusts laptop panel brightness.")

/**
 * gpm_applet_get_brightness:
 * Return value: Success value, or zero for failure
 **/
static gboolean
gpm_applet_get_brightness (GpmBrightnessApplet *applet)
{
	GError  *error = NULL;
	gboolean ret;
	int policy_brightness = 0;

	if (applet->proxy == NULL) {
		printf ("WARNING: not connected\n");
		return FALSE;
	}

	ret = dbus_g_proxy_call (applet->proxy, "GetBrightness", &error,
				 G_TYPE_INVALID,
				 G_TYPE_INT, &policy_brightness,
				 G_TYPE_INVALID);
	if (error) {
		printf ("DEBUG: ERROR: %s\n", error->message);
		g_error_free (error);
	}
	if (ret == TRUE) {
		applet->level = policy_brightness;
	} else {
		/* abort as the DBUS method failed */
		printf ("WARNING: GetBrightness failed!\n");
	}

	return ret;
}

/**
 * gpm_applet_set_brightness:
 * Return value: Success value, or zero for failure
 **/
static gboolean
gpm_applet_set_brightness (GpmBrightnessApplet *applet)
{
	GError  *error = NULL;
	gboolean ret;
	int retval, level = 0;

	if (applet->proxy == NULL) {
		printf ("WARNING: not connected");
		return FALSE;
	}

	if(applet->level <= 0) {
		level = 0;
	} else if(applet->level >= 100) {
		level = 100;
	} else {
		level = bval[applet->level];
	}
	ret = dbus_g_proxy_call (applet->proxy, "SetBrightness", &error,
				 G_TYPE_INT, (int)level,
				 G_TYPE_INVALID, G_TYPE_INT, &retval,
				 G_TYPE_INVALID);
	if (error) {
		g_error("%s\n", error->message);
		g_error_free (error);
	}

	if (ret == FALSE) {
		g_error("WARNING: SetBrightness failed!\n");
	}

	return ret;
}

/**
 * gpm_applet_get_icon:
 * @applet: Brightness applet instance
 *
 * retrieve an icon from stock with a size adapted to panel
 **/
static void
gpm_applet_get_icon (GpmBrightnessApplet *applet)
{
	const gchar *icon;

	/* free */
	if (applet->icon != NULL) {
		g_object_unref (applet->icon);
		applet->icon = NULL;
	}

	if (applet->size <= 2) {
		return;
	}

	icon = GPM_BRIGHTNESS_APPLET_ICON;

	applet->icon = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
						 icon, applet->size - 2, 0, NULL);

	if (applet->icon == NULL) {
		printf ("WARNING: Cannot find %s!\n", icon);
	} else {
		printf ("DEBUG: got icon %s!\n", icon);
		/* update size cache */
		applet->icon_height = gdk_pixbuf_get_height (applet->icon);
		applet->icon_width = gdk_pixbuf_get_width (applet->icon);
	}
}

/**
 * gpm_applet_check_size:
 * @applet: Brightness applet instance
 *
 * check if panel size has changed and applet adapt size
 **/
static void
gpm_applet_check_size (GpmBrightnessApplet *applet)
{
	/* we don't use the size function here, but the yet allocated size because the
	   size value is false (kind of rounded) */
	if (applet->size != GTK_WIDGET(applet)->allocation.height) {
		applet->size = GTK_WIDGET(applet)->allocation.height;
		gpm_applet_get_icon (applet);
		gtk_widget_set_size_request (GTK_WIDGET(applet), applet->icon_width + 2, applet->size);
	}
}

/**
 * gpm_applet_draw_cb:
 * @applet: Brightness applet instance
 *
 * draws applet content (background + icon)
 **/
static gboolean
gpm_applet_draw_cb (GpmBrightnessApplet *applet)
{
	gint w, h, bg_type;
	GdkColor color;
	GdkGC *gc;
	GdkPixmap *background;

	if (GTK_WIDGET(applet)->window == NULL) {
		return FALSE;
	}

	/* retrieve applet size */
	gpm_applet_get_icon (applet);
	gpm_applet_check_size (applet);
	if (applet->size <= 2) {
		return FALSE;
	}

	/* if no icon, then don't try to display */
	if (applet->icon == NULL) {
		return FALSE;
	}

	w = GTK_WIDGET(applet)->allocation.width;
	h = GTK_WIDGET(applet)->allocation.height;

	gc = gdk_gc_new (GTK_WIDGET(applet)->window);

#if 0
	/* draw pixmap background */
	bg_type = panel_applet_get_background (PANEL_APPLET (applet), &color, &background);
	if (bg_type == PANEL_PIXMAP_BACKGROUND && !applet->popped) {
		/* fill with given background pixmap */
		gdk_draw_drawable (GTK_WIDGET(applet)->window, gc, background, 0, 0, 0, 0, w, h);
	}
	
	/* draw color background */
	if (bg_type == PANEL_COLOR_BACKGROUND && !applet->popped) {
		gdk_gc_set_rgb_fg_color (gc,&color);
		gdk_gc_set_fill (gc,GDK_SOLID);
		gdk_draw_rectangle (GTK_WIDGET(applet)->window, gc, TRUE, 0, 0, w, h);
	}

	/* fill with selected color if popped */
	if (applet->popped) {
		color = gtk_rc_get_style (GTK_WIDGET(applet))->bg[GTK_STATE_SELECTED];
		gdk_gc_set_rgb_fg_color (gc,&color);
		gdk_gc_set_fill (gc,GDK_SOLID);
		gdk_draw_rectangle (GTK_WIDGET(applet)->window, gc, TRUE, 0, 0, w, h);
	}
#endif

	/* draw icon at center */
	gdk_draw_pixbuf (GTK_WIDGET(applet)->window, gc, applet->icon,
			 0, 0, (w - applet->icon_width)/2, (h - applet->icon_height)/2,
			 applet->icon_width, applet->icon_height,
			 GDK_RGB_DITHER_NONE, 0, 0);

	return TRUE;
}

/**
 * gpm_applet_destroy_popup_cb:
 * @applet: Brightness applet instance
 *
 * destroys the popup (called if orientation has changed)
 **/
static gboolean
gpm_applet_destroy_popup_cb (GpmBrightnessApplet *applet)
{
	if (applet->popup != NULL) {
		gtk_widget_set_parent (applet->popup, NULL);
		gtk_widget_destroy (applet->popup);
		applet->popup = NULL;
	}
	return TRUE;
}

/**
 * gpm_applet_update_tooltip:
 * @applet: Brightness applet instance
 *
 * sets tooltip's content (percentage or disabled)
 **/
static void
gpm_applet_update_tooltip (GpmBrightnessApplet *applet)
{
	static gchar buf[101];
	if (applet->popped == FALSE) {
		snprintf (buf, 100, _("LCD brightness : %d%%"), applet->level);
		gtk_widget_set_tooltip_text (GTK_WIDGET(applet), buf);
	} else {
		gtk_widget_set_tooltip_text (GTK_WIDGET(applet), NULL);
	}
}

/**
 * gpm_applet_update_popup_level:
 * @applet: Brightness applet instance
 * @get_hw: set UI value from HW value
 * @set_hw: set HW value from UI value
 *
 * updates popup and hardware level of brightness
 * FALSE FAlSE -> set UI from cached value
 * TRUE  FAlSE -> set UI from HW value
 * TRUE  FALSE -> set HW from UI value, then set UI from HW value
 * FALSE TRUE  -> set HW from UI value
 **/
static void
gpm_applet_update_popup_level (GpmBrightnessApplet *applet)
{
	if (applet->popup != NULL) {
		gtk_widget_set_sensitive (applet->btn_plus, applet->level < 100);
		gtk_widget_set_sensitive (applet->btn_minus, applet->level > 0);
		gtk_range_set_value (GTK_RANGE(applet->slider), (guint) applet->level);
	}
	gpm_applet_update_tooltip (applet);
}

/**
 * gpm_applet_plus_cb:
 * @widget: The sending widget (plus button)
 * @applet: Brightness applet instance
 *
 * callback for the plus button
 **/
static gboolean
gpm_applet_plus_cb (GtkWidget *w, GpmBrightnessApplet *applet)
{
	if (applet->level < 100) {
		applet->level++;
	}
	applet->call_worked = gpm_applet_set_brightness (applet);
	gpm_applet_update_popup_level (applet);
	return TRUE;
}

/**
 * gpm_applet_minus_cb:
 * @widget: The sending widget (minus button)
 * @applet: Brightness applet instance
 *
 * callback for the minus button
 **/
static gboolean
gpm_applet_minus_cb (GtkWidget *w, GpmBrightnessApplet *applet)
{
	if (applet->level > 0) {
		applet->level--;
	}
	applet->call_worked = gpm_applet_set_brightness (applet);
	gpm_applet_update_popup_level (applet);
	return TRUE;
}

/**
 * gpm_applet_slide_cb:
 * @widget: The sending widget (slider)
 * @applet: Brightness applet instance
 *
 * callback for the slider
 **/
static gboolean
gpm_applet_slide_cb (GtkWidget *w, GpmBrightnessApplet *applet)
{
	applet->level = gtk_range_get_value (GTK_RANGE(applet->slider));
	applet->call_worked = gpm_applet_set_brightness (applet);
	gpm_applet_update_popup_level (applet);
	return TRUE;
}

/**
 * gpm_applet_slide_cb:
 * @applet: Brightness applet instance
 * @event: The key press event
 *
 * callback handling keyboard
 * mainly escape to unpop and arrows to change brightness
 **/
static gboolean
gpm_applet_key_press_cb (GpmBrightnessApplet *applet, GdkEventKey *event)
{
	int i;
	
	switch (event->keyval) {
	case GDK_KP_Enter:
	case GDK_ISO_Enter:
	case GDK_3270_Enter:
	case GDK_Return:
	case GDK_space:
	case GDK_KP_Space:
	case GDK_Escape:
		/* if yet popped, release focus and hide then redraw applet unselected */
		if (applet->popped) {
			gdk_keyboard_ungrab (GDK_CURRENT_TIME);
			gdk_pointer_ungrab (GDK_CURRENT_TIME);
			gtk_grab_remove (GTK_WIDGET(applet));
			gtk_widget_set_state (GTK_WIDGET(applet), GTK_STATE_NORMAL);
			gtk_widget_hide (applet->popup);
			applet->popped = FALSE;
			gpm_applet_draw_cb (applet);
			gpm_applet_update_tooltip (applet);
			return TRUE;
		} else {
			return FALSE;
		}
		break;
	case GDK_Page_Up:
		for (i = 0;i < 10;i++) {
			gpm_applet_plus_cb (NULL, applet);
		}
		return TRUE;
		break;
	case GDK_Left:
	case GDK_Up:
		gpm_applet_plus_cb (NULL, applet);
		return TRUE;
		break;
	case GDK_Page_Down:
		for (i = 0;i < 10;i++) {
			gpm_applet_minus_cb (NULL, applet);
		}
		return TRUE;
		break;
	case GDK_Right:
	case GDK_Down:
		gpm_applet_minus_cb (NULL, applet);
		return TRUE;
		break;
	default:
		return FALSE;
		break;
	}

	return FALSE;
}

/**
 * gpm_applet_scroll_cb:
 * @applet: Brightness applet instance
 * @event: The scroll event
 *
 * callback handling mouse scrolls, either when the applet
 * is not popped and the mouse is over the applet, or when
 * the applet is popped and no matter where the mouse is.
 **/
static gboolean
gpm_applet_scroll_cb (GpmBrightnessApplet *applet, GdkEventScroll *event)
{
	int i;

	if (event->type == GDK_SCROLL) {
		if (event->direction == GDK_SCROLL_UP) {
			for (i = 0;i < 5;i++) {
				gpm_applet_plus_cb (NULL, applet);
			}
			
		} else {
			for (i = 0;i < 5;i++) {
				gpm_applet_minus_cb (NULL, applet);
			}
		}
		return TRUE;
	}

	return FALSE;
}

/**
 * gpm_applet_create_popup:
 * @applet: Brightness applet instance
 *org.freedesktop.Hal.Device.LaptopPanel
 * cretes a new popup according to orientation of panel
 **/
static void
gpm_applet_create_popup (GpmBrightnessApplet *applet)
{
	static GtkWidget *box, *frame;

	gpm_applet_destroy_popup_cb (applet);

	/* slider */
	applet->slider = gtk_vscale_new_with_range (0, 100, 1);
	gtk_widget_set_size_request (applet->slider, -1, 100);

	gtk_range_set_inverted (GTK_RANGE(applet->slider), TRUE);
	gtk_scale_set_draw_value (GTK_SCALE(applet->slider), FALSE);
	gtk_range_set_value (GTK_RANGE(applet->slider), applet->level);
	g_signal_connect (G_OBJECT(applet->slider), "value-changed", G_CALLBACK(gpm_applet_slide_cb), applet);

	/* minus button */
	applet->btn_minus = gtk_button_new_with_label ("\342\210\222"); /* U+2212 MINUS SIGN */
	gtk_button_set_relief (GTK_BUTTON(applet->btn_minus), GTK_RELIEF_NONE);
	g_signal_connect (G_OBJECT(applet->btn_minus), "pressed", G_CALLBACK(gpm_applet_minus_cb), applet);

	/* plus button */
	applet->btn_plus = gtk_button_new_with_label ("+");
	gtk_button_set_relief (GTK_BUTTON(applet->btn_plus), GTK_RELIEF_NONE);
	g_signal_connect (G_OBJECT(applet->btn_plus), "pressed", G_CALLBACK(gpm_applet_plus_cb), applet);

	/* box */
	box = gtk_vbox_new (FALSE, 1);
	gtk_box_pack_start (GTK_BOX(box), applet->btn_plus, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(box), applet->slider, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(box), applet->btn_minus, FALSE, FALSE, 0);

	/* frame */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER(frame), box);

	/* window */
	applet->popup = gtk_window_new (GTK_WINDOW_POPUP);
	GTK_WIDGET_UNSET_FLAGS (applet->popup, GTK_TOPLEVEL);
	gtk_widget_set_parent (applet->popup, GTK_WIDGET(applet));
	gtk_container_add (GTK_CONTAINER(applet->popup), frame);
}

/**
 * gpm_applet_popup_cb:
 * @applet: Brightness applet instance
 *
 * pops and unpops
 **/
static gboolean
gpm_applet_popup_cb (GpmBrightnessApplet *applet, GdkEventButton *event)
{
	gint x, y;

	/* react only to left mouse button */
	if (event->button != 1) {
		return FALSE;
	}

	/* if yet popped, release focus and hide then redraw applet unselected */
	if (applet->popped) {
		gdk_keyboard_ungrab (GDK_CURRENT_TIME);
		gdk_pointer_ungrab (GDK_CURRENT_TIME);
		gtk_grab_remove (GTK_WIDGET(applet));
		gtk_widget_set_state (GTK_WIDGET(applet), GTK_STATE_NORMAL);
		gtk_widget_hide (applet->popup);
		applet->popped = FALSE;
		gpm_applet_draw_cb (applet);
		gpm_applet_update_tooltip (applet);
		return TRUE;
	}

	/* update UI for current brightness */
	gpm_applet_update_popup_level (applet);

	/* otherwise pop */
	applet->popped = TRUE;
	gpm_applet_draw_cb (applet);

	/* create a new popup (initial or if panel parameters changed) */
	if (applet->popup == NULL) {
		gpm_applet_create_popup (applet);
	}

	/* update UI for current brightness */
	gpm_applet_update_popup_level (applet);

	gtk_widget_show_all (applet->popup);

	/* retrieve geometry parameters and move window appropriately */
	gdk_window_get_origin (GTK_WIDGET(applet)->window, &x, &y);
	y += GTK_WIDGET(applet)->allocation.y + GTK_WIDGET(applet)->allocation.height;
	gtk_window_move (GTK_WINDOW (applet->popup), x, y);

	/* grab input */
	gtk_widget_grab_focus (GTK_WIDGET(applet));
	gtk_grab_add (GTK_WIDGET(applet));
	gdk_pointer_grab (GTK_WIDGET(applet)->window, TRUE,
			  GDK_BUTTON_PRESS_MASK |
			  GDK_BUTTON_RELEASE_MASK |
			  GDK_POINTER_MOTION_MASK,
			  NULL, NULL, GDK_CURRENT_TIME);
	gdk_keyboard_grab (GTK_WIDGET(applet)->window,
			   TRUE, GDK_CURRENT_TIME);
	gtk_widget_set_state (GTK_WIDGET(applet), GTK_STATE_SELECTED);

	return TRUE;
}

/**
 * gpm_applet_theme_change_cb:
 *
 * Updtes icon when theme changes
 **/
static void
gpm_applet_theme_change_cb (GtkIconTheme *icon_theme, gpointer data)
{
	GpmBrightnessApplet *applet = GPM_BRIGHTNESS_APPLET (data);
	gpm_applet_get_icon (applet);
}

/**
 * gpm_applet_stop_scroll_events_cb:
 *
 * Prevents scroll events from reaching the tooltip
 **/
static void
gpm_applet_stop_scroll_events_cb (GtkWidget *widget, GdkEvent  *event)
{
	if (event->type == GDK_SCROLL)
		g_signal_stop_emission_by_name (widget, "event-after");
}

/**
 * gpm_applet_destroy_cb:
 * @object: Class instance to destroy
 **/
static void
gpm_applet_destroy_cb (GtkObject *object)
                                 
{
	GpmBrightnessApplet *applet = GPM_BRIGHTNESS_APPLET(object);

	if (applet->icon != NULL) {
		gdk_pixbuf_unref (applet->icon);
	}
}

/**
 * gpm_brightness_applet_class_init:
 * @klass: Class instance
 **/
static void
gpm_brightness_applet_class_init (GpmBrightnessAppletClass *class)
{
	/* nothing to do here */
}

static void
brightness_changed_cb (DBusGProxy          *proxy,
		       guint	            brightness,
		       GpmBrightnessApplet *applet)
{
	printf ("DEBUG: BrightnessChanged detected: %i\n", brightness);
	applet->level = brightness;
}

/**
 * gpm_brightness_applet_dbus_connect:
 **/
gboolean
gpm_brightness_applet_dbus_connect (GpmBrightnessApplet *applet)
{
	DBusGProxy *proxy = NULL;
	GError *error = NULL;
	gchar **names = NULL;
	int i;

	if(applet->connection == NULL)
	{
	  applet->connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
	  if (error) {
	    printf ("WARNING: Could not connect to DBUS daemon: %s", 
	      error->message);
	    g_error_free (error);
	    applet->connection = NULL;
	    return FALSE;
	  }
	  proxy = dbus_g_proxy_new_for_name (applet->connection,
	    HAL_DBUS_SERVICE, HAL_DBUS_PATH_MANAGER, 
	    HAL_DBUS_INTERFACE_MANAGER);
	  dbus_g_proxy_call (proxy, "FindDeviceByCapability", &error,
	    G_TYPE_STRING, "laptop_panel", G_TYPE_INVALID,
	    G_TYPE_STRV, &names, G_TYPE_INVALID);
	  if (error) {
	    g_warning ("%s", error->message);
	    g_error_free (error);
	    applet->connection = NULL;
	    return FALSE;
	  }
	  if (names == NULL || names[0] == NULL) {
	    g_warning ("No devices of capability laptop_panel");
	    if(names) g_free(names);
	    applet->connection = NULL;
	    return FALSE;
	  }
	  applet->udi = g_strdup (names[0]);
	  for (i=0; names[i]; i++) {
	    g_free (names[i]);
	  }
	  g_free (names);
	}

	if (applet->proxy == NULL) {
	  applet->proxy = dbus_g_proxy_new_for_name(applet->connection,
	    HAL_DBUS_SERVICE, applet->udi,
	    HAL_DBUS_INTERFACE_LAPTOP_PANEL);
	  if (applet->proxy == NULL) {
	    g_warning("Cannot connect, maybe the daemon is not running\n");
	    return FALSE;
	  }

	  dbus_g_proxy_add_signal (applet->proxy, "BrightnessChanged",
	    G_TYPE_UINT, G_TYPE_INVALID);
	  dbus_g_proxy_connect_signal (applet->proxy, "BrightnessChanged",
	    G_CALLBACK (brightness_changed_cb), applet, NULL);
	  /* reset, we might be starting race */
	  applet->call_worked = gpm_applet_get_brightness (applet);
	}
	return TRUE;
}

/**
 * gpm_brightness_applet_dbus_disconnect:
 **/
gboolean
gpm_brightness_applet_dbus_disconnect (GpmBrightnessApplet *applet)
{
	if (applet->proxy != NULL) {
		printf ("DEBUG: removing proxy\n");
		g_object_unref (applet->proxy);
		applet->proxy = NULL;
	}
	return TRUE;
}

/**
 * gpm_brightness_applet_init:
 * @applet: Brightness applet instance
 **/
static void
gpm_brightness_applet_init (GpmBrightnessApplet *applet)
{
	/* initialize fields */
	applet->size = 0;
	applet->call_worked = TRUE;
	applet->popped = FALSE;
	applet->popup = NULL;
	applet->icon = NULL;
	applet->connection = NULL;
	applet->proxy = NULL;

	gpm_brightness_applet_dbus_connect (applet);

	/* coldplug */
	applet->call_worked = gpm_applet_get_brightness (applet);
	gpm_applet_update_popup_level (applet);

	/* show */
	gtk_widget_show_all (GTK_WIDGET(applet));

	/* set appropriate size and load icon accordingly */
	gpm_applet_draw_cb (applet);

	/* connect */
	g_signal_connect (G_OBJECT(applet), "button-press-event",
			  G_CALLBACK(gpm_applet_popup_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "scroll-event",
			  G_CALLBACK(gpm_applet_scroll_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "key-press-event",
			  G_CALLBACK(gpm_applet_key_press_cb), NULL);

	/* We use g_signal_connect_after because letting the panel draw
	 * the background is the only way to have the correct
	 * background when a theme defines a background picture. */
	g_signal_connect_after (G_OBJECT(applet), "expose-event",
				G_CALLBACK(gpm_applet_draw_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "change-orient",
			  G_CALLBACK(gpm_applet_draw_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "change-orient",
			  G_CALLBACK(gpm_applet_destroy_popup_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "destroy",
			  G_CALLBACK(gpm_applet_destroy_cb), NULL);

	/* prevent scroll events from reaching the tooltip */
	g_signal_connect (G_OBJECT (applet), "event-after", G_CALLBACK (gpm_applet_stop_scroll_events_cb), NULL);

	g_signal_connect (gtk_icon_theme_get_default (), "changed", G_CALLBACK (gpm_applet_theme_change_cb), applet);
}

MoblinBrightnessApplet*
moblin_brightness_applet_new()
{
MoblinBrightnessApplet *obj;

  if(!gpm_brightness_applet_get_type())
  {
    g_error("Failed to register type GpmBrightnessApplet\n");
    return NULL;
  }
  obj = g_new0(MoblinBrightnessApplet, 1);
  obj->applet = (GpmBrightnessApplet*)g_object_new(gpm_brightness_applet_get_type(), NULL);

  return obj;
}

void *brightness_initialize(HildonStatusBarItem *item, GtkWidget **button)
{
  MoblinBrightnessApplet *obj = moblin_brightness_applet_new();

  if(!obj)
  {
    g_error("Failed to initialize the brightness applet\n");
    return NULL;
  }

  obj->item = item;
  *button = obj->button = GTK_WIDGET(obj->applet);

  gpm_applet_draw_cb (obj->applet);

  return obj;
}

void brightness_update(void *data, gint value1, gint value2, const gchar *str)
{

}

void brightness_destroy(void *data)
{
}

gint brightness_get_priority(void *data)
{
	return 1;
}

void brightness_entry (HildonStatusBarPluginFn_st *fn)
{
	if (fn == NULL)
	{
		g_error("bad setup call\n");
		return;
	}

	fn->initialize = brightness_initialize;
	fn->destroy = brightness_destroy;
	fn->update = brightness_update;
	fn->get_priority = brightness_get_priority;
}
