#!/usr/bin/python -ttu
# vim: ai ts=4 sts=4 et sw=4
#
# media_info.py: Keeps track of file meta data, will store it in a SQLite
# database, so that it won't need to reread the files.
# 
# Copyright (c) 2007, 2008 Intel Corporation
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; version 2 of the License
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc., 59
# Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import mutagen
import os
import sqlite3

import utils

class MediaInfoError(Exception):
    def __init__(self, message):
        self.message = message

    def __str__(self):
        return self.message

class MediaDatabaseError(Exception):
    def __init__(self, message):
        self.message = message

    def __str__(self):
        return self.message

class MediaDatabase(object):
    """Database to hold the metadata for our media files"""
    def __init__(self, path):
        self.__connect = sqlite3.connect(path)
        self.__cursor = self.__connect.cursor()
        try:
            # Attempt to create a new table, and if that succeeds then
            self.__cursor.execute('''create table MediaInfo
                                     (uri text primary key,
                                     title text,
                                     artist text,
                                     album text,
                                     tracknum text,
                                     date text,
                                     length text,
                                     rawlen integer)''')
        except:
            # Not an error condition, but just the second time anyone
            # has ever attempted to query this database table
            pass

    def get(self, uri):
        self.__cursor.execute("select * from MediaInfo where uri=?", (uri,))
        try:
            return self.__cursor.fetchall()[0]
        except IndexError:
            raise MediaDatabaseError("record for %s does not exists" % (uri))

    def insert(self, uri, title, artist, album, tracknum, date, length, rawlen):
        self.__cursor.execute("insert into MediaInfo (uri, title, artist, album, tracknum, date, length, rawlen) values (?, ?, ?, ?, ?, ?, ?, ?)",
                              (uri,
                               title,
                               artist,
                               album,
                               tracknum,
                               date,
                               length,
                               rawlen))
        self.__connect.commit()

    def update(self, uri, key, value):
        self.__cursor.execute("update MediaInfo set %s='%s' where uri='%s'" %
                              (key, value, uri))
        self.__connect.commit()

    def close(self):
        self.__connect.close()


class MediaInfo(object):
    """Stores the metadata for a media object, if it is already in the database
    then the data will be retrieved from the database.  Otherwise it will
    create a new database object and store the metadata in there"""
    def __init__(self, uri, db):
        self.__db = db
        if uri[0] == '/':
            self.__uri = 'file://' + uri
        else:
            self.__uri = uri
        try:
            self.__proto, self.__full_path = self.__uri.split('://')
        except ValueError:
            raise MediaInfoError("Invalid URI: %s!" % (self.__uri))
        self.__directory, self.__filename = os.path.split(self.__full_path)
        self.__name, tmp = os.path.splitext(self.__filename)
        self.__ext = tmp.replace('.', '')
        self.__info = {}
        try:
            res = self.__db.get(uri=self.__uri)
            self.__info['uri'] = self.__uri
            self.__info['title'] = str(res[1])
            self.__info['artist'] = str(res[2])
            self.__info['album'] = str(res[3])
            self.__info['tracknum'] = str(res[4])
            self.__info['date'] = str(res[5])
            self.__info['length'] = str(res[6])
            self.__info['rawlen'] = res[7]
        except MediaDatabaseError:
            if self.__proto == 'file':
                try:
                    self.__metadata = mutagen.File(self.__full_path)
                    if 'audio/mp4' in self.__metadata.mime:
                        self.__parse_mp4_file()
                    elif 'audio/mp3' in self.__metadata.mime:
                        self.__parse_mp3_file()
                    else:
                        self.__fill_in_empty_info()
                except:
                    self.__fill_in_empty_info()
            else:
                self.__fill_in_empty_info()
            self.__db.insert(uri = self.__uri,
                             title = self.__info['title'],
                             artist = self.__info['artist'],
                             album = self.__info['album'],
                             tracknum = self.__info['tracknum'],
                             date = self.__info['date'],
                             length = self.__info['length'],
                             rawlen = self.__info['rawlen'])

    def __fill_in_empty_info(self):
        self.__info['title'] = self.__name
        self.__info['artist'] = None
        self.__info['album'] = None
        self.__info['tracknum'] = None
        self.__info['date'] = None
        self.__info['length'] = None
        self.__info['rawlen'] = -1

    def __parse_mp3_file(self):
        try:
            self.__info['title'] = str(self.__metadata['TIT2'])
        except:
            self.__info['title'] = self.__name
        try:
            self.__info['artist'] = str(self.__metadata['TPE1'])
        except:
            self.__info['artist'] = None
        try:
            self.__info['album'] = str(self.__metadata['TALB'])
        except:
            self.__info['album'] = None
        try:
            self.__info['tracknum'] = str(self.__metadata['TRCK'])
        except:
            self.__info['tracknum'] = None
        try:
            self.__info['date'] = str(self.__metadata['TDRC'])
        except:
            self.__info['date'] = None
        try:
            info = self.__metadata.info
            self.__info['rawlen'] = info.length
            self.__info['length'] = utils.convertSecondsToString(info.length)
        except:
            self.__info['length'] = None
            self.__info['rawlen'] = -1

    def __parse_mp4_file(self):
        try:
            self.__info['title'] = str(self.__metadata['\xa9nam'][0])
        except:
            self.__info['title'] = self.__name
        try:
            self.__info['artist'] = str(self.__metadata['\xa9ART'][0])
        except:
            self.__info['artist'] = None
        try:
            self.__info['album'] = str(self.__metadata['\xa9alb'][0])
        except:
            self.__info['album'] = None
        try:
            self.__info['tracknum'] = str(self.__metadata['trkn'][0])
        except:
            self.__info['tracknum'] = None
        try:
            self.__info['date'] = str(self.__metadata['\xa9day'][0])
        except:
            self.__info['date'] = None
        try:
            info = self.__metadata.info
            self.__info['rawlen'] = info.length
            self.__info['length'] = utils.convertSecondsToString(info.length)
        except:
            self.__info['length'] = None
            self.__info['rawlen'] = -1
            
    def __getitem__(self, index):
        return self.__info[index]

    def __setitem__(self, key, value):
        if type(value) == type(''):
            value = value.replace('\\\'', '') 
        self.__db.update(self.__uri, key, value)
        self.__info[key] = value

    def __len__(self):
        return len(self.__info)

    def __iter__(self):
        return self

    def __str__(self):
        return self.__info.__str__()

    def next(self):
        try:
            item = self.__info[self.__index]
            self.__index = self.__index + 1
            return item
        except:
            self.__index = 0
            raise StopIteration()

if __name__ == '__main__':
    import sys

    if len(sys.argv) < 3:
        print "USAGE: %s DATABASE_PATH FILE [FILE ...]" % (sys.argv[0])
        sys.exit(1)

    db = MediaDatabase(sys.argv[1])
    for d in sys.argv[2:]:
        if not os.path.isdir(d):
            print MediaInfo(d, db)
        else:
            for f in os.listdir(d):
                print MediaInfo('file://' + os.path.join(d, f), db)
