
#include <libbtutil.h>
#include <libbtpeer/types/btp_torrent.h>
#include <libbtpeer/io.h>

#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <malloc.h>

#include <apr.h>
#include <apr_file_io.h>
#include <apr_pools.h>
#include <apr_strings.h>
#include <apr_version.h>
#include <apr_errno.h>

apr_status_t btp_torrent_io_create_files(btp_torrent* torrent) {
    int           i, ret;
    char          path[BT_PATH_LEN];
    char*         p;
    struct stat   sb;
    apr_file_t*   file;

    for(i=0; i<torrent->info->file_count; i++) {
        if(
            snprintf(
                path, BT_PATH_LEN, "%s/%s", torrent->destination,
                torrent->info->files[i].name
            )
            > BT_FILE_LEN
        ) {
            fprintf(
                stderr,
                "btp_torrent_io_create_files(): Pathname for %s is "
                "longer than %i bytes!\n",
                path, BT_PATH_LEN
            );
	    return APR_EBADPATH;
	}

        /* Create folders */
        p = path;
        while((p = strchr( p, '/' )))
        {
            *p = '\0';
            if(stat(path, &sb)) {
	        if(
                    (
                        ret = apr_dir_make(
                            path, APR_FOPEN_READ|APR_FOPEN_WRITE,
                            torrent->pool
                        )
                    ) != APR_SUCCESS
                ) {
                    fprintf(
                        stderr, "btp_torrent_io_create_files(): "
                        "creating directory %s failed: %s",
                        path, apr_strerror(ret, path, BT_PATH_LEN)
                    );
                    return ret;
	        }
            } else if((sb.st_mode & S_IFMT) != S_IFDIR) {
                fprintf(
                    stderr, "btp_torrent_io_create_files(): can't create "
                    "directory %s: file already exists\n", path
                );
                return APR_EEXIST;
            }
            *p = '/';
            p++;
        }

        if(stat(path, &sb)) {
            if(
                (ret = apr_file_open(
                    &file, path, APR_WRITE|APR_CREATE,
                    APR_FOPEN_READ|APR_FOPEN_WRITE, torrent->pool
                )) != APR_SUCCESS
            ) {
                fprintf(
                    stderr, "btp_torrent_io_create_files(): "
                    "can't create file %s: %s", path,
                    apr_strerror(ret, path, BT_PATH_LEN)
                );
                return ret;
            }
            apr_file_close(file);
        } else if((sb.st_mode & S_IFMT ) != S_IFREG) {
            fprintf(
                stderr, "btp_torrent_io_create_files(): can't create file %s: "
                "a non-file with that name already exists\n", path
            );
            return APR_EEXIST;
        }
    }
    
    return APR_SUCCESS;
}

apr_status_t btp_torrent_io_buffer(
    btp_torrent* t, uint64_t offset, int size, uint8_t* buf, int write,
    bt_off_t *actual
) {
    int piece = offset / t->info->piece_size;
    int begin = offset % t->info->piece_size;
    int left = size;
    int fn;
    apr_status_t ret;
    bt_off_t fpos, fcur, actual_i;
    apr_file_t* fh;
    apr_off_t afpos;
    apr_size_t asize;
    char path[BT_PATH_LEN];
    char err[255];
 
    /* we don't want to think about more than a piece at once.
     * thinking is memory.
     */
    if(bt_metainfo_piece_size(t->info, piece) < begin + size)
        return APR_ENOMEM;
    
    /* we can't manipulate something on the harddrive that's outside
     * of the files' bounds. harddrive is space.
     */
    if((fn = bt_metainfo_pos2file(t->info, offset, &fpos)) < 0)
        return APR_ENOSPC;
    
    if(!write)
        bzero(buf, size);
    
    actual_i = 0;
    
    while(left > 0) {
        if(t->info->files[fn].length < left + fpos)
            fcur = t->info->files[fn].length - fpos;
        else
            fcur = left;
        
        if(
            snprintf(
                path, sizeof(path), "%s/%s", t->destination,
                t->info->files[fn].name
            ) > sizeof(path)
        )
            return APR_ENAMETOOLONG;
        
        if((ret = btp_file_pool_get(t->files, path, &fh)) != APR_SUCCESS) {
            fprintf(stderr, 
                "btp_torrent_io_buffer: failed to get %s %s: %s\n",
                t->destination, t->info->files[fn].name,
                apr_strerror(ret, err, sizeof(err))
            );
            return ret;
        }
        
        afpos = fpos;
        if((ret = apr_file_seek(fh, APR_SET, &afpos)) != APR_SUCCESS)
            return ret;
        
        if(afpos != fpos)
            return APR_EOF;
        
        if(write) {
            if(
                (ret = apr_file_write_full(fh, buf, fcur, &asize))
                != APR_SUCCESS
            )
                return ret;
        } else {
            ret = apr_file_read_full(fh, buf, fcur, &asize);
            
            if(ret != APR_SUCCESS && ret != APR_EOF)
                return ret;

        }
        
        actual_i += asize;
        fn++;
        buf += fcur;
        left -= fcur;
        fpos = 0;
    }
    
    if(actual)
        *actual = actual_i;
    
    return APR_SUCCESS;
}

apr_status_t btp_torrent_io_piece_buffer(
    btp_torrent* t, int piece, uint8_t* buf, int write,
    bt_off_t *actual
) {
    int size = bt_metainfo_piece_size(t->info, piece);
    uint64_t offset = piece * t->info->piece_size;
    return btp_torrent_io_buffer(t, offset, size, buf, write, actual);
}
