
#include <libbtpeer/types/btp_torrent.h>
#include <libbtpeer/types/btp_torrent/pieces.h>
#include <libbtpeer/bitfield.h>


btp_torrent_pieces* btp_torrent_pieces_init(btp_torrent* torrent) {
    btp_torrent_pieces* rv;
    int i;
    
    rv = apr_pcalloc(torrent->pool, sizeof(btp_torrent_pieces));
    rv->block_count = 0;
    rv->block_bits = apr_pcalloc(
        torrent->pool, (torrent->block_count + 7) / 8
    );
    rv->block_downloaders = apr_pcalloc(torrent->pool, torrent->block_count);
    rv->block_missing = apr_palloc(
        torrent->pool, torrent->info->piece_count * sizeof(int)
    );
    rv->piece_bits = apr_pcalloc(
        torrent->pool, (torrent->info->piece_count + 7) / 8
    );

    for(i=0; i<torrent->info->piece_count; i++)
        rv->block_missing[i] = btp_torrent_piece_count_blocks(torrent, i);

    return rv;
}

float btp_torrent_pieces_complete(btp_torrent* torrent) {
    return (float)torrent->pieces->block_count / (float)torrent->block_count;
}

int btp_torrent_piece_complete(btp_torrent* torrent, int piece) {
    return btp_bitfield_has(torrent->pieces->piece_bits, piece);
}

void btp_torrent_piece_add(btp_torrent* torrent, int piece) {
    int i;
    int start = BTP_TORRENT_PIECE_START(torrent, piece);
    int end = start + BTP_TORRENT_PIECE_BLOCKS(torrent, piece);
    for(i=start; i<end; i++) btp_torrent_piece_add_block(torrent, i);
    btp_bitfield_add(torrent->pieces->piece_bits, piece);
}

void btp_torrent_piece_add_block(btp_torrent* torrent, int block) {
    if(!btp_bitfield_has(torrent->pieces->block_bits, block)) {
        torrent->pieces->block_count++;
        if(!torrent->pieces->block_downloaders[block])
            torrent->pieces->block_missing[
                BTP_TORRENT_BLOCK_PIECE(torrent, block)
            ] --;
    }
    btp_bitfield_add(torrent->pieces->block_bits, block);
}

void btp_torrent_block_add_downloader(btp_torrent* torrent, int block) {
    if(
        !torrent->pieces->block_downloaders[block] &&
        !btp_bitfield_has(torrent->pieces->block_bits, block)
    )
        torrent->pieces->block_missing[
            BTP_TORRENT_BLOCK_PIECE(torrent,block)
        ]--;
    
    torrent->pieces->block_downloaders[block]++;
}

void btp_torrent_block_remove_downloader(btp_torrent* torrent, int block) {
    if(
        !torrent->pieces->block_downloaders[block] &&
        !btp_bitfield_has(torrent->pieces->block_bits, block)
    )
        torrent->pieces->block_missing[
            BTP_TORRENT_BLOCK_PIECE(torrent,block)
        ]++;
    
    torrent->pieces->block_downloaders[block]--;
}

void btp_torrent_block_remove(btp_torrent* torrent, int block) {
    if(btp_bitfield_has(torrent->pieces->block_bits, block)) {
        if(!torrent->pieces->block_downloaders[block])
            torrent->pieces->block_missing[
                BTP_TORRENT_BLOCK_PIECE(torrent,block)
            ] ++;
        torrent->pieces->block_count--;
    }
    btp_bitfield_remove(torrent->pieces->block_bits, block);
}

void btp_torrent_block_bits_set(btp_torrent* torrent) {
    int piece, block, start, end, complete;
    
    for(piece=0; piece<torrent->info->piece_count; piece++) {
        start = BTP_TORRENT_PIECE_START(torrent, piece);
        end = start + BTP_TORRENT_PIECE_BLOCKS(torrent, piece);
        complete = 1;
        
        for(block=start; block<end; block++) {
            if(btp_bitfield_has(torrent->pieces->block_bits, block))
                btp_torrent_piece_add_block(torrent, block);
            else
                complete = 0;
        }
        
        if(complete)
            btp_torrent_piece_add(torrent, piece);
    }
}

int btp_torrent_piece_missing_block(btp_torrent* torrent, int piece) {
    int start, end, block;
    
    start = BTP_TORRENT_PIECE_START(torrent, piece);
    end = start + BTP_TORRENT_PIECE_BLOCKS(torrent, piece);
    
    for(block=start; block<end; block++) {
        if(
            (!btp_bitfield_has(torrent->pieces->block_bits, block)) &&
            (!torrent->pieces->block_downloaders[block])
        )
            return block;
    }
    
    return -1;
}

int btp_torrent_piece_most_missing_block(
    btp_torrent* torrent, int piece, int* downloaders
) {
    int start, end, count, block;
    int min = 255;
    int most_missing = -1;
    
    start = BTP_TORRENT_PIECE_START(torrent, piece);
    count = BTP_TORRENT_PIECE_BLOCKS(torrent, piece);
    end = start + count;

    for(block=start; block<end; block++) {
        if(
            (!btp_bitfield_has(torrent->pieces->block_bits, block)) &&
            (!torrent->pieces->block_downloaders[block])
        ) {
            if(torrent->pieces->block_downloaders[block] < min)
                most_missing = block;
        }
    }
    
    if(most_missing >= 0)
        *downloaders = min;
    
    return most_missing;
}
