/* Copyright (c) 2006, Nokia Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * * Neither the name of the Nokia Corporation nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <glib/gi18n.h>
#include <gtk/gtk.h>

#include <widgets/modest-account-view-window.h>
#include <widgets/modest-account-view.h>

#include <modest-runtime.h>
#include <modest-account-mgr-helpers.h>
#include <string.h>
#include "modest-account-assistant.h"
#include "modest-tny-platform-factory.h"

/* 'private'/'protected' functions */
static void                            modest_account_view_window_class_init   (ModestAccountViewWindowClass *klass);
static void                            modest_account_view_window_init         (ModestAccountViewWindow *obj);
static void                            modest_account_view_window_finalize     (GObject *obj);

/* list my signals */
enum {
	/* MY_SIGNAL_1, */
	/* MY_SIGNAL_2, */
	LAST_SIGNAL
};

typedef struct _ModestAccountViewWindowPrivate ModestAccountViewWindowPrivate;
struct _ModestAccountViewWindowPrivate {
	GtkWidget           *add_button;
	GtkWidget           *edit_button;
	GtkWidget           *remove_button;
	GtkWidget	    *default_button;
	ModestAccountView   *account_view;
};
#define MODEST_ACCOUNT_VIEW_WINDOW_GET_PRIVATE(o)      (G_TYPE_INSTANCE_GET_PRIVATE((o), \
                                                        MODEST_TYPE_ACCOUNT_VIEW_WINDOW, \
                                                        ModestAccountViewWindowPrivate))
/* globals */
static GtkDialogClass *parent_class = NULL;

/* uncomment the following if you have defined any signals */
/* static guint signals[LAST_SIGNAL] = {0}; */

GType
modest_account_view_window_get_type (void)
{
	static GType my_type = 0;
	if (!my_type) {
		static const GTypeInfo my_info = {
			sizeof(ModestAccountViewWindowClass),
			NULL,		/* base init */
			NULL,		/* base finalize */
			(GClassInitFunc) modest_account_view_window_class_init,
			NULL,		/* class finalize */
			NULL,		/* class data */
			sizeof(ModestAccountViewWindow),
			1,		/* n_preallocs */
			(GInstanceInitFunc) modest_account_view_window_init,
			NULL
		};
		my_type = g_type_register_static (GTK_TYPE_DIALOG,
		                                  "ModestAccountViewWindow",
		                                  &my_info, 0);
	}
	return my_type;
}

static void
modest_account_view_window_class_init (ModestAccountViewWindowClass *klass)
{
	GObjectClass *gobject_class;
	gobject_class = (GObjectClass*) klass;

	parent_class            = g_type_class_peek_parent (klass);
	gobject_class->finalize = modest_account_view_window_finalize;

	g_type_class_add_private (gobject_class, sizeof(ModestAccountViewWindowPrivate));
}

static void
modest_account_view_window_init (ModestAccountViewWindow *obj)
{
	/* empty */
}

static void
modest_account_view_window_finalize (GObject *obj)
{
	G_OBJECT_CLASS(parent_class)->finalize (obj);
}


static void
on_selection_changed (GtkTreeSelection *sel, ModestAccountViewWindow *self)
{
	ModestAccountViewWindowPrivate *priv;
	GtkTreeModel                   *model;
	GtkTreeIter                    iter;
	gboolean                       has_selection;
	gchar                         *account_name;
	gchar                         *default_account_name;
	
	priv = MODEST_ACCOUNT_VIEW_WINDOW_GET_PRIVATE(self);

	has_selection =
		gtk_tree_selection_get_selected (sel, &model, &iter);

	gtk_widget_set_sensitive (priv->edit_button, has_selection);
	gtk_widget_set_sensitive (priv->remove_button, has_selection);	

	account_name = modest_account_view_get_selected_account (priv->account_view);
	default_account_name = modest_account_mgr_get_default_account(
		modest_runtime_get_account_mgr());
	gtk_widget_set_sensitive (priv->default_button,
				  default_account_name == NULL || account_name == NULL ||
				  strcmp (default_account_name, account_name) != 0);
	g_free (account_name);
	g_free (default_account_name);
}

static void
on_remove_button_clicked (GtkWidget *button, ModestAccountViewWindow *self)
{
	ModestAccountViewWindowPrivate *priv;
	ModestAccountMgr *account_mgr;
	gchar *account_name;
	
	priv = MODEST_ACCOUNT_VIEW_WINDOW_GET_PRIVATE(self);

	account_mgr = modest_runtime_get_account_mgr();	
	account_name = modest_account_view_get_selected_account (priv->account_view);

	if (account_name) {
		gboolean removed;
		GtkWidget *dialog;
		gchar *txt;

		dialog = gtk_dialog_new_with_buttons (_("Confirmation dialog"),
						      GTK_WINDOW (self),
						      GTK_DIALOG_MODAL,
						      GTK_STOCK_CANCEL,
						      GTK_RESPONSE_REJECT,
						      GTK_STOCK_OK,
						      GTK_RESPONSE_ACCEPT,
						      NULL);
		txt = g_strdup_printf (_("Do you really want to delete the account %s?"), account_name);
		gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), 
				    gtk_label_new (txt), FALSE, FALSE, 0);
/* 		gtk_widget_show_all (GTK_WIDGET(GTK_DIALOG(dialog)->vbox)); */
		g_free (txt);

		if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
			/* Remove account. If succeeded it removes also 
			   the account from the ModestAccountView */
			removed = modest_account_mgr_remove_account (account_mgr,
								     account_name);
			if (removed) {
				/* Show confirmation dialog ??? */
			} else {
				/* Show error dialog ??? */
				g_warning ("Error removing account %s", account_name);
			}
		}
		gtk_widget_destroy (dialog);
		g_free (account_name);
	}
}

/** Check whether any connections are active, and cancel them if 
 * the user wishes.
 * Returns TRUE is there was no problem, 
 * or if an operation was cancelled so we can continue.
 * Returns FALSE if the user chose to cancel his request instead.
 */
static gboolean
check_for_active_account (ModestAccountViewWindow *self, const gchar* account_name)
{
	ModestTnyAccountStore *acc_store;
	ModestMailOperationQueue* queue;
	TnyConnectionStatus store_conn_status, transport_conn_status;
	TnyAccount *store_account = NULL, *transport_account = NULL;
	gboolean retval = TRUE;

	acc_store = modest_runtime_get_account_store ();
	queue = modest_runtime_get_mail_operation_queue ();

	store_account = 
		modest_tny_account_store_get_server_account (acc_store,
							     account_name,
							     TNY_ACCOUNT_TYPE_STORE);
	transport_account = 
		modest_tny_account_store_get_server_account (acc_store,
							     account_name,
							     TNY_ACCOUNT_TYPE_TRANSPORT);

	store_conn_status = tny_account_get_connection_status (store_account);
	transport_conn_status = tny_account_get_connection_status (transport_account);

	if (store_conn_status == TNY_CONNECTION_STATUS_CONNECTED ||
	    transport_conn_status == TNY_CONNECTION_STATUS_CONNECTED) {
		gint response;

		response = modest_platform_run_confirmation_dialog (GTK_WINDOW (self), 
								_("emev_nc_disconnect_account"));
		if (response == GTK_RESPONSE_OK) {
			/* FIXME: We should only cancel those of this account */
			modest_mail_operation_queue_cancel_all (queue);

			/* Also disconnect the account */
			if (tny_account_get_connection_status (store_account) == TNY_CONNECTION_STATUS_CONNECTED) {
				tny_account_cancel (store_account);
				tny_camel_account_set_online (TNY_CAMEL_ACCOUNT (store_account),
							      FALSE, NULL, NULL);
			}
			if (tny_account_get_connection_status (transport_account) == TNY_CONNECTION_STATUS_CONNECTED) {
				tny_account_cancel (transport_account);
				tny_camel_account_set_online (TNY_CAMEL_ACCOUNT (transport_account),
							      FALSE, NULL, NULL);
			}
			retval = TRUE;
		} else {
			retval = FALSE;
		}
	}

	/* Frees */
	g_object_unref (store_account);
	g_object_unref (transport_account);
	
	return retval;
}

static void
on_edit_button_clicked (GtkWidget *button, ModestAccountViewWindow *self)
{
  	ModestAccountViewWindowPrivate *priv = MODEST_ACCOUNT_VIEW_WINDOW_GET_PRIVATE (self);
	
	gchar* account_name = modest_account_view_get_selected_account (priv->account_view);
	if (!account_name)
		return;
		
	if (check_for_active_account (self, account_name)) {
		modest_tny_account_store_show_account_settings_dialog (modest_runtime_get_account_store (), account_name);
		
	}
	
	g_free (account_name);
}

static void
on_add_button_clicked (GtkWidget *button, ModestAccountViewWindow *self)
{
	GtkWidget *assistant;
	ModestAccountViewWindowPrivate *priv;
	
	priv = MODEST_ACCOUNT_VIEW_WINDOW_GET_PRIVATE(self);
	assistant = modest_account_assistant_new (modest_runtime_get_account_mgr());
	gtk_window_set_transient_for (GTK_WINDOW(assistant),
				      GTK_WINDOW(self));

	gtk_widget_show (GTK_WIDGET(assistant));
}


static void
on_default_button_clicked (GtkWidget *button, ModestAccountViewWindow *self)
{
	ModestAccountViewWindowPrivate *priv;
	ModestAccountMgr *account_mgr;
	gchar *account_name;
	
	priv = MODEST_ACCOUNT_VIEW_WINDOW_GET_PRIVATE(self);

	account_mgr = modest_runtime_get_account_mgr();	
	account_name = modest_account_view_get_selected_account (priv->account_view);

	modest_account_mgr_set_default_account (account_mgr, account_name);

	g_free (account_name);
}


static GtkWidget*
button_box_new (ModestAccountViewWindow *self)
{

	GtkWidget *button_box;
	ModestAccountViewWindowPrivate *priv;
	
	priv = MODEST_ACCOUNT_VIEW_WINDOW_GET_PRIVATE(self);
	
	button_box   = gtk_vbutton_box_new ();
	gtk_button_box_set_spacing (GTK_BUTTON_BOX (button_box), 6);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (button_box), 
				   GTK_BUTTONBOX_START);
	
	priv->add_button     = gtk_button_new_from_stock(GTK_STOCK_ADD);
	priv->default_button = gtk_button_new_with_label(_("Make default"));
	priv->remove_button  = gtk_button_new_from_stock(GTK_STOCK_REMOVE);
	priv->edit_button    = gtk_button_new_from_stock(GTK_STOCK_EDIT);
	
	g_signal_connect (G_OBJECT(priv->add_button), "clicked",
			  G_CALLBACK(on_add_button_clicked),
			  self);
	g_signal_connect (G_OBJECT(priv->remove_button), "clicked",
			  G_CALLBACK(on_remove_button_clicked),
			  self);
	g_signal_connect (G_OBJECT(priv->edit_button), "clicked",
			  G_CALLBACK(on_edit_button_clicked),
			  self);
	g_signal_connect (G_OBJECT(priv->default_button), "clicked",
			  G_CALLBACK(on_default_button_clicked),
			  self);
	
	gtk_box_pack_start (GTK_BOX(button_box), priv->add_button, FALSE, FALSE,2);
	gtk_box_pack_start (GTK_BOX(button_box), priv->default_button, FALSE, FALSE,2);
	gtk_box_pack_start (GTK_BOX(button_box), priv->remove_button, FALSE, FALSE,2);
	gtk_box_pack_start (GTK_BOX(button_box), priv->edit_button, FALSE, FALSE,2);

	gtk_widget_set_sensitive (priv->edit_button, FALSE);
	gtk_widget_set_sensitive (priv->remove_button, FALSE);	
	gtk_widget_set_sensitive (priv->default_button, FALSE);
	
	return button_box;
}

static GtkWidget*
window_vbox_new (ModestAccountViewWindow *self)
{
	ModestAccountViewWindowPrivate *priv;
	GtkTreeSelection *sel;
	GtkWidget *main_hbox, *main_vbox, *button_box;
	GtkWidget *scrolled_window;

	priv = MODEST_ACCOUNT_VIEW_WINDOW_GET_PRIVATE(self);

	main_vbox     = gtk_vbox_new (FALSE, 0);
	main_hbox     = gtk_hbox_new (FALSE, 12);
	
	priv->account_view = modest_account_view_new (modest_runtime_get_account_mgr());
	gtk_widget_set_size_request (GTK_WIDGET(priv->account_view), 300, 400);

	sel = gtk_tree_view_get_selection (GTK_TREE_VIEW(priv->account_view));
	g_signal_connect (G_OBJECT(sel), "changed",  G_CALLBACK(on_selection_changed),
			  self);
	
	button_box = button_box_new (self);
	
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_window), GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (scrolled_window), GTK_WIDGET (priv->account_view));
	gtk_box_pack_start (GTK_BOX(main_hbox), scrolled_window, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(main_hbox), button_box, FALSE, FALSE,0);

	gtk_box_pack_start (GTK_BOX(main_vbox), main_hbox, TRUE, TRUE, 0);

	gtk_widget_show_all (main_vbox);
	return main_vbox;
}


GtkWidget*
modest_account_view_window_new (void)
{
	GObject *obj;
	ModestAccountViewWindowPrivate *priv;
	
	obj  = g_object_new(MODEST_TYPE_ACCOUNT_VIEW_WINDOW, NULL);
	priv = MODEST_ACCOUNT_VIEW_WINDOW_GET_PRIVATE(obj);

	gtk_window_set_resizable (GTK_WINDOW(obj), TRUE);
	gtk_window_set_title (GTK_WINDOW(obj), _("mcen_ti_emailsetup_accounts"));
	gtk_window_set_type_hint (GTK_WINDOW(obj), GDK_WINDOW_TYPE_HINT_DIALOG);
	gtk_window_set_default_size (GTK_WINDOW (obj), 640, 480);
	
	gtk_window_set_modal (GTK_WINDOW(obj), TRUE);
				     
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(obj)->vbox),
			    window_vbox_new (MODEST_ACCOUNT_VIEW_WINDOW(obj)),
			    TRUE, TRUE, 12);

	gtk_dialog_add_button (GTK_DIALOG (obj), GTK_STOCK_CLOSE, GTK_RESPONSE_OK);
		
	return GTK_WIDGET(obj);
}
