#include <stdlib.h>
#include <string.h>

#include "config.h"

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif

#include "buffer.h"

/**
 * init the buffer 
 * 
 */

buffer* buffer_init() {
	buffer *b;
	
	b = malloc(sizeof(buffer));
	
	b->ptr = 0;
	b->size = 0;
	b->used = 0;
	
	return b;
}

/**
 * free the buffer 
 * 
 */

void buffer_free(buffer *b) {
	if (!b) return;
	
	if (b->size) {
		free(b->ptr);
		b->size = 0;
		b->used = 0;
	}
	free(b);
}

/**
 * 
 * allocate (if neccessary) enough space for 'size' bytes and 
 * set the 'used' coutner to 0
 * 
 */

int buffer_prepare_copy(buffer *b, size_t size) {
	if (!b) return -1;
	
	if (0 == b->size) {
		b->size = size;
		b->ptr = malloc(b->size);
	} else if (size > b->size) {
		free(b->ptr);
		
		b->size = size;
		b->ptr = malloc(b->size);
	}
	b->used = 0;
	return 0;
}

/**
 * 
 * increase the internal buffer (if neccessary) to append another 'size' byte
 * ->used isn't changed
 * 
 */

int buffer_prepare_append(buffer *b, size_t size) {
	if (!b) return -1;
	
	if (0 == b->size) {
		b->size = size;
		b->ptr = malloc(b->size);
		b->used = 0;
	} else if (b->used + size > b->size) {
		b->size += size;
		b->ptr = realloc(b->ptr, b->size);
	}
	return 0;
}

int buffer_strcpy(buffer *b, const char *s) {
	size_t s_len;
	
	if (!s || !b) return -1;
	
	s_len = strlen(s);
	if (buffer_prepare_copy(b, s_len + 1)) return -1;
	
	memcpy(b->ptr, s, s_len + 1);
	b->used = s_len + 1;
	
	return 0;
}

int buffer_strcpy_len(buffer *b, const char *s, size_t s_len) {
	if (!s || !b) return -1;
	
	if (s_len == 0) return 0;
	
	if (buffer_prepare_copy(b, s_len + 1)) return -1;
	
	memcpy(b->ptr, s, s_len);
	b->ptr[s_len] = '\0';
	b->used = s_len + 1;
	
	return 0;
}


int buffer_strcat(buffer *b, const char *s) {
	size_t s_len;
	
	if (!s || !b) return -1;
	
	/* the buffer is empty, fallback to copy */
	if (b->used == 0) {
		return buffer_strcpy(b, s);
	}
	
	if (b->ptr[b->used - 1] != '\0') {
		char *p = NULL;
		/* seg-fault */
		*p = 1;
	}
	
	s_len = strlen(s);
	if (buffer_prepare_append(b, s_len)) return -1;
	
	memcpy(b->ptr + b->used - 1, s, s_len + 1);
	b->used += s_len;
	
	return 0;
}

int buffer_strcat_len(buffer *b, const char *s, size_t s_len) {
	if (!s || !b) return -1;
	
	if (s_len == 0) return 0;
	
	/* the buffer is empty, fallback to copy */
	if (b->used == 0) {
		return buffer_strcpy_len(b, s, s_len);
	}
	
	if (b->ptr[b->used - 1] != '\0') {
		char *p = NULL;
		/* seg-fault */
		*p = 1;
		
	}
	
	if (buffer_prepare_append(b, s_len)) return -1;
	
	memcpy(b->ptr + b->used - 1, s, s_len);
	b->ptr[b->used + s_len - 1] = '\0';
	b->used += s_len;
	
	return 0;
}


int ltostr(char *s, long l) {
	int i, sign = 0;
	
	if (l < 0) {
		sign = 1;
		l = -l;
	}
	
	for (i = 0; l > 9; l /= 10, i++) {
		s[i] = '0' + (l % 10);
	}
	
	s[i] = '0' + l;
	if (sign) {
		s[++i] = '-';
	}
	s[i+1] = '\0';
	
	/* swap bytes again :) */
	if (i > 0) {
		int li = i;
		for (; i > li / 2; i--) {
			char c;
			
			c = s[i];
			s[i] = s[li - i];
			s[li - i] = c;
		}
	}
	
	return 0;
}


int ultostr(char *s, unsigned long l) {
	int i;
	
	for (i = 0; l > 9; l /= 10, i++) {
		s[i] = '0' + (l % 10);
	}
	
	s[i] = '0' + l;
	s[i+1] = '\0';
	
	/* swap bytes again :) */
	if (i > 1) {
		int li = i;
		for (; i > li / 2; i--) {
			char c;
			
			c = s[i];
			s[i] = s[li - i];
			s[li - i] = c;
		}
	}
	
	return 0;
}

int ultohex(char *s, unsigned long l) {
	int i;
	
	for (i = 0; l > 15; l /= 16, i++) {
		s[i] = (l % 16) > 9 ? '0' + (l % 16) : 'a' + (l % 16) - 10;
	}
	
	s[i] = '0' + l;
	s[i+1] = '\0';
	
	/* swap bytes again :) */
	if (i > 1) {
		int li = i;
		for (; i > li / 2; i--) {
			char c;
			
			c = s[i];
			s[i] = s[li - i];
			s[li - i] = c;
		}
	}
	
	return 0;
}

