/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: plugin_config.c,v 1.38 2003/04/18 18:46:15 ostborn Exp $
*/

#include <libintl.h>
#include <locale.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>
#include <limits.h>
#include <string.h>
#include <assert.h>
#include <string.h>
#include <sys/stat.h>
#include <errno.h>

#include "mconfig.h"
#include "mstate.h"
#include "mlocale.h"
#include "mhash.h"
#include "mlist.h"
#include "mdatatypes.h"
#include "mplugins.h"
#include "misc.h"

#include "plugin_config.h"
#include "mtree.h"
#include "web.h"
#include "mail.h"

#include "datatypes/string/datatype.h"
#include "datatypes/count/datatype.h"

#define M_PLUGIN_NAME "output_template"

int mplugins_output_template_dlinit(mconfig *ext_conf) {
	config_output *conf = NULL;

	if (0 != strcmp(ext_conf->version, VERSION)) {
		M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "version string doesn't match: (mla) %s != (plugin) %s\n", ext_conf->version, VERSION);
		return -1;
	}

	conf = malloc(sizeof(config_output));
	memset(conf, 0, sizeof(config_output));

	conf->variables = mlist_init();
	conf->reports = mlist_init();
	conf->files = mlist_init();
	conf->menuentry = mlist_init();
	conf->menutitle = mlist_init();
	conf->col_circle = mlist_init();
	conf->col_vhostcircle = mlist_init();
	
	conf->tmp_buf = buffer_init();
	conf->tmp_output = buffer_init();

	ext_conf->plugin_conf = conf;

	return 0;
}

int mplugins_output_template_dlclose(mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf;
	
	buffer_free(conf->tmp_buf);
	buffer_free(conf->tmp_output);

	mlist_free(conf->variables);
	mlist_free(conf->reports);
	mlist_free(conf->files);
	mlist_free(conf->menuentry);
	mlist_free(conf->menutitle);
	mlist_free(conf->col_circle);
	mlist_free(conf->col_vhostcircle);

	mtree_free(conf->menu);

	/* these pointer are static, but have to die at the end */
	generate_web_cleanup(ext_conf);
	generate_mail_cleanup(ext_conf);

	if (conf->template_name) free(conf->template_name);
	if (conf->template_path) free(conf->template_path);
	if (conf->assumedprotocol) free(conf->assumedprotocol);
	if (conf->hostname) free(conf->hostname);
	if (conf->index_filename) free(conf->index_filename);
	if (conf->filename_pattern) free(conf->filename_pattern);

	free(ext_conf->plugin_conf);
	ext_conf->plugin_conf = NULL;

	return 0;
}

int mplugins_output_template_parse_config(mconfig *ext_conf, const char *filename, const char *section) {
	config_output *conf = ext_conf->plugin_conf;

	const mconfig_values config_values[] = {
		/* strings */
		{"template_name", M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->template_name)},
		{"template_path", M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->template_path)},
		{"assumedprotocol", M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->assumedprotocol)},
		{"hostname", M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->hostname)},
		{"index_filename", M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->index_filename)},
		{"filename_pattern", M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->filename_pattern)},
		{"outputdir", M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->outputdir)},

		/* integer */
		{"show_available_reports_and_die", M_CONFIG_TYPE_INT,	M_CONFIG_VALUE_OVERWRITE, &(conf->show_reports)},

		/* string lists */
		{"variable", M_CONFIG_TYPE_STRING_LIST, M_CONFIG_VALUE_APPEND, &(conf->variables)},
		{"report", M_CONFIG_TYPE_STRING_LIST, M_CONFIG_VALUE_APPEND, &(conf->reports)},
		{"menuentry", M_CONFIG_TYPE_STRING_LIST, M_CONFIG_VALUE_APPEND, &(conf->menuentry)},
		{"menutitle", M_CONFIG_TYPE_STRING_LIST, M_CONFIG_VALUE_APPEND, &(conf->menutitle)},


		{NULL, M_CONFIG_TYPE_INT, 0, NULL}
	};

	return mconfig_parse_section(ext_conf, filename, section, config_values);
}

/**
 * get the title for a menu section
 *
 * checks the reports titles and the configfile options for menu-title
 *
 * @return pointer to the string (no copy !!)
 */

const char *get_menu_title (mconfig *ext_conf, const tmpl_reports * reports, const char *name) {
	config_output *conf = ext_conf->plugin_conf;
	mlist *l;
	int i;


	for (i = 0; reports[i].key; i++) {
		if (0 == strcmp(reports[i].key, name))
			return reports[i].title;
	}

	for (l = conf->menutitle; l && l->data; l = l->next) {
		char *parent, *child;
		
		/* split string */
		parent = strdup(l->data->key);

		if ((child = strchr(parent, ',')) == NULL) {
			return NULL;
		}
		/* remove the ',' and finish the parent-field */
		*child++ = '\0';

		if (0 == strcmp(parent, name)) {
			free(parent);
			/* let 'child' point the original string as that one is persistent */
			child = l->data->key + (child - parent);

			/* remove starting whitespaces */
			while (*child == ' ') child++;

			return child;
		} else {
			free(parent);
		}
	}

	return NULL;
}

int prepare_menu_structure(mconfig *ext_conf, tmpl_reports *reports) {
	config_output *conf = ext_conf->plugin_conf;
	mlist *l;
	mtree *menu = mtree_init();
	assert(menu);

	for (l = conf->menuentry; l && l->data; l = l->next) {
		char *parent, *child;
		mtree *t;
		
		/* split string */
		parent = strdup(l->data->key);

		if ((child = strchr(parent, ',')) == NULL) {
			return -1;
		}
		/* remove the ',' and finish the parent-field */
		*child++ = '\0';

		/* remove starting whitespaces */
		while (*child == ' ') child++;

		if (mtree_is_empty(menu)) {
			/* first element */
			mdata *data;
			
			/* name -- title */
			data = mdata_String_create(parent, get_menu_title(ext_conf, reports, parent) /* title */ );

			menu->data = data;
		}

		if ((t = mtree_search(menu, parent))) {
			mdata * data;
			mtree * cd;

			cd = mtree_init();

			/* retreive the menutitle */

			data = mdata_String_create(child, get_menu_title(ext_conf, reports, child) /* title */ );
			cd->data = data;

			mtree_add_child(t, cd);
		} else {
			fprintf(stderr, "%s.%d: parent '%s' not found in menu-tree\n",
				__FILE__, __LINE__, parent);
			return -1;
		}

		free(parent);
	}

	conf->menu = menu;

	return 0;
}

int mplugins_output_template_set_defaults(mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf;
	char *fn;
	int ret, i;
	char *str;

	const mconfig_values config_values[] = {
		/* strings */
		{"menu", M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->tmpl_menu)},  /* filename of the template for the menu */
		{"table", M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->tmpl_table)}, /* filename of the template for the inner tables */
		{"outer", M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->tmpl_outer)}, /* filename of the template for the framework itself */
		{"index", M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->tmpl_index)}, /* filename of the template for the index file */

		/* colors */
		{"col_background",	M_CONFIG_TYPE_COLTRIPPL, M_CONFIG_VALUE_OVERWRITE, &(conf->col_backgnd)},
		{"col_foreground",	M_CONFIG_TYPE_COLTRIPPL, M_CONFIG_VALUE_OVERWRITE, &(conf->col_foregnd)},
		{"col_shadow",	M_CONFIG_TYPE_COLTRIPPL, M_CONFIG_VALUE_OVERWRITE, &(conf->col_shadow)},
		{"col_border",	M_CONFIG_TYPE_COLTRIPPL, M_CONFIG_VALUE_OVERWRITE, &(conf->col_border)},
		{"col_pages",	M_CONFIG_TYPE_COLTRIPPL, M_CONFIG_VALUE_OVERWRITE, &(conf->col_pages)},
		{"col_files",	M_CONFIG_TYPE_COLTRIPPL, M_CONFIG_VALUE_OVERWRITE, &(conf->col_files)},
		{"col_visits",	M_CONFIG_TYPE_COLTRIPPL, M_CONFIG_VALUE_OVERWRITE, &(conf->col_visits)},
		{"col_xfer",	M_CONFIG_TYPE_COLTRIPPL, M_CONFIG_VALUE_OVERWRITE, &(conf->col_xfer)},
		{"col_hits",	M_CONFIG_TYPE_COLTRIPPL, M_CONFIG_VALUE_OVERWRITE, &(conf->col_hits)},

		/* flat menu flag */
		{"flat_menu", M_CONFIG_TYPE_INT,	M_CONFIG_VALUE_OVERWRITE, &(conf->flat_menu)},

		/* cell classnames */
		{"cell_class_title",       M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_title)},
		{"cell_class_leftheader",  M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_leftheader)},
		{"cell_class_header",      M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_header)},
		{"cell_class_rightheader", M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_rightheader)},
		{"cell_class_leftline",    M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_leftline)},
		{"cell_class_line",        M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_line)},
		{"cell_class_rightline",   M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_rightline)},
		{"cell_class_leftgline",   M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_leftgline)},
		{"cell_class_gline",       M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_gline)},
		{"cell_class_rightgline",  M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_rightgline)},
		{"cell_class_leftfooter",  M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_leftfooter)},
		{"cell_class_footer",      M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_footer)},
		{"cell_class_rightfooter", M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_rightfooter)},
		{"cell_class_lefthline",   M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_lefthline)},
		{"cell_class_hline",       M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_hline)},
		{"cell_class_righthline",  M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_class_righthline)},

		/* cell tags */
		{"cell_tags_title",       M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_title)},
		{"cell_tags_leftheader",  M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_leftheader)},
		{"cell_tags_header",      M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_header)},
		{"cell_tags_rightheader", M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_rightheader)},
		{"cell_tags_leftline",    M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_leftline)},
		{"cell_tags_line",        M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_line)},
		{"cell_tags_rightline",   M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_rightline)},
		{"cell_tags_leftgline",   M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_leftgline)},
		{"cell_tags_gline",       M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_gline)},
		{"cell_tags_rightgline",  M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_rightgline)},
		{"cell_tags_leftfooter",  M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_leftfooter)},
		{"cell_tags_footer",      M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_footer)},
		{"cell_tags_rightfooter", M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_rightfooter)},
		{"cell_tags_lefthline",   M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_lefthline)},
		{"cell_tags_hline",       M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_hline)},
		{"cell_tags_righthline",  M_CONFIG_TYPE_STRING, M_CONFIG_VALUE_OVERWRITE, &(conf->cell_tags_righthline)},

		/* mlist - StrInt */
		{"country_circle",	M_CONFIG_TYPE_STRING_LIST, M_CONFIG_VALUE_APPEND, &(conf->col_circle)},
		{"vhost_circle",        M_CONFIG_TYPE_STRING_LIST, M_CONFIG_VALUE_APPEND, &(conf->col_vhostcircle)},

		/* string lists */
		{"files", M_CONFIG_TYPE_STRING_LIST, M_CONFIG_VALUE_APPEND, &(conf->files)}, /* name the files which have to be copied to the outputdir */

		{NULL, M_CONFIG_TYPE_INT, 0, NULL}
	};

	for (i = 0; i < M_TMPL_MAX_REPORTS; i++) {
		conf->avail_reports[i].key = NULL;
		conf->avail_reports[i].title = NULL;
		conf->avail_reports[i].func = NULL;
	}

	/* register reports */
	register_reports_web(ext_conf, conf->avail_reports);
	register_reports_mail(ext_conf, conf->avail_reports);

	if (conf->show_reports) {
		fprintf(stderr, "-- 'displaying the available reports & die' feature enabled\n");
		for (i = 0; i < M_TMPL_MAX_REPORTS; i++) {
			if (conf->avail_reports[i].key) {
				fprintf(stderr, "-- %2d. %s\n", i
					, conf->avail_reports[i].key);
			}
		}

		fprintf(stderr, "-- done - doing down now\n");
		return -1;
	}

	/* use the patch-functions from generate.c */


	if (NULL == (str = (mconfig_get_value(ext_conf, conf->template_name)))) {
		fprintf(stderr, "ERROR: [%s] no template name is set ( template_name = ... )\n",
			M_PLUGIN_NAME);
		return -1;
	}
	free(str);

	if (NULL == (str = (mconfig_get_value(ext_conf, conf->template_path)))) {
		fprintf(stderr, "ERROR: [%s] no path to the templates is set ( template_path = ... )\n",
			M_PLUGIN_NAME);
		return -1;
	}
	free(str);

	if (NULL == conf->hostname) {
		fprintf(stderr, "ERROR: [%s] no hostname is specified ( hostname = ... )\n",
			M_PLUGIN_NAME);
		return -1;
	}
	
	if (NULL == conf->assumedprotocol) {
		fprintf(stderr, "ERROR: [%s] no hostname is specified ( assumedprotocol = ... )\n",
			M_PLUGIN_NAME);
		return -1;
	}

	if (NULL == (str = (mconfig_get_value(ext_conf, conf->outputdir)))) {
		fprintf(stderr, "ERROR: [%s] no output-directory was set ( outputdir = ... )\n",
			M_PLUGIN_NAME);
		return -1;
	}

	if (dir_check_perms(str)) {
		return -1;
	}
	free(str);

	/* load the config file for the theme */
	fn = malloc(strlen(conf->template_name) + strlen(conf->template_path) + strlen("theme.conf") + 2 + 1);

	sprintf(fn, "%s/%s/theme.conf",
		conf->template_path,
		conf->template_name);

	/* load template */
	if ((ret = mconfig_parse_section(ext_conf, fn, conf->template_name, config_values)) != 0) {
		free(fn);
		return ret;
	}

	free(fn);

	if (prepare_menu_structure(ext_conf, conf->avail_reports)) {
		fprintf(stderr, "[%s] preparing menu-structure failed\n",
			M_PLUGIN_NAME
			);
		return -1;
	}
	
	if (NULL == (str = (mconfig_get_value(ext_conf, conf->index_filename)))) {
		fprintf(stderr, "ERROR: [%s] index_filename is not set. Please include 'output_template' from modlogan.def.conf.\n",
			M_PLUGIN_NAME);
		return -1;
	}
	free(str);
	
	if (NULL == (str = (mconfig_get_value(ext_conf, conf->filename_pattern)))) {
		fprintf(stderr, "ERROR: [%s] filename_pattern is not set. Please include 'output_template' from modlogan.def.conf.\n",
			M_PLUGIN_NAME);
		return -1;
	}
	free(str);

	/* check colors */

	return 0;
}

int mplugins_init(mplugin *func) {
	func->dlinit = mplugins_output_template_dlinit;
	func->dlclose = mplugins_output_template_dlclose;
	func->parse_config = mplugins_output_template_parse_config;
	func->set_defaults = mplugins_output_template_set_defaults;
	func->get_next_record = NULL;
	func->insert_record = NULL;
	func->gen_report = mplugins_output_generate_monthly_output;
	func->gen_history = mplugins_output_generate_history_output;

	return 0;
}

/**
 * handle the global-variable in the config
 *
 * add a shadow-config which holds the original, unpatched values
 * and include the transformed value in the normal config
 *
 * @see mplugin_output_template_unpatch_config
 */

int mplugins_output_template_patch_config(mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf, *orig_conf;
	mlist *l;

	if (conf->old_conf == NULL) {
		orig_conf = malloc(sizeof(config_output));
		memset(orig_conf, 0, sizeof(config_output));
#define PATCH(x) \
		orig_conf->x = conf->x; conf->x = mconfig_get_value(ext_conf, orig_conf->x);

		PATCH(template_name);
		PATCH(template_path);
		PATCH(assumedprotocol);
		PATCH(hostname);
		PATCH(index_filename);
		PATCH(outputdir);

		orig_conf->variables = conf->variables;

		conf->variables = mlist_init();
		for (l = orig_conf->variables; l && l->data; l = l->next) {
			mdata * data;
			char *s;
			
			s = mconfig_get_value(ext_conf, l->data->key);

			data = mdata_Count_create(s, 1, M_DATA_STATE_PLAIN);
			mlist_insert(conf->variables, data);

			free(s);
		}


		conf->old_conf = orig_conf;
#undef PATCH
		return 1;
	} else {
		return 0;
	}
}

/**
 * restore the original config
 *
 * @see mplugin_output_template_unpatch_config
 */

int mplugins_output_template_unpatch_config(mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf, *orig_conf;

	if (conf->old_conf != NULL) {
		orig_conf = conf->old_conf;
#define PATCH(x) \
		if (conf->x) free(conf->x); conf->x = orig_conf->x;

		PATCH(template_name);
		PATCH(template_path);
		PATCH(assumedprotocol);
		PATCH(hostname);
		PATCH(index_filename);
		PATCH(outputdir);

		mlist_free(conf->variables); conf->variables = orig_conf->variables;

		free(orig_conf);

		conf->old_conf = NULL;
#undef PATCH
		return 1;
	} else {
		return 0;
	}
}
