/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: generate.c,v 1.39 2003/01/06 11:19:21 ostborn Exp $
*/

#include <libintl.h>
#include <locale.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>

#include "mconfig.h"
#include "misc.h"
#include "mstate.h"
#include "mlocale.h"
#include "mhash.h"
#include "mlist.h"
#include "mdatatypes.h"
#include "datatypes/count/datatype.h"
#include "mplugins.h"

#include "pictures.h"
#include "plugin_config.h"

#define HIGHLIGHT	1
#define GROUPING	2
#define VISITS		4
#define INDEX		8
#define NO_VCOUNT	16
#define COUNTRY 	32

int mhash_get_value(mhash *h, const char *key) {
	int c = 0, i;
	if (!h) return 0;

	for ( i = 0; i < h->size; i++) {
		mlist *l;
		
		for (l = h->data[i]->list; l && l->data; l = l->next) {
			mdata *data = l->data;
			
			if (0 == strcmp(key, data->key)) {
				return mdata_get_count(data);
			}
		}
		if (c != 0) break;
	}

	return c;
}

int show_mhash (mconfig *ext_conf, FILE *f, mhash *h, int count, int opt) {
	mlist *l, *first;
	int i = 0;
	config_output *conf = ext_conf->plugin_conf;

	if (!h) return 0;

	first = l = mlist_init();

	mhash_unfold_sorted_limited(h, l, count);

	while (l && (i++ < count)) {
		mdata *data = l->data;

		if (data) {
			char *enc_url;
			int cut_url;
			
			enc_url = html_encode(data->key);
			cut_url = strlen(enc_url) > 40;

			if (cut_url) {
				enc_url[40] = '\0';
			}

			fprintf(f,"<TR>");

			if (opt & INDEX) {
				fprintf(f,"<TD width=\"5%%\" align=right>%d</TD>", i);
			}

			fprintf(f,"<TD width=\"15%%\" align=right>%d</TD>", mdata_get_count(data));
			if (!(opt & NO_VCOUNT)) {
				fprintf(f,"<TD width=\"15%%\" align=right>%.0f</TD>", mdata_get_vcount(data));
			}

			if ((opt & GROUPING) &&
			    mdata_is_grouped(data)) {
				fprintf(f,"<TD bgcolor=\"%s\">%s%s</TD>", conf->col_grouping, enc_url, cut_url ? "..." : "");
			} else {
				if (opt & HIGHLIGHT) {
					fprintf(f,"<TD><a href=\"%s\">%s</a>%s</TD>", data->key, enc_url, cut_url ? "..." : "");
				} else if (opt & COUNTRY) {
					char *c = html_encode(misoname(data->key));
					fprintf(f,"<td>%s</td>\n", c);
					free(c);
				} else {
					fprintf(f,"<TD>%s%s</TD>", enc_url, cut_url ? "..." : "");
				}
			}
			fprintf(f,"</TR>\n");

			free(enc_url);
		}

		l = l->next;
	}

	mlist_free(first);

	return 0;
}

int mhash_status_unfold_sorted_limited(mhash *h, mlist *l, int count ) {
	int i, j;
	mdata *data, *ins_data;
	char __dummy__[] = "999";
	char __dummy_2_[] = "";
	char *max, *last = __dummy_2_;

	for ( j = 0; j < count; j++) {
		data = NULL;

		max = __dummy__;

		for ( i = 0; i < h->size; i++) {
			if (h->data[i]->list) {
				mlist *hl;

				for (hl = h->data[i]->list; hl && hl->data; hl = hl->next) {
					if ( strcmp(hl->data->key, max) < 0 &&
					     strcmp(hl->data->key, last) > 0) {
						max = hl->data->key;
						data = hl->data;
					}
				}
			}
		}

		if (data) {
			if (data->type == M_DATA_TYPE_COUNT) {
				ins_data = mdata_Count_create(data->key, data->data.count.count, M_DATA_STATE_PLAIN);

				mlist_insert(l, ins_data);
			} else {
				fprintf(stderr, "%s.%d: ARGS\n", __FILE__, __LINE__);
			}
			last = data->key;
		}
	}

	return 0;
}

int show_status_mhash (FILE *f, mhash *h, int count) {
	mlist *l, *first;
	int i = 0;

	if (!h) return 0;

	first = l = mlist_init();

	mhash_status_unfold_sorted_limited(h, l, count);

	while (l && (i++ < count)) {
		mdata *data = l->data;

		if (data) {
			fprintf(f,"<TR><TD width=\"15%%\" align=right>%i</TD><TD>%s - %s</TD></TR>\n", data->data.count.count, data->key, mhttpcodes(strtol(data->key, NULL, 10)));
		}

		l = l->next;
	}

	mlist_free(first);

	return 0;
}

void table_start(FILE *f, char *str, int colspan) {
	fprintf(f,"<P><CENTER><TABLE BORDER=1 WIDTH=400 BGCOLOR=\"#eeeeee\"><TR><TH colspan=%i>%s</TH></TR>\n", colspan, str);
}

void table_end(FILE *f) {
	fprintf(f,"</TABLE></CENTER><P>");
}

void file_start(FILE *f, mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf;

	fprintf(f, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">\n\n");
	fprintf(f, "<HTML><HEAD><TITLE>Statistics</TITLE></HEAD><meta name=\"ROBOTS\" content=\"NOINDEX, NOARCHIVE, NOFOLLOW\">\n<BODY bgcolor=\"%s\">\n", conf->col_body);

	fprintf(f, "%s %s<HR><br>\n", _("Statistics for"), conf->hostname);
}

void file_end(FILE *f) {
/* Tell the users that this is valid HTML 4.0  :) */
	fprintf(f, "<HR><a href=\"http://validator.w3.org/check/referer\"><img border=0 "\
		"src=\"http://validator.w3.org/images/vh40\" "\
		"alt=\"Valid HTML 4.0!\" height=31 width=88 align=\"right\"></a>");
	fprintf(f, "Output generated by <a href=\"%s\">%s %s</a>\n", "http://www.modlogan.org/", PACKAGE, VERSION);
	fprintf(f, "</BODY></HTML>\n");
}

char *table_header(int maxcount, int count, char *str) {
	static char trans_buf[255];

	sprintf(trans_buf, "%i %s %i %s", (maxcount > count) || (maxcount < 0) ? count : maxcount, _("of"), count, str);

	return trans_buf;

}

int mplugins_output_webalizer_generate_monthly_output(mconfig *ext_conf, mstate *state, const char *subpath) {
	unsigned int i, s_200, s_304;
	double d = 0;
	FILE *f;
	char filename[255];
	char *ref;
	data_WebHistory sumdat, maxdat;
	int last_day = 1;
	config_output *conf = ext_conf->plugin_conf;

	mstate_web *staweb = NULL;

	if (!state->ext) return -1;

	if (state->ext_type != M_STATE_TYPE_WEB) return -1;

	staweb = state->ext;

	if (subpath) {
		sprintf(filename, "%s/%s/",
			ext_conf->plugin_conf ? conf->outputdir : ".",
			subpath);
		mkdir(filename, 0755);

		sprintf(filename, "%s/%s/m_usage_%04i%02i.html",
			conf->outputdir ? conf->outputdir : ".",
			subpath,
			state->year, state->month);
	} else {
		sprintf(filename, "%s/m_usage_%04i%02i.html",
			conf->outputdir ? conf->outputdir : ".",
			state->year, state->month);
	}

	if (!(f = fopen(filename, "w"))) {
		return -1;
	}

	file_start(f,ext_conf);

	sumdat.files	= maxdat.files		= 0;
	sumdat.xfersize	= maxdat.xfersize	= 0;
	sumdat.hits	= maxdat.hits		= 0;
	sumdat.hosts	= maxdat.hosts		= 0;
	sumdat.pages	= maxdat.pages		= 0;
	sumdat.visits	= maxdat.visits		= 0;

	/* count the values */
	for ( i = 0; i < 31; i++) {
		if (staweb->days[i].hits) last_day = i+1;
		sumdat.files	+= staweb->days[i].files;
		sumdat.xfersize	+= staweb->days[i].xfersize;
		sumdat.hits	+= staweb->days[i].hits;
		sumdat.hosts	+= staweb->days[i].hosts;
		sumdat.pages	+= staweb->days[i].pages;
		sumdat.visits	+= staweb->days[i].visits;

		if (maxdat.files < staweb->days[i].files)
			maxdat.files	= staweb->days[i].files;
		if (maxdat.hits < staweb->days[i].hits)
			maxdat.hits	= staweb->days[i].hits;
		if (maxdat.hosts < staweb->days[i].hosts)
			maxdat.hosts	= staweb->days[i].hosts;
		if (maxdat.pages < staweb->days[i].pages)
			maxdat.pages	= staweb->days[i].pages;
		if (maxdat.visits < staweb->days[i].visits)
			maxdat.visits	= staweb->days[i].visits;
		if (maxdat.xfersize < staweb->days[i].xfersize)
			maxdat.xfersize	= staweb->days[i].xfersize;
	}



	table_start(f, _("Summary"), 3);

	/* Totals */
	fprintf(f,"<TR><TD>%s</TD><TD colspan=2 align=right>%li</TD></TR>\n", _("Total Hits"), sumdat.hits);
	fprintf(f,"<TR><TD>%s</TD><TD colspan=2 align=right>%li</TD></TR>\n", _("Total Files"), sumdat.files);
	fprintf(f,"<TR><TD>%s</TD><TD colspan=2 align=right>%li</TD></TR>\n", _("Total Pages"), sumdat.pages);
/*	fprintf(f,"<TR><TD>%s</TD><TD colspan=2 align=right>%li</TD></TR>\n", _("Total Hosts"), sumdat.hosts);*/
	fprintf(f,"<TR><TD>%s</TD><TD colspan=2 align=right>%li</TD></TR>\n", _("Total Visits"), sumdat.visits);
	fprintf(f,"<TR><TD>%s</TD><TD colspan=2 align=right>%.0f</TD></TR>\n", _("Transfered KBytes"), sumdat.xfersize / 1024);

	fprintf(f,"<TR><TH>&nbsp;</TH><TH>%s</TH><TH>%s</TH></TR>\n", _("avg"), _("max"));

	fprintf(f,"<TR><TD>%s</TD><TD align=right>%li</TD><TD align=right>%li</TD></TR>\n", _("Hits per Day"), sumdat.hits / last_day, maxdat.hits);
	fprintf(f,"<TR><TD>%s</TD><TD align=right>%li</TD><TD align=right>%li</TD></TR>\n", _("Files per Day"), sumdat.files / last_day, maxdat.files);
	fprintf(f,"<TR><TD>%s</TD><TD align=right>%li</TD><TD align=right>%li</TD></TR>\n", _("Pages per Day"), sumdat.pages / last_day, maxdat.pages);
/*	fprintf(f,"<TR><TD>%s</TD><TD align=right>%li</TD><TD align=right>%li</TD></TR>\n", _("Hosts per Day"), sumdat.hosts / last_day, maxdat.hosts);*/
	fprintf(f,"<TR><TD>%s</TD><TD align=right>%li</TD><TD align=right>%li</TD></TR>\n", _("Visits per Day"), sumdat.visits / last_day, maxdat.visits);
	fprintf(f,"<TR><TD>%s</TD><TD align=right>%.0f</TD><TD align=right>%.0f</TD></TR>\n", _("Transfered Kbytes per Day"), (sumdat.xfersize / 1024) / last_day, maxdat.xfersize / 1024);
#if 0
	if (sumdat.visits) {
		d = (staweb->allvisittime / sumdat.visits);
		min = d / 60;
		sec = (int)floor(d) % 60;
	} else {
		min = 0;
		sec = 0;
	}
	fprintf(f,"<TR><TD>%s</TD><TD align=right>%i:%02i %s</TD><TD align=right>%s</TD></TR>\n", _("Time per visit"), min, sec, _("min"), "---");

	if (sumdat.visits) {
		d = (double)staweb->allvisitlength / sumdat.visits;
	} else {
		d = 0;
	}
	fprintf(f,"<TR><TD>%s</TD><TD align=right>%.2f</TD><TD align=right>%s</TD></TR>\n", _("Pages per visit"), d, "---");
#endif
	s_200 = mhash_get_value(staweb->status_hash, "200");
	s_304 = mhash_get_value(staweb->status_hash, "304");

	d = ((double)s_304/(s_200+s_304)) * 100;
	fprintf(f,"<TR><TD>%s</TD><TD align=right>%.2f%%</TD><TD align=right>%s</TD></TR>\n", _("Cache Hit ratio"), d, "---");

	table_end(f);

	ref = create_pic_31_day(ext_conf, state, subpath);

	if (ref && strlen(ref)) {
		fprintf(f, "%s", ref);
	}



	table_start(f, _("Daily Statistics"), 6);
	fprintf(f,"<TR><TH>%s</TH><TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH></TR>\n",
		_("Day"),
		conf->col_hits,
		_("Hits"),
		conf->col_files,
		_("Files"),
		conf->col_pages,
		_("Pages"),
		conf->col_visits,
		_("Visits"),
		conf->col_xfer,
		_("KBytes")
		);
	for ( i = 0; i < last_day; i++) {
		fprintf(f,"<TR><TD>%i</TD><TD align=right>%li</TD><TD align=right>%li</TD><TD align=right>%li</TD><TD align=right>%li</TD><TD align=right>%.0f</TD></TR>\n",
			i+1,
			staweb->days[i].hits,
			staweb->days[i].files,
			staweb->days[i].pages,
			staweb->days[i].visits,
			staweb->days[i].xfersize / 1024
			);
	}
	table_end(f);

	ref = create_pic_24_hour(ext_conf, state, subpath);

	if (ref && strlen(ref)) {
		fprintf(f, "%s", ref);
	}

	table_start(f, _("Hourly Statistics"), 6);
	fprintf(f,"<TR><TH>%s</TH><TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH></TR>\n",
		_("Hour"),
		conf->col_hits,
		_("Hits"),
		conf->col_files,
		_("Files"),
		conf->col_pages,
		_("Pages"),
		conf->col_visits,
		_("Visits"),
		conf->col_xfer,
		_("KBytes")
		);
	for ( i = 0; i < 24; i++) {
		fprintf(f,"<TR><TD>%i</TD><TD align=right>%li</TD><TD align=right>%li</TD><TD align=right>%li</TD><TD align=right>%li</TD><TD align=right>%.0f</TD></TR>\n",
			i,
			staweb->hours[i].hits,
			staweb->hours[i].files,
			staweb->hours[i].pages,
			staweb->hours[i].visits,
			staweb->hours[i].xfersize / 1024
			);
	}
	table_end(f);

	if (conf->max_req_urls > 0) {
		table_start(f, table_header(conf->max_req_urls, mhash_count(staweb->req_url_hash),_("Requested URL's")), 3);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			_("URL")
			);
		show_mhash(ext_conf, f,staweb->req_url_hash, conf->max_req_urls, HIGHLIGHT | GROUPING | INDEX | NO_VCOUNT);
		table_end(f);
	}

	if (conf->max_os > 0) {
		table_start(f, table_header(conf->max_os, mhash_count(staweb->os_hash), _("Used Operating Systems")), 4);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			conf->col_visits,
			_("Visits"),
			_("Operating System")
			);
		show_mhash(ext_conf, f,staweb->os_hash, conf->max_os, GROUPING | VISITS | INDEX);
		table_end(f);
	}

	if (conf->max_hosts > 0) {
		table_start(f, table_header(conf->max_hosts, mhash_count(staweb->host_hash),_("Hosts")), 4);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			conf->col_visits,
			_("Visits"),
			_("Host")
			);
		show_mhash(ext_conf, f,staweb->host_hash, conf->max_hosts, GROUPING | INDEX);
		table_end(f);
	}

	if (conf->max_ref_urls > 0) {
		table_start(f, table_header(conf->max_ref_urls, mhash_count(staweb->ref_url_hash),_("Referring URL's")), 3);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			_("Referrer")
			);
		show_mhash(ext_conf, f,staweb->ref_url_hash, conf->max_ref_urls, HIGHLIGHT | GROUPING | INDEX);
		table_end(f);
	}
#if 0
	if (conf->max_entry_pages > 0) {
		table_start(f, table_header(conf->max_entry_pages, mhash_count(staweb->entry_pages),_("Entry Pages")), 3);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			_("Entry Page")
			);
		show_mhash(ext_conf, f,staweb->entry_pages, conf->max_entry_pages, HIGHLIGHT | INDEX);
		table_end(f);
	}

	if (conf->max_exit_pages > 0) {
		table_start(f, table_header(conf->max_exit_pages, mhash_count(staweb->exit_pages),_("Exit Pages")), 3);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			_("Exit Page")
			);
		show_mhash(ext_conf, f,staweb->exit_pages, conf->max_exit_pages, HIGHLIGHT | INDEX);
		table_end(f);
	}
#endif
	if (conf->max_indexed_pages > 0) {
		table_start(f, table_header(conf->max_indexed_pages, mhash_count(staweb->indexed_pages),_("Indexed Pages")), 3);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			_("Indexed Page")
			);
		show_mhash(ext_conf, f,staweb->indexed_pages, conf->max_indexed_pages,HIGHLIGHT | INDEX);
		table_end(f);
	}

	if (conf->max_ua > 0) {
		table_start(f, table_header(conf->max_ua, mhash_count(staweb->ua_hash),_("Used Browsers")), 4);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			conf->col_visits,
			_("Visits"),
			_("Browser")
			);
		show_mhash(ext_conf, f,staweb->ua_hash, conf->max_ua, GROUPING | VISITS | INDEX);
		table_end(f);
	}

	if (conf->max_req_prot > 0) {
		table_start(f, table_header(conf->max_req_prot, mhash_count(staweb->req_prot_hash),_("Used Request Protocol")), 3);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			_("Protocol")
			);
		show_mhash(ext_conf, f,staweb->req_prot_hash, conf->max_req_prot,0 | INDEX);
		table_end(f);
	}

	if (conf->max_req_meth > 0) {
		table_start(f, table_header(conf->max_req_meth, mhash_count(staweb->req_meth_hash),_("Used Request Method")), 3);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			_("Method")
			);
		show_mhash(ext_conf, f,staweb->req_meth_hash,conf->max_req_meth,0 | INDEX);
		table_end(f);
	}

	if (conf->max_status_codes > 0) {
		table_start(f, table_header(conf->max_status_codes, mhash_count(staweb->status_hash),_("Status Code")), 2);
		fprintf(f,"<TR><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			_("Status Code")
			);
		show_status_mhash(f,staweb->status_hash, conf->max_status_codes);
		table_end(f);
	}

	if (conf->max_robots > 0) {
		table_start(f, table_header(conf->max_robots, mhash_count(staweb->robots),_("Robots")), 3);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			_("Robot")
			);
		show_mhash(ext_conf, f,staweb->robots, conf->max_robots,0 | INDEX);
		table_end(f);
	}

	if (conf->max_bookmarks > 0) {
		table_start(f, table_header(conf->max_bookmarks, mhash_count(staweb->bookmarks),_("Bookmarked Pages")), 3);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			_("Bookmarked Page")
			);
		show_mhash(ext_conf, f,staweb->bookmarks,conf->max_bookmarks,HIGHLIGHT | INDEX);
		table_end(f);
	}

	if (conf->max_broken_links > 0) {
		table_start(f, table_header(conf->max_broken_links, mhash_count(staweb->status_missing_file),_("Missing File / Broken Link")), 3);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			_("Broken Link")
			);
		show_mhash(ext_conf, f, staweb->status_missing_file, conf->max_broken_links, GROUPING | INDEX);
		table_end(f);
	}

	if (conf->max_internal_errors > 0) {
		table_start(f, table_header(conf->max_internal_errors, mhash_count(staweb->status_internal_error),_("Internal Errors")), 2);
		show_mhash(ext_conf, f,staweb->status_internal_error,conf->max_internal_errors,0);
		table_end(f);
	}

	if (conf->max_search_strings > 0) {
		table_start(f, table_header(conf->max_search_strings, mhash_count(staweb->searchstring),_("SearchStrings")), 3);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			_("Search String")
			);
		show_mhash(ext_conf, f,staweb->searchstring, conf->max_search_strings,0 | INDEX);
		table_end(f);
	}

	if (conf->max_search_engines > 0) {
		table_start(f, table_header(conf->max_search_engines, mhash_count(staweb->searchsite),_("SearchEngines")), 3);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			_("Search Engine")
			);
		show_mhash(ext_conf, f,staweb->searchsite, conf->max_search_engines, HIGHLIGHT | GROUPING | INDEX);
		table_end(f);
	}

	if (conf->max_countries > 0) {
		ref = create_pic_countries(ext_conf, state, subpath);

		if (ref && strlen(ref)) {
			fprintf(f, "%s", ref);
		}
		table_start(f, table_header(conf->max_countries, mhash_count(staweb->country_hash),_("Countries")), 4);
		fprintf(f,"<TR><TH>#</TH><TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH><TH>%s</TH></TR>\n",
			conf->col_hits,
			_("Hits"),
			conf->col_visits,
			_("Visits"),
			_("Country")
			);
		show_mhash(ext_conf, f, staweb->country_hash, conf->max_countries, VISITS | INDEX | COUNTRY);
		table_end(f);
	}

	file_end(f);

	fclose(f);

	return 0;
}

int mplugins_output_webalizer_generate_history_output(mconfig *ext_conf, mlist *history, const char *subpath) {
	mlist *l = history;
	FILE *f;
	char filename[255];
	char *ref;
	config_output *conf = ext_conf->plugin_conf;

	sprintf(filename, "%s%s%s/index.html",
		conf->outputdir ? conf->outputdir : ".",
		subpath ? "/" : "",
		subpath ? subpath : "" );

	if (!(f = fopen(filename, "w"))) {
		return -1;
	}

	file_start(f,ext_conf);

	ref = create_pic_12_month(ext_conf, history, subpath);

	if (ref && strlen(ref)) {
		fprintf(f, "%s", ref);
	}

	table_start(f, _("History"), 11);

	fprintf(f,"<TR><TH>&nbsp;</TH><TH colspan=5>%s</TH><TH colspan=5>%s</TH></TR>",
		_("Average/day"),
		_("Totals")
		);
	fprintf(f,"<TR><TH>%s</TH>" \
		"<TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH>" \
		"<TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH>" \
		"<TH bgcolor=\"%s\">%s</TH>" \
		"<TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH>" \
		"<TH bgcolor=\"%s\">%s</TH><TH bgcolor=\"%s\">%s</TH>" \
		"<TH bgcolor=\"%s\">%s</TH>" \
		"</TR>\n",
		_("Month"),
		conf->col_hits,
		_("Hits"),
		conf->col_files,
		_("Files"),
		conf->col_pages,
		_("Pages"),
		conf->col_visits,
		_("Visits"),
		conf->col_xfer,
		_("KBytes"),
		conf->col_hits,
		_("Hits"),
		conf->col_files,
		_("Files"),
		conf->col_pages,
		_("Pages"),
		conf->col_visits,
		_("Visits"),
		conf->col_xfer,
		_("KBytes")
		);

	/* go to the last element */
	while (l->next) l = l->next;

	while (l) {
		mdata *data = l->data;

		if (!data) break;

		fprintf(f,"<TR><TD><A HREF=\"m_usage_%04i%02i.html\">%s %04i</A></TD>" \
			"<TD align=\"right\">%li</TD><TD align=\"right\">%li</TD>" \
			"<TD align=\"right\">%li</TD><TD align=\"right\">%li</TD>" \
			"<TD align=\"right\">%.0f</TD>" \
			"<TD align=\"right\">%li</TD><TD align=\"right\">%li</TD>" \
			"<TD align=\"right\">%li</TD><TD align=\"right\">%li</TD>" \
			"<TD align=\"right\">%.0f</TD>" \
			"</TR>\n",
			data->data.hist->year,
			data->data.hist->month,
			get_month_string(data->data.hist->month,1),
			data->data.hist->year,

			data->data.hist->data.web.hits / data->data.hist->days_passed,
			data->data.hist->data.web.files / data->data.hist->days_passed,
			data->data.hist->data.web.pages/ data->data.hist->days_passed,
			data->data.hist->data.web.visits/ data->data.hist->days_passed,
			data->data.hist->data.web.xfersize / 1024 / data->data.hist->days_passed,
			data->data.hist->data.web.hits,
			data->data.hist->data.web.files,
			data->data.hist->data.web.pages,
			data->data.hist->data.web.visits,
			data->data.hist->data.web.xfersize / 1024

			);
		l = l->prev;
	}

	table_end(f);

	file_end(f);

	fclose(f);

	return 0;
}
