// 
// MoveToUsagesHandler.cs
//  
// Author:
//       Mike Krüger <mkrueger@novell.com>
// 
// Copyright (c) 2010 Novell, Inc (http://www.novell.com)
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
using System;
using MonoDevelop.Components.Commands;
using System.Linq;
using MonoDevelop.Ide;
using MonoDevelop.Ide.Gui.Content;
using Mono.TextEditor;

namespace MonoDevelop.CSharp.Highlighting
{
	public enum MoveToUsagesCommand {
		PrevUsage,
		NextUsage
	}
	
	public class MoveToPrevUsageHandler : CommandHandler
	{
		protected override void Update (CommandInfo info)
		{
			MonoDevelop.Ide.Gui.Document doc = IdeApp.Workbench.ActiveDocument;
			if (doc == null || doc.Editor == null) {
				info.Visible = info.Enabled = false;
				return;
			}
			
			info.Visible = info.Enabled = true;
		}
		
		internal static HighlightUsagesExtension GetHighlightUsageExtension (MonoDevelop.Ide.Gui.Document doc)
		{
			ITextEditorExtension ext = doc.EditorExtension;
			while (ext != null) {
				if (ext is HighlightUsagesExtension)
					return (HighlightUsagesExtension)ext;
				ext = ext.Next;
			}
			return null;
		}
		protected override void Run ()
		{
			MonoDevelop.Ide.Gui.Document doc = IdeApp.Workbench.ActiveDocument;
			HighlightUsagesExtension ext = GetHighlightUsageExtension (doc);
			if (ext == null)
				return;
			if (ext.IsTimerOnQueue)
				ext.ForceUpdate ();
			if (ext.Markers.Count == 0)
				return;
			
			
			if (ext.Markers.ContainsKey (doc.Editor.Caret.Line)) {
				var marker = ext.Markers[doc.Editor.Caret.Line];
				ISegment segment = null;
				for (int i = 0; i < marker.Usages.Count; i++) {
					if (marker.Usages[i].EndOffset < doc.Editor.Caret.Offset)
						segment = marker.Usages[i];
				}
				if (segment != null) {
					MoveToNextUsageHandler.MoveToSegment (doc, segment);
					return;
				}
			}
			
			int max = int.MinValue;
			foreach (var pair in ext.Markers) {
				if (pair.Key > max && pair.Key < doc.Editor.Caret.Line)
					max = pair.Key;
			}
			if (max >= 0) {
				MoveToNextUsageHandler.MoveToSegment (doc, ext.Markers[max].Usages.Last ());
				return;
			}
			MoveToNextUsageHandler.MoveToSegment (doc, ext.Markers.Last ().Value.Usages.Last ());
		}
	}
	
	public class MoveToNextUsageHandler : CommandHandler
	{
		protected override void Update (CommandInfo info)
		{
			MonoDevelop.Ide.Gui.Document doc = IdeApp.Workbench.ActiveDocument;
			if (doc == null || doc.Editor == null) {
				info.Visible = info.Enabled = false;
				return;
			}
			
			info.Visible = info.Enabled = true;
		}
		
		protected override void Run ()
		{
			MonoDevelop.Ide.Gui.Document doc = IdeApp.Workbench.ActiveDocument;
			HighlightUsagesExtension ext = MoveToPrevUsageHandler.GetHighlightUsageExtension (doc);
			if (ext == null)
				return;
			if (ext.IsTimerOnQueue)
				ext.ForceUpdate ();
			if (ext == null || ext.Markers.Count == 0)
				return;
			
			if (ext.Markers.ContainsKey (doc.Editor.Caret.Line)) {
				var marker = ext.Markers[doc.Editor.Caret.Line];
				ISegment segment = null;
				for (int i = 0; i < marker.Usages.Count; i++) {
					if (marker.Usages[i].Offset > doc.Editor.Caret.Offset) {
						segment = marker.Usages[i];
						break;
					}
				}
				if (segment != null) {
					MoveToSegment (doc, segment);
					return;
				}
			}
			
			int max = int.MinValue;
			foreach (var pair in ext.Markers) {
				if (pair.Key > doc.Editor.Caret.Line) {
					max = pair.Key;
					break;
				}
			}
			
			if (max >= 0) {
				MoveToSegment (doc, ext.Markers[max].Usages.First ());
				return;
			}
			MoveToSegment (doc, ext.Markers.First ().Value.Usages.First ());
		}
		
		public static void MoveToSegment (MonoDevelop.Ide.Gui.Document doc, ISegment segment)
		{
			TextEditorData data = doc.Editor;
			data.Caret.Offset = segment.Offset;
			data.Parent.ScrollTo (segment.EndOffset);
			
			var loc = data.Document.OffsetToLocation (segment.EndOffset);
			if (data.Parent.TextViewMargin.ColumnToX (data.Document.GetLine (loc.Line), loc.Column) < data.HAdjustment.PageSize)
				data.HAdjustment.Value = 0;
		}
	}
}

