//
// The provider for a tree of ECMA documents
//
// Authors:
//   Miguel de Icaza (miguel@ximian.com)
//   Joshua Tauberer (tauberer@for.net)
//
// (C) 2002, 2003 Ximian, Inc.
// (C) 2003 Joshua Tauberer.
//
// TODO:
//   Should cluster together constructors
//
// Easy:
//   Should render attributes on the signature.
//   Include examples as well.
//
namespace Monodoc {
using System;
using System.Reflection;
using System.IO;
using System.Xml;
using System.Xml.XPath;
using System.Xml.Xsl;
using System.Text;
using System.Collections;
using ICSharpCode.SharpZipLib.Zip;

using BF = System.Reflection.BindingFlags;

//
// Helper routines to extract information from an Ecma XML document
//
public class EcmaDoc {
	public static string GetFullClassName (XmlDocument doc)
	{
		return doc.SelectSingleNode ("/Type").Attributes ["FullName"].InnerText;
	}
	
	public static string GetClassName (XmlDocument doc)
	{
		string name = doc.SelectSingleNode ("/Type").Attributes ["Name"].InnerText;
		return name.Replace("+", ".");
	}

	public static string GetClassAssembly (XmlDocument doc)
	{
		return doc.SelectSingleNode ("/Type/AssemblyInfo/AssemblyName").InnerText;
	}

	public static string GetClassNamespace (XmlDocument doc)
	{
		string s = doc.SelectSingleNode ("/Type").Attributes ["FullName"].InnerText;

		return s.Substring (0, s.LastIndexOf ("."));
	}
	
	public static string GetTypeKind (XmlDocument doc)
	{
		XmlNode node = doc.SelectSingleNode ("/Type/Base/BaseTypeName");

		if (node == null){
			if (GetFullClassName (doc) == "System.Object")
				return "Class";
			return "Interface";
		}

		switch (node.InnerText){

		case "System.Delegate":
		case "System.MulticastDelegate":
			return "Delegate";
		case "System.ValueType":
			return "Structure";
		case "System.Enum":
			return "Enumeration";
		default:
			return "Class";
		}
	}

	//
	// Utility function: converts a fully .NET qualified type name into a C#-looking one
	//
	public static string ConvertCTSName (string type)
	{
		if (!type.StartsWith ("System."))
			return type;

		if (type.EndsWith ("*"))
			return ConvertCTSName(type.Substring(0, type.Length - 1)) + "*";
		if (type.EndsWith ("&"))
			return ConvertCTSName(type.Substring(0, type.Length - 1)) + "&";
		if (type.EndsWith ("[]"))
			return ConvertCTSName(type.Substring(0, type.Length - 2)) + "[]";

		switch (type) {
		case "System.Byte": return "byte";
		case "System.SByte": return "sbyte";
		case "System.Int16": return "short";
		case "System.Int32": return "int";
		case "System.Int64": return "long";
			
		case "System.UInt16": return "ushort";
		case "System.UInt32": return "uint";
		case "System.UInt64": return "ulong";
			
		case "System.Single":  return "float";
		case "System.Double":  return "double";
		case "System.Decimal": return "decimal";
		case "System.Boolean": return "bool";
		case "System.Char":    return "char";
		case "System.String":  return "string";
			
		case "System.Object":  return "object";
		case "System.Void":  return "void";
		}

		if (type.LastIndexOf(".") == 6)
			return type.Substring(7);
		
		return type;
	}
}

//
// The Ecma documentation provider:
//
// It populates a tree with contents during the help assembly
//
public class EcmaProvider : Provider {
	string basedir;

	public EcmaProvider (string base_directory)
	{
		basedir = base_directory;

		if (!Directory.Exists (basedir))
			throw new FileNotFoundException (String.Format ("The directory `{0}' does not exist", basedir));
	}
	
	public override void PopulateTree (Tree tree)
	{
		string [] namespaces = Directory.GetDirectories (basedir);
		

		
		foreach (string ns in namespaces){
			string [] files = Directory.GetFiles (ns);
			Node ns_node = null;
			string tn = null;
			
			Hashtable nsnodes = new Hashtable();

			foreach (string file in files){
				if (!file.EndsWith (".xml"))
					continue;

				if (ns_node == null) {
					tn = Path.GetFileName (ns);
					Console.Error.WriteLine ("Processing namespace {0}", tn);
					ns_node = tree.LookupNode (tn, "N:" + tn);
					string ns_summary_file = basedir + "/" + tn + ".xml";
					
					nsnodes[ns_node] = nsnodes;
					
					if (File.Exists (ns_summary_file)) {
						XmlDocument nsSummaryFile = new XmlDocument ();
						nsSummaryFile.Load (ns_summary_file);
						
						namespace_summaries [tn] = nsSummaryFile.SelectSingleNode ("Namespace/Docs/summary");
						namespace_remarks [tn] = nsSummaryFile.SelectSingleNode ("Namespace/Docs/remarks");
						
					} else if (!namespace_summaries.ContainsKey (tn)) {
						namespace_summaries [tn] = null;
						namespace_remarks [tn] = null;
					}
				}
				Console.Error.WriteLine ("    Processing input file {0}", Path.GetFileName (file));

				PopulateClass (tn, ns_node, file);
			}
			
			// Sort the list of types in each namespace
			foreach (Node ns_node2 in nsnodes.Keys)
				ns_node2.Sort();
		}

	}
		
	struct TypeInfo : IComparable {
		public string type_assembly;
		public string type_name;
		public string type_full;
		public string type_kind;
		public XmlNode type_doc;

		public TypeInfo (string k, string a, string f, string s, XmlNode n)
		{
			type_assembly = a;
			type_name = s;
			type_doc = n;
			type_kind = k;
			type_full = f;
		}

		public int CompareTo (object b)
		{
			TypeInfo na = this;
			TypeInfo nb = (TypeInfo) b;

			return String.Compare (na.type_full, nb.type_full);
		}
	}
	
	//
	// Packs a file with the summary data
	//
	public override void CloseTree (HelpSource hs, Tree tree)
	{
		foreach (DictionaryEntry de in class_summaries){
			XmlDocument doc = new XmlDocument ();
			string ns = (string) de.Key;
			
			ArrayList list = (ArrayList) de.Value;
			list.Sort();

			XmlElement elements = doc.CreateElement ("elements");
			doc.AppendChild (elements);
			
			if (namespace_summaries [ns] != null)
				elements.AppendChild (doc.ImportNode ((XmlNode)namespace_summaries [ns],true));
			else
				elements.AppendChild (doc.CreateElement("summary"));
			
			if (namespace_remarks [ns] != null)
				elements.AppendChild (doc.ImportNode ((XmlNode)namespace_remarks [ns],true));
			else
				elements.AppendChild (doc.CreateElement("remarks"));
			
			Console.Error.WriteLine ("Have {0} elements in the {1}", list.Count, ns);
			foreach (TypeInfo p in list){
				XmlElement e = null;
				
				switch (p.type_kind){
				case "Class":
					e = doc.CreateElement ("class"); 
					break;
					
				case "Enumeration":
					e = doc.CreateElement ("enum");
					break;
					
				case "Structure":
					e = doc.CreateElement ("struct");
					break;
					
				case "Delegate":
					e = doc.CreateElement ("delegate");
					break;
					
				case "Interface":
					e = doc.CreateElement ("interface");
					break;
				}
				
				e.SetAttribute ("name", p.type_name);
				e.SetAttribute ("fullname", p.type_full);
				e.SetAttribute ("assembly", p.type_assembly);
				XmlNode copy = doc.ImportNode (p.type_doc, true);
				e.AppendChild (copy);
				elements.AppendChild (e);
			}
			hs.PackXml ("xml.summary." + ns, doc);
		}
		
		
		XmlDocument nsSummary = new XmlDocument ();
		XmlElement root = nsSummary.CreateElement ("elements");
		nsSummary.AppendChild (root);
		
		foreach (DictionaryEntry de in namespace_summaries) {
			XmlNode n = (XmlNode)de.Value;
			XmlElement summary = nsSummary.CreateElement ("namespace");
			summary.SetAttribute ("ns", (string)de.Key);
			root.AppendChild (summary);
			if (n != null)
				summary.AppendChild (nsSummary.ImportNode (n,true));
			else
				summary.AppendChild (nsSummary.CreateElement("summary"));
			
		}
		tree.HelpSource.PackXml ("mastersummary.xml", nsSummary);
	}
	       
	static Hashtable class_summaries = new Hashtable ();
	static Hashtable namespace_summaries = new Hashtable ();
	static Hashtable namespace_remarks = new Hashtable ();
	XmlDocument doc;
	
	void PopulateClass (string ns, Node ns_node, string file)
	{
		doc = new XmlDocument ();
		doc.Load (file);
		
		string name = EcmaDoc.GetClassName (doc);
		string assembly = EcmaDoc.GetClassAssembly (doc);
		string kind = EcmaDoc.GetTypeKind (doc);
		string full = EcmaDoc.GetFullClassName (doc);

		Node class_node;
		string file_code = ns_node.tree.HelpSource.PackFile (file);

		XmlNode class_summary = doc.SelectSingleNode ("/Type/Docs/summary");
		ArrayList l = (ArrayList) class_summaries [ns];
		if (l == null){
			l = new ArrayList ();
			class_summaries [ns] = (object) l;
		}
		l.Add (new TypeInfo (kind, assembly, full, name, class_summary));
	       
		class_node = ns_node.LookupNode (String.Format ("{0} {1}", name, kind), "ecma:" + file_code + "#" + name + "/");
		
		if (kind == "Delegate") {
			if (doc.SelectSingleNode("/Type/ReturnValue") == null)
				Console.Error.WriteLine("Delegate " + name + " does not have a ReturnValue node.  See the ECMA-style updates.");
		}

		if (kind == "Enumeration")
			return;

		if (kind == "Delegate")
			return;
		
		//
		// Always add the Members node
		//
		class_node.CreateNode ("Members", "*");

		PopulateMember (name, class_node, "Constructor", "Constructors");
		PopulateMember (name, class_node, "Method", "Methods");
		PopulateMember (name, class_node, "Property", "Properties");
		PopulateMember (name, class_node, "Field", "Fields");
		PopulateMember (name, class_node, "Event", "Events");
		PopulateMember (name, class_node, "Operator", "Operators");
	}

	class NodeIndex {
		public XmlNode node;
		public int     index;

		public NodeIndex (XmlNode node, int index)
		{
			this.node = node;
			this.index = index;
		}
	}

	struct NodeCompare : IComparer {
		public int Compare (object a, object b)
		{
			NodeIndex na = (NodeIndex) a;
			NodeIndex nb = (NodeIndex) b;

			return String.Compare (na.node.Attributes ["MemberName"].InnerText,
					       nb.node.Attributes ["MemberName"].InnerText);
		}
	}

	static NodeCompare NodeComparer = new NodeCompare ();

	string GetMemberName (XmlNode node)
	{
		return node.Attributes ["MemberName"].InnerText;
	}
	
	//
	// Performs an XPath query on the document to extract the nodes for the various members
	// we also use some extra text to pluralize the caption
	//
	void PopulateMember (string typename, Node node, string type, string caption)
	{
		string select = type;
		if (select == "Operator") select = "Method";
		
		XmlNodeList list1 = doc.SelectNodes (String.Format ("/Type/Members/Member[MemberType=\"{0}\"]", select));
		ArrayList list = new ArrayList();
		int i = 0;
		foreach (XmlElement n in list1) {
			n.SetAttribute("assembler_index", (i++).ToString());
			if (type == "Method" && GetMemberName(n).StartsWith("op_")) continue;
			if (type == "Operator" && !GetMemberName(n).StartsWith("op_")) continue;
			list.Add(n);
		}
		
		int count = list.Count;
		
		if (count == 0)
			return;

		Node nodes_node;
		string key = type.Substring (0, 1);
		nodes_node = node.CreateNode (caption, key);
		
		switch (type) {
			case "Event":
			case "Field":
				foreach (XmlElement n in list)
					nodes_node.CreateNode (GetMemberName (n), n.GetAttribute("assembler_index"));
				break;

			case "Constructor":
				foreach (XmlElement n in list)
					nodes_node.CreateNode (EcmaHelpSource.MakeSignature(n, typename), n.GetAttribute("assembler_index"));
				break;

			case "Property": // properties with indexers can be overloaded too
			case "Method":
			case "Operator":
				foreach (XmlElement n in list) {
					bool multiple = false;
					foreach (XmlNode nn in list) {
						if (n != nn && GetMemberName(n) == nn.Attributes ["MemberName"].InnerText) {
							multiple = true;
							break;
						}
					}
					
					string group, name, sig;
					if (type != "Operator") {
						name = GetMemberName(n);
						sig = EcmaHelpSource.MakeSignature(n, null);
						group = name;
					} else {
						EcmaHelpSource.MakeOperatorSignature(n, out name, out sig);
						group = name;
					}
					
					if (multiple) {
						nodes_node.LookupNode (group, group)
							.CreateNode (sig, n.GetAttribute("assembler_index"));
					} else {
						nodes_node.CreateNode (name, n.GetAttribute("assembler_index"));
					}
				}
				
				foreach (Node n in nodes_node.Nodes) {
					if (!n.IsLeaf)
						n.Sort ();
				}
				
				break;
				
			default:
				throw new InvalidOperationException();
		}
		
		nodes_node.Sort ();
	}

}

//
// The Ecma HelpSource provider
//
public class EcmaHelpSource : HelpSource {

	public EcmaHelpSource (string base_file, bool create) : base (base_file, create)
	{
		ExtObject = new ExtensionObject (this);
	}

	public override string GetText (string url, out Node match_node)
	{
		match_node = null;
		
		if (url == "root:")
		{
			XmlReader summary = GetHelpXml ("mastersummary.xml");
			if (summary == null)
				return null;

			XsltArgumentList args = new XsltArgumentList();
			args.AddExtensionObject("monodoc://extensions", ExtObject);
			args.AddParam("show", "", "masteroverview");
			return Htmlize(new XPathDocument (summary), args);
		}
		
		if (url.StartsWith ("ecma:"))
			return GetTextFromUrl (url);

		return null;
	}


	string RenderMemberLookup (string typename, string member, ref Node type_node)
	{
		if (type_node.Nodes == null)
			return null;

		string lasturl = "";
		XmlNode doc = null;

		string membername = member;
		string[] argtypes = null;
		if (member.IndexOf("(") > 0) {
			membername = membername.Substring(0, member.IndexOf("("));
			member = member.Replace("@", "&");
			
			// reform the member signature with CTS names

			string x = member.Substring(member.IndexOf("(")+1);
			argtypes = x.Substring(0, x.Length-1).Split(',', ':'); // operator signatures have colons

			if (membername == ".ctor")
				membername = typename;

			member = membername + "(";
			for (int i = 0; i < argtypes.Length; i++) {
				argtypes[i] = EcmaDoc.ConvertCTSName(argtypes[i]);
				if (i > 0) member += ",";
				member += argtypes[i];
			}
			member += ")";
		}
		
		// Check if a node caption matches exactly
		
		bool isoperator = false;
		
		if ((membername == "op_Implicit" || membername == "op_Explicit") && argtypes.Length == 2) {
			isoperator = true;
			membername = "Conversion";
			member = argtypes[0] + " to " + argtypes[1];
			Console.WriteLine(member);
		} else if (membername.StartsWith("op_")) {
			isoperator = true;
			membername = membername.Substring(3);
		}

		foreach (Node x in type_node.Nodes){
			if (x.Nodes == null)
				continue;
			if (isoperator && x.Caption != "Operators")
				continue;
			
			foreach (Node m in x.Nodes) {
				string caption = m.Caption;
				if (m.IsLeaf) {
					// No overloading (usually), is just the member name.  The whole thing for constructors.
					if (caption == membername || caption == member) {
						type_node = m;
						return GetTextFromUrl (m.URL);
					}
				} else if (caption == member) {
					// Though there are overloads, no arguments are in the url, so use this base node
					type_node = m;
					return GetTextFromUrl (m.URL);
				} else {
					// Check subnodes which are the overloads -- must match signature
					foreach (Node mm in m.Nodes) {
						if (mm.Caption == member) {
							type_node = mm;
							return GetTextFromUrl (mm.URL);
						}
					}
				}
			}
		}
		
		return null;
	}

	public static string MakeSignature (XmlNode n, string cstyleclass)
	{
		string sig;

		if (cstyleclass == null)
			sig = n.Attributes["MemberName"].InnerText;
		else {
			// constructor style
			sig = cstyleclass;
		}
	
		/*if (n.SelectSingleNode("MemberType").InnerText == "Method" || n.SelectSingleNode("MemberType").InnerText == "Constructor") {*/ // properties with indexers too
			XmlNodeList paramnodes = n.SelectNodes("Parameters/Parameter");
			sig += "(";
			bool first = true;
			foreach (XmlNode p in paramnodes) {
				if (!first) sig += ",";
				string type = p.Attributes["Type"].InnerText;
				type = EcmaDoc.ConvertCTSName(type);
				sig += type;
				first = false;
			}
			sig += ")";
		//}

		return sig;
	}
	
	public static void MakeOperatorSignature (XmlNode n, out string nicename, out string sig)
	{
		string name;

		name = n.Attributes["MemberName"].InnerText;
		nicename = name.Substring(3);
		
		switch (name) {
			// unary operators: no overloading possible
			case "op_UnaryPlus": case "op_UnaryNegation": case "op_LogicalNot": case "op_OnesComplement":
			case "op_Increment": case "op_Decrement": 
			case "op_True": case "op_False":
				sig = nicename;
				break;
			
			// binary operators: overloading is possible based on parameter types
			case "op_Addition": case "op_Subtraction": case "op_Multiply": case "op_Division": case "op_Modulus":
			case "op_BitwiseAnd": case "op_BitwiseOr": case "op_ExclusiveOr":
			case "op_LeftShift": case "op_RightShift":
			case "op_Equality": case "op_Inequality":
			case "op_GreaterThan": case "op_LessThan": case "op_GreaterThanOrEqual": case "op_LessThanOrEqual":
				XmlNodeList paramnodes = n.SelectNodes("Parameters/Parameter");
				sig = nicename + "(";
				bool first = true;
				foreach (XmlNode p in paramnodes) {
					if (!first) sig += ",";
					string type = p.Attributes["Type"].InnerText;
					type = EcmaDoc.ConvertCTSName(type);
					sig += type;
					first = false;
				}
				sig += ")";
				break;
			
			// overloading based on parameter and return type
			case "op_Implicit": case "op_Explicit":
				nicename = "Conversion";
				string arg = n.SelectSingleNode("Parameters/Parameter/@Type").InnerText;
				string ret = n.SelectSingleNode("ReturnValue/ReturnType").InnerText;
				sig = EcmaDoc.ConvertCTSName(arg) + " to " + EcmaDoc.ConvertCTSName(ret);
				break;
				
			default:
				throw new InvalidOperationException();
		}	
	}

	//
	// This routine has to perform a lookup on a type.
	//
	// Example: T:System.Text.StringBuilder
	//
	// The prefix is the kind of opereation being requested (T:, E:, M: etc)
	// ns is the namespace being looked up
	// type is the type being requested
	//
	// This has to walk our toplevel (which is always a namespace)
	// And then the type space, and then walk further down depending on the request
	//
	public override string RenderTypeLookup (string prefix, string ns, string type, string member, out Node match_node)
	{
		string url = GetUrlForType (prefix, ns, type, member, out match_node);
		if (url == null) return null;
		return GetTextFromUrl (url);
	}

	public string GetIdFromUrl (string prefix, string ns, string type)
	{
		Node tmp_node = new Node (Tree, "", "");
		string url = GetUrlForType (prefix, ns, type, null, out tmp_node);
		if (url == null) return null;
		return GetFile (url.Substring (5), out url);
	}

	public string GetUrlForType (string prefix, string ns, string type, string member, out Node match_node)
	{
		if (!prefix.EndsWith(":")) throw new ArgumentException("prefix");
			
		// If a nested type, compare only inner type name to node list.
		// This should be removed when the node list doesn't lose the containing type name.
		type = type.Replace("+", ".");

		string nsp = prefix + ns;

		foreach (Node ns_node in Tree.Nodes){
			string ns_node_namespace = ns_node.Element.Substring (2);

			if (ns_node_namespace != ns)
				continue;
			
			foreach (Node type_node in ns_node.Nodes){
				string element = type_node.Element;
				int pidx = element.IndexOf ("#");
				int sidx = element.IndexOf ("/");
				
				string cname = element.Substring (pidx + 1, sidx-pidx-1);
				
				//Console.WriteLine ("t:{0} cn:{1} p:{2}", type, cname, prefix);

				if (type == cname && prefix == "T:") {
					match_node = type_node;
					return type_node.URL;
				} else if (type.StartsWith (cname)){
					int p = cname.Length;

					match_node = type_node;
					if (type == cname){
						string ret = RenderMemberLookup (type, member, ref match_node);
						if (ret == null)
							return type_node.URL;
						return match_node.URL;

					} else if (type [p] == '/'){
						//
						// This handles summaries
						//
						match_node = null;
						foreach (Node nd in type_node.Nodes) {
							if (nd.Element [nd.Element.Length - 1] == type [p + 1]) {
								match_node = nd;
								break;
							}
						}
						
						return type_node.URL + type.Substring (p + 1);
					}
				}
			}
		}

		match_node = null;
		return null;
	}
	
	public override string GetNodeXPath (XPathNavigator n)
	{
		if (n.Matches ("/Type/Docs/*")) {
			string type_name = (string) n.Evaluate ("string (ancestor::Type/@FullName)");
			
			return String.Format ("/Type [@FullName = '{0}']/Docs/{1}", type_name, n.Name);
		}
		
		if (n.Matches ("/Type/Members/Member [MemberType != 'Constructor' and MemberType != 'Method']/Docs/*")) {
			string type_name = (string) n.Evaluate ("string (ancestor::Type/@FullName)");
			string member_name = (string) n.Evaluate ("string (ancestor::Member/@MemberName)");
			string param_name = (string) n.Evaluate ("string (@name)");
			
			if (param_name == null || param_name == "") {
				return String.Format ("/Type [@FullName = '{0}']/Members/Member [@MemberName = '{1}']/Docs/{2}",
					type_name, member_name, n.Name);
			} else {
				return String.Format ("/Type [@FullName = '{0}']/Members/Member [@MemberName = '{1}']/Docs/param [@name = '{2}']",
					type_name, member_name, param_name);
			}
		}
		
		if (n.Matches ("/Type/Members/Member [MemberType = 'Constructor' or MemberType = 'Method']/Docs/*")) {
			string type_name = (string) n.Evaluate ("string (ancestor::Type/@FullName)");
			string member_name = (string) n.Evaluate ("string (ancestor::Member/@MemberName)");
			string member_sig = (string) n.Evaluate ("string (ancestor::Member/MemberSignature [@Language='C#']/@Value)");
			string param_name = (string) n.Evaluate ("string (@name)");
			
			if (param_name == null || param_name == "") {
				return String.Format (
				"/Type [@FullName = '{0}']/Members/Member [@MemberName = '{1}'][MemberSignature [@Language='C#']/@Value = '{2}']/Docs/{3}",
				type_name, member_name, member_sig, n.Name);
			} else {
				return String.Format (
				"/Type [@FullName = '{0}']/Members/Member [@MemberName = '{1}'][MemberSignature [@Language='C#']/@Value = '{2}']/Docs/param [@name = '{3}']",
				type_name, member_name, member_sig, param_name);
			}
		}
		
		Console.WriteLine ("WARNING: Was not able to get clean XPath expression for node {0}", EditingUtils.GetXPath (n));
		return base.GetNodeXPath (n);
	}


	public override string RenderNamespaceLookup (string nsurl, out Node match_node)
	{
		foreach (Node ns_node in Tree.Nodes){
			if (ns_node.Element != nsurl)
				continue;

			match_node = ns_node;
			string ns_name = nsurl.Substring (2);
			
			XmlReader summary = GetHelpXml ("xml.summary." + ns_name);
			if (summary == null)
				return null;

			XsltArgumentList args = new XsltArgumentList();
			args.AddExtensionObject("monodoc://extensions", ExtObject);
			args.AddParam("show", "", "namespace");
			args.AddParam("namespace", "", ns_name);
			return Htmlize(new XPathDocument (summary), args);

		}
		match_node = null;
		return null;
	}

	private string SelectString(XmlNode node, string xpath) {
		XmlNode ret = node.SelectSingleNode(xpath);
		if (ret == null) return "";
		return ret.InnerText;
	}
	private int SelectCount(XmlNode node, string xpath) {
		return node.SelectNodes(xpath).Count;
	}

	//
	// Returns the XmlDocument from the given url, and fills in `file' and `rest'
	//
	XmlDocument GetXmlFromUrl (string url, out string file, out string rest)
	{
		//
		// Remove ecma:
		//
		url = url.Substring (5);
		file = GetFile (url, out rest);

		// Console.WriteLine ("Got [{0}] and [{1}]", file, rest);
		XmlDocument doc = GetHelpXmlWithChanges (file);
		if (doc == null)
			return null;

		return doc;
	}
	
	string GetTextFromUrl (string url)
	{
		string file, rest;

		XmlDocument doc = GetXmlFromUrl (url, out file, out rest);
		
		// Load base-type information so the stylesheet can draw the inheritance
		// tree and (optionally) include inherited members in the members list.
		XmlElement basenode = (XmlElement)doc.SelectSingleNode("Type/Base");
		XmlElement membersnode = (XmlElement)doc.SelectSingleNode("Type/Members");
		XmlNode basetype = doc.SelectSingleNode("Type/Base/BaseTypeName");
		int baseindex = 0;
		while (basetype != null) {
			// Add the ParentType node to Type/Base
			XmlElement inheritancenode = doc.CreateElement("ParentType");
			inheritancenode.SetAttribute("Type", basetype.InnerText);
			inheritancenode.SetAttribute("Order", (baseindex++).ToString());
			basenode.AppendChild(inheritancenode);
			
			// Load the base type XML data
			int dot = basetype.InnerText.LastIndexOf('.');
			string ns = basetype.InnerText.Substring(0, dot == -1 ? 0 : dot);
			string n = basetype.InnerText.Substring(dot == -1 ? 0 : dot+1);
			string basetypexml = GetIdFromUrl("T:", ns, n);
			XmlDocument basetypedoc = null;

			if (basetypexml != null)
				basetypedoc = GetHelpXmlWithChanges (basetypexml);
			if (basetypedoc == null) {
				inheritancenode.SetAttribute("Incomplete", "1");
				break;
			}
			
			if (SettingsHandler.Settings.ShowInheritedMembers) {
				// Add inherited members
				foreach (XmlElement member in basetypedoc.SelectNodes("Type/Members/Member[not(MemberType='Constructor')]")) {
					string sig = SelectString(member, "MemberSignature[@Language='C#']/@Value");
					if (sig.IndexOf(" static ") >= 0) continue;
					
					// If the signature of member matches the signature of a member already in the XML,
					// don't add it.
					string xpath = "@MemberName='" + SelectString(member, "@MemberName") + "'";
					xpath       += " and ReturnValue/ReturnType='" + SelectString(member, "ReturnValue/ReturnType") + "'";
					xpath       += " and count(Parameters/Parameter)=" + SelectCount(member, "Parameters/Parameter") + "";
					int pi = 1;
					foreach (XmlElement p in member.SelectNodes("Parameters/Parameter")) {
						xpath   += " and Parameters/Parameter[position()=" + pi + "]/@Type = '" + p.GetAttribute("Type") + "'";
						pi++;
					}
					
					// If a signature match is found, don't add.
					XmlNode match = membersnode.SelectSingleNode("Member[" + xpath + "]");
					if (match != null)
						continue;
					
					member.SetAttribute("DeclaredIn", basetype.InnerText);
					membersnode.AppendChild(membersnode.OwnerDocument.ImportNode(member, true));				
				}
			}
			
			basetype = basetypedoc.SelectSingleNode("Type/Base/BaseTypeName");
		}

		XsltArgumentList args = new XsltArgumentList();

		args.AddExtensionObject("monodoc://extensions", ExtObject);
		
		if (rest == "") {
			args.AddParam("show", "", "typeoverview");
			return Htmlize(doc, args);
		}
		
		string [] nodes = rest.Split (new char [] {'/'});
		
		switch (nodes.Length) {
			case 1:
				args.AddParam("show", "", "members");
				args.AddParam("index", "", "all");
				break;
			case 2:
				// Could either be a single member, or an overload thingy
				try {
					int dummy = int.Parse (nodes [1]); // is it an int
					
					args.AddParam("show", "", "member");
					args.AddParam("index", "", nodes [1]);
				} catch {
					args.AddParam("show", "", "overloads");
					args.AddParam("index", "", nodes [1]);
				}
				break;
			case 3:
				args.AddParam("show", "", "member");
				args.AddParam("index", "", nodes [2]);
				break;
			default:
				return "What the hell is this URL " + url;
		}

		switch (nodes [0]){
		case "C":
			args.AddParam("membertype", "", "Constructor");
			break;
		case "M":
			args.AddParam("membertype", "", "Method");
			break;
		case "P":
			args.AddParam("membertype", "", "Property");
			break;
		case "F":
			args.AddParam("membertype", "", "Field");
			break;
		case "E":
			args.AddParam("membertype", "", "Event");
			break;
		case "O":
			args.AddParam("membertype", "", "Operator");
			break;
		case "*":
			args.AddParam("membertype", "", "All");
			break;
		default:
			return "Unknown url: " + url;
		}

		return Htmlize(doc, args);
	}

	
	public override void RenderPreviewDocs (XmlNode newNode, XmlWriter writer)
	{
		XsltArgumentList args = new XsltArgumentList ();
		args.AddExtensionObject ("monodoc://extensions", ExtObject);
		
		Htmlize (newNode, args, writer);
	}

	static XslTransform ecma_transform;

	public static string Htmlize (IXPathNavigable ecma_xml)
	{
		return Htmlize(ecma_xml, null);
	}
	
	static string Htmlize (IXPathNavigable ecma_xml, XsltArgumentList args)
	{
		EnsureTransform ();
		
		StringWriter output = new StringWriter ();
		ecma_transform.Transform (ecma_xml, args, output);
		return output.ToString ();
	}
	
	static void Htmlize (IXPathNavigable ecma_xml, XsltArgumentList args, XmlWriter w)
	{

		EnsureTransform ();
		
		if (ecma_xml == null)
			return;

		ecma_transform.Transform (ecma_xml, args, w);
	}
	
	static void EnsureTransform ()
	{
		if (ecma_transform == null) {
			ecma_transform = new XslTransform ();
			Assembly assembly = System.Reflection.Assembly.GetCallingAssembly ();
			Stream stream = assembly.GetManifestResourceStream ("mono-ecma.xsl");

			XmlReader xml_reader = new XmlTextReader (stream);
			ecma_transform.Load (xml_reader);
		}
	}

	// This ExtensionObject stuff is used to check at run time whether
	// types and members have been implemented and whether there are any
	// MonoTODO attributes left on them. 

	readonly ExtensionObject ExtObject;
	class ExtensionObject {
		readonly EcmaHelpSource hs;
		
		public ExtensionObject (EcmaHelpSource hs)
		{
			this.hs = hs;
		}
		
		public string Colorize(string code, string lang) {
			return(Mono.Utilities.Colorizer.Colorize(code,lang));
		}
		// Used by stylesheet to nicely reformat the <see cref=> tags. 
		public string MakeNiceSignature(string sig)
		{
			if (sig.Length < 3)
				return sig;
			if (sig[1] != ':')
				return sig;

			char s = sig[0];
			sig = sig.Substring(2);

			switch (s) {
				case 'N': return sig;
				case 'T': return ShortTypeName(sig);

				case 'C': case 'M': case 'P': case 'F': case 'E':
					int paren = sig.IndexOf("(");
					if (paren > 0) {
						string name = sig.Substring(0, paren);
						string[] args = sig.Substring(paren, sig.Length-paren-1).Split(',');
						
						// avoid (() on no args
						if (args.Length == 1 && args[0] == "(")
							return ShortTypeName(sig);
							
						for (int i = 0; i < args.Length; i++)
							args[i] = ShortTypeName(args[i]);
						return ShortTypeName(name) + "(" + String.Join(", ", args) + ")";
					} else {
						return ShortTypeName(sig);
					}

				default:
					return sig;
			}
		}
		
		public string EditUrl (XPathNodeIterator itr)
		{
			if (itr.MoveNext ())
				return hs.GetEditUri (itr.Current);
			
			return "";
		}

		private static string ShortTypeName(string name)
		{
			//if (name.LastIndexOf(".") > 0) name = name.Substring(name.LastIndexOf(".")+1);
			return name;
		}


		public string MonoImpInfo(string assemblyname, string typename, string membername, string arglist, bool strlong)
		{
			ArrayList a = new ArrayList();
			if (arglist != "") a.Add(arglist);
			return MonoImpInfo(assemblyname, typename, membername, a, strlong);
		}

		public string MonoImpInfo(string assemblyname, string typename, string membername, XPathNodeIterator itr, bool strlong)
		{
			ArrayList rgs = new ArrayList ();
			while (itr.MoveNext ())
				rgs.Add (itr.Current.Value);
			
			return MonoImpInfo (assemblyname, typename, membername, rgs, strlong);
		}
		
		public string MonoImpInfo(string assemblyname, string typename, string membername, ArrayList arglist, bool strlong)
		{
			try {
				Assembly assembly = null;
				
				try {
					assembly = Assembly.LoadWithPartialName(assemblyname);
				} catch (Exception e) { }
				if (assembly == null) {
					/*if (strlong) return "The assembly " + assemblyname + " is not available to MonoDoc.";
					else return "";*/
					return ""; // silently ignore
				}

				Type t = assembly.GetType(typename, false);
				if (t == null) {
					if (strlong)
						return typename + " has not been implemented.";
					else
						return "Not implemented.";
				}

				MemberInfo[] mis = t.GetMember(membername, BF.Static | BF.Instance | BF.Public | BF.NonPublic);

				if (mis.Length == 0)
					return "This member has not been implemented.";
				if (mis.Length == 1)
					return MonoImpInfo(mis[0], "member", strlong);

				// Scan for the appropriate member
				foreach (MemberInfo mi in mis) {
					System.Reflection.ParameterInfo[] pis;

					if (mi is MethodInfo || mi is ConstructorInfo)
						pis = ((MethodBase)mi).GetParameters();
					else if (mi is PropertyInfo)
						pis = ((PropertyInfo)mi).GetIndexParameters();
					else
						pis = null;
					
					if (pis != null) {
						bool good = true;
						if (pis.Length != arglist.Count) continue;
						for (int i = 0; i < pis.Length; i++) {
							if (pis[i].ParameterType.FullName != (string)arglist[i]) { good = false; break; }
						}
						if (!good) continue;
					}

					return MonoImpInfo(mi, "member", strlong);
				}

				return "This member has not been implemented.";

			} catch (Exception e) {
				return "";
			}
		}
		
		public string MonoImpInfo(System.Reflection.MemberInfo mi, string itemtype, bool strlong)
		{
			string s = "";

			object[] atts = mi.GetCustomAttributes(true);
			int todoctr = 0;
			foreach (object att in atts) if (att.GetType().Name == "MonoTODOAttribute") todoctr++;

			if (todoctr > 0) {
				if (strlong)
					s = "This " + itemtype + " is marked as being unfinished.<BR/>\n";
				else 
					s = "Unfinished.";
			}

			return s;
		}

		public string MonoImpInfo(string assemblyname, string typename, bool strlong)
		{
			try {
				if (assemblyname == "")
					return "";

				Assembly assembly = Assembly.LoadWithPartialName(assemblyname);
				if (assembly == null)
					return "";

				Type t = assembly.GetType(typename, false);
				if (t == null) {
					if (strlong)
						return typename + " has not been implemented.";
					else
						return "Not implemented.";
				}

				string s = MonoImpInfo(t, "type", strlong);

				if (strlong) {
					MemberInfo[] mis = t.GetMembers(BF.Static | BF.Instance | BF.Public | BF.NonPublic);

					// Scan members for MonoTODO attributes
					int mctr = 0;
					foreach (MemberInfo mi in mis) {
						string mii = MonoImpInfo(mi, null, false);
						if (mii != "") mctr++; 
					}
					if (mctr > 0) {
						s += "This type has " + mctr + " members that are marked as unfinished.<BR/>";
					}
				}

				return s;

			} catch (Exception e) {
				return "";
			}			
		}

		public bool MonoEditing ()
		{
			return SettingsHandler.Settings.EnableEditing && Settings.RunningGUI;
		}
	}

	//
	// This takes one of the ecma urls, which look like this:
	// ecma:NUMERIC_ID#OPAQUE/REST
	//
	// NUMERIC_ID is the numeric ID assigned by the compressor
	// OPAQUE is opaque for node rendering (it typically contains T:System.Byte for example)
	// REST is the rest of the argument used to decode information
	//
	static string GetFile (string url, out string rest)
	{
		int pound = url.IndexOf ("#");
		int slash = url.IndexOf ("/");
		
		string fname = url.Substring (0, pound);
		rest = url.Substring (slash+1);

		return fname;
	}

	// This should have a little cache or something.
	static XmlDocument GetDocument (HelpSource hs, string fname)
	{
		Stream s = hs.GetHelpStream (fname);
		if (s == null){
			Console.Error.WriteLine ("Could not fetch document {0}", fname);
			return null;
		}
		
		XmlDocument doc = new XmlDocument ();

		doc.Load (s);
		
		return doc;
	}

	string GetKindFromCaption (string s)
	{
		int p = s.LastIndexOf (' ');
		if (p > 0)
			return s.Substring (p + 1);
		return null;
	}
	
	//
	// Populates the index.
	//
	public override void PopulateIndex (IndexMaker index_maker)
	{
		foreach (Node ns_node in Tree.Nodes){
			foreach (Node type_node in ns_node.Nodes){
				string typename = type_node.Caption.Substring (0, type_node.Caption.IndexOf (' '));
				string full = ns_node.Caption + "." + typename;

				string doc_tag = GetKindFromCaption (type_node.Caption);
				string url = "T:" + full;
					
				if (doc_tag == "Class" || doc_tag == "Structure" || doc_tag == "Interface"){

					index_maker.Add (type_node.Caption, typename, url);
					index_maker.Add (full + " " + doc_tag, full, url);

					foreach (Node c in type_node.Nodes){
						switch (c.Caption){
						case "Constructors":
							index_maker.Add ("  constructors", typename+"0", url + "/C");
							break;
						case "Fields":
							index_maker.Add ("  fields", typename+"1", url + "/F");
							break;
						case "Events":
							index_maker.Add ("  events", typename+"2", url + "/E");
							break;
						case "Properties":
							index_maker.Add ("  properties", typename+"3", url + "/P");
							break;
						case "Methods":
							index_maker.Add ("  methods", typename+"4", url + "/M");
							break;
						case "Operators":
							index_maker.Add ("  operators", typename+"5", url + "/O");
							break;
						}
					}

					//
					// Now repeat, but use a different sort key, to make sure we come after
					// the summary data above, start the counter at 6
					//
					string keybase = typename + "6.";
					
					foreach (Node c in type_node.Nodes){
						switch (c.Caption){
						case "Constructors":
							break;
						case "Fields":
							foreach (Node nc in c.Nodes){
								string res = nc.Caption;

								string nurl = String.Format ("F:{0}.{1}", full, res);
								index_maker.Add (String.Format ("{0}.{1} field", typename, res),
										 keybase + res, nurl);
								index_maker.Add (String.Format ("{0} field", res), res, nurl);
							}

							break;
						case "Events":
							foreach (Node nc in c.Nodes){
								string res = nc.Caption;
								string nurl = String.Format ("E:{0}.{1}", full, res);
								
								index_maker.Add (String.Format ("{0}.{1} event", typename, res),
										 keybase + res, nurl);
								index_maker.Add (String.Format ("{0} event", res), res, nurl);
							}
							break;
						case "Properties":
							foreach (Node nc in c.Nodes){
								string res = nc.Caption;
								string nurl = String.Format ("P:{0}.{1}", full, res);
								index_maker.Add (String.Format ("{0}.{1} property", typename, res),
										 keybase + res, nurl);
								index_maker.Add (String.Format ("{0} property", res), res, nurl);
							}
							break;
						case "Methods":
							foreach (Node nc in c.Nodes){
								string res = nc.Caption;
								int p = res.IndexOf ("(");
								if (p > 0)
									res = res.Substring (0, p); 
								string nurl = String.Format ("M:{0}.{1}", full, res);
								index_maker.Add (String.Format ("{0}.{1} method", typename, res),
										 keybase + res, nurl);
								index_maker.Add (String.Format ("{0} method", res), res, nurl);
							}
					
							break;
						case "Operators":
							foreach (Node nc in c.Nodes){
								string res = nc.Caption;
								string nurl = String.Format ("O:{0}.{1}", full, res);
								index_maker.Add (String.Format ("{0}.{1} operator", typename, res),
										 keybase + res, nurl);
							}
							break;
						}
					}
				} else if (doc_tag == "Enumeration"){
					//
					// Enumerations: add the enumeration values
					//
					index_maker.Add (type_node.Caption, typename, url);
					index_maker.Add (full + " " + doc_tag, full, url);

					// Now, pull the values.
					string file, rest;
					XmlDocument x = GetXmlFromUrl (type_node.URL, out file, out rest);
					if (x == null)
						continue;
					
					XmlNodeList members = x.SelectNodes ("/Type/Members/Member");

					if (members == null)
						continue;

					foreach (XmlNode member_node in members){
						string enum_value = member_node.Attributes ["MemberName"].InnerText;
						string caption = enum_value + " value";
						index_maker.Add (caption, caption, url);
					}
				} else if (doc_tag == "Delegate"){
					index_maker.Add (type_node.Caption, typename, url);
					index_maker.Add (full + " " + doc_tag, full, url);
				}
			}
		}
	}
}
}
