module Weak_store = struct
  type 'a t = 
      { mutable w : 'a Weak.t ; mutable free : int }

  let create () =
    { w = Weak.create 8 ; free = 0 }

  let register s v =
    let len = Weak.length s.w in
    assert (len > 0) ;
    if s.free < len
    then begin
      Weak.set s.w s.free (Some v) ;
      s.free <- s.free + 1
    end
    else begin
      let i = ref 0 in
      let full = ref true in
      while !full && !i < Weak.length s.w do
	full := Weak.check s.w !i ; 
	if !full then incr i
      done ;
      if !full
      then begin
	let n_s = Weak.create (2 * len) in
	Weak.blit s.w 0 n_s 0 len ;
	s.w <- n_s ; 
	s.free <- len + 1 ;
	i := len 
      end ;
      Weak.set s.w !i (Some v)
    end
end



type db
type stmt
type argument
type sql_type = [`INTEGER|`FLOAT|`TEXT|`BLOB|`NULL]
type sql_value = [
  | `INT of int
  | `INT64 of int64
  | `FLOAT of float
  | `TEXT of string
  | `BLOB of string
  | `VALUE of argument
  | `NULL ]

type error_code =
  | ERROR
  | INTERNAL
  | PERM
  | ABORT
  | BUSY
  | LOCKED
  | NOMEM
  | READONLY
  | INTERRUPT
  | IOERR
  | CORRUPT
  | NOTFOUND
  | FULL
  | CANTOPEN
  | PROTOCOL
  | EMPTY
  | SCHEMA
  | TOOBIG
  | CONSTRAINT
  | MISMATCH
  | MISUSE
  | NOLFS
  | AUTH
  | FORMAT
  | RANGE
  | NOTADB
exception Error of error_code * string

let init =
  Callback.register_exception "mlsqlite3_exn" (Error (ERROR, ""))


external open_db  : string -> db = "ml_sqlite3_open"
external _close_db : db -> unit = "ml_sqlite3_close"

external set_stmt_store : db -> stmt Weak_store.t option -> unit = "ml_sqlite3_set_stmt_store"
external get_stmt_store : db -> stmt Weak_store.t = "ml_sqlite3_get_stmt_store"

external finalize_stmt : stmt -> unit = "ml_sqlite3_finalize_noerr"

let stmt_store db =
  try get_stmt_store db
  with Not_found ->
    let s = Weak_store.create () in 
    set_stmt_store db (Some s) ;
    s
let register_stmt db stmt =
  Gc.finalise finalize_stmt stmt ;
  Weak_store.register (stmt_store db) stmt

let close_db db =
  begin
    try 
      let store = (get_stmt_store db).Weak_store.w in
      for i = 0 to Weak.length store - 1 do
	match Weak.get store i with
	| Some stmt -> finalize_stmt stmt
	| None -> ()
      done ;
      set_stmt_store db None ;
    with Not_found -> ()
  end ;
  _close_db db




external interrupt : db -> unit = "ml_sqlite3_interrupt"
external is_complete : string -> bool = "ml_sqlite3_complete"
external _version  : unit -> string = "ml_sqlite3_version"
let version = _version ()
external last_insert_rowid : db -> int64 = "ml_sqlite3_last_insert_rowid"
external changes       : db -> int = "ml_sqlite3_changes"
external total_changes : db -> int = "ml_sqlite3_total_changes"
(* external get_autocommit : db -> bool = "ml_sqlite3_get_autocommit" *)
external sleep : int -> unit = "ml_sqlite3_sleep"

external busy_set : db -> (int -> [`FAIL|`RETRY]) -> unit
   = "ml_sqlite3_busy_handler"
external busy_unset : db -> unit = "ml_sqlite3_busy_handler_unset"
external busy_timeout : db -> int -> unit = "ml_sqlite3_busy_timeout"

external trace_set   : db -> (string -> unit) -> unit = "ml_sqlite3_trace"
external trace_unset : db -> unit = "ml_sqlite3_trace_unset"

external progress_handler_set : db -> int -> (unit -> unit) -> unit 
  = "ml_sqlite3_progress_handler"
external progress_handler_unset : db -> unit = "ml_sqlite3_progress_handler_unset"

(* type vm *)
(* type stmt = { *)
(*     mutable vm : vm ; *)
(*     db         : db ; *)
(*     sql        : string ; *)
(*     sql_off    : int *)
(*   } *)

external prepare : db -> string -> int -> stmt option * int = "ml_sqlite3_prepare"

let _prepare_one db sql =
  match prepare db sql 0 with
  | Some stmt, _ -> 
      register_stmt db stmt ;
      stmt
  | None, _ -> failwith "Sqlite3.prepare_one: empty statement"

let prepare_one db sql =
  _prepare_one db (String.copy sql)

let prepare_one_f db fmt =
  Printf.kprintf (_prepare_one db) fmt

external reset : stmt -> unit = "ml_sqlite3_reset"
external expired : stmt -> bool = "ml_sqlite3_expired"
external step : stmt -> [`DONE|`ROW] = "ml_sqlite3_step"

external bind : stmt -> int -> sql_value -> unit = "ml_sqlite3_bind"
external bind_parameter_count : stmt -> int = "ml_sqlite3_bind_parameter_count"
external bind_parameter_index : stmt -> string -> int = "ml_sqlite3_bind_parameter_index"
external bind_parameter_name : stmt -> int -> string = "ml_sqlite3_bind_parameter_name"
external clear_bindings : stmt -> unit = "ml_sqlite3_clear_bindings"
external transfer_bindings : stmt -> stmt -> unit = "ml_sqlite3_transfer_bindings"

external column_blob : stmt -> int -> string = "ml_sqlite3_column_blob"
external column_double : stmt -> int -> float = "ml_sqlite3_column_double"
external column_int : stmt -> int -> int = "ml_sqlite3_column_int"
external column_int64 : stmt -> int -> int64 = "ml_sqlite3_column_int64"
external column_text : stmt -> int -> string = "ml_sqlite3_column_text"
external column_type : stmt -> int -> sql_type = "ml_sqlite3_column_type"
external data_count : stmt -> int = "ml_sqlite3_data_count"
external column_count : stmt -> int = "ml_sqlite3_column_count"
external column_name : stmt -> int -> string = "ml_sqlite3_column_name"
external column_decltype : stmt -> int -> string = "ml_sqlite3_column_decltype"


external value_blob   : argument -> string = "ml_sqlite3_value_blob"
external value_double : argument -> float  = "ml_sqlite3_value_double"
external value_int    : argument -> int    = "ml_sqlite3_value_int"
external value_int64  : argument -> int64  = "ml_sqlite3_value_int64"
external value_text   : argument -> string = "ml_sqlite3_value_text"
external value_type   : argument -> sql_type = "ml_sqlite3_value_type"
external _create_function : 
  db -> string -> int -> (argument array -> sql_value) -> unit
    = "ml_sqlite3_create_function"

let create_fun_N db name f =
  _create_function db name (-1) f

let create_fun_0 db name f =
  _create_function db name 0 (fun _ -> f ())

let create_fun_1 db name f =
  _create_function db name 1 (fun args -> f args.(0))

let create_fun_2 db name f =
  _create_function db name 2 (fun args -> f args.(0) args.(1))

let create_fun_3 db name f =
  _create_function db name 3 (fun args -> f args.(0) args.(1) args.(2))

external delete_function : db -> string -> unit = "ml_sqlite3_delete_function"

let fold_prepare db sql f init =
  let rec loop acc off =
    if off >= String.length sql
    then acc
    else
      match prepare db sql off with
      | Some stmt, nxt -> 
	  let acc =
	    try f acc stmt
	    with exn -> finalize_stmt stmt ; raise exn in
	  finalize_stmt stmt ;
	  loop acc nxt
      | None, nxt -> 
	  loop acc nxt in
  loop init 0

let rec fold_rows f acc stmt =
  match step stmt with
  | `DONE -> acc
  | `ROW  ->
      fold_rows f (f acc stmt) stmt


let _exec db sql =
  fold_prepare 
    db sql
    (fold_rows
       (fun () _ -> ()))
    ()

let exec db sql =
  _exec db (String.copy sql)

let exec_f db fmt =
  Printf.kprintf (_exec db) fmt


let fetch_one ?column_names f init stmt =
  begin
    match column_names with
    | None -> ()
    | Some n ->
	n := Array.init (column_count stmt) (column_name stmt)
  end ;
  fold_rows
    (fun acc stmt ->
      let row =
	Array.init
	  (data_count stmt)
	  (column_blob stmt) in
      f acc row)
    init 
    stmt

let _fetch db sql ?column_names f init = 
  fold_prepare
    db sql
    (fetch_one ?column_names f)
    init

let fetch db sql ?column_names f init =
  _fetch db (String.copy sql) ?column_names f init

let fetch_f db ?column_names f init fmt =
  Printf.kprintf (fun q -> _fetch db q ?column_names f init) fmt
