# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: blt2d.tcl,v 2.62 2004/01/01 11:39:06 jfontain Exp $


class blt2DViewer {

    set (axisTickFont) $font::(smallNormal)

    proc blt2DViewer {this parentPath path {labelsColorHeight 0}} viewer {} {
        grid propagate $parentPath 0           ;# so that -width and -height options acting on container frame in derived class work
        grid rowconfigure $parentPath 0 -weight 1                                               ;# so that viewer expands vertically
        $path configure -background $viewer::(background) -cursor {} -highlightthickness 0\
            -plotpadx 1 -plotpady 2                          ;# use minimum padding for extreme values, flat zero line to be visible
        $path yaxis configure -tickshadow {} -title {} -tickfont $(axisTickFont)
#       $path yaxis configure -tickshadow {} -title {} -tickfont $(axisTickFont) -minorticks {0.1 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9}
        $path legend configure -hide 1                                ;# use custom labeler instead allowing coloring for thresholds
        set labels [new colorLabels $parentPath -colorheight $labelsColorHeight]
        viewer::setupDropSite $this $parentPath                               ;# allow dropping of data cells in whole viewer window
        set menu [menu $path.menu -tearoff 0]
        set ($this,help) [new menuContextHelp $menu]
        ::update idletasks                                                                      ;# so that ordinate below is correct
        set y [expr {[$path extents topmargin] + [$path cget -plotborderwidth] + [lindex [$path cget -plotpady] 0]}]
        $menu add command -label [mc Maximum]... -state disabled\
            -command "viewer::limitEntry $this $path w 0 $y -ymaximum maximum $(axisTickFont) {blt2DViewer::yMaximumEntered $this}"
        menuContextHelp::set $($this,help) 0 [mc {set ordinate maximum value or automatic scaling}]
        bindtags $parentPath [concat [bindtags $parentPath] PopupMenu$this]
        bindtags $path [concat [bindtags $path] PopupMenu$this]
        set ($this,colorsMenu) [viewer::createColorsMenu $path "blt2DViewer::setColor $this \$colorLabels::label::(clicked) %c"]
        if {!$global::readOnly} {
            bind PopupMenu$this <ButtonPress-3> "tk_popup $menu %X %Y"
        }
        set label [new imageLabel $path -font $(axisTickFont) -bindtags PopupMenu$this]
        set ($this,drop) [new dropSite -path $path -formats DATACELLS -state disabled\
            -regioncommand "blt2DViewer::dropRegion $this" -command "blt2DViewer::yMaximumCellDrop $this"\
        ]
        set ($this,tips) {}
        if {!$global::readOnly} {
            lappend ($this,tips)\
                [new widgetTip -path $path -rectangle [list 0 0 $viewer::(limitAreaWidth) $viewer::(limitAreaHeight)]]
            lappend ($this,tips) [new widgetTip -path $widget::($label,path)]
        }
        set ($this,maximum) $label
        set ($this,yMaximum) {}
        set ($this,elements) {}
        set ($this,path) $path                                                                                    ;# BLT viewer path
        set ($this,labels) $labels
        set ($this,menu) $menu
        updateTip $this
#       bind $path <Configure> "blt2DViewer::yUpdateGraduations $this"                                         ;# track size updates
    }

    proc ~blt2DViewer {this} {
        if {[info exists ($this,drag)]} {
            delete $($this,drag)
        }
        eval delete $($this,elements)                                                                    ;# delete existing elements
        if {[info exists ($this,selector)]} {
            delete $($this,selector)
        }
        eval delete $($this,labels) $($this,help) $($this,drop) $($this,maximum) $($this,tips)
    }

    proc supportedTypes {this} {
        return $global::numericDataTypes
    }

    # colors of soon to be created cells when initializing from file (invoked from derived classes option procedure)
    proc setCellColors {this list} {
        set ($this,nextCellIndex) 0                                                       ;# initialize cell index in list of colors
    }

    proc dragData {this format} {
        set legends [selector::selected $($this,selector)]
        set selectedElements {}
        foreach element $($this,elements) {
            if {[lsearch -exact $legends $($this,legend,$element)] < 0} continue
            lappend selectedElements $element
        }
        switch $format {
            OBJECTS {
                if {[llength $selectedElements] > 0} {
                    return $selectedElements                                            ;# return selected elements if there are any
                } elseif {[llength $($this,elements)] == 0} {
                    return $this                                                   ;# return graph itself if it contains no elements
                } else {
                    return {}                                                                            ;# return nothing otherwise
                }
            }
            DATACELLS {
                return [cellsFromElements $this $selectedElements]
            }
        }
    }

    proc validateDrag {this legend x y} {
        if {($legend == 0) && ([llength $($this,elements)] == 0)} {
            return 1                                                                                   ;# allow drag of empty viewer
        } elseif {[lsearch -exact [selector::selected $($this,selector)] $legend] >= 0} {
            return 1                                                                     ;# allow dragging from selected legend only
        } else {
            return 0
        }
    }

    proc monitorCell {this array row column} {
        set cell ${array}($row,$column)
        if {![canMonitor $this $array]} return
        if {[lsearch -exact [cellsFromElements $this $($this,elements)] $cell] >= 0} return                ;# already charted, abort
        viewer::registerTrace $this $array
        if {[info exists ($this,nextCellIndex)]} {                           ;# recreate data cell color from recorded configuration
            set color [lindex $composite::($this,-cellcolors) $($this,nextCellIndex)]
            if {[string length $color] == 0} {                 ;# colors list exhausted: we are done initializing from recorded data
                unset color ($this,nextCellIndex)
            } else {
                incr ($this,nextCellIndex)                                                            ;# get ready for upcoming cell
            }
        }
        if {![info exists color]} {set color [viewer::getDisplayColor $cell]}
        set element [newElement $this $($this,path) -color $color]
        set labels $($this,labels)
        if {[llength $($this,elements)] == 0} {                                           ;# display labels on the first new element
            grid $widget::($labels,path) -row 0 -column 1 -sticky nw -ipadx 2 -ipady 2 ;# padding needed for selection sunken relief
        }
        if {$global::readOnly} {
            set legend [colorLabels::new $labels {} -color $color]
        } else {
            set legend [colorLabels::new $labels $($this,colorsMenu) -color $color]
        }
        # keep track of element existence
        switched::configure $element -deletecommand "blt2DViewer::deletedElement $this $array $element"
        lappend ($this,elements) $element
        set ($this,label,$element) [viewer::label $array $row $column]
        set ($this,legend,$element) $legend
        set ($this,cell,$element) $cell
        if {$composite::($this,-draggable)} {                                       ;# selector may not exist if dragging disallowed
            set labelPath $composite::($legend,label,path)
            set drag [new dragSite -path $labelPath -validcommand "blt2DViewer::validateDrag $this $legend"]
            dragSite::provide $drag OBJECTS "blt2DViewer::dragData $this"
            dragSite::provide $drag DATACELLS "blt2DViewer::dragData $this"
            set ($this,drag,$element) $drag
            set selector $($this,selector)
            selector::add $selector $legend
            bind $labelPath <ButtonPress-1> "blt2DViewer::buttonPress $selector $legend"
            bind $labelPath <Control-ButtonPress-1> "selector::toggle $selector $legend"
            bind $labelPath <Shift-ButtonPress-1> "selector::extend $selector $legend"
            bind $labelPath <ButtonRelease-1> "blt2DViewer::buttonRelease $selector $legend 0"
            bind $labelPath <Control-ButtonRelease-1> "blt2DViewer::buttonRelease $selector $legend 1"
            bind $labelPath <Shift-ButtonRelease-1> "blt2DViewer::buttonRelease $selector $legend 1"
        }
        if {[string first ? $($this,label,$element)] >= 0} {                                       ;# label cannot be determined yet
            set ($this,relabel,$element) {}
        } else {                                 ;# display label without value, which suffices when displaying data cells histories
            composite::configure $($this,legend,$element) -text $($this,label,$element)
        }
        switched::configure $($this,drop) -state normal
        $($this,menu) entryconfigure 0 -state normal
        modified $this [llength $($this,elements)]
        if {[info exists ($this,repositionMaximum)]} {
            positionMaximum $this 1
            unset ($this,repositionMaximum)                                                                                  ;# done
        }
    }

    virtual proc canMonitor {this array} {
        return 1
    }

    proc cells {this} {
        return [cellsFromElements $this $($this,elements)]
    }

    proc deletedElement {this array element} {
        viewer::unregisterTrace $this $array                                          ;# trace may no longer be needed on this array
        ldelete ($this,elements) $element
        if {$composite::($this,-draggable)} {
            delete $($this,drag,$element)
            selector::remove $($this,selector) $($this,legend,$element)
        }
        colorLabels::delete $($this,labels) $($this,legend,$element)
        set length [llength $($this,elements)]
        if {$length == 0} {                                                                                    ;# nothing to monitor
            grid forget $widget::($($this,labels),path)                                   ;# no need to display an empty labels area
            composite::configure $this -ymaximum {} -ymaximumcell {}             ;# possibly remove maximums as Y axis becomes empty
            switched::configure $($this,drop) -state disabled
            $($this,menu) entryconfigure 0 -state disabled
        }
        viewer::returnDisplayColor $($this,cell,$element)
        unset ($this,cell,$element) ($this,label,$element) ($this,legend,$element)
        modified $this $length
    }

    virtual proc update {this array} {          ;# update display using cells data (virtual for viewers that handle cells histories)
        updateTimeDisplay $this [set seconds [clock seconds]]
        foreach element $($this,elements) {
            set cell $($this,cell,$element)
            if {[string first $array $cell] != 0} continue                               ;# check that cell belongs to updated array
            if {[catch {set value [set $cell]}]} {
                updateElement $this $element $seconds ?                                                     ;# data no longer exists
                composite::configure $($this,legend,$element) -text "$($this,label,$element): ?"
            } else {
                if {[info exists ($this,relabel,$element)]} {                              ;# if label is not yet defined, update it
                    viewer::parse $cell ignore row column ignore
                    set label [viewer::label $array $row $column]
                    set ($this,label,$element) $label
                    if {[string first ? $label] < 0} {                                               ;# label now completely defined
                        unset ($this,relabel,$element)
                    }
                }
                if {[string is double -strict $value]} {                                                         ;# check if numeric
                    updateElement $this $element $seconds $value                       ;# may be ? if cell value is meant to be void
                } else {
                    updateElement $this $element $seconds ?                                      ;# do not fail on non-numeric value
                }
                composite::configure $($this,legend,$element) -text "$($this,label,$element): $value"
            }
        }
        if {[info exists cell]} {                                                 ;# no maximum can exist when there are no elements
            yUpdateMaximum $this $array
            if {[info exists ($this,relabelTip)]} {updateTip $this}
#           yUpdateGraduations $this
        }
    }

    virtual proc newElement {this path args}                                       ;# let derived class create an element of its own

    virtual proc updateElement {this element seconds value}      ;# let derived class (such as graph, bar chart, ...) update element

    virtual proc updateTimeDisplay {this seconds} {}        ;# eventually let derived class (such as graph) update axis, for example

    virtual proc initializationConfiguration {this} {
        set colors {}
        foreach element $($this,elements) {
            lappend colors [switched::cget $element -color]
        }
        return [list -cellcolors $colors]
    }

    proc cellsFromElements {this elements} {
        set cells {}
        foreach element $elements {
            lappend cells $($this,cell,$element)
        }
        return $cells
    }

    proc monitored {this cell} {
        foreach element $($this,elements) {
            if {[string equal $($this,cell,$element) $cell]} {
                return 1
            }
        }
        return 0
    }

    proc setLegendsState {this legends select} {
        if {$select} {
            set relief sunken
        } else {
            set relief flat
        }
        foreach legend $legends {
            composite::configure $legend -relief $relief
        }
    }

    proc allowDrag {this} {
        set ($this,drag) [new dragSite -path $($this,path) -validcommand "blt2DViewer::validateDrag $this 0"]    ;# for empty viewer
        dragSite::provide $($this,drag) OBJECTS "blt2DViewer::dragData $this"        ;# drag sites for legends are setup dynamically
        set ($this,selector) [new objectSelector -selectcommand "blt2DViewer::setLegendsState $this"]
    }

    proc setCellColor {this cell color} {
        foreach element $($this,elements) {
            if {[string equal $($this,cell,$element) $cell]} {
                composite::configure $($this,legend,$element) -background $color
                return                                                       ;# done since there cannot be duplicate monitored cells
            }
        }
    }

    virtual proc modified {this monitored} {}                                               ;# number of monitored cells has changed

    proc buttonPress {selector legend} {
        foreach selected [selector::selected $selector] {
            if {$selected == $legend} return                               ;# in an already selected legend, do not change selection
        }
        selector::select $selector $legend
    }

    proc buttonRelease {selector legend extended} {                 ;# extended means that there is an extended selection in process
        if {$extended} return
        set list [selector::selected $selector]
        if {[llength $list] <= 1} return                                          ;# nothing to do if there is no multiple selection
        foreach selected $list {
            if {$selected == $legend} {                                                             ;# in an already selected legend
                selector::select $selector $legend                                                   ;# set selection to sole legend
                return
            }
        }
    }

    proc yResetMaximum {this} {
        composite::configure $($this,maximum) -text {} -image {}
        $($this,path) yaxis configure -max {}
        positionMaximum $this 0
        set ($this,yMaximum) {}                                                                                       ;# reset cache
        updateTip $this
#       yUpdateGraduations $this
    }

    proc yUpdateMaximum {this {array {}}} {
        if {[string length $composite::($this,-ymaximum)] > 0} {                                                    ;# fixed maximum
            set value $composite::($this,-ymaximum)
            if {[string equal $value $($this,yMaximum)]} return                                                ;# no change in cache
            ### when -ymaximum is set at BLT chart creation time, left margin extent, as returned by the BLT library, is not correct
            ### until an element is created, so trigger another maximum update at that time from here:
            set ($this,repositionMaximum) {}
        } elseif {([string length $array] > 0) && ![catch {set cell $($this,yMaximumCell)}]} {                       ;# maximum cell
            if {[string first $array $cell] != 0} return                                 ;# check that cell belongs to updated array
            set value ?; catch {set value [set $cell]}
            if {[string equal $value $($this,yMaximum)]} return                                                ;# no change in cache
        } else return                                                                                   ;# no maximum, nothing to do
        set ($this,yMaximum) $value                                                                                         ;# cache
        if {[string equal $value ?]} {                                                                         ;# void maximum value
            set value {}
            composite::configure $($this,maximum) -text ? -image $viewer::(rightDarkGrayArrow)
        } else {                                                                                              ;# valid maximum value
            composite::configure $($this,maximum) -text $value -image $viewer::(rightRedArrow)
        }
        $($this,path) yaxis configure -max $value
        positionMaximum $this 1
#       yUpdateGraduations $this
    }

    proc dropRegion {this} {
        set X [winfo rootx $($this,path)]; set Y [winfo rooty $($this,path)]
        return [list $X $Y [expr {$X + $viewer::(limitAreaWidth)}] [expr {$Y + $viewer::(limitAreaHeight)}]]
    }

    proc yMaximumCellDrop {this} {
        if {[llength $dragSite::data(DATACELLS)] != 1} {
            lifoLabel::flash $global::messenger {only one data cell can be used for maximum ordinate}
            bell
            return
        }
        set cell [lindex $dragSite::data(DATACELLS) 0]
        viewer::parse $cell ignore ignore ignore type
        if {[lsearch -exact [supportedTypes $this] $type] < 0} {                          ;# type must exist since cell array exists
            viewer::wrongTypeMessage $type
        } else {
            composite::configure $this -ymaximumcell $cell
        }
    }

    proc ySetMaximum {this value} {                                                ;# invoked by derived class when -ymaximum is set
        if {[string length $value] == 0} {                                             ;# remove fixed maximum value or maximum cell
            yResetMaximum $this
        } else {
            composite::configure $this -ymaximumcell {}
            yUpdateMaximum $this
            updateTip $this
        }
    }

    proc ySetMaximumCell {this cell} {                                         ;# invoked by derived class when -ymaximumcell is set
        if {[info exists ($this,yMaximumCell)]} {                                                        ;# there was a maximum cell
            viewer::parse $($this,yMaximumCell) array ignore ignore ignore
            viewer::unregisterTrace $this $array                                                                ;# no longer monitor
            unset ($this,yMaximumCell)
        }
        if {[string length $cell] == 0} {
            yResetMaximum $this
        } else {
            composite::configure $this -ymaximum {}
            viewer::parse $cell array ignore ignore ignore
            viewer::registerTrace $this $array
            set ($this,yMaximumCell) $cell
            yUpdateMaximum $this $array
            updateTip $this
        }
    }

    proc positionMaximum {this visible} {
        if {$visible} {
            ::update idletasks                                                             ;# so that measurements below are correct
            set path $($this,path)
            set width [expr {[$path extents leftmargin] - round([$path yaxis cget -ticklength] / 2.0)}]
            composite::configure $($this,maximum) -width $width
            place $widget::($($this,maximum),path) -anchor e -x $width\
                -y [expr {[$path extents topmargin] + [$path cget -plotborderwidth] + [lindex [$path cget -plotpady] 0]}]
        } else {
            place forget $widget::($($this,maximum),path)
        }
    }

    proc yMaximumEntered {this value} {
        if {[string length $value] == 0} {
            composite::configure $this -ymaximumcell {}                             ;# force deletion of maximum cell on empty entry
        }
        composite::configure $this -ymaximum $value
    }

    proc updateTip {this} {
        if {[info exists ($this,yMaximumCell)]} {
            viewer::parse $($this,yMaximumCell) array row column type
            set label [viewer::label $array $row $column]
            if {[string first ? $label] < 0} {                                                           ;# label completely defined
                catch {unset ($this,relabelTip)}
            } else {
                set ($this,relabelTip) {}                                                          ;# label cannot be determined yet
            }
            set text "maximum cell: $label"
        } elseif {![catch {set value $composite::($this,-ymaximum)}] && ([string length $value] > 0)} {
            # note: composite data does not exist yet when this procedure is invoked from the constructor of this class
            set text "fixed maximum value set to $value"
        } else {
            set text {set fixed maximum value or drop cell as dynamic maximum value here}
        }
        foreach tip $($this,tips) {switched::configure $tip -text $text}
    }

    proc setColor {this legend color} {
        foreach element $($this,elements) {
            if {$($this,legend,$element) == $legend} break
        }
        switched::configure $element -color $color
        composite::configure $legend -color $color
    }

    virtual proc updateLabels {this {values 1}} {
        if {$values} {                                                                                   ;# include values in labels
            foreach element $($this,elements) {
                viewer::parse $($this,cell,$element) array ignore ignore ignore
                set ($this,relabel,$element) {}
                set update($array) {}
            }
            foreach array [array names update] {
                update $this $array
            }
        } else {
            foreach element $($this,elements) {
                viewer::parse $($this,cell,$element) array row column ignore
                composite::configure $($this,legend,$element) -text [set ($this,label,$element) [viewer::label $array $row $column]]
            }
        }
        if {[info exists ($this,yMaximumCell)]} {                                                         ;# there is a maximum cell
            updateTip $this
        }
    }

#   proc yUpdateGraduations {this} {
#       ::update idletasks                                                                                ;# make sure data is fresh
#       set path $($this,path)
#       foreach {minimum maximum} [$path yaxis limits] {}
#       set range [expr {$maximum - $minimum}]
#       if {$range <= 0} return
#       set height [$path extents plotheight]
#       if {$height <= 0} return
#       if {\
#           [info exists ($this,range)] && ($range == $($this,range)) &&\
#           [info exists ($this,height)] && ($height == $($this,height))\
#       } return                                                                                                        ;# no change
#       set ($this,range) $range
#       set ($this,height) $height
#       # major ticks step: leave at least 2 pixels between adjacent minor ticks (assuming there are 10 between major ticks)
#       set step [expr {10 * pow(10, ceil(log10((2.0 * $range) / $height)))}]
#       if {$step >= $range} {
#           set step [expr {pow(10, floor(log10($range)))}]
#       }
#       set y [expr {floor($minimum / $step) * $step}]
#       for {set list $y} {$y <= $maximum} {} {
#           set y [expr {$y + $step}]
#           lappend list $y
#       }
#       $path yaxis configure -majorticks $list
#   }

}
