# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu


# various procedures mostly used in moodss.tcl

# $Id: procs.tcl,v 1.50 2004/01/28 22:48:01 jfontain Exp $


proc printUsage {exitCode} {
    puts stderr "Usage: $::argv0 \[OPTION\]... \[MODULE\] \[MODULE\]..."
    puts stderr {  --debug          module errors verbose reporting}
    puts stderr {  -f, --file       configuration file name}
    puts stderr {  -h, --help       display this help and exit}
    puts stderr {  -p, --poll-time  poll time in seconds}
    puts stderr {  -r, --read-only  disable viewer creation, editing, ...}
    puts stderr {  -S, --static     disable internal window manager sizing and moving}
    puts stderr {  --show-modules   try to find available moodss modules}
    puts stderr {  --tabs           notebook pages tabs labels}
    puts stderr {  --version        output version information and exit}
    exit $exitCode
}

proc printVersion {} {
    puts "moodss (a Modular Object Oriented Dynamic SpreadSheet) version $global::applicationVersion"
}

proc loadFromFile {name} {                                       ;# eventually unload all existing modules and load from a save file
    clearModulesAndViewers
    set global::saveFile $name
    set global::fileDirectory [file dirname $name]
    set initializer [new record -file $name]
    record::read $initializer
    configuration::load [record::configurationData $initializer]                                 ;# set global values from save file
    modules::parse [record::modulesWithArguments $initializer]                                                          ;# recursive
    set modules::(initialized) [record::modules $initializer]
    return $initializer
}

proc clearModulesAndViewers {} {
    foreach instance $modules::(instances) {
        dynamicallyUnloadModule $modules::instance::($instance,namespace)
    }
    if {[llength $modules::(instances)] > 0} {
        error {internal moodss error: please report to author with error trace}
    }
    foreach viewer $viewer::(list) {                                         ;# delete all existing viewers except thresholds viewer
        set class [classof $viewer]
        switch $class {
            ::store - ::store::dialog - ::thresholdLabel - ::thresholds {
                ${class}::reset $viewer                                         ;# make sure to remove now obsolete existing entries
            }
            default {
                delete $viewer
            }
        }
    }
    if {[info exists databaseInstances::singleton]} {
        delete $databaseInstances::singleton
    }
}

proc clear {} {                  ;# return true if user went ahead and cleared main window or return false in case of change in mind
    static busy 0

    if {$busy} return             ;# protection against overlapping invocations from fast clicking, as a lot of updates occur within
    if {[needsSaving]} {                             ;# see if there are changes that the user may want to be saved before reloading
        switch [inquireSaving] {
            yes {
                save
                if {[needsSaving]} {return 0}                        ;# data was not saved: assume the user wants to abort reloading
            }
            cancel {return 0}
        }
    }
    set busy 1
    if {[info exists ::initializer]} {
        delete $::initializer
        unset ::initializer
    }
    clearModulesAndViewers
    databaseConnection 0                                                                                 ;# disconnect from database
    set global::saveFile {}
    updateFileSaveHelp {}                                                                      ;# to reflect save file disappearance
    updateTitle
    updateMenuWidget
    updateToolBar
    updateDragAndDropZone
    configuration::load [preferences::read]                                                             ;# reinitialize from rc file
    record::snapshot   ;# take a snapshot of new configuration os that user is not asked to save wnen trying to open a new file next
    set busy 0
    return 1
}

proc reload {} {                                                                      ;# restart by loading modules from a save file
    if {[needsSaving]} {                                    ;# there are changes that the user may want to be saved before reloading
        switch [inquireSaving] {
            yes {
                save
                if {[needsSaving]} return                            ;# data was not saved: assume the user wants to abort reloading
            }
            cancel return
        }
    }
    set file [tk_getOpenFile\
        -title {moodss: Open} -initialdir $global::fileDirectory -defaultextension .moo -filetypes {{{moodss data} .moo}}\
    ]
    if {[string length $file] == 0} return                                                                  ;# user canceled loading
    databaseConnection 0                                                                                 ;# disconnect from database
    set global::fileDirectory [file dirname $file]
    if {[info exists ::initializer]} {
        delete $::initializer
        unset ::initializer
    }
    set ::initializer [loadFromFile $file]
    foreach {width height} [record::sizes $::initializer] {}                                                 ;# used stored geometry
    wm geometry . ${width}x$height
    modules::initialize 0 initializationErrorMessageBox
    modules::setPollTimes [record::pollTime $::initializer]
    foreach instance $modules::(instances) {
        displayModule $instance $::draggable
    }
    summaryTable::reset; currentValueTable::reset                    ;# so that data table names are generated as when first started
    createSavedViewers $::initializer
    pages::manageScrolledCanvas                                                                      ;# so that first page is on top
    updateTitle
    updateMenuWidget
    updateToolBar
    updateDragAndDropZone
    updateFileSaveHelp $file                                                                  ;# since current save file has changed
    pages::eventuallyRefresh
    refresh                                                                               ;# make sure data is immediately displayed
    update                                               ;# required so that table and viewer windows sizes are correct for snapshot
    record::snapshot                       ;# take a snap shot of new configuration so any future changes are detected as meaningful
}

proc createCellsViewer {class cells draggable static {pollTime {}}} {
    if {[string length $pollTime] == 0} {
        set viewer [new $class $global::canvas -draggable $draggable]
    } else {
        set viewer [new $class $global::canvas -draggable $draggable -interval $pollTime]
    }
    if {[viewer::view $viewer $cells]} {
        manageViewer $viewer 1 -static $static -dragobject $viewer
        return $viewer
    } else {                                                                   ;# do not create new viewer if cells cannot be viewed
        delete $viewer
        return 0
    }
}

proc manageViewer {viewer destroyable args} {        ;# viewer or table, arguments are window manager configuration switched options
    set path $widget::($viewer,path)
    canvasWindowManager::manage $global::windowManager $path $viewer
    eval canvasWindowManager::configure $global::windowManager $path $args
    if {$destroyable} {
        composite::configure $viewer -deletecommand "canvasWindowManager::unmanage $global::windowManager $path"
    }
}

proc save {{ask 0}} {                           ;# save current configuration in user defined file or currently defined storage file
    if {$ask || ([string length $global::saveFile] == 0)} {
        set file [tk_getSaveFile\
            -title {moodss: Save} -initialdir $global::fileDirectory -defaultextension .moo -filetypes {{{moodss data} .moo}}\
            -initialfile $global::saveFile
        ]
        if {[string length $file] == 0} return                                                               ;# user canceled saving
        set global::saveFile $file
        set global::fileDirectory [file dirname $file]
        updateFileSaveHelp $file
    }
    lifoLabel::push $global::messenger "saving in $global::saveFile..."
    update idletasks                                                                                 ;# make sure message is visible
    set record [new record -file $global::saveFile]
    set error [catch {record::write $record} message]
    lifoLabel::pop $global::messenger
    if {$error} {
        tk_messageBox -title {moodss: Save} -type ok -icon error -message $message
    }
    delete $record
    if {!$error} record::snapshot                                                                  ;# remember current configuration
}

proc refresh {} {
    static updateEvent

    catch {after cancel $updateEvent}                                                             ;# eventually cancel current event
    if {[llength $modules::(synchronous)] == 0} return                                                              ;# nothing to do
    foreach instance $modules::(synchronous) {
        set namespace $modules::instance::($instance,namespace)
        ${namespace}::update                                                                ;# ask module to update its dynamic data
    }
    if {$global::pollTime > 0} {                                                                   ;# any synchronous modules loaded
        set updateEvent [after [expr {1000 * $global::pollTime}] refresh]                                 ;# convert to milliseconds
    }
}

# invoked by thresholds code when a threshold condition occurs (color and level are empty for resetting)
proc cellThresholdCondition {array row column color level summary} {    ;# summary is a short description of the threshold condition
    dataTable::cellThresholdCondition $array $row $column
    viewer::cellThresholdCondition $array $row $column $color $level $summary
}

proc inquireSaving {} {
    set message {There are unsaved configuration changes. Do you want them saved to file}
    if {[string length $::global::saveFile] > 0} {                                                           ;# there is a save file
        append message ": $::global::saveFile"
    }
    append message ?
    array set answer {0 yes 1 no 2 cancel}
    return $answer([tk_dialog .saveorexit {moodss: Save} $message question 0 Yes No Cancel])
}

proc needsSaving {} {                                                              ;# no need to save if there are no loaded modules
    return [expr {[record::changed] && ([llength $modules::(instances)] > 0)}]
}

proc manageToolBar {{save 1}} {                                                              ;# whether to save state in preferences
    set bar [updateToolBar]
    if {$global::showToolBar} {
        grid $bar -row 0 -column 0 -sticky we
    } else {
        grid forget $bar
    }
    if {$save} {
        preferences::update
    }
}

proc createSavedViewers {record} {                                                                   ;# strictly viewers, not tables
    if {[llength [set range [record::databaseRange $record]]] > 0} {                               ;# saved in database history mode
        monitorDatabaseInstances $range                                                    ;# display loaded instance modules viewer
    }
    foreach {class cells x y width height level switchedOptions} [record::viewersData $record] {
        switch $class {
            ::store - ::thresholds {
                set viewer [set ${class}::singleton]         ;# store and thresholds viewers are special cases and are not displayed
                eval switched::configure $viewer $switchedOptions
            }
            ::thresholdLabel {
                set viewer [set ${class}::singleton]                ;# threshold label viewer is a special case and is not displayed
                eval composite::configure $viewer $switchedOptions
            }
            default {
                set viewer [eval new $class $global::canvas $switchedOptions -draggable $::draggable]
                foreach list [composite::configure $viewer] {
                    if {[string equal [lindex $list 0] -interval]} {                              ;# viewer supports interval option
                        composite::configure $viewer -interval $global::pollTime                          ;# so use current interval
                        break                                                                                                ;# done
                    }
                }
                if {[viewer::manageable $viewer]} {
                    manageViewer $viewer 1 -static $global::static -setx $x -sety $y -setwidth $width -setheight $height\
                        -level $level -dragobject $viewer
                }
            }
        }
        set viewerCells($viewer) $cells                                                                              ;# gather cells
    }
    # monitor cells now that all viewers exist (for example, summary tables have their own data and thus need be created before
    # other viewers can reference them)
    foreach {viewer cells} [array get viewerCells] {
        viewer::view $viewer $cells
    }
}

# must be invoked only when the application is running, that is after all the save file and command line modules have been loaded
proc dynamicallyLoadModules {arguments} {     ;# arguments list format is: module [-option [value] -option ...] module [-option ...]
    set instances $modules::(instances)
    modules::parse $arguments
    modules::initialize                                               ;# initializes only modules that have not yet been initialized
    modules::setPollTimes                                            ;# recalculate valid poll times but do not change current value
    set first 1
    foreach instance $modules::(instances) {
        if {[lsearch -exact $instances $instance] >= 0} continue
        # new module or instance:
        if {$first} {                ;# reset next module table coordinates so that successively loaded modules do not go off screen
            displayModule $instance $::draggable 1
            set first 0
        } else {
            displayModule $instance $::draggable
        }
    }
    updateTitle
    updateMenuWidget
    updateToolBar
    refresh                                                                               ;# make sure data is immediately displayed
}

proc dynamicallyUnloadModule {namespace} {
    foreach instance $modules::(instances) {
        if {[string equal $modules::instance::($instance,namespace) $namespace]} break                                      ;# found
    }
    if {[lindex $modules::instance::($instance,times) 0] >= 0} {                                    ;# then if module is synchronous
        ldelete modules::(synchronous) $instance                                                              ;# remove it from list
    }
    foreach table $dataTable::(list) {                                                                      ;# delete related tables
        if {[string equal [modules::namespaceFromArray [composite::cget $table -data]] $namespace]} {
            canvasWindowManager::unmanage $global::windowManager $widget::($table,path)
            delete $table
        }
    }
    modules::instance::empty $instance                             ;# empty module data so that related viewers can show empty cells
    modules::unload $instance
    modules::setPollTimes                                            ;# recalculate valid poll times but do not change current value
    updateTitle
    updateMenuWidget
    updateToolBar
}

proc residentTraceModule {display} {                                      ;# initialize and eventually display resident trace module
    if {![winfo exists .trace]} {
        toplevel .trace
        wm withdraw .trace
        wm group .trace .                                            ;# for proper window manager (windowmaker for example) behavior
        wm title .trace {moodss: trace}
        set namespace $modules::instance::($modules::(trace),namespace)
        set table [new dataTable .trace -data ${namespace}::data]
        dataTable::update $table                                      ;# force refreshing of the display so data appears immediately
        # handle closing via window manager:
        wm protocol .trace WM_DELETE_WINDOW "wm withdraw .trace; set global::showTrace 0"
        pack $widget::($table,path) -fill both -expand 1
    }
    if {$display} {
        wm deiconify .trace
    } else {
        wm withdraw .trace
    }
    after idle {focus .}                                                                        ;# keep the focus on the main window
}

proc displayModule {instance draggable {resetOrigin 0}} {
    static x
    static y

    if {![info exists x] || $resetOrigin} {
        foreach {x y dummy dummy} [$global::canvas cget -scrollregion] {}                           ;# next module table coordinates
    }
    if {[lindex $modules::instance::($instance,times) 0] >= 0} {                                         ;# if module is synchronous
### should be done in modules code ###
        lappend modules::(synchronous) $instance
    }
    if {[info exists modules::instance::($instance,views)]} {                             ;# check whether several views are defined
        set viewMembers $modules::instance::($instance,views)                             ;# create and manage a table for each view
    } else {
        set viewMembers {{}}                                                           ;# there is a single table (the default view)
    }
    set index 0
    set namespace $modules::instance::($instance,namespace)
    foreach members $viewMembers {
        set initialize [expr {[info exists ::initializer] && ([lsearch -exact $modules::(initialized) $namespace] >= 0)}]
        if {$initialize} {
            set arguments [record::tableOptions $::initializer $namespace $index]                ;# extra stored arguments for table
        } else {
            set arguments {}
        }
        if {![catch {set ${namespace}::data(resizableColumns)} resizable]} {
            lappend arguments -resizablecolumns $resizable
        }
        if {[llength $members] > 0} {                                                                                ;# it is a view
            array set ::view$instance $members
            set table\
                [eval new dataTable $global::canvas -data ${namespace}::data -view ::view$instance -draggable $draggable $arguments]
            unset ::view$instance
        } else {                                                                                     ;# use single default data view
            set table [eval new dataTable $global::canvas -data ${namespace}::data -draggable $draggable $arguments]
        }
        if {[info exists modules::instance::($instance,identifier)]} {                                 ;# set a title for data table
            set title $modules::instance::($instance,identifier)                                    ;# favor identifier if it exists
        } else {
            set title $namespace                                                                      ;# else simply use module name
        }
        if {$initialize} {                                                                       ;# setup initialized modules tables
            # use stored window manager initialization data for table if it exists:
            set list [record::tableWindowManagerData $::initializer $namespace $index]
            if {[llength $list] > 0} {
                foreach {x y width height level xIcon yIcon} $list {}
                manageViewer $table 0 -static $global::static -setx $x -sety $y -setwidth $width -setheight $height\
                    -level $level -title $title -iconx $xIcon -icony $yIcon
            } else {
                manageViewer $table 0 -static $global::static -setx $x -sety $y -title $title
            }
        } else {
            manageViewer $table 0 -static $global::static -setx $x -sety $y -title $title
        }
        set x [expr {$x + $global::xWindowManagerInitialOffset}]                          ;# offset tables to achieve a nicer layout
        set y [expr {$y + $global::yWindowManagerInitialOffset}]
        incr index                                                                                      ;# next view for initializer
    }
}

proc initializationErrorMessageBox {namespace message} {                                                  ;# namespace of the module
    set top [new toplevel .]
    set path $widget::($top,path)
    wm transient $path .
    wm title $path "moodss: Error initializing module \"$namespace\""
    wm group $path .                                                 ;# for proper window manager (windowmaker for example) behavior
    wm protocol $path WM_DELETE_WINDOW "delete $top"                                                                 ;# self cleanup
    set text [message $path.message -text $message -font $font::(mediumNormal) -justify left -width 640]
    # stop sign from ::tk::MessageBox{} in Tk library msgbox.tcl:
    set canvas [canvas $path.icon -width 32 -height 32 -highlightthickness 0]
    $canvas create oval 0 0 31 31 -fill red -outline black
    $canvas create line 9 9 23 23 -fill white -width 4
    $canvas create line 9 23 23 9 -fill white -width 4
    grid rowconfigure $path 0 -weight 1
    grid columnconfigure $path 1 -weight 1
    grid $canvas -row 0 -column 0 -sticky nw -padx 2 -pady 2
    grid $text -row 0 -column 1 -sticky nw
    grid [frame $path.separator -relief sunken -borderwidth 1 -height 2] -row 1 -column 0  -columnspan 100 -stick we -pady 2
    grid [button $path.close -text Close -command "destroy $path"] -row 2 -column 0 -columnspan 100 -padx 2 -pady 2
}

proc databaseConnection {connect} {                                               ;# boolean: or disconnect, procedure is idempotent
    if {$connect} {
        if {$global::database != 0} return                                                                      ;# already connected
        lifoLabel::push $global::messenger {connecting to database...}
    } else {
        if {$global::database == 0} return                                                                   ;# already disconnected
        lifoLabel::push $global::messenger {disconnecting from database...}
    }
    busy 1 .
    if {$connect} {
        set database [eval new database $global::databaseOptions]
        if {[string length $database::($database,error)] > 0} {              ;# there was a problem probably due to misconfiguration
            tk_messageBox -title {moodss: database} -type ok -icon error -message $database::($database,error)
            delete $database
        } else {
            if {$database::($database,created)} {
                modules::trace {} moodss(database) {created table(s) in moodss database}
            }
            set global::database $database
        }
    } else {
        delete $global::database                                                                                       ;# disconnect
        set global::database 0
    }
    busy 0 .
    lifoLabel::pop $global::messenger
}

proc loadFromDatabase {draggable static} {                                    ;# get ready to load data cell histories from database
    if {![info exists databaseInstances::singleton]} {                                               ;# not in database mode already
        if {![clear]} return                                                    ;# eventually unload all modules unless user aborted
        databaseConnection 1                                                                                  ;# connect to database
        if {$global::database == 0} return                                                             ;# database connection failed
    }
    database::displayAndSelectInstances             ;# show dialog box with modules instances, data cells histories, ... in database
    createInstancesViewer $draggable $static
    updateMenuWidget
    updateToolBar
    updateDragAndDropZone
    # monitor selected instance if any and eventually delete empty instances viewer when closed:
    switched::configure $database::(dialog) -command "databaseInstances::monitor $databaseInstances::singleton"\
       -deletecommand {after idle {databaseInstances::deleteEmpty}}  ;# note: avoid loop in mutual destruction with instances viewer
}

proc createInstancesViewer {draggable static} {
    if {[info exists databaseInstances::singleton]} return
    set instances [new databaseInstances $global::canvas -draggable $draggable]               ;# create database instances singleton
    set path $widget::($instances,path)
    canvasWindowManager::manage $global::windowManager $path $instances
    set title {database module instances}
    if {[info exists ::initializer] && ([llength [set list [record::databaseViewerWindowManagerData $::initializer]]] > 0)} {
        foreach {x y width height xIcon yIcon} $list {}                                     ;# recorded window manager configuration
        canvasWindowManager::configure $global::windowManager $path\
            -setx $x -sety $y -setwidth $width -setheight $height -iconx $xIcon -icony $yIcon\
            -static $static -title $title -level $global::integerMinimum                     ;# always stack below all other windows
    } else {
        canvasWindowManager::configure $global::windowManager $path\
            -static $static -title $title -level $global::integerMinimum                     ;# always stack below all other windows
    }
    # clear on self deletion to avoid recursion in clear{}, delayed to avoid loop in mutual destruction with instances dialog box
    composite::configure $instances -selfdeletecommand {after idle clear}\
        -deletecommand "canvasWindowManager::unmanage $global::windowManager $path; database::removeInstances"
}

proc databaseRecording {start} {                                                                           ;# boolean: start or stop
    if {$start} {
        databaseConnection 1                                                                                  ;# connect to database
        if {$global::database == 0} return                                                             ;# database connection failed
        if {[llength [viewer::cells $store::singleton]] == 0} {
            tk_messageBox -title {moodss: cells database history} -type ok -icon error\
                -message {no cells were set to be monitored (use Edit Database menu)}
            return
        }
        refresh                         ;# attempt data storage from store class so that eventual errors can be detected immediately
    } else {
        databaseConnection 0                                                                             ;# disconnect from database
    }
    updateMenuWidget
    updateToolBar
}

proc monitorDatabaseInstances {presetRange} {                                  ;# used when loading a database history configuration
    databaseConnection 1                                                                                      ;# connect to database
    if {$global::database == 0} {exit 1}                                             ;# database connection failure is a fatal error
    createInstancesViewer [expr {!$global::readOnly}] $global::static
    foreach instance $modules::(instances) {                                     ;# display all instance modules in instances viewer
        foreach {name index} [modules::decoded $modules::instance::($instance,namespace)] {}    ;# index is database instance number
        if {![string equal $name instance]} {error "not an instance module in history mode: $name"}
        array set option $modules::instance::($instance,arguments)
        databaseInstances::monitor $databaseInstances::singleton [list $index $name $option(-identifier) $option(-arguments)] 0
    }
    eval databaseInstances::setCursors $databaseInstances::singleton $presetRange
}

proc updateCanvasSize {args} {                                                                             ;# ignore trace arguments
    $global::canvas configure -width $global::canvasWidth -height $global::canvasHeight\
        -scrollregion [list 0 0 $global::canvasWidth $global::canvasHeight]
}

proc updateCanvasBackground {args} {                                                                       ;# ignore trace arguments
    $global::canvas configure -background $global::canvasBackground
}

proc traceDialog {title message {exit 0}} {            ;# with text area so that message cutting and pasting is possible by the user
    set dialog [new dialogBox . -title $title -buttons x -default x -x [winfo pointerx .] -y [winfo pointery .]]
    if {$exit} {composite::configure $dialog -labels {x Exit}}
    set frame [frame $widget::($dialog,path).frame]
    set scroll [new scroll text $frame -height 100]
    composite::configure $dialog -deletecommand "delete $scroll; set ::traceDialogDone {}"
    set text $composite::($scroll,scrolled,path)
    $text insert end $message
    $text configure -font $font::(mediumNormal) -wrap word -state disabled
    pack $widget::($scroll,path) -fill both -expand 1
    dialogBox::display $dialog $frame
    vwait ::traceDialogDone
}
