# copyright (C) 1997-2004 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: viewer.tcl,v 2.56 2004/01/01 11:39:06 jfontain Exp $


class viewer {                                                                             ;# handle viewers related functionalities
    # viewers do not derive from a common interface class but rather support a common set a options through the composite class

    set (list) {}
if {$global::withGUI} {
    set (background) $widget::option(label,background)              ;# so that all viewers have the same background on all platforms
    set (rightRedArrow) [image create photo -data {R0lGODlhBQAJAIAAAP8AAP8AACH5BAEKAAEALAAAAAAFAAkAAAIMRB5gp9v2YlJsJRQKADs=}]
    set (rightDarkGrayArrow) [image create photo -data {R0lGODlhBQAJAIAAAKCgoP///yH5BAEKAAEALAAAAAAFAAkAAAIMRB5gp9v2YlJsJRQKADs=}]
    set (limitAreaWidth) 40
    set (limitAreaHeight) 20
}

    proc viewer {this} {
        lappend (list) $this
    }

    proc ~viewer {this} {
        dataTrace::unregister $this                                                                       ;# remove all array traces
        if {[info exists ($this,drop)]} {
            delete $($this,drop)
        }
        ldelete (list) $this
if {$global::withGUI} {
        pages::monitorActiveCells                ;# refresh pages monitored cells since cells with thresholds could have disappeared
        thresholdLabel::monitorActiveCells                                               ;# refresh global thresholds viewer as well
}
    }

    virtual proc supportedTypes {this}

    proc view {this cells} {                                                                  ;# cells is a list of data array cells
        set list {}
        foreach cell $cells {
            parse $cell array row column type
            # no corresponding module (happens when loading a viewer from a save file without the corresponding module):
            if {![info exists $array]} continue
            if {[lsearch -exact [supportedTypes $this] $type] < 0} {                      ;# type must exist since cell array exists
if {$global::withGUI} {
                wrongTypeMessage $type
}
                return 0                                                                                     ;# give up on all cells
            }
            set update($array) {}
            lappend list $array $row $column
        }
        # warning: it is important to monitor cell in the original order as some viewers, such as freetext, expect it:
        foreach {array row column} $list {
            monitorCell $this $array $row $column
if {$global::withGUI} {
            # eventually initialize cell thresholds data:
            foreach {color level summary} [thresholds::cellData $array $row $column] {
                thresholdCondition $this $array $row $column $color $level $summary
            }
            setCellColor $this ${array}($row,$column) [cellThresholdColor $array $row $column]
}
        }
        foreach array [array names update] {                                        ;# update viewer with current values immediately
            update $this $array
        }
        return 1                                                                                       ;# cells accepted for viewing
    }

    virtual proc monitorCell {this array row column}

    # public procedure. note that type will be set only if the cell array actually exists when this procedure is invoked
    proc parse {dataCell arrayName rowName columnName typeName} {
        upvar 1 $arrayName cellArray $rowName cellRow $columnName cellColumn $typeName cellType

        if {([scan $dataCell {%[^(](%u,%u)} array row column] != 3) || ($row < 0) || ($column < 0)} {
            error "\"$dataCell\" is not a valid array cell"
        }
        set cellArray $array; set cellRow $row; set cellColumn $column; catch {set cellType [set ${array}($column,type)]}
    }

    proc updateInterval {value} {                                             ;# static procedure for updating all viewers intervals
        foreach viewer $(list) {
            catch {composite::configure $viewer -interval $value}                 ;# some viewers do not support the interval option
        }
    }

    # derive a label from cell using the tabular data index columns. array name must be qualified
    proc label {array row column {identify {}}} {                                                     ;# allow forcing module header
        set label {}
        if {[string length $identify] == 0} {
            set identify $global::cellsLabelModuleHeader                                                     ;# use default behavior
        }
        if {$identify} {
            set identifier [modules::identifier $array]                                         ;# see if array needs identification
            if {[string length $identifier] > 0} {               ;# data comes from a module array (could come from a summary table)
                set label "$identifier: "
            }
        }
        if {[catch {set ${array}(indexColumns)} columns]} {                                                      ;# no index columns
            set columns 0                                                                 ;# use first column as single index column
        }
        foreach index $columns {                                                                      ;# use index columns for label
            if {[catch {set ${array}($row,$index)} value]} {
                append label {? }               ;# cell may no longer exist if originating from a save file, give user some feedback
            } elseif {[string length $value] > 0} {                                                            ;# ignore empty cells
                append label "$value "
            }
        }
        append label [set ${array}($column,label)]
        return $label
    }

    virtual proc update {this array}                                                              ;# update display using cells data

    proc registerTrace {this array {last 0}} {
        dataTrace::register $this $array "viewer::update $this $array" $last
    }

    proc unregisterTrace {this array} {
        dataTrace::unregister $this $array
    }

    virtual proc cells {this}

    virtual proc initializationConfiguration {this} { ;# configuration with switch / value option pairs for initialization from file
        return {}
    }

if {$global::withGUI} {

    proc setupDropSite {this path} {                                 ;# allow dropping of data cells, viewer mutation or kill action
        set ($this,drop) [new dropSite -path $path -formats {DATACELLS VIEWER KILL} -command "viewer::handleDrop $this"]
    }

    proc handleDrop {this} {
        if {![catch {set dragSite::data(DATACELLS)} data]} {
            view $this $data
        } elseif {![catch {set dragSite::data(VIEWER)} data]} {
            mutate $this $data
        } elseif {[info exists dragSite::data(KILL)]} {
            delete $this                                                                                           ;# self destructs
        }
    }

    proc mutate {this class} {
        if {[string equal $class [classof $this]]} return                            ;# no need to create a viewer of the same class
        set viewer [eval new $class $global::canvas -draggable [composite::cget $this -draggable]]
        foreach list [composite::configure $viewer] {
            if {[string equal [lindex $list 0] -interval]} {                                      ;# viewer supports interval option
                composite::configure $viewer -interval $global::pollTime                                  ;# so use current interval
                break                                                                                                        ;# done
            }
        }
        # only attempt to view cells that still exist (for example, some cells may come from a vanished summary table)
        set cells {}
        set count 0
        foreach cell [cells $this] {
            if {[info exists $cell]} {
                lappend cells $cell
            }
            incr count
        }
        if {[llength $cells] > 0} {
            view $viewer $cells
        }
        if {[llength $cells] < $count} {                                                                 ;# warn user when necessary
            lifoLabel::flash $global::messenger {some data cells no longer exist}
        }
        foreach {x y width height} [canvasWindowManager::getGeometry $global::windowManager $widget::($this,path)] {}
        set level [canvasWindowManager::getStackLevel $global::windowManager $widget::($this,path)]
        delete $this                                                                                       ;# delete existing viewer
        # viewer is as destroyable as previously deleted viewer
        manageViewer $viewer 1 -static $global::static -setx $x -sety $y -setwidth $width -setheight $height -level $level\
            -dragobject $viewer
    }

    proc cellThresholdColor {array row column} {
        set manager [new thresholdsManager]
        set cell ${array}($row,$column)
        foreach {color level summary} [thresholds::cellData $array $row $column] {
            thresholdsManager::condition $manager $cell $color $level $summary
        }
        set color [lindex [lindex [thresholdsManager::colorsAndTexts $manager] 0] 0]
        delete $manager
        return $color
    }

    proc cellThresholdCondition {array row column color level summary} {
        set topColor [cellThresholdColor $array $row $column]
        foreach viewer $(list) {
            thresholdCondition $viewer $array $row $column $color $level $summary
            setCellColor $viewer ${array}($row,$column) $topColor
        }
    }

    # viewers should implement only one of the following procedures:

    # latest threshold condition (reset if summary is not defined (empty)):
    virtual proc thresholdCondition {this array row column color level summary} {}
    # gives the color of the most important and recent threshold (reset if color is empty):
    virtual proc setCellColor {this cell color} {}

    virtual proc manageable {this} {return 1}                         ;# whether it should be managed by the internal window manager

    proc monitoring {cell} {                                                                            ;# viewers monitoring a cell
        set viewers {}
        foreach viewer $(list) {
            if {[monitored $viewer $cell]} {
                lappend viewers $viewer
            }
        }
        return $viewers
    }

    virtual proc monitored {this cell}

    # If this is the first time a color is requested for that cell, give a new color, else give the color already used by that cell,
    # so that a cell always has the same color in all its container viewers, until it is no longer monitored by any viewer.
    # When the cell is no longer displayed by the viewer, the color must be returned returnDisplayColor{}, so that the color can
    # eventually be made available for other cells.
    proc getDisplayColor {cell} {
        variable colorIndex                                          ;# cell to color index mapping (index in global viewers colors)
        variable usageCount                                                   ;# number of times a cell is displayed using its color

        if {![info exists colorIndex($cell)]} {                           ;# return a new color, if possible, not used by any viewer
            set colors [llength $global::viewerColors]
            for {set index 0} {$index < $colors} {incr index} {
                set count($index) 0
            }
            foreach {name index} [array get colorIndex] {                                                        ;# scan used colors
                incr count($index)
            }
            set color 0
            set minimum $global::integerMaximum
            for {set index 0} {$index < $colors} {incr index} {                                    ;# find the next least used color
                if {$count($index) < $minimum} {
                    set minimum $count($index)
                    set color $index
                }
            }
            set colorIndex($cell) $color
            set usageCount($cell) 0
        }
        incr usageCount($cell)
        return [lindex $global::viewerColors $colorIndex($cell)]
    }
    proc returnDisplayColor {cell} {                                 ;# use to return display color obtained using getDisplayColor{}
        variable colorIndex
        variable usageCount

        if {[catch {incr usageCount($cell) -1}]} return                                            ;# cell color came from save file
        if {$usageCount($cell) == 0} {                                                          ;# cell no longer displayed in color
            unset colorIndex($cell) usageCount($cell)
        }
    }

    proc limitEntry {this path anchor x y option name font command} {
        set entry [entry $path.maximum -borderwidth 0 -highlightthickness 0 -width 10 -font $font]
        setupEntryValidation $entry {{checkUnsignedInteger %P}}                                       ;# filter on positive integers
        lifoLabel::push $global::messenger "enter $name value (empty for automatic scale, Return to valid, Escape to abort)"
        # when Return or Enter is pressed, pass new value as limit, else if Escape is pressed, abort:
        foreach key {<KP_Enter> <Return>} {
            bind $entry $key "
                $command \[string trimleft \[%W get\] 0\]                               ;# avoid problems with invalid octal numbers
                destroy %W
                lifoLabel::pop $global::messenger
            "
        }
        bind $entry <Escape> "destroy %W; lifoLabel::pop $global::messenger"
        $entry insert 0 [composite::cget $this $option]
        $entry selection range 0 end                                  ;# initially select all characters, which is a common behavior
        place $entry -anchor $anchor -x $x -y $y
        focus $entry
        ::update idletasks                                                                     ;# so entry is visible and grab works
        grab $entry
    }

    proc wrongTypeMessage {type} {
        lifoLabel::flash $global::messenger "cannot display data of type $type"
        bell
        if {![info exists (wrongTypeDrop)]} {
            set (wrongTypeDrop) {}
            tk_messageBox -title {moodss: drag and drop} -type ok -icon info -message\
                "Some viewers only accept a limited set of data types (if this case, this viewer cannot display data of type $type)"
        }
    }

    proc createColorsMenu {parentPath command} {
        set menu [menu $parentPath.colorsMenu -tearoff 0]
        set spaces {   }                            ;### Tk bug: use 6 spaces for windows because otherwise labels look too thin ###
        if {[string equal $::tcl_platform(platform) windows]} {set spaces {      }}
        set rows 0
        set index 0
        foreach color $global::viewerColors {
            $menu add command -label $spaces -background $color -activebackground $color
            regsub -all %c $command $color string                                                            ;# use color in command
            $menu entryconfigure $index -hidemargin 1 -command $string
            if {$rows >= 3} {
                $menu entryconfigure $index -columnbreak 1
                set rows 0
            }
            incr rows
            incr index
        }
        return $menu
    }

    virtual proc updateLabels {this} {}

}

    proc numericType {type} {
        switch $type {
            integer - real {return 1}
            default {return 0}
        }
    }

    virtual proc saved {this} {return 1}                               ;# whether it is supposed to be saved in a configuration file

}
