/*
 * Mooix method library.
 *
 * Copyright 2002-2003 by Joey Hess <joey@mooix.net>
 * under the terms of the modified BSD license given in full in the
 * file COPYING.
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include "moomethod.h"

int methinit (void) {
	char *dir = getenv("THIS");
	if (dir == NULL)
		return 0;
	return chdir(getenv("THIS"));
}

void freeparam (param *param) {
	free(param->name);
	free(param->value);
	free(param);
}

char *mooix_getline (FILE *f, int killquotes) {
	int size = 0;
	char *ret = NULL;

	if (feof(f))
		return NULL;
	
	do {
		ret = realloc(ret, size + 128 + 1);
		if (! fgets(ret + size, 128, f)) {
			if (size == 0) {
				free(ret);
				return NULL; /* reached eof with empty string */
			}
			else {
				ret[size]='\0';
				break;
			}
		}
		size = strlen(ret);
	} while (size > 0 && ret[size - 1] != '\n');

	/* remove trailing newline */
	if (ret[size - 1] == '\n')
		ret[--size] = '\0';
	
	/* Remove quotes? */
	if (killquotes) {
		if (size > 1 && ret[0] == '"' && ret[size - 1] == '"') {
			ret[--size] = '\0';
			memmove(ret, ret + 1, size); /* left shift by one char */
		}
	}
	return ret;
}

char *getkey () {
	return mooix_getline(stdin, 1);
}

char *escape (const char *s) {
	/* Change embedded newlines to \\n, and double slashes. Add quotes. */
	if (strchr(s, '\n') || strchr(s, '\\')) {
		char *q, *t = malloc(strlen(s) * 2 + 3);
		const char *p;
		
		for (p = s, q = t + 1; p[0] != '\0'; p++, q++) {
			if (p[0] == '\n') {
				q[0] = '\\';
				q[1] = 'n';
				q++;
			}
			else if (p[0] == '\\') {
				q[0] = '\\';
				q[1] = '\\';
				q++;
			}
			else {
				q[0] = p[0];
			}
		}
		t[0] = q[0] = '"';
		q[1] = '\0';
		return t;
	}
	else {
		int len = strlen(s);
		char *t = malloc(len + 3);
		t[0] = '"';
		strcpy(t+1, s);
		t[0] = t[len + 1] = '"';
		t[len + 2] = '\0';
		return t;
	}
}

char *unescape (char *s) {
	char *p = s;
	
	while (p && (p = strstr(p, "\\"))) {
		int len = strlen(s);

		if (p[1] == '\\') {
			/* memmove below will remove first slash */
		}
		else if (p[1] == 'n' && (p == s || p[-1] != '\\')) {
			/* Turn "\n" into a literal newline. */
			p[1] = '\n';
		}
		else {
			p++;
			continue;
		}
		
		/* Copy remainder of line over slash. */
		memmove(p, p+1, len  - (p - s) + 1);
		p++;
	}
	
	return s;
}

char *fgetvalue (FILE *f) {
	return unescape(mooix_getline(f, 1));
}

char *getvalue () {
	return fgetvalue(stdin);
}

char **fgetallvals (FILE *f) {
	int size=16, count=0;
	char **ret=malloc(size * sizeof(char *));
	char *s;
	
	while ((s = fgetvalue(f))) {
		ret[count] = s;
		count++;
		if (count >= size) {
			size *= 2;
			ret = realloc(ret, size * sizeof(char *));
		}
	}
	ret[count] = NULL;
	return ret;
}

char **getallvals () {
	return fgetallvals(stdin);
}

param *getparam (void) {
	param *ret = malloc(sizeof(param));
	
	ret->name = NULL;
	ret->value = NULL;
	
	if ((ret->name = getkey()) == NULL ||
	    (ret->value = getvalue()) == NULL) {
	    	freeparam(ret);
	    	return NULL;
	}

	return ret;
}

int truefield (object *obj, const char *field) {
	char *file, *value;
	
	file = fieldfile(obj, field);
	if (! file)
		return 0;

	value = getfield(file);
	if (! value)
		return 0;
	if (! strlen(value))
		return 0;
	if (strcmp(value, "0") == 0)
		return 0;
	else
		return 1;
}

char *getfield (const char *field) {
	char *ret;
	FILE *f = fopen(field, "r");
	if (f == NULL)
		return NULL;
	ret = mooix_getline(f, 0);
	fclose(f);
	return ret;
}

int setfield (const char *field, const char *value) {
	FILE *f = fopen(field, "w");
	if (f == NULL)
		return 0;
	fprintf(f, "%s", value);
	fclose(f);
	return 1;
}

char *fieldfile (object *obj, const char *field) {
	int size, ods, fs, len;
	char *ret;
	char *p;
	struct stat buf;
	int depth = 0;

	/* set up ret to hold obj->dir/field */
	ods = strlen(obj->dir);
	fs = strlen(field);
	len = ods + 1;
	size = len + fs + 128;
	ret=malloc(size * sizeof(char));
	ret[0]='\0';
	strcat(ret, obj->dir);
	strcat(ret, "/");
	p = ret + ods + 1;
	
	for (;;) {
		/* Add field to end and see if anything turns up. */
		len += fs;
		if (len >= size) {
			size *= 2;
			ret=realloc(ret, size * sizeof(char));
			/* ret might move, and thus so must p */
			p = ret + len - fs;
		}
		strcat(ret, field);
		if (stat(ret, &buf) == 0) {
			return ret;
		}
		
		/* jump back to end of directory */
		len -= fs;
		p[0]='\0';

		/* Add parent/ to end; make sure there is a parent */
		len += 7;
		if (len >= size) {
			size *= 2;
			ret=realloc(ret, size * sizeof(char));
			p = ret + len - 7;
		}
		strcat(ret, "parent/");
		p += 7; /* points to end of parent/ */
		if (stat(ret, &buf) != 0) {
			free(ret);
			return NULL; /* no more parents */
		}

		/* Just in case.. */
		depth++;
		if (depth > 200) {
			fprintf(stderr, "possible recursive parent loop: %s\n", ret);
			exit(1);
		}
	}
}

object *derefobj (const char *s) {
	object *ret;

	if (! s || strncmp(s, "mooix:", 6) != 0)
		return NULL;
	
	ret = malloc(sizeof(object));
	ret->dev = 0;
	ret->dir = strdup(s + 6);

	return ret;
}

object *getobj (char *s) {
	object *ret = malloc(sizeof(object));
	ret->dev = 0;
	ret->dir = s;
	return ret;
}

void freeobj (object *obj) {
	free(obj->dir);
	free(obj);
}

/* This is very similar to _runmethod in the Mooix::Thing perl module.. */
FILE **runmethod_raw (object *obj, const char *method) {
	static FILE *ret[2];
	int pipe1[2], pipe2[2];
	int parent_rdr, child_wtr;
	int child_rdr, parent_wtr;
	pid_t pid;

	/* Parent and child communication pipes. */
	pipe(pipe1);
	parent_rdr=pipe1[0];
	child_wtr=pipe1[1];
	pipe(pipe2);
	child_rdr=pipe2[0];
	parent_wtr=pipe2[1];

	pid = fork();
	if (pid == -1) {
		close(parent_rdr);
		close(parent_wtr);
		close(child_rdr);
		close(child_wtr);
		return NULL;
	}
	else if (pid != 0) {
		ret[0]=fdopen(child_wtr, "w");
		ret[1]=fdopen(child_rdr, "r");
		
		/* Ignore sigpipes, which can easily occur if the child is
		 * very quick to run and does not read its input. */
		signal(SIGPIPE, SIG_IGN);

		close(parent_rdr);
		close(parent_wtr);

		return ret;
	}
	else {
		char *qualmethod;
		
		close(child_rdr);
		close(child_wtr);

		close(0);
		dup2(parent_rdr, 0);
		close(parent_rdr);
		close(1);
		dup2(parent_wtr, 1);
		close(parent_wtr);

		if (chdir(obj->dir) != 0)
			exit(1);

		qualmethod=fieldfile(getobj("."), method);
		if (! qualmethod) {
			exit(1);
		}
		
		if (getenv("THIS")) { /* in the moo */
			execlp(qualmethod, qualmethod, NULL);
		}
		else {
			execlp("runmeth", "runmeth", qualmethod, NULL);
		}
		fprintf(stderr, "failed to exec %s %s\n", obj->dir, qualmethod);
		exit(1);
	}
}

FILE *runmethod (object *obj, const char *method, char **params) {
	FILE *wtr, *rdr, **fds;
	
	fds = runmethod_raw(obj, method);
	if (fds == NULL)
		return NULL;
	wtr = fds[0];
	rdr = fds[1];
	
	/* Pass params to child. */
	if (params) {
		int i;
		for (i = 0; params[i] != NULL; i++)
			fprintf(wtr, "%s\n", params[i]);
	}
	fclose(wtr); /* let child know we're done so it can run */
	
	return rdr;
}	

int statobj (object *obj) {
	struct stat buf;
	if (stat(obj->dir, &buf) != 0)
		return 0;
	obj->dev = buf.st_dev;
	obj->ino = buf.st_ino;
	return 1;
}

int objcmp (object *a, object *b) {
	/* The stat info is cached between calls. */
	if (! a->dev) {
		if (! statobj(a))
			return -1;
	}
	if (! b->dev) {
		if (! statobj(b))
			return -1;
	}

	/* Return as does strcmp. */
	if (a->dev != b->dev)
		return (a->dev > b->dev) - (a->dev < b->dev);
	else
		return (a->ino > b->ino) - (a->ino < b->ino);
}
