#!/usr/bin/perl

package Mooix::MCP;

#Turn off in production code!
use strict;
use warnings;

=head1 NAME

Mooix::MCP - MUD client protocol support for mooix

=head1 SYNOPSIS

  use Mooix::MCP;
  my $mcp=Mooix::MCP->new($some_obj, "\n");
 
  ...

  $some_obj->mcpsession_start;
 
  ...

  print $client $mcp->encode($message, @params);
  
  ...
  
  $command=$mcp->decode(<$client>);
  if (defined $command) {
  	# pass command to parser
	...
  	print $client $mcp->escape($result);
  }

=head1 DESCRIPTION

This module implements about half of mcp; the other half is implemented
in the mooix:mixin/mcp/ family of objects.

=cut

# From the BNF in the spec.
my $quoted_string=qr/"(?:[^"]|\\")*"/;
my $unquoted_string=qr/[-~`!@#$%^&)(=+}{\]\[|';\/><.,A-Za-z0-9_]+/;
my $ident=qr/[a-zA-Z_][-a-zA-Z_]*/;

=head1 METHODS

=over 4

=item new

Create a new object. Should be passed a Mooix::Thing object that has the
mcpsession class mixed into it. Later on when mcp commands are received,
its mcpsession_receive method will be called to receive the commands. The
second argument is the line separator, typially "\n" or "\r\n".

=cut

sub new {
	my $proto=shift;
	my $class=ref($proto) || $proto;
	my $this=bless({thing => shift(), sep => shift(), pending => {}}, 
		       $class);
	return $this;
}

=item escape

This method should be used for all output to the client. It will escape
any lines that look like mcp. It returns the escaped line.

=cut

sub escape {
	my $this=shift;
	my $line=shift;
	if ($line=~/#\$(?:"|#)/) {
		return '#$"'.$line;
	}
	else {
		return $line;
	}
}

=item decode

This method should be called on each line of input from the client.
It will do mcp network line translation on the line. If the line is out of
band, the out of band data will be returned. If it is in-band, it will be
processed, a mcp message may be sent, and undef will be returned.

=cut

sub decode {
	my $this=shift;
	my $line=shift;

	$line=~s/$this->{sep}$//;
	
	if ($line=~/^#\$"/) {
		$line=~s/^#\$"//;
		return $line;
	}
	elsif ($line=~/^#\$#/) {
		# message
		if ($line=~/^#\$#($ident)\s+($unquoted_string)(?:$|\s+(.*))/) {
			my $message=$1;
			my $auth_key=$2;
			my $keyvals=$3;
			if ($auth_key ne $this->auth_key) {
				# ignore message with bad key
				return;
			}
			$this->handle_message($message, $keyvals)
		}
		# multi-line data
		elsif ($line=~/^#\$#\*\s+($unquoted_string)\s+($ident): (.*)/) {
			my $data_tag=$1;
			my $data_key=$2;
			my $data_value=$3;
			$this->add_value($data_tag, $data_key, $data_value);
		}
		# end of multi-line data
		elsif ($line=~/^#\$#:\s+($unquoted_string)\s*$/) {
			my $data_tag=$1;
			$this->send_pending_message($data_tag);
		}
		# message w/o auth key
		elsif ($line=~/^#\$#($ident)\s+(.*)/) {
			my $message=$1;
			my $keyvals=$2;
			if (length $this->auth_key) {
				# ignore bogus message with no auth key
				return;
			}
			$this->handle_message($message, $keyvals);
		}

		return;
	}
	else {
		return $line;
	}
}

=item encode

This method encodes an mcp message and returns the encoded message for
sending. The first parameter is the message name. Subsequent parameters
are key and vlaue pairs. 

A pair can be repeated to send a multi-line value. However, if the message
requires that a given value always be multi-line, and there may only be one
line, you should add an asterisk to the end of the name of the key.

=cut

sub encode {
	my $this=shift;
	my $message=shift;
	
	my %params;
	my @keylist;
	while (@_) {
		my $key=shift;
		my $value=shift;
		push @keylist, $key unless exists $params{$key};
		push @{$params{$key}}, $value;
	}

	my $s="#\$#$message";
	if (length $this->auth_key) {
		$s.=" ".$this->auth_key;
	}
	my $e="";
	my $data_tag='';
	foreach my $key (@keylist) {
		if (@{$params{$key}} > 1 || $key =~ /\*$/) {
			# multi-line
			my $keyname=$key;
			$keyname=~s/\*$//;
			$s.=" $keyname\*: \"\"";
			if (! length $data_tag) {
				$data_tag=int(rand(1000000));
				$s.=" _data-tag: $data_tag";
			}
			foreach (@{$params{$key}}) {
				$e.="#\$#* $data_tag $keyname\: ".$_.$this->{sep};
			}
		}
		else {
			my $value=${$params{$key}}[0];
			if ($value !~ /^$unquoted_string$/) {
				$value=~s/\"/\\"/g; # escape quotes
				$value='"'.$value.'"';
			}
			$s.=" $key: $value";
		}
	}
	if (length $data_tag) {
		$e.='#$#: '.$data_tag.$this->{sep}; # end of multiline data
	}
	return $s.$this->{sep}.$e;
}

=item auth_key

Get the session auth key from the thing object. It might not have
the key to start with, if it's not a client. Assume that once it
gets the key, the key will remain valid for the entire session.

=cut

sub auth_key {
	my $this=shift;
	
	if (exists $this->{auth_key}) {
		return $this->{auth_key};
	}
	else {
		my $k=$this->{thing}->mcpsession_auth_key;
		if (defined $k && length $k) {
			$this->{auth_key}=$k;
			return $k;
		}
	}
	return "";
}

=item known_data_tag

Returns true if the passed data tag is one that the object knows about.

=cut

sub known_data_tag {
	my $this=shift;
	my $tag=shift;
	foreach my $t (@{$this->{data_tags}}) {
		return 1 if $tag eq $t;
	}
	return 0;
}

=item handle_message

Called to handle each message that's decoded.
Passed the message name, and a keyvals parameter, that contains the as-yet
unparsed keyvals part of the message, in the form "key: value ...".

If the message has no multiline components, it dispatches it right away.
If there is multiline, the message is added to the pending messages array,
to be handled when it is fully received.

=cut

sub handle_message {
	my $this=shift;
	my $message=shift;
	my $keyvals=shift;
	
	# first, parse the keyvals
	my (@keyvals)=$keyvals=~/\s*($ident)(\*?):\s+($unquoted_string|$quoted_string)\s*/g;
	my %params;
	my %multilineparams;
	while (@keyvals) {
		my $key=shift @keyvals;
		my $multiline=shift @keyvals;
		my $value=shift @keyvals;
		if ($value=~/^"/) {
			# de-quote.
			$value=~s/^"//;
			$value=~s/"$//;
			$value=~s/\\"/"/g;
		}
		if ($multiline eq '*') {
			$multilineparams{$key}=[];
		}
		else {
			# spec says no duplicate keys
			return if exists $params{$key};
			$params{$key}=$value;
		}
	}

	# This subroutine will be called to add values to the message, and
	# when the message is complete.
	my $callback=sub {
		if (@_) {
			# Add a value.
			my $key=shift;
			my $value=shift;
			return unless $multilineparams{$key};
			push @{$multilineparams{$key}}, $value;
		}
		else {
			# Message is done, so send it.
			my @params;
			foreach my $key (keys %params) {
				push @params, $key, $params{$key};
			}
			foreach my $key (keys %multilineparams) {
				foreach my $value (@{$multilineparams{$key}}) {
					push @params, $key, $value;
				}
			}
			$this->{thing}->mcpsession_receive($message, @params);
		}
	};

	if (keys %multilineparams) {
		return unless exists $params{'_data-tag'} && length $params{'_data-tag'};
		$this->{pending}->{$params{'_data-tag'}}=$callback;
		delete $params{'_data-tag'};
	}
	else {
		# message is ready!
		$callback->();
	}
}

=item add_value

Adds a value to a pending message. Pass the data tag, the key, and the value.

=cut

sub add_value {
	my $this=shift;
	my $tag=shift;
	my $key=shift;
	my $value=shift;

	return unless ref $this->{pending}->{$tag};
	$this->{pending}->{$tag}->($key, $value);
}

=item send_pending_message

Sends a pending message. Pass the data tag.

=cut

sub send_pending_message {
	my $this=shift;
	my $tag=shift;

	return unless ref $this->{pending}->{$tag};
	$this->{pending}->{$tag}->();
	delete $this->{pending}->{$tag};
}

=head1 COPYRIGHT

Copyright 2003 by Joey Hess <joey@mooix.net>
under the terms of the modified BSD license given in full in the file
COPYRIGHT.

=head1 AUTHOR

Joey Hess <joey@mooix.net>

=cut

1
