"""Thing.py
This module implements the Thing class for mooix.

This file is part of the Python 2.2.1+ binding for mooix
Copyright (c)2002,2003 Nicholas D. Borko.  All Rights Reserved.
The author can be reached at nick@dd.revealed.net

The Python binding for mooix is free software; you can redistribute
it and/or modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

The Python binding for mooix is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with mooix; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA."""

import inspect, os, os.path, sys, traceback
from fcntl import flock, LOCK_UN, LOCK_SH, LOCK_EX, LOCK_NB
from debug import *
from util import *
import variables
from Result import *

class MooixError(StandardError):
    """A generic exception for something going wrong."""
    pass

def _prettyname(thing):
    return (thing.article and '%s ' % thing.article) + thing.name

class Thing(object):
    """This is the mooix object class for Python. It provides a standard class
    interface to the decidedly non-standard mooix objects.

    Please do not change directory after instantiating objects in this
    class; relative paths are used extensively, for security reasons.

    Available attributes:

    path: the absolute path that was used to instantiate the Thing.
    id: the same as path, for binding consistency
    """

    def __init__(self, path):
        """Thing(path) -> a mooix "Thing"

        path: the path to the object you want to instantiate,
              no mooix: prefix
        """
        debug('creating thing from path %s' % path)

        if not os.path.exists(path):
            raise MooixError, '%s is not a mooix object' % path

        self.__dict__['path'] = os.path.abspath(os.path.realpath(path))

    def __find_field(self, filename, dir = os.curdir, mixin = None):
        """__find_field(filename [, dir [, mixin]]) -> directory

        filename: the name of the field you're looking for
        dir: optionally a place to start looking, os.curdir by default
        mixin: treat this as a mix-in if not None

        Utility function to find the directory an attribute lives in.  It will
        recursively look through parent directories, returning None if the
        field can't be found.

        N.B. There is no recursion check, which is left to Python to raise an
        exception at sys.getrecursionlimit().  If you've set this exceptionally
        high, you're gonna eat your sytem's resources.  If you want to catch
        this, be sure to wrap a call in try and look for a RuntimeError."""
        debug('looking for %s' % os.path.join(dir, filename))

        parent = os.path.join(dir, 'parent')
        if not os.path.exists(os.path.join(self.path, parent)):
            parent = None

        if mixin is None:
            if os.path.exists(os.path.join(self.path, dir, filename)):
                return dir
            elif parent is not None:
                return self.__find_field(filename, parent)
            else:
                return None
        else:
            if os.path.exists(os.path.join(self.path, dir, mixin)):
                return self.__find_field(os.path.join(dir, mixin))
            elif parent is not None:
                return self.__find_field(filename, parent, mixin)

        return None

    def fieldfile(self, name, abs_path = False):
        """fieldfile(name, [ abs_path ]) -> path to file that contains
                                            the attribute/method

           name: name of attribute or method to find (string)
           abs_path: If true, return an absolute path, otherwise return a
                     relative path.  Default is False.

           Returns the path to the named attribute or method as it exists
           in the file system.  Raises MoooxError on failure."""

        if name[0] != '_':
            dir = self.__find_field(name)
            if dir is None and '_' in name[1:-1]:
                mixin, name = name.split('_', 1)
                dir = self.__find_field(name, mixin = mixin)
            if dir is not None:
                if abs_path:
                    return os.path.join(self.path, dir, name)
                else:
                    return os.path.join(dir, name)
        raise MooixError, '%s is not an attribute of %s' % (name, self)

    def get_path(self):
        return self.__dict__['path']

    path = property(get_path, None)
    id = property(get_path, None)

    def Return(self, value = None):
        """Return([value])

        value: None for nothing, or else a single value or list

        Returns a value (string, Thing, or a list of them) from the method
        being called."""
        if value is not None:
            if isinstance(value, list) or isinstance(value, tuple):
                print '\n'.join([ mooix_parse(i) for i in value ])
            else:
                print mooix_parse(value)
        sys.exit(0)

    def __str__(self):
        return 'mooix:%s' % self.path

    def __repr__(self):
        return '%s (%s) ' % (self, object.__repr__(self))

    def __eq__(self, other):
        if isinstance(other, Thing):
            return self.id == other.id
        else:
            return self.id == str(other)

    def __ne__(self, other):
        return not self.__eq__(other)

    def __setattr__(self, name, value):
        # if this is not me, then call the setfield method
        if self.id != os.environ.get('THIS', None):
            debug('calling setfield, "%s" != "%s"' % \
                  (self.id, os.environ.get('THIS', None)))
            if isinstance(value, list) or isinstance(value, tuple):
                result = self.setfield(name, *value)
            else:
                result = self.setfield(name, value)
            if result.status:
                self.croak('Permission denied, cannot set "%s"' % name,
                           result.status)
            else:
                return

        if name[0] != '_':
            # always set attributes locally, never in parents
            attr_file = os.path.join(self.path, name)

            # this only does one os.stat -- less stats, better performance
            attr_stat = stat(attr_file)

            # don't mess with directories -- BAD
            if attr_stat.isdir and not attr_stat.islink:
                raise MooixError, '%s is a full-fledged object' % name

            # ignore setting methods, because there's not a sane way I
            # can think of to handle that
            elif attr_stat.isexec and not attr_stat.islink:
                raise MooixError, '%s is a method' % name

            # if it's a Thing, then create a symlink
            elif isinstance(value, Thing):
                if attr_stat.exists:
                    try:
                        os.unlink(attr_file)
                    except:
                        traceback.print_exc()
                        return
                try:
                    os.symlink(value.path, attr_file)
                except:
                    traceback.print_exc()
                    return

            # otherwise, just try writing the value to a file
            else:
                if attr_stat.exists and attr_stat.islink:
                    # if previous file was a symlink, remove it first
                    try:
                        os.unlink(attr_file)
                    except:
                        traceback.print_exc()
                        return
                try:
                    if isinstance(value, list) or isinstance(value, tuple):
                        is_object_list = 1
                        for v in value:
                            if not isinstance(v, Thing):
                                is_object_list = 0
                                break
                        open(attr_file, 'w+').write(
                                       '\n'.join([ mooix_parse(a, raw = True) \
                                                   for a in value ]))
                        if is_object_list:
                            if attr_stat.exists and not attr_stat.islink:
                                # preserve previous permissions, except symlinks
                                os.chmod(attr_file,
                                         01644 & (attr_stat.bits & 0777))
                            else:
                                os.chmod(attr_file, 01644)
                        elif attr_stat.exists and attr_stat.issticky:
                            # remove sticky bit if previously an object list
                            chmod(attr_file, attr_stat.bits & 0777)
                    else:
                        open(attr_file, 'w+').write('%s\n' % value)
                        if attr_stat.exists and attr_stat.issticky:
                            # remove sticky bit if previously an object list
                            chmod(attr_file, attr_stat.bits & 0777)
                except:
                    traceback.print_exc()
        else:
            raise AttributeError, name

    def __getattr__(self, name):
        try:
            rel_path = self.fieldfile(name)
            attr_file = os.path.join(self.path, rel_path)
        except MooixError:
            raise AttributeError, name
        
        debug('attr_file is %s' % attr_file)

        # this only does one os.stat -- less stats, better performance
        attr_stat = stat(attr_file)

        # for directories or links, try returning a Thing
        if attr_stat.isdir or attr_stat.islink:
            debug('getattr returning mooix:%s' % attr_file)
            return Thing(attr_file)

        # check to see if it's a method
        elif attr_stat.isread and attr_stat.isexec:
            debug('returning method %s' % attr_file)
            return Method(self, rel_path)

        # otherwise it's probably just a file (property)
        elif attr_stat.isread:
            debug('accessing attribute %s' % attr_file)
            is_object_list = attr_stat.issticky
            return Property(self, attr_file, is_object_list)

        # who knows what, maybe just not accessible
        else:
            raise MooixError, 'could not access %s' % name

    def defines(self, name):
        """defines(name) -> filename if the object defines name, None if not

        name: string representing the field name

        Checks to see if an object contains a field.  Inheriting the
        field from a parent does not count; the field must be part of
        the object the method is called on.  If the object does define
        the field, returns the filename of the field in the object."""

        filename = os.path.join(self.path, name)
        if os.path.exists(filename):
            return filename
        else:
            return None

    def hasattr(self, name):
        """hasattr(name) -> true if named attribute is available, false if not

        name: string representing the attribute name

        This does the same thing as the builtin hasattr() function, but
        it does not call getattr() to make its determination.  Always
        use this method instead of the hasattr() builtin."""

        try:
            self.fieldfile(name)
            return True
        except MooixError:
            return False

    def msg(self, event, *arg, **kw):
        """msg(self, event [, arg1 [, ...]])

        message: a event to send as a message

        Sends an event to the outermost container of the object.  The
        global args are automatically passed, so you don't have to
        explicitly pass them as in other language bindings.  If you want
        to override a global arg, just specify it as an argument to msg
        and it will be overridden."""
        kw['event'] = event

        # I'm making the assumption that args should always be passed.
        # other bindings make you explicitly pass global args, but we'll
        # do it implicitly while letting the caller override anything
        for k, v in variables.args.iteritems():
            kw.setdefault(k, v)

        # call the external msg function
        return Method(self, self.fieldfile('msg'))(**kw)

    def isa(self, other):
        """isa(other) -> true or false

        other: a Thing or an absolute path as a string

        Returns true of Things are equal or the path is the same as the
        our path."""
        if self == other:
            return True
        
        me = self
        while me.hasattr('parent'):
            if me == other:
                return True
            else:
                me = me.parent
        return False

    def super(self, *arg, **kw):
        """super([arg1 = value [ ,...]])

        Call the predecessor's corresponding method and return whatever
        it returns.  The global args are automatically passed, so you don't
        have to explicitly pass them as in other language bindings.  If you
        want to override a global arg, just specify it as an argument to
        super and it will be overridden."""
        for k, v in variables.args.iteritems():
            kw.setdefault(k, v)
        caller, method = os.path.split(os.environ['METHOD'])
        dir = self.__find_field(method,
                                os.path.join(caller, 'parent'))
        if dir is not None:
            debug('super calling %s' % os.path.join(dir, method))
            return Method(self, os.path.join(dir, method))(**kw)
        else:
            debug('super did not find a parent method')
            return NullMethod(self)

    def getlock(self, locktype = LOCK_EX, field = '.mooix'):
        """getlock([locktype[, filename]) -> open file object

        locktype: A valid lock type for flock() from the fcntl module.
                  Defaults to LOCK_EX
        field: The name of the field to lock.  Defaults to '.mooix'

        Return an open file object that represents a lock.  Close the file
        to drop the lock.  The field to lock must exist on the object and
        not be inherited, or else an IOError exception will be raised."""
        lock = file(os.path.join(self.path, field))
        flock(lock, locktype)
        return lock

    def usage(self, *arg):
        """usage([arg1[, ...]])

        Displays usage help on the currently running method (by calling
        the object's getusage method (which must exist if you want this to
        work)) and exits.  Any arguments are printed out.
        """
        caller, method = os.path.split(sys.argv[0])
        if self.hasattr('getusage'):
            usage = self.getusage(field = method).list + [ '' ]
            usage[0] = 'Usage: ' + usage[0]
            sys.stderr.write('\n'.join([ str(a) for a in arg ] + usage))
            sys.exit(0)

    def croak(self, error, status = 0):
        """croak(error[, status])

        error: reason for failing
        status: result status to exit with, defaults to 0
        
        Display a mooix calltrace instead of a Python traceback."""
        frame = inspect.stack()[1]
        sys.stderr.write('Error: %s in %s.%s line %d\n' % \
                         (error, self.id, os.path.basename(sys.argv[0]), frame[2]))
        sys.stderr.write('calltrace:\n\t')
        sys.stderr.write('\n\t'.join(self.calltrace()))
        sys.stderr.write('\n')
        sys.exit(status)

    def encapsulator(self):
        """encapsulator() -> object that contains this one

        Return the mooix object that the object is encapsulated in.  That
        is, the object that the mooix object is a subdirectory underneath."""
        return Thing(os.path.join(self.path, os.pardir))

    def safegetfield(self, field):
        """safegetfield(field) -> value of safe attribute or method

        This method can be used to safely get the value of a field or
        method, which might be supplied by a non-programmer (e.g., as part
        of a message), without accidentially calling destructive methods
        like destroy.

        It only allows getting values of fields that are not private.

        It only allows calling of methods that are marked as safe by the
        existence of a field named .<method>-safe with a true value.
        """
        if field[0] != '_':
            if self.hasattr(field):
                if not self.can(field):
                    filename = self.fieldfile(field)
                    filestat = stat(filename)
                    if not (filestat.isdir or filestat.islink):
                        safefield = '.%s-safe' % field
                        if hasattr(self, safefield) and \
                               getattr(self, safefield):
                            return getattr(self, field)
        warnings.warn('%s is not safe, None returned', warnings.RuntimeWarning,
                      stacklevel = 2)
        return None

    def prettyname(self):
        """prettyname() -> the object's name with any article prepended"""
        return _prettyname(self)

    def prettylist(self, object_list):
        """prettylist(object_list) -> a pretty-printed list of objects

        Generates a very pretty-printed list of objects, and returns it.
        The object it's run on will appear in the list as "you"."""
        object_list = tolist(object_list)
        if object_list:
            return (len(object_list) > 2
                    and ', '
                    or ' and ').join([ _prettyname(o)
                                       for o in object_list
                                       if isinstance(o, Thing) ])
        else:
            return 'nothing'


__all__ = [ 'MooixError', 'Thing', 'LOCK_UN', 'LOCK_SH', 'LOCK_EX', 'LOCK_NB' ]
