require "mooix"

=begin

= Name
thing.rb - Mooix Thing class for Ruby

= Synopsis

  #!/usr/bin/ruby
  require "mooix"
  require "mooix/thing"
  . . .
  Mooix.run do
  . . .
  end

= Description

thing.rb provides a Ruby interface to
<((<Mooix|URL:http://mooix.net>))>'s Thing class. Usually used in
conjunction with the top-level mooix module.

=end

=begin

= Thing class

This class almost transparently wraps Mooix objects in Ruby. You must, however, use:
  self.variable
and not:
  variable
for instance variable accesses or method calls.

=end

class Thing
  attr_reader :path
  attr_writer :args

=begin

= Methods
--- Thing.new(path [, ROOT = false])
Instantiates a Thing wrapping the object specified by ((|path|)).
    :Parameters:
      : ((|path|))
        path to this object.

=end

  def initialize(path, root=false)
    if (not root) and (File.lstat(path).symlink?)
      @path = File.readlink(path)
    else
      @path = path
    end
    @root = root
  end

=begin
--- Thing.path
Returns the full path to this object.
    :Returns:
      complete, expanded path to the caller.

=end

  def path
    File.expand_path(@path)
  end

=begin
--- == obj
Compares two Thing instances for equality.
    :Parameters:
      : ((|obj|))
        object to compare with.
    :Returns:
      true if paths are equal, false if not, or false if ((|obj|)) isn't a Thing.
=end

  def ==(obj) # comparison
    return false if not obj.instance_of?(Thing)
    # symlinks are followed upon init
    return path == obj.path
  end

=begin
--- Thing.to_s
Returns a reference to this Thing in the form of
'mooix:/path/to/self'. This reference is used by various bindings to
transparently create objects from their paths.
    :Returns:
      mooix:path reference to this object.
=end

  def to_s
    "mooix:#{self.path}"
  end
  alias old_method_missing method_missing
  def method_missing(methId, *args)
    Dir.chdir(@path) if !@root
    val = invoke_method(methId.id2name, args)
    Dir.chdir($root) if !@root and $in_run
    Dir.chdir(self.path) if !$in_run
    return val
  end

=begin
--- Thing.member_path(name[, base = @path])
Returns the full path to the field or method specified by ((|name|)).
    :Parameters:
      : ((|name|))
        name of the method or field.
    :Returns:
      path to field specified by ((|name|)), nil otherwise.
=end

  def member_path(name, base = @path)
    dir = Mooix.find_method_path(name, base)
    if dir == nil && name =~ /(.+?)_(.*)/ # maybe a mixin
      name = $2
      mixin = $1
      dir = Mooix.find_method_path(name, base, mixin)
    end
    return nil if dir == nil
    return dir+"/"+name
  end

=begin
--- Thing.has?(name)
Predicate to determine whether the caller has a method or field.
    :Parameters:
      : ((|name|))
        name of the method or field.
    :Returns:
      true if the field exists on the caller, nil otherwise.
=end

  def has?(name)
    path = member_path(name)
    return false if path == nil
    return File.stat(path).readable?
  end

=begin
--- Thing.defines?(name)
Predicate to determine whether the caller defines a method or field.
    :Parameters:
      : ((|name|))
        name of the method or field.
    :Returns:
      true if the field is defined on the caller, nil otherwise.
=end

  def defines?(name)
    File.exists?(name)
  end

=begin
--- Thing.respond_to?(name)
Predicate to determine whether the caller responds to a specific method.
    :Parameters:
      : ((|name|))
        name of the method.
    :Returns:
      true if ((|name|)) is a method to which ((|self|)) responds, false otherwise.
=end

  def respond_to?(name)
    has?(name) and File.executable?(member_path(name))
  end

=begin
--- Thing.implements?(name)
Same as ((|Thing.responds_to?|)).
    :Parameters:
      : ((|name|))
        name of the method.
    :Returns:
      true if ((|name|)) is a method to which ((|self|)) responds, false otherwise.
=end

  alias implements? respond_to?

  def invoke_method(name, args, start = ".") #start = "."
    oldname = String.new(name)
    name.chop! if name =~ /(.+)\=$/
    cd = @path
    $stderr.puts("\nMethod <#{name}>, path <#{cd}>, cwd <#{Dir.getwd}>") if $debug != nil
    file = member_path(name, start)
    if file == nil
      if oldname =~ /\=$/
	file = "#{name}"
      else
	raise "Could not find field or method <#{name}>"
      end
    end
    if oldname =~ /(.+)\=$/ # field write
      $stderr.print("\tfield write: ") if $debug != nil
      if File.executable?(file) and (not (File.symlink?(file) or File.directory?(file))) # it's really a method
	$stderr.puts("method, invoking <#{file}>") if $debug != nil
	IO.popen("#{file}", "w") { |p| p.write(Mooix.ruby_to_mooix(args)) }
      elsif (File.symlink?(file) or (not File.exists?(file))) and ((args.size == 1) and args[0].instance_of?(Thing))
	p = args[0].path
	$stderr.print("object reference to <#{p}>") if $debug != nil
	if File.exists?(name)
	  $stderr.print(", deleting old #{name}") if $debug != nil
	  File.delete(name)
	end
	$stderr.print(", making symlink to #{name}") if $debug != nil
	File.symlink(p, name)
	$stderr.puts(", symlink made") if $debug != nil
      else # it's just a file, make sure to use local copy
	$stderr.puts("file, writing <#{name}>") if $debug != nil
        target = Thing.new(File.dirname(name))
	File.open("#{name}", File::CREAT|File::TRUNC|File::WRONLY) { |f| f.write(args.join("\n")) } if Dir.pwd == $ZERO
        rc = 0
        # We need to temporarily disable auto-conversion to obtain a "real" value.
        oldconvert = $autoconvert
        $autoconvert = false
        rc = target.setfield(File.basename(name), args.join("\n")) if $ZERO != target.path
        $autoconvert = oldconvert
        raise(Mooix::SetfieldFailure, "failed to set field") if rc == 1
      end
    else # field read / method access
      $stderr.print("\tfield read: ") if $debug != nil
      ret = []
      if File.directory?(file) || File.symlink?(file) # object reference
	path = File.expand_path(file)
	$stderr.print("\tobject reference to <#{path}>") if $debug != nil
	ret = ["mooix:#{path}"]
      elsif File.executable?(file)
	$stderr.puts("method, invoking <#{file}>") if $debug != nil
	IO.popen("#{file}", "r+") do |p|
	  p.write(Mooix.ruby_to_mooix(args))
	  p.close_write
	  ret = p.readlines
	end
      elsif File.readable?(file) # simple file read
	$stderr.puts("file, reading <#{file}>") if $debug != nil
	File.open(file, "r") { |file| ret = file.readlines }
      else
	raise "Don't know how to handle <#{name}>"
      end
      ret.collect! { |val| Mooix.parse_mooix(val.chomp) }
      return nil if ret.length == 0
      return ret[0] if ret.length == 1
      return ret if ret.length > 1
    end
  end

=begin
--- Thing.msg(name, @args)
Wrapper around Mooix's msg method.
    :Parameters:
      : ((|name|))
        name of the message to be sent.
      : ((|args|))
        additional arguments, usually the @args array available to verbs.
=end

  def msg(name, params={})
    params = Hash[*params] if params.class != Hash
    params["event"] = name
        self.invoke_method("msg", params)
  end
  # Don't call this or bad things will happen!
  # Well, presumably you DON'T want deprecation errors going to stderr. :)
  def message(name, *varg)
    $stderr.puts("Error: Thing.message is deprecated. Call msg instead. I'll do it for you because I'm nice, but this method will go away soon!")
    self.msg(name, @args)
  end
  #private :invoke_method

=begin
--- Thing.is_thing?(other)
Predicate to determine whether the first argument is an instance of this.
    :Parameters:
      : ((|other|))
        a string or object denoting the other thing for comparison.
    :Returns:
      true if ((|other|)) is an instance of ((|this|)), false otherwise.
=end

  def is_thing?(o)
    other = nil
    if (o.instance_of?(String))
      other = Thing.new(o)
    elsif (o.instance_of?(Thing))
      other = o
    else
      return false
    end
    if self == other
      return true
    elsif not self.has?("parent")
      return false
    else
      return self.parent.is_thing?(other)
    end
  end

=begin
--- Thing.super(*args)
Calls the same method on superclasses if they exist. Note that this must be called using ((|self|)) as the receivor, else the Ruby super() method is used instead.
    :Parameters:
      : ((|*args|))
        arguments to the overridden method.
    :Returns:
      value of the superclass's method.
=end

  def super(*args)
    path = $ZERO
    raise "invoke_super on non-root object" if not @root
    cmd = File.basename(path)
    loc = File.dirname(path)
    $stderr.puts("path: <#{path}>, cmd: <#{cmd}>, loc: <#{loc}>") if $debug != nil
    self.invoke_method(cmd, args, "#{loc}/parent")
  end

=begin
--- Thing.getlock(type = File::LOCK_EX) [do . . . end]
Locks ((|this|)) and, if passed a block, executes the contained code before releasing the lock.
    :Parameters:
      : ((|type|)) = ((|File::LOCK_EX|))
        type of lock. Currently useful values are ((|File::LOCK_EX|)) (the default) and ((|File::LOCK_SH|)).
    :Returns:
      file handle to the lock. If no block is given, this can be closed to release the lock.
=end

  def getlock(type = File::LOCK_EX)
    lock = File.open(".mooix")
    lock.flock(type)
    if block_given?
      yield
      lock.close
      lock = nil
    end
    lock
  end

=begin
--- Thing.prettyname
Returns a name suitable for pretty-printing. Adds an article if one is defined, else adds nothing.
    :Returns:
      pretty-printable name for this.
=end

  def prettyname
    if self.has?("article") and self.article != nil
      return "#{article} #{name}"
    else
      return "#{name}"
    end
  end

=begin
--- Thing.prettylist(objects)
Prepares ((|objects|)) for pretty-printing.
    :Parameters:
      : ((|objects|))
        list of objects or strings whose value is to be pretty-printed.
    :Returns:
      string listing all objects and strings contained within ((|objects|)) in a format suitable for printing.
=end

  def prettylist(objects)
    return "nothing" if objects.length == 0
    result = ""
    objects.each_index do |x|
      if objects[x] == self
	result += "you"
      else
	if objects[x].class != String
	  result += "#{objects[x].prettyname}"
	else
	  result += "#{objects[x]}"
	end
      end
      result += ", " if x <= (objects.length-3)
      result += " and " if x == (objects.length-2)
    end
    return result
  end
end
