/*
 * mooix call stack support library
 *
 * Copyright 2002-2003 by Joey Hess <joey@mooix.net>
 * under the terms of the modified BSD license given in full in the file
 * COPYRIGHT.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <assert.h>
#include <unistd.h>
#include <stdarg.h>
#include <stdlib.h>
#include <syslog.h>
#include <string.h>
#include <dirent.h>

#include "moocallstack.h"

#define EXT ".stack"

char *callstack_file (uid_t uid) {
	static char buf[128]; /* A static buffer is returned.. */
	snprintf(buf, 128, "%s/%i" EXT, RUNDIR, uid);
	return buf;
}

int callstack_unlink (uid_t uid) {
	return unlink(callstack_file(uid));
}

void callstack_walk (void (*function)(uid_t uid)) {
	uid_t uid;
	callstack_iter *iter;
	
	iter = callstack_iter_start();
	while ((uid = callstack_iter_next(iter))) {
		(function)(uid);
	}
	callstack_iter_end(iter);
}

callstack_iter *callstack_iter_start (void) {
	DIR *dir;

	dir = opendir(RUNDIR);
	assert(dir != NULL);
	return (callstack_iter *) dir;
}

uid_t callstack_iter_next (callstack_iter *iter) {
	struct dirent *ent;
	int extlen = strlen(EXT);
	int namelen;

	while ((ent =readdir((DIR *) iter)) != NULL) {
		/* See if the filename ends in EXT. */
		namelen = strlen(ent->d_name);
		if (namelen >= extlen &&
		    strcmp(ent->d_name + namelen - extlen, EXT) == 0) {
			return atoi(ent->d_name);
		}
	}
	return 0;
}

void callstack_iter_end (callstack_iter *iter) {
	closedir((DIR *) iter);
}

void callstack_free (struct callstack *s) {
	struct callstack *prev = NULL;
	while (s) {
		prev = s;
		s = s->next;
		free(prev);
	}
}

int callstack_size (struct callstack *stack) {
	int i = 0;
	while (stack) {
		i++;
		stack = stack->next;
	}
	return i;
}

struct callstack *callstack_push (struct callstack *s, int objectfd, char *method) {
	struct callstack *new;
	struct stat st_buf;
	int res;
	
	res = fstat(objectfd, &st_buf);
	assert(res == 0);

	/*
	 * Rudimentry compression: don't add the same thing to a stack
	 * that's already on the top of it.
	 */
	if (s && st_buf.st_dev == s->dev && st_buf.st_ino == s->inode)
		return s;
	
	new = malloc(sizeof(struct callstack));
	assert(new != NULL);
	new->next = s;
	new->dev = st_buf.st_dev;
	new->inode = st_buf.st_ino;
	new->method = method;
	new->boundry = 0;
	return new;
}
/* Note that there is no command to pop a stack. Mood never needs to! */

struct callstack *callstack_copy (struct callstack *s) {
	struct callstack *new, *ret = NULL, *retend = NULL;
	
	while (s) {
		new = malloc(sizeof(struct callstack));
		assert(new != NULL);
		memcpy(new, s, sizeof(struct callstack));
		if (retend)
			retend = retend->next = new;
		else
			ret = retend = new;
		
		s = s->next;
	}

	return ret;
}

struct callstack *callstack_extend (struct callstack *head, struct callstack *extention) {
	struct callstack *s = head;
	if (head != NULL) {
		while (s->next) {
			s = s->next;
		}
		s->next = extention;
		return head;
	}
	else {
		return extention;
	}
}

int callstack_dedup (struct callstack *s) {
	/* This is quite primitive, just removes congruent duplicates.
	 * It'd be cool if it could remove any duplicates, even
	 * non-congruent. But as it stands, it removes a ton of duplicates,
	 * in a simple O(n) sweep.
	 */
	struct callstack *cur, *prev;
	int count = 0;

	if (! s) return 0;
	
	cur = s->next;
	prev = s;
	while (cur) {
		if (cur->dev == prev->dev && cur->inode == prev->inode) {
			count++;
			prev->next = cur->next;
			free(cur);
			cur = prev->next;
		}
		else {
			prev = cur;
			cur = cur->next;
		}
	}
	
	return count;
}

struct callstack *callstack_invert (struct callstack *s) {
	struct callstack *t, *ret = s;

	if (s == NULL)
		return NULL;
	
	/* Find end. */
	while (ret->next)
		ret = ret->next;

	/* Move each element to the end until the end is reached. */
	while (s != ret) {
		t = s;
		s = s->next;
		t->next = ret->next;
		ret->next = t;
	}

	return ret;
}

int callstack_is_subset(struct callstack *subset, struct callstack *superset) {
	struct callstack *sub, *super;

	sub = subset;
	while (sub != NULL) {
		if (sub->boundry)
			return 1;
		
		for (super = superset; super != NULL; super = super->next) {
			if (super->dev == sub->dev && super->inode == sub->inode) {
				goto NEXT;
			}
		}
		return 0;

NEXT:
		sub = sub->next;
	}
	return 1;
}

int callstack_save (uid_t dest, struct callstack *stack) {
	FILE *out;

	if ((out = fopen(callstack_file(dest), "w")) == NULL)
		return -1;
	callstack_dump(stack, out);
	fclose(out);
	return 0;
}

void callstack_dump (struct callstack *stack, FILE *out) {
	while (stack) {
		if (stack->boundry) fprintf(out, "\n");
		if (stack->method)
			fprintf(out, "%i\n%i\n%s\n",
			        (int) stack->dev, (int) stack->inode,
				stack->method);
		else
			fprintf(out, "%i\n%i\n\n",
			        (int) stack->dev, (int) stack->inode);
		stack = stack->next;
	}
}

/* This is brken out only for the catstack test program, and should be
 * inlined int callstack_load. */
inline struct callstack *callstack_fromfd (FILE *in, int onlytop) {
	char *s;
	char buf[PATH_MAX];
	int on_boundry = 0;
	struct callstack *ret = NULL, *new;
	
	while (fgets(buf, PATH_MAX - 1, in) != NULL) {
		if (buf[0] == '\n') {
			/* This empty line is a stack boundry. Add the
			 * boundry flag to the next item read. */
			on_boundry = 1;
		}
		else {
			new = malloc(sizeof(struct callstack));
			assert(new != NULL);
			new->next = ret;
			ret = new;
		
			ret->dev = atoi(buf);
			/* Load inode line in. */
			fgets(buf, PATH_MAX, in);
			ret->inode = atoi(buf);
			
			if (onlytop)
				goto OUT;
			
			ret->boundry = on_boundry;
			on_boundry = 0;
		
			/* Load in method line too. */
			fgets(buf, PATH_MAX, in);
			if ((s = strchr(buf, '\n')) != NULL)
				s[0] = '\0'; /* kill newline */
			ret->method = strdup(buf);
		}
	}

OUT:
	ret=callstack_invert(ret);
	return ret;
}

struct callstack *callstack_load (const char *filename, int onlytop) {
	FILE *in;
	struct callstack *ret = NULL;
	
	if ((in = fopen(filename, "r")) == NULL)
		return NULL;
	ret=callstack_fromfd(in, onlytop);
	fclose(in);
	return ret;
}
