/*
 * Mooix proxy wrapper library.
 * Helper functions for communication with mood.
 *
 * Copyright 2001-2003 by Joey Hess <joey@mooix.net>
 * under the terms of the modified BSD license given in full in the
 * file COPYRIGHT.
 */

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <errno.h>
#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <sys/stat.h>
#include <libgen.h>
#include <fcntl.h>
#include <sys/uio.h>
#include "libmooproxy.h"

int sendfd (int sock, int fd) {
	struct msghdr msg;
	struct cmsghdr* p_cmsg;
	struct iovec vec;
	char cmsgbuf[CMSG_SPACE(sizeof(fd))];
	int* p_fds;
	char sendchar = 0;

	msg.msg_control = cmsgbuf;
	msg.msg_controllen = sizeof(cmsgbuf);
	p_cmsg = CMSG_FIRSTHDR(&msg);
	p_cmsg->cmsg_level = SOL_SOCKET;
	p_cmsg->cmsg_type = SCM_RIGHTS;
	p_cmsg->cmsg_len = CMSG_LEN(sizeof(fd));
	p_fds = (int*)CMSG_DATA(p_cmsg);
	*p_fds = fd;
	msg.msg_controllen = p_cmsg->cmsg_len;
	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_iov = &vec;
	msg.msg_iovlen = 1;
	msg.msg_flags = 0;
	/* "To pass file descriptors or credentials you need to send/read
	 * at least one byte" (man 7 unix) */
	vec.iov_base = &sendchar;
	vec.iov_len = sizeof(sendchar);
	return sendmsg(sock, &msg, 0);
}

/* Receive a file descriptor on a socket. */
int getfd(int client) {
	struct msghdr msg;
	char recvchar;
	struct iovec vec;
	int recv_fd;
	char cmsgbuf[CMSG_SPACE(sizeof(recv_fd))];
	struct cmsghdr* p_cmsg;
	int* p_fd;

	vec.iov_base = &recvchar;
	vec.iov_len = sizeof(recvchar);
	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_iov = &vec;
	msg.msg_iovlen = 1;
	msg.msg_control = cmsgbuf;
	msg.msg_controllen = sizeof(cmsgbuf);
	msg.msg_flags = 0;
	p_fd = (int*) CMSG_DATA(CMSG_FIRSTHDR(&msg));
	*p_fd = -1;
	if (recvmsg(client, &msg, 0) != 1) {
		mooix_debug(PRIORITY_WARNING, "bad recvmsg in getfd", NULL);
		return -1;
	}
	p_cmsg = CMSG_FIRSTHDR(&msg);
	if (p_cmsg == NULL) {
		mooix_debug(PRIORITY_WARNING, "null p_cmsg in getfd", NULL);
		return -1;
	}
	p_fd = (int*) CMSG_DATA(p_cmsg);
	recv_fd = *p_fd;
	return recv_fd;
}

int setup_socket (char *path) {
	int sock;
	struct sockaddr_un sunx;
	
	memset(&sunx, 0, sizeof(sunx));
	sunx.sun_family = AF_UNIX;
	strncpy(sunx.sun_path, path, sizeof(sunx.sun_path));
	
	if ((sock = socket(PF_UNIX, SOCK_STREAM, PF_UNSPEC)) == -1 ||
	    connect(sock, (struct sockaddr *) &sunx, 
		    sizeof(sunx.sun_family) + strlen(sunx.sun_path)) == -1) {
		return -1;
	}
	
	return sock;	
}

/* Ensure that there is a proxy handle open, and return it. */
int getproxy (void) {
	char *sockfile;
	
	if (proxysock != -1)
		return proxysock;
	
	sockfile = getenv("MOOSOCK");
	if (sockfile == NULL) {
		mooix_debug(PRIORITY_ERROR, "MOOSOCK not set", NULL);
		return -1;
	}
	if ((proxysock = setup_socket(getenv("MOOSOCK"))) == -1) {
		mooix_debug(PRIORITY_ERROR, "unable to connect to %s: %s",
				getenv("MOOSOCK"), strerror(errno), NULL);
		return -1;
	}

	return proxysock;
}

/*
 * Get a result from the client. Sets errno to the err field, and returns
 * the ret field.
 */
int getresult (int sock) {
	struct result_t result;
	if (read(sock, &result, sizeof(result)) != sizeof(result)) {
		mooix_debug(PRIORITY_WARNING, "short read from proxy", NULL);
		return -1;
	}
	errno = result.err;
	return result.ret;
}
 
/* Sends a command to the proxy. */
int sendcommand (int sock, int command, ...) {
	struct command_t cmd;
	va_list ap;
	char *s;
	int offset = 0, len, bufsize = 64;
	
	cmd.type = command;
	va_start(ap, command);
	cmd.buf = malloc(bufsize * sizeof(char));
	for (cmd.argc = 0; (s = va_arg(ap, char *)) != NULL; cmd.argc++) {
		len = strlen(s) + 1;
		while (offset + len >= bufsize) { /* cheezy, but effective */
			bufsize = bufsize * 2;
			cmd.buf = realloc(cmd.buf, bufsize * sizeof(char));
		}
		memcpy(cmd.buf + offset, s, len * sizeof(char));
		offset += len;
		cmd.buf[offset-1] = '\0';
	}
	cmd.len = offset;

	if (write(sock, &cmd, sizeof(cmd)) == -1 ||
	    write(sock, cmd.buf, cmd.len) == -1) {
		mooix_debug(PRIORITY_WARNING, "write to proxy: %s",
			strerror(errno), NULL);
		free(cmd.buf);
		va_end(ap);
		return -1;
	}
	
	free(cmd.buf);
	va_end(ap);

	return 0;
}

/*
 * Sorta like sendcommand except it takes an argv after the first three args.
 */
int sendcommandv (int sock, int command, const char *firstarg,
		  const char *secondarg, const char *thirdarg,
		  char *const argv[]) {
	struct command_t cmd;
	int offset = 0, len, bufsize = 128;
	
	cmd.type = command;
	cmd.buf = malloc(bufsize * sizeof(char));

	/* First and second args may be NULL. */
	if (firstarg) {
		len = strlen(firstarg) + 1;
		while (offset + len >= bufsize) {
			bufsize = bufsize * 2;
			cmd.buf = realloc(cmd.buf, bufsize * sizeof(char));
		}
		memcpy(cmd.buf + offset, firstarg, len * sizeof(char));
		offset += len;
	}
	else {
		offset += 1;
	}
	cmd.buf[offset-1] = '\0';
	if (secondarg) {
		len = strlen(secondarg) + 1;
		while (offset + len >= bufsize) {
			bufsize = bufsize * 2;
			cmd.buf = realloc(cmd.buf, bufsize * sizeof(char));
		}
		memcpy(cmd.buf + offset, secondarg, len * sizeof(char));
		offset += len;
	}
	else {
		offset += 1;
	}
	cmd.buf[offset-1] = '\0';
	
	len = strlen(thirdarg) + 1;
	while (offset + len >= bufsize) {
		bufsize = bufsize * 2;
		cmd.buf = realloc(cmd.buf, bufsize * sizeof(char));
	}
	memcpy(cmd.buf + offset, thirdarg, len * sizeof(char));
	offset += len;
	cmd.buf[offset-1] = '\0';
	
	for (cmd.argc = 0; argv[cmd.argc] != NULL; cmd.argc++) {
		len = strlen(argv[cmd.argc]) + 1;
		while (offset + len >= bufsize) {
			bufsize = bufsize * 2;
			cmd.buf = realloc(cmd.buf, bufsize * sizeof(char));
		}
		memcpy(cmd.buf + offset, argv[cmd.argc], len * sizeof(char));
		offset += len;
		cmd.buf[offset-1] = '\0';
	}
	cmd.len = offset;

	cmd.argc=cmd.argc+3; // first 3 args

	if (write(sock, &cmd, sizeof(cmd)) == -1 ||
	    write(sock, cmd.buf, cmd.len) == -1) {
		mooix_debug(PRIORITY_WARNING, "write to proxy: %s",
			strerror(errno), NULL);
		free(cmd.buf);
		return -1;
	}

	free(cmd.buf);
	return 0;
}
