/*
 * Mooix proxy wrapper library.
 * Proxy versions of functions, that pass function calls on to mood.
 *
 * Copyright 2001-2003 by Joey Hess <joey@mooix.net>
 * under the terms of the modified BSD license given in full in the file
 * COPYRIGHT.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <assert.h>
#include <signal.h>

#include "libmooproxy.h"
#include "client.h"
#include "real.h"

#define INTLEN 8

int proxy_open (char *name, int flags, mode_t mode) {
	int proxy=getproxy();
	char s_flags[INTLEN], s_mode[INTLEN];
	snprintf(s_flags, INTLEN, "%i", flags);
	snprintf(s_mode, INTLEN, "%i", mode & ~mask);
	
	if (sendcommand(proxy, command_open, name, s_flags, s_mode, NULL) == -1 ||
	    getresult(proxy) == -1)
		return -1;
	return getfd(proxy);
}

int proxy_unlink (const char *pathname) {
	if (strcmp(pathname, MOOFILE) == 0) {
		/* This is not a regular unlink, it is the signal to
		 * destroy the object. */
		int socket;
		int cwdfd = real_open(".", O_RDONLY, 0);

		if ((socket = setup_socket(MOOSOCK)) == -1)
			return -1;
	
		if (sendfd(socket, cwdfd) == -1 ||
		    sendcommand(socket, command_destroy, NULL) == -1 ||
		    close(cwdfd) == -1)
			return -1;

		return getresult(socket);
	}
	else {
		int proxy = getproxy();
		if (sendcommand(proxy, command_unlink, pathname, NULL) == -1)
			return -1;
		return getresult(proxy);
	}
}

int proxy_symlink (const char *oldpath, const char *newpath) {
	int proxy = getproxy();
	if (sendcommand(proxy, command_symlink, oldpath, newpath, NULL) == -1)
		return -1;
	return getresult(proxy);
}

int proxy_mkdir (const char *pathname, mode_t mode) {
	int proxy = getproxy();
	char s_mode[INTLEN];
	snprintf(s_mode, INTLEN, "%i", mode & ~mask);
	if (sendcommand(proxy, command_mkdir, pathname, s_mode, NULL) == -1)
		return -1;
	return getresult(proxy);
}

int proxy_rmdir (const char *pathname) {
	int proxy = getproxy();
	if (sendcommand(proxy, command_rmdir, pathname, NULL) == -1)
		return -1;
	return getresult(proxy);
}

int proxy_chmod (const char *path, mode_t mode) {
	int proxy = getproxy();
	char s_mode[INTLEN];
	snprintf(s_mode, INTLEN, "%i", mode);
	if (sendcommand(proxy, command_chmod, path, s_mode, NULL) == -1)
		return -1;
	return getresult(proxy);
}

int proxy_rename (const char *oldpath, const char *newpath) {
	int proxy = getproxy();
	if (sendcommand(proxy, command_rename, oldpath, newpath, NULL) == -1)
		return -1;
	return getresult(proxy);
}

int proxy_connect (int sockfd, const struct sockaddr_un *serv_addr, socklen_t addrlen) {
	int proxy = getproxy();

	if (sendcommand(proxy, command_connect, serv_addr->sun_path, NULL) == -1 ||
	    sendfd(proxy, sockfd) == -1) {
		fprintf(stderr, "connect failed\n");
		return -1;
	}
	return getresult(proxy);
}

int proxy_kill (pid_t pid, int sig) {
	int socket;
	char s_pid[INTLEN];
	char s_sig[INTLEN];
	int cwdfd = real_open(".", O_RDONLY, 0);
	
	if ((socket = setup_socket(MOOSOCK)) == -1)
		return -1;
	
	snprintf(s_pid, INTLEN, "%i", -1 * pid);
	snprintf(s_sig, INTLEN, "%i", sig);
	if (sendfd(socket, cwdfd) == -1 ||
	    sendcommand(socket, command_kill, s_pid, s_sig, NULL) == -1 ||
	    close(cwdfd) == -1)
		return -1;
	return getresult(socket);
}

/* The environment is ignored as it is not accepted by mood. */
int proxy_execve (const char *filename, char * const argv[], char * const envp[]) {
	int socket, ret, shcircuit=0; /* hi, Brian! */
	int cwdfd = real_open(".", O_RDONLY, 0);
	char *newcallstack = NULL;

	/* Block hup, so the parent process that execed can get sighups, 
	 * without this forked process interrupting its read by getting
	 * them.
	 */
	signal(SIGHUP, SIG_IGN);
	
	if (getenv("MOOIX_NONSTANDARD")) {
		newcallstack = getenv("MOOIX_NEW_CALLSTACK");
		shcircuit = (getenv("MOOIX_SHORT_CIRCUIT") != NULL);
	}
	
	if ((socket = setup_socket(MOOSOCK)) == -1)
		return -1;
	
	if (sendfd(socket, cwdfd) == -1 ||
	    sendcommandv(socket, command_execv, newcallstack, getenv("MOOIX_DEBUG"), filename, argv) == -1 ||
	    close(cwdfd) == -1 ||
	    sendfd(socket, 0) == -1 ||
	    sendfd(socket, 1) == -1 ||
	    sendfd(socket, 2) == -1)
		return getresult(socket);
	
	if (shcircuit)
		exit(0);
	
	ret = getresult(socket);
	if (errno == 0) {
		/* Exec done, so exit. */
		exit(WEXITSTATUS(ret));
	}
	return ret;
}
