/*
 * embedding python into mood speeds execution of python methods
 *
 * Copyright 2003 by Nick Borko <nick@dd.revealed.net>
 * under the terms of the GNU GPL.
 */
#ifdef EMBED_PYTHON

#include <Python.h>
#include <stdlib.h>

/* not in Python.h, but I don't want a warning. */
extern int Py_Main(int, char**);

/* global reference to the mooix module */
PyObject *mooix_module;

/*
 * I'm aware that there are some leaking references; but these routines are
 * only run once, and the interpreter is short-lived.
 */

void init_python(void) {
	PyObject *sys_module;

	Py_Initialize();

	/* add to sys to signal that this is an embedded interp */
	sys_module = PyImport_ImportModule("sys");
	PyObject_SetAttrString(sys_module, "MOOD_EMBEDDED", PyInt_FromLong(1L));
	Py_DECREF(sys_module);

	/* import mooix here; slightly more efficient */
	mooix_module = PyImport_ImportModule("mooix");
}

void run_python(int my_argc, char **my_argv, const char *this,
                const char *moosock, const char *method,
                const char *mooix_debug) {
	PyObject *os_module, *os_environ, *environ_dict, *init_func;

	/* we have to set environment variables in Python manually, since
	  os.environ was already set up in Py_Initialize.  Note that this
	  *doesn't* change the actual environment, so you have to do that
	  separately */
	os_module = PyImport_ImportModule("os");
	os_environ = PyObject_GetAttrString(os_module, "environ");

	/* os.environ is more like a UserDict; hopefully this implementation
	   won't change, since calling __setitem__ is cumbersome */
	environ_dict = PyObject_GetAttrString(os_environ, "data");

	setenv("THIS", this, 1);
	PyDict_SetItemString(environ_dict, "THIS",
	                     PyString_FromString(this));

	setenv("MOOSOCK", moosock, 1);
	PyDict_SetItemString(environ_dict, "MOOSOCK",
	                     PyString_FromString(moosock));

	setenv("METHOD", my_argv[0], 1);
	PyDict_SetItemString(environ_dict, "METHOD",
	                     PyString_FromString(my_argv[0]));

	if (mooix_debug) {
		setenv("MOOIX_DEBUG", mooix_debug, 1);
		PyDict_SetItemString(environ_dict, "MOOIX_DEBUG",
			             PyString_FromString(mooix_debug));
	}

	/* shouldn't really be necessary since we're going to exit anyway,
	   but let's be formal about it */
	Py_DECREF(os_module);
	Py_DECREF(os_environ);
	Py_DECREF(environ_dict);

	/* set sys.argv */
	/* the interpreter needs to be the first argument */
	my_argv[1] = my_argv[0];
	my_argv[0] = "python";
	PySys_SetArgv(my_argc, my_argv);

	/* initialize mooix vars */
	init_func = PyObject_GetAttrString(mooix_module, "initialize");
	PyObject_CallFunction(init_func, NULL);
	Py_DECREF(init_func);

	/* there is no easy way to do this in C, so let the interp do it */
	PyRun_SimpleString("from mooix import *");

	/* run the method, using argv */
	Py_Main(my_argc, my_argv);
}

void end_python(void) {
	Py_Finalize();
}

#endif /* EMBED_PYTHON */
