/*
 * Unix socket stuff.
 *
 * Copyright 2001-2003 by Joey Hess <joey@mooix.net>
 * under the terms of the GNU GPL.
 */

#include "mood.h"
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <syslog.h>
#include <string.h>
#include <sys/un.h>
#include <sys/uio.h>

/* Create and bind to a unix socket given the path to it. */
int bindsocket (char *path, int perms) {
	int sock;
	struct sockaddr_un sunx;
	
	unlink(path);
	
	memset(&sunx, 0, sizeof(sunx));
	sunx.sun_family = AF_UNIX;
	strncpy(sunx.sun_path, path, sizeof(sunx.sun_path));

	if ((sock = socket(PF_UNIX, SOCK_STREAM, PF_UNSPEC)) == -1) {
		perror("socket");
		exit(1);
	}
	
	if (bind(sock, (struct sockaddr *) &sunx,
	         sizeof(sunx.sun_family)+strlen(sunx.sun_path)) == -1 ||
	    chmod(path, perms) == -1 ||
	    listen(sock, 5) == -1) {
		perror("setup socket");
		close(sock);
		exit(1);
	}
	
	return sock;
}

/* Get credentials of the client connected to the socket. */
struct ucred *getcreds (int client) {
#if defined(__linux__)
	int passcred = -1;
	static struct ucred creds;
	socklen_t so_len = sizeof(passcred);
	
	if ((setsockopt(client, SOL_SOCKET, SO_PASSCRED, &passcred, so_len) != 0)) {
		perror("could not pass credentials");
		close(client);
		return NULL;
	}

	so_len = sizeof(creds);

	if (getsockopt(client, SOL_SOCKET, SO_PEERCRED, &creds, &so_len) == -1 ||
	    so_len != sizeof(creds)) {
		perror("did not get valid credentials");
		close(client);
		return NULL;
	}
	
	/* Now turn off credentials passing again, so fd passing can work. 
	 * I don't know exactly why it fails w/o this.. */
	passcred = 0;
	if ((setsockopt(client, SOL_SOCKET, SO_PASSCRED, &passcred, so_len) != 0)) {
		perror("setsockopt");
		close(client);
		return NULL;
	}
	
	return &creds;
#elif defined(__FreeBSD__)
	static struct ucred creds;
	uid_t uid;
	gid_t gid;

	if ((getpeereid(client, &uid, &gid) != 0)) {
		perror("getpeereid");
		close(client);
		return NULL;
	}
	creds.cr_uid = uid;
	creds.cr_gid = gid;
	return &creds;
#else
#error "getcreds() yet implemented for this system"
	return NULL;
#endif
}

/* Send a file descriptor through a socket. */
void sendfd(int client, int send_fd) {
	struct msghdr msg;
	struct cmsghdr* p_cmsg;
	struct iovec vec;
	char cmsgbuf[CMSG_SPACE(sizeof(send_fd))];
	int* p_fds;
	char sendchar = 0;

	msg.msg_control = cmsgbuf;
	msg.msg_controllen = sizeof(cmsgbuf);
	p_cmsg = CMSG_FIRSTHDR(&msg);
	p_cmsg->cmsg_level = SOL_SOCKET;
	p_cmsg->cmsg_type = SCM_RIGHTS;
	p_cmsg->cmsg_len = CMSG_LEN(sizeof(send_fd));
	p_fds = (int*)CMSG_DATA(p_cmsg);
	*p_fds = send_fd;
	msg.msg_controllen = p_cmsg->cmsg_len;
	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_iov = &vec;
	msg.msg_iovlen = 1;
	msg.msg_flags = 0;
	/* "To pass file descriptors or credentials you need to send/read at
	 * least one byte" (man 7 unix) */
	vec.iov_base = &sendchar;
	vec.iov_len = sizeof(sendchar);
	rassert(sendmsg(client, &msg, 0) != -1);
}

/* Receive a file descriptor on a socket. */
int getfd(int client) {
	struct msghdr msg;
	char recvchar;
	struct iovec vec;
	int recv_fd;
	char cmsgbuf[CMSG_SPACE(sizeof(recv_fd))];
	struct cmsghdr* p_cmsg;
	int* p_fd;

	vec.iov_base = &recvchar;
	vec.iov_len = sizeof(recvchar);
	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_iov = &vec;
	msg.msg_iovlen = 1;
	msg.msg_control = cmsgbuf;
	msg.msg_controllen = sizeof(cmsgbuf);
	msg.msg_flags = 0;
	p_fd = (int*) CMSG_DATA(CMSG_FIRSTHDR(&msg));
	*p_fd = -1;
	rassert(recvmsg(client, &msg, 0) == 1);
	p_cmsg = CMSG_FIRSTHDR(&msg);
	rassert(p_cmsg != NULL);
	p_fd = (int*) CMSG_DATA(p_cmsg);
	recv_fd = *p_fd;
	rassert(recv_fd != -1);
	return recv_fd;
}
