/*
 * Uid space management.
 *
 * Mood uses a large uid space to run methods in, so different running
 * methods cannot interfere with each other. (The matching gids are used
 * as well.)
 * 
 * Copyright 2001-2003 by Joey Hess <joey@mooix.net>
 * under the terms of the GNU GPL.
 */

#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <stdio.h>
#include <signal.h>
#include <assert.h>
#include <sys/types.h>
#include <unistd.h>
#include <syslog.h>
#include <sys/wait.h>
#include <pwd.h>
#include <sys/types.h>
#include <errno.h>
#include <string.h>
#include "mood.h"

gid_t entrygid;
uid_t firstuid;
uid_t lastuid;

/* Holds states of our uids. Indexed by (uid number - firstuid). */
unsigned int *uidstate = NULL;
/* States for uidstate. */
#define FREE 0
#define USED 1
#define UNAVAIL 2

/*
 * For speed of finding free uids, a doubly linked list of free uids. Just
 * to keep things interesting, the links back form a cycle, while the
 * forward links do not.
 */
struct uidlist_t {
	uid_t uid;
	struct uidlist_t *next;
	struct uidlist_t *prev;
} *freelist = NULL;

/* Check that a uid is valid. */
int validuid (uid_t uid) {
	if (uid >= firstuid && uid <= lastuid)
		return 1;
	else
		return 0;
}

/*
 * Validate that the thing connecting to us has the uid (and gid) of a known
 * moo method. No one else, except entrygid (and root), is allowed to talk
 * with us.
 */
int validate (struct ucred *creds) {

/* "portable" access to uid and gid fields */
#ifdef __linux__
#define creds_uid uid
#define creds_gid gid
#else
#define creds_uid cr_uid
#define creds_gid cr_gid
#endif

	if (creds == NULL) {
		warn("null creds");
		return 0;
	}
	else if (creds->creds_gid == entrygid || creds->creds_uid == 0)
		return 1;
	else if (uidstate && creds->creds_uid >= firstuid &&
                             creds->creds_uid <= lastuid &&
	         creds->creds_uid == creds->creds_gid && 
		 uidstate[creds->creds_uid - firstuid] == USED)
		return 1;
	warn("connection denied to uid %i", creds->creds_uid);
	return 0;
}

/* Allocates a uid for a method. */
uid_t uidalloc (void) {
	struct uidlist_t *next;
	uid_t uid;

	if (freelist == NULL)
		return -1;

	next = freelist;
	freelist = freelist->next;
	if (freelist)
		freelist->prev = next->prev;

	uid = next->uid;
	free(next);
	assert(uidstate[uid - firstuid] == FREE);
	uidstate[uid - firstuid] = USED;
	
	return uid;
}

/* Kill anything that might still be running under a uid, and mark it free. */
void reclaim (uid_t uid, int reallykill) {
	int status;
	int ret;
	pid_t pid;
	struct uidlist_t *new;
	char sockfile[64];

	/* Never reclaim unavailable uid's. */
	if (uidstate[uid - firstuid] == UNAVAIL)
		return;
	
	if (reallykill) {
		pid = fork();
		assert(pid != -1);
		if (pid == 0) {
			ret = setuid(uid);
			assert(ret != -1);
			kill(-1, SIGTERM); /* just being nice */
			ret = kill(-1, SIGKILL);
			assert(ret != -1);
			exit(0);
		}
		ret = waitpid(pid, &status, 0);
		assert(ret == pid);
		if (WIFEXITED(status) != 0 && WEXITSTATUS(status) != 0)
			die("error cleaning uid %i: %s", uid, strerror(errno));
	}
	
	uidstate[uid - firstuid] = FREE;
	callstack_unlink(uid);
	
	/* Remove socket file. */
	snprintf(sockfile, 64, "%s/%i.sock", RUNDIR, uid);
	unlink(sockfile);
	
	/*
	 * Add reclaimed uid to end of freelist, so
	 * it won't be reused for as long as is possible.
	 *
	 * This is why the freelist is set up as a circular linked list in
	 * one direction, but not in the other, BTW.
	 */
	new = malloc(sizeof(struct uidlist_t));
	new->uid = uid;
	new->next = NULL;
	if (freelist) {
		new->prev = freelist->prev;
		new->prev->next = new;
		freelist->prev = new;
	}
	else {
		new->prev = new;
		freelist = new;
	}
}

/* The devil's own subroutine. */
void clearuidspace (uid_t low, uid_t high, unsigned int reallykill) {
	uid_t uid;
	assert(low > 0); /* No this code is NOT here because of a bitter
			    experience. Why do you ask? */
	for (uid = low; uid <= high; uid++)
		reclaim(uid, reallykill);
}

/* Perform various sanity checks on a uid space to prevent foot injury. */
void checkuidspace (uid_t first, uid_t last) {
	struct passwd *passwd;
	int numgood;
	
	if (first > last)
		die("last uid (%i) is lower than first uid (%i)",
				last, first);
	/* 
	 * I don't think there's a spec for system uid's but this seems
	 * like a good start at a sanity check.
	 */
	if (first < 100)
		die("first uid (%i) is less than 100", first);
	
	numgood = last - first + 1;
	while ((passwd = getpwent())) {
		if (passwd->pw_uid >= first && passwd->pw_uid <= last) {
			/* Since the uid appeared, it must be used for
			 * something, so is unavailable to mood. */
			warn("not using uid %i: present in password file", passwd->pw_uid);
			uidstate[passwd->pw_uid - firstuid] = UNAVAIL;
			numgood--;
		}
	}
	endpwent();

	if (numgood < 1)
		die("no good uids!");
	if (numgood < 100)
		warn("only %i good uids found", numgood);
}

/* Setup the uid space. */
void uidspace (uid_t first, uid_t last, gid_t entry) {
	firstuid=first;
	lastuid=last;
	entrygid=entry;
	assert(uidstate == NULL); // only call this function once..
	uidstate = malloc((lastuid - firstuid + 1) * sizeof(unsigned int));
	assert(uidstate != NULL);
	checkuidspace(firstuid, lastuid);
}
