/*
 * This method is written in C for speed.
 *
 * TODO: optimize. It uses several arrays to look up keyed values that
 * should be done with hashes or something like that. Also, it leaks like
 * a sieve. 
 *
 * XXX as an optimization, check to see if the message is
 * targeted at a short list of specific objects. If so, rather than
 * searching through all the containers with propigate, just loop
 * over the destination objects, filtering the message through all
 * containers in between the obj and the destination, and
 * send the message. This optimization could also be used for the
 * onlyto parameter.
 */

#include <time.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <moomethod.h>
#include <ctype.h>

#define DEFAULTINTENSITY 50

/* Globals and data types {{{ */

/* Global holds the parameters passed to this method. */
param **params;

/* Global holds an object to skip, if skip parameter was passed. */
object *skip;
/* And this one holds an object that should be the only one to get a
 * message, if onlyto was passed. */
object *onlyto;
object *avatar;
/* The object that originatd the object. */
object *originator;

/* The types of message criteria. */
enum criteria_type {
	crit_sense,
	crit_object,
	crit_session,
};

/* Message criteria of sense type can have an associated intensity, and the
 * criteria field is used to hold either the sense or the name of the
 * object. */
struct criteria {
	enum criteria_type type;
	int intensity;
	char *criteria;
};

/* A message is a set of criteria plus message text to send. */
struct message {
	struct criteria **criteria;
	int numcriteria;
	int criteriaspace; /* amount of space currently alloced for criteria */
	char *message;
};

/* A message block is a set of messages. Messages are tried in order, and
 * delivered when their criteria match.
 */
struct message_block {
	struct message **messages;
	int num;
};

/* This is a list of the senses that matter when seeing if criteria allow a
 * message to be delivered. */
char **senses = NULL;
int senses_size = 0;
int num_senses = 0;

/* }}} End of globals and data types. */

/* Adds a sense to the senses list, if it is not in there already. */
void add_sense (char *sense) { /* {{{ */
	int i;
	int seen = 0;

	for (i = 0; ! seen && i < num_senses; i++)
		if (strcmp(sense, senses[i]) == 0)
			seen =1;
	if (! seen) {
		num_senses++;
		if (senses_size < num_senses) {
			senses_size += 8;
			senses = realloc(senses, sizeof(char *) * senses_size);
		}
		senses[num_senses - 1] = sense;
	}
} /* }}} */

/* Creates a new filter array. A filter is just an integer array of length
 * num_senses; so it has room for filter info for each sense on the senses
 * list. */
int *new_filter (void) { /* {{{ */
	int *ret;
	int i;

	ret = malloc(num_senses * sizeof(int *));
	for (i = 0; i < num_senses; i++)
		ret[i] = 0;
	return ret;
} /* }}} */

/* Creates a copy of an existing filter. */
int *copy_filter (const int *filter) { /* {{{ */
	int *ret;
	int i;
	
	ret = malloc(num_senses * sizeof(int *));
	for (i = 0; i < num_senses; i++)
		ret[i] = filter[i];
	return ret;
} /* }}} */

/* Prints a filter to stderr for debugging. */
void dump_filter(const int *filter, const char *msg) { /* {{{ */
	int i;
	if (! filter) {
		fprintf(stderr, "%s: (null)\n", msg);
	}
	else {
		fprintf(stderr, "%s: ", msg);
		for (i = 0; i < num_senses; i++)
			fprintf(stderr, "%s = %i ", senses[i], filter[i]);
		fprintf(stderr, "\n");
	}
} /* }}} */

/* Read all parameters, return NULL terminated array */
void getparams (void) { /* {{{ */
	param *p;
	int numparams=4;
	int curparam=0;

	params = malloc(sizeof(param *) * (numparams + 1));
	while ((p = getparam())) {
		params[curparam++]=p;
		if (curparam >= numparams) {
			numparams = numparams * 2;
			params=realloc(params, sizeof(param *) * (numparams + 1));
		}
	}
	params[curparam]=NULL;
} /* }}} */

/* Look up a parameter from an array by name. */
char *findparam (const char *key) { /* {{{ */
	/* TODO: optimize. hash? tsearch? */
	int i;
	for (i=0; params[i] != NULL; i++)
		if (strcmp(key, params[i]->name) == 0)
			return params[i]->value;
	return NULL;
} /* }}} */

/* Parses a criteria string into a criteria structure, and adds it to the
 * list in the passed message structure. */
void parsecriteria (char *cstring, struct message *m) { /* {{{ */
	char *end;
	struct criteria *criteria = malloc(sizeof(struct criteria));

	m->numcriteria++;
	if (m->numcriteria >= m->criteriaspace) {
		if (m->criteriaspace == 0)
			m->criteriaspace = 4;
		else
			m->criteriaspace *= 2;
		m->criteria = realloc(m->criteria, m->criteriaspace *
						   sizeof(struct criteria *));
	}
	criteria->intensity = DEFAULTINTENSITY;
	m->criteria[m->numcriteria - 1] = criteria;

	/* There can be whitespace both before and after the criteria
	 * string; remove it. */
	while (isspace(cstring[0]))
		cstring++;

	if (cstring[0] == '\0')
		return; /* empty */
	
	/* Remove trailing whitespace and make end point to the last
	 * character of the criteria string. */
	end = cstring + strlen(cstring) - 1;
	if (end > cstring) {
		while (isspace(end[0]))
			end--;
		end[1] = '\0';
	}
	
	/* Determine what type of criteria we have. */
	if (cstring[0] == '$') {
		criteria->type = crit_object;
		cstring++;
	}
	else if (strcmp(cstring, "session") == 0) {
		criteria->type = crit_session;
	}
	else {
		criteria->type = crit_sense;
		/* Is there an intensity? */
		if (end[0] == ')') {
			char *istart = strchr(cstring, '(');
			if (istart) {
				criteria->intensity = atoi(istart+1);
				istart[0] = '\0';
			}
		}
		add_sense(cstring);
	}

	criteria->criteria = cstring;
} /* }}} */

/* Parses a line of text into a message structure. */
struct message *parse_message (char *line) { /* {{{ */
	char *s;
	struct message *ret = malloc(sizeof(struct message));
	
	ret->numcriteria = 0;
	ret->criteriaspace = 0;
	ret->criteria = NULL;
	
	/* Split into message and criteria at the colon, with optional
	 * whitespace after. */
	s = strchr(line, ':');
	if (! s) {
		fprintf(stderr, "message parse error near \"%s\"\n", line);
		exit(1);
	}
	ret->message = s+1;
	s[0] = '\0';
	while (isspace(ret->message[0]))
		ret->message++;
	
	/* Criteria are delimited by commas. */
	while ((s = strchr(line, ',')) != NULL) {
		s[0] = '\0';
		parsecriteria(line, ret);
		line = s+1;
	}
	if (line[0] != '\0')
		parsecriteria(line, ret);
	
	return ret;
} /* }}} */

/* Reads messaages from the passed filename. If there are multiple blocks,
 * picks one at random. Parses the selected message block and returns it in
 * a structure. */
struct message_block *read_messages (const char *filename) { /* {{{ */
	char **lines;
	char *line;
	signed int *blocks;
	int line_count, block_count, lines_size, blocks_size;
	int start, end, pick, i;
	struct message_block *ret = malloc(sizeof(struct message_block));
	FILE *f = fopen(filename, "r");
	
	if (! f)
		return (struct message_block *) NULL;
	
	/* Read in all lines, keeping track of block boundries. */
	lines_size = 8;
	lines = malloc(lines_size * sizeof(char *));
	blocks_size = 3;
	blocks = malloc(blocks_size * sizeof(int *));
	block_count=0;
	blocks[block_count] = -1;
	for (line_count = 0; (line = mooix_getline(f, 0)); line_count++) {
		if (line_count == lines_size) {
			lines_size = line_count * 2;
			lines = realloc(lines, lines_size * sizeof(char *));
		}
		lines[line_count] = line;
		if (line[0] == '\0') {
			block_count++;
			if (block_count == blocks_size) {
				blocks_size = blocks_size * 2;
				blocks = realloc(blocks, blocks_size * sizeof(int));
			}
			blocks[block_count] = line_count;
		}
	}
	if (line_count > blocks[block_count]) {
		block_count++;
		if (block_count == blocks_size) {
			blocks_size = blocks_size * 2;
			blocks = realloc(blocks, blocks_size * sizeof(int));
		}
		blocks[block_count] = line_count;
	}

	if (line_count == 0)
		return (struct message_block *) NULL;

	/* Select a block at random. */
	srand(getpid() * (int) time(NULL));
	pick = (float) block_count * rand() / (RAND_MAX + 1.0);
	start = blocks[pick] + 1;
	end = blocks[pick + 1] - 1;
	
	/* Parse block. */
	ret->num = end - start;
	ret->messages = malloc((ret->num + 1) * sizeof(struct message *));
	for (i = start; i <= end; i++)
		ret->messages[i - start] = parse_message(lines[i]);

	return ret;
} /* }}} */

/* Generate a prettified name for an object, from the POV of the recipient. */
char *prettyname (object *obj, object *recipient) { /* {{{ */
	char *name, *article, *file;
	struct stat buf;
	
	if (objcmp(obj, recipient) == 0)
		return "you";
	
	file = fieldfile(obj, "name");
	if (! file)
		return "";
	/* The name might be a method to be called with no parameters.
	 * Rarely, but worth the stat for consistency. */
	if (stat(file, &buf) != 0)
		return "";
	if (((buf.st_mode & S_IXUSR) == S_IXUSR) ||
	    ((buf.st_mode & S_IXGRP) == S_IXGRP) ||
	    ((buf.st_mode & S_IXOTH) == S_IXOTH)) {
		/* Only allow running of methods that are marked as safe. */
		if (! truefield(obj, ".name-safe")) {
			return "";
		}
		else {
			FILE *f = runmethod(obj, "name", NULL);
			if (! f)
				return "";
			name = fgetvalue(f);
			fclose(f);
		}
	}
	else {
		name = getfield(file);
	}
	
	/* Hmm, article could be a method too, but it seems a little silly
	 * to support that. */
	file = fieldfile(obj, "article");
	if (! file)
		return "";
	article = getfield(file);
	if (! article || ! strlen(article)) {
		return name;
	}
	else {
		char *ret = malloc(strlen(article) + 1 + strlen(name) + 1);
		sprintf(ret, "%s %s", article, name);
		return ret;
	}
} /* }}} */

/* Used to expand things of the form "object->field" in messages. */
char *expandfield (char *var, char *field, object *recipient) { /* {{{ */
	struct stat buf;
	char *file, *val;
	object *obj;
	
	if (strcmp(var, "this") == 0) {
		obj = originator;
	}
	else {
		val = findparam(var);
		if (! val)
			return "";
		obj = derefobj(val);
		if (! obj)
			return "";
	}

	/* This is a special case to get the pronoun right for gender mixin
	 * lookups. It also works around fieldfile not supporting mixins.. */
	if (strncmp(field, "gender_", strlen("gender_")) == 0) {
		object *newobj;
		char *dir = fieldfile(obj, "gender");
		if (! dir) {
			return "";
		}
		newobj = getobj(dir);
		
		if (objcmp(obj, recipient) == 0) {
			dir = fieldfile(newobj, "self");
			if (! dir) {
				return "";
			}
			newobj = getobj(dir);
		}

		obj = newobj;
		
		field = field + strlen("gender_");
		
		if (! obj)
			return "";
	}
	
	/* Don't allow expansion of private fields. */
	if (field[0] == '.')
		return "";
	
	file = fieldfile(obj, field);
	if (! file)
		return "";

	if (stat(file, &buf) != 0)
		return "";
	
	/* If the file is another object, then pretty-print its name. */
	if (S_ISDIR(buf.st_mode))
		return prettyname(getobj(file), recipient);
		
	/* The "field" might be a method to be called with no parameters. */
	if (((buf.st_mode & S_IXUSR) == S_IXUSR) ||
	    ((buf.st_mode & S_IXGRP) == S_IXGRP) ||
	    ((buf.st_mode & S_IXOTH) == S_IXOTH)) {
		/* Only allow running of methods that are marked as safe. */
		char *safefield = malloc(1 + strlen(field) + strlen("-safe") + 1);
		sprintf(safefield, ".%s-safe", field);
		if (! truefield(obj, safefield)) {
			return "";
		}
		else {
			FILE *f = runmethod(obj, field, NULL);
			char *ret;
			if (! f)
				return "";
			ret = fgetvalue(f);
			fclose(f);
			return ret;
		}
	}
	else {
		/* Return the entire file, no matter how many lines. */
		FILE *f = fopen(file, "r");
		int size = 0;
		char *ret = NULL;
	
		if (! f || feof(f))
			return "";
		
		do {
			ret = realloc(ret, size + 128 + 1);
			if (! fgets(ret + size, 128, f)) {
				if (size == 0) {
					free(ret);
					return ""; /* eof with empty string */
				}
				else {
					ret[size]='\0';
					break;
				}
			}
			size = strlen(ret);
		} while (size > 0);

		/* Remove trailing newlines. */
		while (ret[size - 1] == '\n')
			ret[--size] = '\0';
		
		return ret;
	}
} /* }}} */

/* Used to expand things of the form "object->method(arg)" in messages */
char *expandmethod (char *var, char *method, char *param, object *recipient) { /* {{{ */
	/* Params are only currently supported for a special "verb"
	 * pseudo-method, that does verb conjugaton. */
	if (strlen(param) == 0) {
		return expandfield(var, method, recipient);
	}
	else if (strcmp(method, "verb") == 0) {
		char *comma;
		char *val;
		object *obj;

		if (strcmp(var, "this") == 0) {
			obj = originator;
		}
		else {
			val = findparam(var);
			if (! val)
				return "";
		
			obj = derefobj(val);
			if (! obj)
				return "";
		}
	
		comma = strchr(param, ',');
		if (! comma) {
			/* Crummy basic conjugation: remove 's' for d.o. */
			if (objcmp(recipient, obj) == 0) {
				int len = strlen(param);
				if (param[len - 1] == 's')
					param[len - 1] = '\0';
			}
			return param;
		}
		else {
			/* Conjugations supplied in the param, pick right one. */
			if (objcmp(recipient, obj) == 0) {
				return comma + 1;
			}
			else {
				comma[0] = '\0';
				return param;
			}
		}
	}
	else {
		return "";
	}
} /* }}} */

/* Escape xml tags and entities in the input. */
char *escapexml (char *text) { /* {{{ */
	char *s, *p=text, *ret=NULL;
	
	while ((s = strpbrk(p, "<>&"))) {
		if (! ret) {
			/* Worst case, every character must be escaped to
			 * &amp;, so it grows 4 times as large. */
			ret = malloc(strlen(text) * 4 + 1);
			ret[0] = '\0';
		}
		if (s > p)
			strncat(ret, p, s - p);
		switch (s[0]) {
			case '<':
				strcat(ret, "&lt;");
				break;
			case '>':
				strcat(ret, "&gt;");
				break;
			case '&':
				strcat(ret, "&amp;");
				break;
		}
		p = s+1;
	}

	if (! ret) {
		return text;
	}
	else {
		strcat(ret, p);
		return ret;
	}
} /* }}} */

/* Generate a message for an object from a template. */
char *expandmessage (const char *template, object *recipient) { /* {{{ */
	char *tmpl = strdup(template);
	char *s, *result;
	int retlen = 0;
	int retsize = strlen(template) + 80;
	char *p, *ret = malloc(retsize * sizeof(char));
	char *retlinestart = ret;
	int resultissender = 0;
	ret[0] = '\0';
	
	/* Break the template at \\n strings, and process each
	 * independently. Rejoin results with newlines. */
	while (tmpl) {
		char *nl = strstr(tmpl, "\\n");
		if (nl) 
			nl[0] = '\0';
		
		/* Do not allow spoofing of <sender> tags in the template. */
		if ((p = strstr(tmpl, "<sender>"))) {
			tmpl = escapexml(tmpl); /* No more Mr. nice guy. */
		}
		
		/* Looking for things of the forms:
		 * $var
		 * $var->field
		 * $var->method(param)
		 * Cannot modify the template, and s should be set to the
		 * end of anything found.
		 */
		while ((s = strchr(tmpl, '$'))) {
			char *var, *varstart = s + 1;
			
			/* Add anything before the var to ret. */
			s[0] = '\0';
			if (s > tmpl) {
				int span = s - tmpl;
				if (retlen + span >= retsize) {
					retsize = retlen + span + 1;
					ret = realloc(ret, retsize * sizeof(char));
				}
				strcat(ret, tmpl);
				retlen += span;		
			}
			result = NULL;
			
			/* Find var. */
			s++;
			while (isalnum(s[0]) || s[0] == '_')
				s++;
			if (s > varstart) {
				var = malloc(s - varstart + 1);
				bzero(var, s - varstart + 1);
				memcpy(var, varstart, s - varstart);
				
				/* Find field or method. */
				if (s[0] == '-' && s[1] == '>') {
					char *fm, *fmstart = s + 2;
					s+=2;
					while (isalnum(s[0]) || s[0] == '_')
						s++;
					if (s > fmstart) {
						fm = malloc(s - fmstart + 1);
						bzero(fm, s - fmstart + 1);
						memcpy(fm, fmstart, s - fmstart);

						/* Find param. */
						if (s[0] == '(') {
							char *param, *paramstart = s + 1;
							s++;
							while (s && s[0] != ')')
								s++;
							if (s[0] == ')') {
								param = malloc(s - paramstart + 1);
								bzero(param, s - paramstart + 1);
								memcpy(param, paramstart, s - paramstart);
								s++;
								
								result = expandmethod(var, fm, param, recipient);
							}
						}
						else {
							result = expandfield(var, fm, recipient);
						}
					}
				}
				else {
					/* Expand a variable. */
					object *obj = NULL;
					char *val = NULL;

					if (strcmp(var, "this") == 0) {
						obj = originator;
					}
					else {
						val = findparam(var);
						obj = derefobj(val);
						if (! obj) {
							result = val;
						}
					}
					
					if (obj) {
						result = prettyname(obj, recipient);
						if (avatar && objcmp(obj, avatar) == 0)
							resultissender = 1;
					}
				}

				if (result) {
					int span;

					result = escapexml(result);
					
					span = strlen(result);

					if (resultissender) {
						/* Will add tag for antispoofing display. */
						span += strlen("<sender></sender>");
						/* Adjust where the beginning of the real line (to uppercase) is, skipping over the tag. */
						if (ret[0] == '\0' || ret[-1] == '\n')
							retlinestart += strlen("<sender>");
					}
					
					if (retlen + span >= retsize) {
						retsize = retlen + span + 1;
						ret = realloc(ret, retsize * sizeof(char));
					}
					
					if (resultissender) {
						strcat(ret, "<sender>");
					}
					strcat(ret, result);
					if (resultissender) {
						strcat(ret, "</sender>");
					}
					resultissender = 0;
					retlen += span;
				}
			}

			tmpl = s;

		}

		/* Add anything after the last expansion to ret. */
		if (tmpl) {
			int span = strlen(tmpl);
			if (retlen + span >= retsize) {
				retsize = retlen + span + 1;
				ret = realloc(ret, retsize * sizeof(char));
			}
			strcat(ret, tmpl);
			retlen += span;
		}
		
		if (nl) {
			/* Add newline to ret. */
			if (retlen + 1 >= retsize) {
				retsize = retlen + 2;
				ret = realloc(ret, retsize * sizeof(char));
			}
			ret[retlen] = '\n';
			ret[retlen + 1] = '\0';
			retlen++;
			
			tmpl = nl+2;
		}
		else {
			tmpl = nl; /* break loop */
		}

		/* Upper-case the beginning of the line in ret. */
		retlinestart[0] = toupper(retlinestart[0]);
		retlinestart = ret + strlen(ret);
	}
	
	return ret;
} /* }}} */

/* Check criteria and maybe deliver a message to an object. */
int deliver_message (object *obj, const struct message *msg, const int *filter) { /* {{{ */
	int i, j;
	int isthis;
	int send = 1;
	struct criteria *crit;
	object *pobj, *onlyto_session = NULL;
	int *usedsenses = new_filter(); /* not strictly a filter, but same 
				           data structure will do */
	
	for (i = 0; i < msg->numcriteria && send; i++) {
		crit = msg->criteria[i];
		switch (crit->type) {
			case crit_sense:
				for (j = 0; j < num_senses; j++) {
					if (strcmp(senses[j], crit->criteria) == 0) {
						if (crit->intensity < filter[j])
							send = 0;
						usedsenses[j] = crit->intensity;
						break;
					}
				}
				break;
			case crit_object:
				if (strcmp(crit->criteria, "this") == 0) {
					pobj = originator;
					isthis = 1;
				}
				else {
					pobj = derefobj(findparam(crit->criteria));
					isthis = 0;
				}
				if (pobj) {
					if (objcmp(obj, pobj) != 0)
						send = 0;
					if (! isthis)
						free(pobj);
				}
				else {
					send = 0;
				}
				break;
			case crit_session:
				{
					onlyto_session = derefobj(findparam("session"));
					if (avatar && objcmp(obj, avatar) != 0)
						send = 0;
				}
				break;
		}
	}

	if (send) {
		/* Pass the list of senses that this messages uses along
		 * with the expanded message, and all the parameters passed
		 * to this method on to the tell method. */
		FILE **fds, *wtr, *rdr;
		char *message = expandmessage(msg->message, obj);

		/* Use runmethod_raw to avoid having to build up a big
		 * parameters data structure. */
		fds = runmethod_raw(obj, "notice");
		if (fds == NULL)
			return 0;
		wtr = fds[0];
		rdr = fds[1];
		
		/* Let the notice method know it goes only to the one
		 * session. */
		if (onlyto_session) {
			fprintf(wtr, "session\nmooix:%s\n", onlyto_session->dir);
		}
		
		fprintf(wtr, "message\n%s\n", escape(message));
		for (i = 0; i < num_senses; i++) {
			if (usedsenses[i]) {
				fprintf(wtr, "sense\n%s\nintensity\n%i\n",
						senses[i], usedsenses[i]);
			}
		}
		fprintf(wtr, "originator\nmooix:%s\n", originator->dir);
		for (i=0; params[i] != NULL; i++) {
			fprintf(wtr, "%s\n%s\n", params[i]->name, params[i]->value);
		}

		/* Let the method know the params are done. */
		fclose(wtr);
		fgetallvals(rdr);
		free(message);

		/* This method returns a list of objects the message was sent
		 * to. Just print them out.. */
		printf("mooix:%s\n", obj->dir);
	}

	free(usedsenses);
	return send;
} /* }}} */

/* Types of filters. See mooix:filter/base->trigger.inf */
#define TO_TRIGGERED 2
#define IN_TRIGGERED 4
#define OUT_TRIGGERED 8
#define INTER_TRIGGERED 16

/* Checks to see if the passed object has messagefilters. If so,
 * checks to see if each filter will trigger, and if it does
 * accumulates its filters to passed filter arrays. If an array is NULL,
 * does not bother accumulating to that array. */
void filter (object *obj, int *to_f, int *in_f, int *out_f, int *inter_f) { /* {{{ */
	int i;
	char fieldname[32];
	char *file;
	char *value;
	char *line;
	FILE *f = NULL, *df;
	int trigger;
	int *to_filter[4]; /* 4 = number of filters in parameters */
	int num_to_filter = 0;
	char *list;
	object *messagefilter_list, *messagefilter;
	char *messagefilters_dir;

	/* I don't bother to check for inherited messageflter objects. 
	 * Quicker not to. */
	messagefilters_dir = malloc(strlen(obj->dir) + strlen("/messagefilters") + 1);
	sprintf(messagefilters_dir, "%s/messagefilters", obj->dir);
	messagefilter_list = getobj(messagefilters_dir);
	list=fieldfile(messagefilter_list, "list");
	if (list)
		f = fopen(list, "r");
	freeobj(messagefilter_list);
	if (! f)
		return;
	
	while ((line = mooix_getline(f, 0))) {
		messagefilter=derefobj(line);
		free(line);

		file = fieldfile(messagefilter, "trigger");
		if (! file)
			continue;
		value = getfield(file);
		if (! value)
			continue;
		trigger=atoi(value);
		if (! trigger)
			continue;
	
		/* Work out which of the input filters can be affected by
		 * this object based on the value of is_filter. Fill an array
		 * with the filters to act on, and then we can just loop over
		 * it below. */
		num_to_filter=0;
		if (to_f && trigger & TO_TRIGGERED) {
			to_filter[num_to_filter] = to_f;
			num_to_filter++;
		}
		if (in_f && trigger & IN_TRIGGERED) {
			to_filter[num_to_filter] = in_f;
			num_to_filter++;
		}
		if (out_f && trigger & OUT_TRIGGERED) {
			to_filter[num_to_filter] = out_f;
			num_to_filter++;
		}
		if (inter_f && trigger & INTER_TRIGGERED) {
			to_filter[num_to_filter] = inter_f;
			num_to_filter++;
		}
		if (! num_to_filter)
			continue;
	
		/* Static filters. */
		for (i = 0; i < num_senses; i++) {
			snprintf(fieldname, 31, "filter_%s", senses[i]);
			file = fieldfile(messagefilter, fieldname);
			if (file != NULL) {
				FILE *fh = fopen(file, "r");
				if (fh != NULL) {
					value = mooix_getline(fh, 0);
					if (value) {
						int f;
						int v = atoi(value);
						free(value);
						for (f = 0; f < num_to_filter; f++)
							to_filter[f][i] += v;
					}
				}
				fclose(fh);
				free(file);
			}
		}

		/* Dynamic filter. Since this is a mite expensive to do every
		 * time, only do it if the trigger was ORed with 1. */
		if (trigger & 1 &&
		    (df = runmethod(messagefilter, "filtermessage", NULL))) {
			while (! feof(df)) {
				char *sense = fgetvalue(df);
				value = fgetvalue(df);
				if (sense != NULL && filter != NULL) {
					for (i = 0; i < num_senses; i++) {
						if (strcmp(senses[i], sense) == 0) {
							int f;
							int v = atoi(value);
							for (f = 0; f < num_to_filter; f++)
								to_filter[f][i] += v;
							break;
						}
					}
				}
				free(sense);
				free(value);
			}
			fclose(df);
		}
		
		freeobj(messagefilter);
	}
} /* }}} */

/* Propigates a message throughout the current location. The parameters
 * are:
 *
 * obj		The object to propigate to.
 * from		If recursing, the object propigated from.
 * fromloc	Set if from is the object's location.
 * messages	The messages to deliver.
 * traverse_f	Filters accumulated traversing containers on the way to
 * 		this object.
 * inter_f	Filters accumulated from locations of object. If fromloc is
 * 		not set, then this parameter is MODIFIED by this call.
 */
void propigate (object *obj, object *from, int fromloc,
                const struct message_block *messages,
                int *traverse_f, int *inter_f) { /* {{{ */
	int *to_f;
	int *out_f = NULL;
	int *my_inter_f = NULL;
	FILE *c;
	object *location = NULL;
	int traverse_location = 0;
	int is_aware = truefield(obj, "aware");
	char *contents_file = malloc(strlen(obj->dir) + strlen("/contents/list") + 1);
	sprintf(contents_file, "%s/contents/list", obj->dir);
	
	if (fromloc) {
		/* Leaf object optimisation: If the object is not a
		 * container, and the message has propigated from its
		 * location, and the object is not aware, then there is no
		 * point in worrying about filters, or doing delivery, or
		 * any of that. */
		if (! is_aware) {
			struct stat st_buf;
			if (stat(contents_file, &st_buf) != 0) {
				free(contents_file);
				return;
			}
		}
	}
	else {
		/* Find the location, if there is one, and see if it
		 * should be traversed. */
		char *file = malloc(strlen(obj->dir) + strlen("/location") + 1);
		sprintf(file, "%s/location", obj->dir);
		location = getobj(file);
		if (statobj(location) && (! from || objcmp(from, location) != 0)) {
			traverse_location = 1;
			out_f = new_filter();
		}
	}

	/* Always copy traverse_f, to avoid modifying it. */
	traverse_f = copy_filter(traverse_f);

	/* Apply filters for this object. If the message is coming from the
	 * location, apply in_triggered filters to traverse_f. Note that
	 * out_f will be NULL unless there's an untraversed location. */
	to_f=new_filter();
	if (fromloc) {
		/* Since fromloc is set, do not modify inter_f.
		 * Instead, make a copy now. */
		inter_f = copy_filter(inter_f);
		filter(obj, to_f, traverse_f, out_f, inter_f);
	}
	else {
		/* Don't change inter_f yet, since location propigation
		 * needs to use the one that is not filtered through this
		 * object. */
		my_inter_f = new_filter();
		filter(obj, to_f, NULL, out_f, my_inter_f);
	}
	
	if (traverse_location) {
		/* Propigate to the location (and deeper, recursively). As
		 * locations are traversed, inter_f accululates all the
		 * inter_triggered filters for the locations, which is why
		 * locations are traversed first. For traverse_f, 
		 * pass a filter that is the sum of our traverse_f and the
		 * out_triggered filters for this object, if the message is
		 * coming from somthing other than this object. If it comes
		 * directly from this object, then the out_f does not
		 * apply. */
		int *f;
		int i;
		if (from) {
			f = malloc(num_senses * sizeof(int *));
			for (i= 0; i < num_senses; i++)
				f[i] = out_f[i] + traverse_f[i];
		}
		else {
			f = new_filter();
		}
		propigate(location, obj, 0, messages, f, inter_f);
		free(f);
		freeobj(location);
	}
	
	if (! fromloc) {
		/* Merge my_inter_f into inter_f, after location
		 * propigation is complete. */
		int i;
		for (i= 0; i < num_senses; i++)
			inter_f[i] += my_inter_f[i];
		free(my_inter_f);
	}
	
	/* Is this a container? If so, propigate to all the contents. */
	c = fopen(contents_file, "r");
	free(contents_file);
	if (c) {
		/* Get all contents of the container in an array first,
		 * then close the fd, before filtering through them. This
		 * avoids problems with possibly running out of fds if
		 * there are a lot of objects to traverse. */
		int size = 16;
		object **contents = malloc(size * sizeof(object *));
		int i, num_contents = 0;
		char *item;
		
		while ((item = fgetvalue(c))) {
			object *contents_obj = derefobj(item);
			if (contents_obj) {
				contents[num_contents] = contents_obj;
				free(item);
				num_contents++;
				if (num_contents >= size) {
					size *= 2;
					contents = realloc(contents, size * sizeof(object *));
				}
			}
		}
		
		fclose(c);
		
		for (i = 0; i < num_contents; i++) {
			if (! from || objcmp(contents[i], from) != 0) {
				/* Since fromloc is not set, this call will
				 * not modify the passed filters. */
				propigate(contents[i], obj, 1, messages, traverse_f, inter_f);
			}
			freeobj(contents[i]);
		}

		free(contents);
	}

	/* Finally, delivery to this object. */
	if (is_aware && 
	    (! skip || objcmp(obj, skip) != 0) &&
	    (! onlyto || objcmp(obj, onlyto) == 0)) {
		int i;
		/* Combine to_f, traverse_f, and inter_f into one. */
		for (i= 0; i < num_senses; i++)
			to_f[i] += traverse_f[i] + inter_f[i];
		/* Delivery. */
		for (i = 0; i <= messages->num; i++) {
			if (deliver_message(obj, messages->messages[i], to_f))
				break;
		}
	}
	
	/* Cleanup copies made to prevent filter modifications. */
	free(traverse_f);
	if (fromloc)
		free(inter_f);
} /* }}} */

int main (int argc, char **argv) { /* {{{ */
	object *this;
	char *messagefield;
	struct message_block *messages;
	
	methinit();
	getparams();

	this = getobj(getenv("THIS")); /* needs to be an abs path */
	avatar = derefobj(findparam("avatar"));
	skip = derefobj(findparam("skip"));
	onlyto = derefobj(findparam("onlyto"));
	originator = derefobj(findparam("originator"));
	if (! originator)
		originator=this;
	messagefield = strdup(findparam("event"));
	if (! messagefield) {
		fprintf(stderr, "event field required\n");
		exit(1);
	}
	messagefield = realloc(messagefield, strlen(messagefield) + 5);
	messagefield = strcat(messagefield, ".msg");
	
	messages = read_messages(fieldfile(originator, messagefield));
	if (messages == NULL)
		exit(0);
	propigate(this, NULL, 0, messages, new_filter(), new_filter());
	return 0;
} /* }}} */
