# moosic - the client portion of the moosic jukebox system.
#
# Copyright (C) 2001-2003 Daniel Pearson <daniel@nanoo.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import sys, socket, os, os.path, getopt, xmlrpclib, errno, time, locale, re

from utilities import *
from moosic_factory import *
from moosic_dispatcher import *

# Define the True and False constants if they don't already exist.
try: True
except NameError: True = 1
try: False
except NameError: False = 0

# Giving stderr a shorter name is convenient when printing error messages.
err = sys.stderr

# Set the locale to the user's default settings, if possible.
try: locale.setlocale(locale.LC_ALL, '')
except: pass


def main(argv):
    COMMANDS = get_command_docs() \
        + center_text('This Moosic has Super Cow Powers.', pad_char=' ')
    USAGE = "usage: " + os.path.basename(argv[0]) + \
        " [options] <command>" + '''
    Options:
        -d, --shuffle-dir       When a directory is named on the command line,
                                shuffle the result of recursing through the
                                directory before inserting it into the list.
        -a, --shuffle-args      Shuffle only the arguments explicitly specified
                                on the command line.
        -g, --shuffle-global    Shuffle the entire argument list after
                                directory arguments specified on the command
                                line have been replaced with their contents.
                                This is the default behavior.
        -o, --inorder           Don't shuffle the given filelist at all, and
                                maintain the order specified on the command
                                line.
        -s, --sort              Sort the filelist, regardless of the order
                                specified on the command line.
        -i, --ignore-case       Treat any given regular expressions as if they
                                were case-insensitive.
        -r, --no-recurse        Don't replace directories named on the command
                                line with their contents.
        -n, --non-file-args     Don't change any names given in a filelist.
                                Useful if your filelist consists of URLs or
                                other objects that aren't local files.
        -f, --auto-find         Replace each string in the given filelist with
                                the results of a "fuzzy" search for music files
                                which match that string. (Beware: this can
                                be slow with large music collections.)
        -F, --auto-grep         Replace each string in the given filelist with
                                the results of a regular-expression search for
                                music files which match that string. (Beware:
                                this can be slow with large music collections.)
        -m, --music-dir <dir>   Specifies the directory to search when using the
                                "auto-find" and "auto-grep" features.
                                (Default: ~/music/)
        -c, --config-dir <dir>  Specifies the directory where moosic should
                                find the files kept by moosicd.
                                (Default: ~/.moosic/)
        -t, --tcp <host>:<port> Communicate with a Moosic server that is
                                listening to the specified TCP/IP port on the
                                specified host.  (Not recommended.)
        -N, --no-startserver    Don't automatically start moosicd if it isn't
                                already running.
        -S, --showcommands      Print the list of possible commands and exit.
        -h, --help              Print this help text and exit.
        -v, --version           Print version information and exit.
                       This Moosic has Super Cow Powers.'''

    # Option processing.
    def process_options(arglist, opts):
        opts = opts.copy()
        try:
            opt_spec = { 'd':'shuffle-dir',
                         'a':'shuffle-args',
                         'g':'shuffle-global',
                         'o':'inorder',
                         's':'sort',
                         'i':'ignore-case',
                         'r':'no-recurse',
                         'n':'non-file-args',
                         '':'no-file-munge',
                         'f':'auto-find',
                         'F':'auto-grep',
                         'm:':'music-dir=',
                         'c:':'config-dir=',
                         't:':'tcp=',
                         'N':'no-startserver',
                         'S':'showcommands',
                         'h':'help',
                         'v':'version', }
            short_opts = ''.join(opt_spec.keys())
            long_opts = opt_spec.values()
            options, arglist = getopt.getopt(arglist, short_opts, long_opts)
        except getopt.error, e:
            print >>err, 'Option processing error:', e
            sys.exit(1)
        for option, val in options:
            if option == '-d' or option == '--shuffle-dir':
                opts['shuffle-dir'] = True
                opts['shuffle-global'] = False
            if option == '-a' or option == '--shuffle-args':
                opts['shuffle-args'] = True
                opts['shuffle-global'] = False
            if option == '-g' or option == '--shuffle-global':
                opts['shuffle-global'] = True
            if option == '-o' or option == '--inorder':
                opts['shuffle-global'] = False
                opts['shuffle-args'] = False
                opts['shuffle-dir'] = False
            if option == '-s' or option == '--sort':
                opts['shuffle-global'] = False
                opts['shuffle-args'] = False
                opts['shuffle-dir'] = False
                opts['sort'] = True
            if option == '-i' or option == '--ignore-case':
                opts['ignore-case'] = True
            if option == '-r' or option == '--no-recurse':
                opts['dir-recurse'] = False
            if option == '-n' or option == '--no-file-munge' or option == '--non-file-args':
                opts['file-munge'] = False
                opts['dir-recurse'] = False
            if option == '-f' or option == '--auto-find':
                opts['auto-find'] = True
                opts['auto-grep'] = False
            if option == '-F' or option == '--auto-grep':
                opts['auto-grep'] = True
                opts['auto-find'] = False
            if option == '-m' or option == '--music-dir':
                opts['music-dir'] = os.path.abspath(os.path.expanduser(val))
            if option == '-c' or option == '--config-dir':
                opts['config-dir'] = os.path.abspath(os.path.expanduser(val))
            if option == '-t' or option == '--tcp':
                if ":" not in val:
                    print >>err, 'Invalid address:', val
                    print >>err, 'You must specify both a hostname and a port number.',
                    print >>err, 'For example, "example.com:123"'
                    sys.exit(1)
                host, port = val.split(':', 1)
                try:
                    port = int(port)
                except ValueError, e:
                    print >>err, "Invalid port number:", port
                    sys.exit(1)
                opts['tcp-address'] = (host, port)
            if option == '-N' or option == '--no-startserver':
                opts['start moosicd'] = False
            if option == '-S' or option == '--showcommands':
                print COMMANDS
                sys.exit(0)
            if option == '-h' or option == '--help':
                print USAGE
                sys.exit(0)
            if option == '-v' or option == '--version':
                print "moosic", VERSION
                print """
Copyright (C) 2001-2003 Daniel Pearson <daniel@nanoo.org>
This is free software; see the source for copying conditions.  There is NO
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE."""
                sys.exit(0)
        return arglist, opts

    home = os.getenv('HOME', '/tmp')
    # Set the built-in default options.
    opts = {'shuffle-global':True, 'shuffle-args':False, 'shuffle-dir':False,
            'debug':False, 'file-munge':True, 'sort':False, 'dir-recurse':True,
            'config-dir':os.path.join(home, '.moosic', ''), 'tcp-address':None,
            'music-dir':os.path.join(home, 'music', ''), 'auto-find':False,
            'auto-grep':False, 'start moosicd':True, 'ignore-case':False,
            'sort':False, 'rc-filename':os.path.join(home, '.moosicrc')}
    # Gather options specified before the command.
    arglist, opts = process_options(argv[1:], opts)
    # Pluck the command out of the argument list.
    if not arglist:
        print >>err, "You must provide a command."
        print >>err, "Use the --showcommands option to learn what commands are available."
        print >>err, "Use the --help option to learn what options are available."
        print >>err, "usage:", os.path.basename(argv[0]), "[options] <command>"
        sys.exit(1)
    command = arglist.pop(0)
    command = command.lower()
    command = re.sub(r'[\W_]', r'', command)
    # Gather options specified after the command.
    arglist, opts = process_options(arglist, opts)
    # TODO: Gather option values from a config file.
    #file_opts = process_configfile(opts['rc-filename'], opts)
    # Use the options from the command-line to override the options from the
    # config file.
    #file_opts.update(opts)
    #opts = file_opts

    if command not in dispatcher:
        print >>err, 'Error: invalid command: "%s"' % (command)
        print >>err, "Use the --showcommands option or the 'help' command to see"
        print >>err, "the available commands."
        print >>err, "Use the --help option to learn about the possible options."
        print >>err, "usage:", os.path.basename(argv[0]), "[options] <command>"
        sys.exit(1)

    # Check the number of arguments given to the command.
    if not check_args(command, arglist):
        sys.exit(1)

    # Create a proxy object for speaking to the Moosic server.
    if opts['tcp-address']:
        host, port = opts['tcp-address']
        moosic = InetMoosicProxy(host, port)
    else:
        server_address = os.path.join(opts['config-dir'], 'socket')
        moosic = LocalMoosicProxy(server_address)

    # Make sure that our connection to the server is working properly.
    try:
        moosic.no_op()
    except socket.error, e:
        if e[0] in (errno.ECONNREFUSED, errno.ENOENT):
            # The server doesn't seem to be running, so let's try to start it
            # for ourselves.
            if opts['tcp-address']:
                print >>err, wrap("Error: The server (moosicd) doesn't seem " \
                        "to be running, and I was unable to start it myself " \
                        "because:", 79)
                print >>err, "The target Moosic server is on a remote computer."
                sys.exit(1)
            if opts['start moosicd']:
                print >>err, "Notice: The Moosic server isn't running, so it " \
                             "is being started automatically."
                failure_reason = startServer(opts['config-dir'])
            else:
                failure_reason = "Automatic launching of the server is disabled."
            if failure_reason:
                print >>err, wrap("Error: The server (moosicd) doesn't seem " \
                        "to be running, and I was unable to start it myself " \
                        "because:", 79)
                print >>err, failure_reason
                sys.exit(1)
            else:
                # Wait bit to give moosicd time to start up.
                time.sleep(0.25)
                # Test the server connection again.
                try:
                    moosic.no_op()
                except Exception, e:
                    # We tried our best. Finally give up.
                    print >>err, "An attempt was made to start the Moosic " \
                                 "server, but it still can't be contacted."
                    print >>err, "%s: %s" % (str(e.__class__).split('.')[-1], e)
                    sys.exit(1)
        else:
            print >>err, "Socket error:", e
            sys.exit(1)

    try:
        # Dispatch the command.
        sys.exit(dispatcher[command](moosic, arglist, opts))
    except socket.error, e:
        print >>err, "Socket error:", e[1]
        sys.exit(1)
    except xmlrpclib.Fault, e:
        if ':' in e.faultString:
            fault_type, fault_msg = e.faultString.split(':', 1)
            fault_type = fault_type.split('.')[-1]
            print >>err, "Error from Moosic server: [%s] %s" % (fault_type, fault_msg)
        else:
            print >>err, "Error from Moosic server:", e.faultString
        sys.exit(1)
    except xmlrpclib.ProtocolError, e:
        print >>err, "RPC protocol error: %s %s." % (e.errcode, e.errmsg)
        sys.exit(1)
    except ValueError, e:
        print >>err, "Error:", e
        sys.exit(1)
    except Exception, e:
        if isinstance(e, SystemExit):
            raise e
        else:
            print >>err, "%s: %s" % (str(e.__class__).split('.')[-1], e)
            sys.exit(1)

main(sys.argv)
