# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Authors: Florian Boucault <florian@fluendo.com>
#          Olivier Tilloy <olivier@fluendo.com>

"""
Browser controller, used to display controllers sequentially and store the
navigation history.
"""

from twisted.python import reflect
from twisted.internet import task

from elisa.core import common
from elisa.core.utils import defer

from elisa.plugins.pigment.pigment_controller import PigmentController


class EmptyHistory(Exception):

    """
    Raised by L{BrowserController} when trying to access its controllers when
    there are none.
    """

    pass


TRANSITION_FORWARD_IN = 'transition_forward_in'
TRANSITION_FORWARD_OUT = 'transition_forward_out'
TRANSITION_BACKWARD_IN = 'transition_backward_in'
TRANSITION_BACKWARD_OUT = 'transition_backward_out'

TRANSITIONS = (TRANSITION_FORWARD_IN, TRANSITION_FORWARD_OUT,
               TRANSITION_BACKWARD_IN, TRANSITION_BACKWARD_OUT)


class BrowserController(PigmentController):

    """
    A browser controller handles the lifecycle of controllers and provides a
    way to navigate a sequence of controllers pretty much like one would do
    with websites using a web browser. In that analogy websites are controllers
    and URLs are controller paths.

    Visual transitions are applied when displaying a new controller in order to
    replace aesthetically the previously shown controller.

    A browser also keeps track of the history of controllers that were visited.

    @ivar current_controller: the current controller, that is the controller
                              loaded last
    @type current_controller: L{elisa.plugins.pigment.pigment_controller.PigmentController}
    """

    default_config = {
        TRANSITION_FORWARD_IN: 'elisa.plugins.poblesec.transitions.FadeIn',
        TRANSITION_FORWARD_OUT: 'elisa.plugins.poblesec.transitions.FadeOut',
        TRANSITION_BACKWARD_IN: 'elisa.plugins.poblesec.transitions.FadeIn',
        TRANSITION_BACKWARD_OUT: 'elisa.plugins.poblesec.transitions.FadeOut',
                     }

    config_doc = {
        TRANSITION_FORWARD_IN: 'Incomming controller transition when going forward',
        TRANSITION_FORWARD_OUT: 'Outgoing controller transition when going forward',
        TRANSITION_BACKWARD_IN: 'Incomming controller transition when going backward',
        TRANSITION_BACKWARD_OUT: 'Outgoing controller transition when going backward',
                 }

    def __init__(self):
        super(BrowserController, self).__init__()
        self._current_controller = None
        self._pending_load_dfr = None
        self._history = []

    @property
    def current_controller(self):
        return self._current_controller

    def initialize(self):
        dfr = super(BrowserController, self).initialize()
        for transition_name in TRANSITIONS:
            self._create_transition(transition_name)

        return dfr

    def _create_transition(self, name):
        path = self.config[name]

        transitionClass = reflect.namedAny(path)
        transition = transitionClass()

        # store the transition in an instance variable
        setattr(self, '_' + name, transition)

    def clean(self):
        # Unload and clean all the controllers in the history.

        def iterate_history():
            while len(self._history) > 0:
                controller = self._history.pop()
                yield controller.clean()

        def history_cleaned(result):
            return super(BrowserController, self).clean()

        dfr = task.coiterate(iterate_history())
        dfr.addCallback(history_cleaned)
        return dfr

    def load_new_controller(self, path, label='', hints=[], **kwargs):
        """
        Create a new controller from a given L{path} using given arguments
        L{kwargs} and display it instead of the previously loaded one, also
        called the current controller.
        An animated visual transition is applied to both the current controller
        and the newly created one.

        The newly created controller has a reference to the browser itself
        stored in the instance variable 'browser'. This allows the controller
        to use features of the browser that created it.
        For example: from each controller loaded with this  method we can
        load the new controller simply by calling::

            self.browser.load_new_controller(path, label, **kwargs)

        In this manner, we can ensure that the browser for which the new
        controller is loaded is the correct one.

        Return a deferred fired when the new controller is completely
        initialized with the controller as a result. At that point the visual
        transitions are not necessarily finished yet.

        @param path:   path of the controller to be created
        @type path:    C{str}
        @param label:  a display name representing the controller that can be
                       used in the user interface of the browser
        @type label:   C{unicode}
        @param hints:  hints used by the browser on how to display the
                       controller to be created
        @type hints:   C{list} of C{str}
        @param kwargs: keywoard arguments passed to the initialize method of
                       the controller to be created
        @type kwargs:  C{dict}

        @rtype:        L{elisa.core.utils.defer.Deferred}
        """
        # cancel any unfinished loading
        if self._pending_load_dfr is not None:
            self._pending_load_dfr.cancel()
            self._pending_load_dfr = None

        def controller_created(controller):
            # hide previously shown controller and show newly created one
            self._show_controller(controller, self._transition_forward_in)
            self._history.append(controller)

            if self._current_controller is not None:
                self._hide_controller(self._current_controller,
                                      self._transition_forward_out)

            self._current_controller = controller
            self._pending_load_dfr = None

            # set browser for newly created controller 
            controller.browser = self 

            return controller

        def log_controller_creation_error(error, controller_path):
            self._pending_load_dfr = None
            if error.type != defer.CancelledError:
                # as it was not a CancelledError, we want to log it
                log_path = common.application.log_failure(error)
                self.warning("Loading controller with path %s failed. Full" \
                             " failure log at %s" % \
                             (controller_path, log_path))
                return error
            # swallow CancelledErrors

        try:
            creation_dfr = self.frontend.create_controller(path, **kwargs)

            # creation deferred as returned by initialize of components are not
            # cancellable thus the necessity of chaining it with a cancellable
            # one
            cancellable_dfr = defer.Deferred()

            # store it, so that we can cancel it later
            self._pending_load_dfr = cancellable_dfr

            # chain callbacks of creation_dfr to cancellable_dfr
            creation_dfr.chainDeferred(cancellable_dfr)

            cancellable_dfr.addCallback(controller_created)
        except:
            cancellable_dfr = defer.fail()

        # log errors
        cancellable_dfr.addErrback(log_controller_creation_error, path)

        return cancellable_dfr

    def _show_controller(self, controller, transition):
        """
        Display L{controller} applying a visual L{transition}.

        Return the deferred returned by L{transition}.

        @param controller: controller to be shown
        @type controller:  L{elisa.plugins.pigment.pigment_controller.PigmentController}
        @param transition: visual transition to be applied in order to show the
                           controller
        @type transition:  L{elisa.plugins.poblesec.transitions.Transition}

        @rtype:            L{elisa.core.utils.defer.Deferred}
        """
        controller.widget.visible = False
        self.widget.add(controller.widget)
        self.widget.set_focus_proxy(controller.widget)
        controller.widget.position = (0.0, 0.0, 0.0)
        controller.widget.size = (1.0, 1.0)
        controller.prepare()
        controller.widget.set_focus()

        def set_sensitive(controller):
            controller.sensitive = True

        dfr = transition.apply(controller)
        dfr.addCallback(set_sensitive)
        return dfr

    def _hide_controller(self, controller, transition):
        """
        Hide L{controller} applying a visual L{transition}.

        Return the deferred returned by L{transition}.

        @param controller: controller to be hidden
        @type controller:  L{elisa.plugins.pigment.pigment_controller.PigmentController}
        @param transition: visual transition to be applied in order to hide the
                           controller
        @type transition:  L{elisa.plugins.poblesec.transitions.Transition}

        @rtype:            L{elisa.core.utils.defer.Deferred}
        """
        controller.sensitive = False

        def remove_from_canvas(controller):
            self.widget.remove(controller.widget)
            controller.widget.visible = False
            controller.removed()

        dfr = transition.apply(controller)
        dfr.addCallback(remove_from_canvas)
        return dfr

    def load_previous_controller(self):
        """
        Display the controller that was loaded just before the current one.

        The previous controller is popped and an animated visual transition
        is applied to both the current controller and the previous one.
        The current controller is then destroyed.

        Return a deferred fired when the controller is loaded and only when
        the previously shown controller is completely hidden and destroyed.

        @raises EmptyHistory: when there is no previous controller left in the
                              history

        @rtype: L{elisa.core.utils.defer.Deferred}
        """

        if len(self._history) <= 1:
            raise EmptyHistory()

        # hide previously shown controller and show previous one in the history
        current_controller = self._history.pop()
        previous_controller = self._history[-1]

        self._show_controller(previous_controller,
                              self._transition_backward_in)
        dfr_hide = self._hide_controller(current_controller,
                                         self._transition_backward_out)

        def clean_current(result):
            return current_controller.clean()

        dfr_hide.addCallback(clean_current)

        self._current_controller = previous_controller

        return dfr_hide

    def load_initial_controller(self):
        """
        Display the controller that was loaded as the first for this browser.

        All controllers but the first one are popped, and an animated visual
        transition is applied between the last and the first controller 
        in the history.
        All controllers but the first one are then destroyed.
        This method can be used for easy jump to 'the home' of the browser.

        Return a deferred fired when the initial controller is loaded,
        the last controller is completely hidden, and the all other 
        controllers are destroyed.

        @rtype: L{elisa.core.utils.defer.Deferred}
        """
        def iterate_history():
            while len(self._history) > 1:
                controller = self._history.pop()
                yield controller.clean()

        last_controller = self._current_controller
        self._current_controller = self._history[0]
        
        self._show_controller(self._current_controller,
                              self._transition_backward_in)
        dfr = self._hide_controller(last_controller,
                                    self._transition_backward_out)
        dfr.addCallback(lambda result: task.coiterate(iterate_history()))
        return dfr
