/*
  **
  ** conf.c
  **
  ** I originaly wrote conf.c as part of the drpoxy package 
  ** thanks to Matthew Pratt and others for their additions. 
  **
  ** Copyright 1999 Jeroen Vreeken (pe1rxq@chello.nl)
  **
  ** This software is licensed under the terms of the GNU General 
  ** Public License (GPL). Please see the file COPYING for details.
  ** 
  **
*/

/**
 * How to add a config option :
 *
 *   1. think twice, there are settings enough
 *
 *   2. add a field to 'struct config' (conf.h) and 'struct config conf'
 *
 *   4. add a entry to the config_params array below, if your
 *      option should be configurable by the config file.
 */ 
#include "motion.h"
#include "video.h"
#include "conf.h"
#include "track.h"

extern struct trackoptions track_template;

struct config conf_template = {
	width:			DEF_WIDTH,
	height:			DEF_HEIGHT,
	quality:		DEF_QUALITY,
	max_changes:		DEF_CHANGES,
	always_diff:		0,
	new_img:		1,
	motion_img:		0,
	snap_overwrite:		0,
	gap:			DEF_GAP,
	maxmpegtime:		DEF_MAXMPEGTIME,
	locate:			1,
	input:			IN_DEFAULT,
	norm:			0,
	frame_limit:		DEF_MAXFRAMERATE,
	mpeg:			0,
	quiet:			0,
	ppm:			0,
	noise:			DEF_NOISELEVEL,
	mingap:			0,
	lightswitch:		0,
	jpg_cleanup:		0,
	nightcomp:		1,
	adjustrate:		0,
	realmotion:		0,
	oldlayout:		0,
	low_cpu:		0,
	nochild:		0,
	autobright:		0,
	roundrobing_frames:	1,
	roundrobing_skip:	1,
	post_capture:		0,
	switchfilter:		0,
	ffmpeg_cap_new:		0,
	ffmpeg_cap_motion:	0,
	ffmpeg_bps:		DEF_FFMPEG_BPS,
	webcam_port:		0,
	webcam_quality:		30,
	webcam_motion:		0,
	webcam_maxrate:		100,
	frequency:		0,
	timelaps:		0,
	device:			VIDEO_DEVICE,
	vidpipe:		NULL,
	mail_address:		NULL,
	sms_nr:			NULL,
	filepath:		NULL,
	externcommand:		NULL,
	mask_file:		NULL,
	mysql_db:		NULL,
	mysql_host:		NULL,
	mysql_user:		NULL,
	mysql_password:		NULL,
	onsavecommand:		NULL,
	onmpegcommand:		NULL,
	motionvidpipe:		NULL,
	netcam_url:		NULL,
	netcam_userpass:	NULL,
	pgsql_db:		NULL,
	pgsql_host:		NULL,
	pgsql_user:		NULL,
	pgsql_password:		NULL,
	pgsql_port:		5432,
	mpeg_encode_bin:	"/usr/local/bin/mpeg_encode",
	drawtext_changes: 0,
	drawtext_shots: 0,
	drawtext_user:  NULL
};



static struct context **copy_bool(struct context **, char *, int);
static struct context **copy_int(struct context **, char *, int);
static struct context **copy_string(struct context **, char *, int);
static struct context **config_thread(struct context **cnt, char *str, int val);
//static void config_realconfig(char *str, void *val);
static void usage(void);

/* Pointer magic to determine relative addresses of variables to a
   struct context pointer */
#define CNT_OFFSET(varname) ( \
		(int)&((struct context *)NULL)->varname \
	)
#define CONF_OFFSET(varname) ( \
		(int)&((struct context *)NULL)->conf.varname \
	)
#define TRACK_OFFSET(varname) ( \
		(int)&((struct context *)NULL)->track.varname \
	)

config_param config_params[] = {
	{
	"always_changes",
	CONF_OFFSET(always_diff),
	copy_bool
	},
	{
	"daemon",
	CNT_OFFSET(daemon),
	copy_bool
	},
	{
	"snapshots",
	CNT_OFFSET(alarmtime),
	copy_int
	},
	{
	"threshold",
	CONF_OFFSET(max_changes),
	copy_int
	},
	{
	"framerate",
	CONF_OFFSET(frame_limit),
	copy_int
	},
	{
	"locate",
	CONF_OFFSET(locate),
	copy_bool
	},
	{
	"output_motion",
	CONF_OFFSET(motion_img),
	copy_bool
	},
	{
	"output_normal",
	CONF_OFFSET(new_img),
	copy_bool
	},
	{
	"target_dir",
	CONF_OFFSET(filepath),
	copy_string
	},
	{
	"videodevice",
	CONF_OFFSET(device),
	copy_string
	},
	{
	"input",
	CONF_OFFSET(input),
	copy_int
	},
	{
	"norm",
	CONF_OFFSET(norm),
	copy_int
	},
	{
	"quality",
	CONF_OFFSET(quality),
	copy_int
	},
	{
	"quiet",
	CONF_OFFSET(quiet),
	copy_bool
	},
	{
	"ppm",
	CONF_OFFSET(ppm),
	copy_bool
	},
	{
	"width",
	CONF_OFFSET(width),
	copy_int
	},
	{
	"height",
	CONF_OFFSET(height),
	copy_int
	},
	{
	"execute",
	CONF_OFFSET(externcommand),
	copy_string
	},
	{
	"mail",
	CONF_OFFSET(mail_address),
	copy_string
	},
	{
	"sms",
	CONF_OFFSET(sms_nr),
	copy_string
	},
	{
	"snapshot_overwrite",
	CONF_OFFSET(snap_overwrite),
	copy_bool
	},
	{
	"gap",
	CONF_OFFSET(gap),
	copy_int
	},
	{
	"max_mpeg_time",
	CONF_OFFSET(maxmpegtime),
	copy_int	
	},
	{
	"minimum_gap",
	CONF_OFFSET(mingap),
	copy_int
	},
	{
	"mpeg_encode",
	CONF_OFFSET(mpeg),
	copy_bool
	},
	{
	"noise_level",
	CONF_OFFSET(noise),
	copy_int
	},
	{
	"mask_file",
	CONF_OFFSET(mask_file),
	copy_string
	},
	{
	"lightswitch",
	CONF_OFFSET(lightswitch),
	copy_bool
	},
	{
	"video_pipe",
	CONF_OFFSET(vidpipe),
	copy_string
	},
	{
	"jpg_cleanup",
	CONF_OFFSET(jpg_cleanup),
	copy_bool
	},
	{
	"onsave",
	CONF_OFFSET(onsavecommand),
	copy_string
	},
	{
	"onmpeg",
	CONF_OFFSET(onmpegcommand),
	copy_string
	},
	{
	"motion_video_pipe",
	CONF_OFFSET(motionvidpipe),
	copy_string
	},
	{
	"night_compensate",
	CONF_OFFSET(nightcomp),
	copy_bool
	},
	{
	"adjust_rate",
	CONF_OFFSET(adjustrate),
	copy_bool
	},
	{
	"realmotion",
	CONF_OFFSET(realmotion),
	copy_bool
	},
	{
	"oldlayout",
	CONF_OFFSET(oldlayout),
	copy_bool
	},
	{
	"low_cpu",
	CONF_OFFSET(low_cpu),
	copy_bool
	},
	{
	"thread",
	0,
	config_thread
	},
	{
	"auto_brightness",
	CONF_OFFSET(autobright),
	copy_bool
	},
	{
	"webcam_port",
	CONF_OFFSET(webcam_port),
	copy_int
	},
	{
	"webcam_quality",
	CONF_OFFSET(webcam_quality),
	copy_int
	},
	{
	"webcam_motion",
	CONF_OFFSET(webcam_motion),
	copy_bool
	},
	{
	"webcam_maxrate",
	CONF_OFFSET(webcam_maxrate),
	copy_int
	},
	{
	"mpeg_encode_bin",
	CONF_OFFSET(mpeg_encode_bin),
	copy_string
	},
	{
	"frequency",
	CONF_OFFSET(frequency),
	copy_int
	},
	{
	"drawtext_changes",
	CONF_OFFSET(drawtext_changes),
	copy_bool
	},
	{
	"drawtext_shots",
	CONF_OFFSET(drawtext_shots),
	copy_bool
	},
	{
	"drawtext_user",
	CONF_OFFSET(drawtext_user),
	copy_string
	},
#ifdef HAVE_CURL
	{
	"netcam_url",
	CONF_OFFSET(netcam_url),
	copy_string
	},
	{
	"netcam_userpass",
	CONF_OFFSET(netcam_userpass),
	copy_string
	},
#endif
#ifdef HAVE_MYSQL
	{
	"mysql_db",
	CONF_OFFSET(mysql_db),
	copy_string
	},
	{
	"mysql_host",
	CONF_OFFSET(mysql_host),
	copy_string
	},
	{
	"mysql_user",
	CONF_OFFSET(mysql_user),
	copy_string
	},
	{
	"mysql_password",
	CONF_OFFSET(mysql_password),
	copy_string
	},
#endif
	{
	"track_port",
	TRACK_OFFSET(port),
	copy_string	
	},
	{
	"track_motorx",
	TRACK_OFFSET(motorx),
	copy_int
	},
	{
	"track_maxx",
	TRACK_OFFSET(maxx),
	copy_int
	},
	{
	"track_iomojo_id",
	TRACK_OFFSET(iomojo_id),
	copy_int
	},
	{
	"track_speed",
	TRACK_OFFSET(speed),
	copy_int
	},
	{
	"track_stepsize",
	TRACK_OFFSET(stepsize),
	copy_int
	},
	{
	"roundrobing_frames",
	CONF_OFFSET(roundrobing_frames),
	copy_int
	},
	{
	"roundrobing_skip",
	CONF_OFFSET(roundrobing_skip),
	copy_int
	},
	{
	"switchfilter",
	CONF_OFFSET(switchfilter),
	copy_bool
	},
	{
	"post_capture",
	CONF_OFFSET(post_capture),
	copy_int
	},
#ifdef HAVE_PGSQL
	{
	"pgsql_db",
	CONF_OFFSET(pgsql_db),
	copy_string
	},
	{
	"pgsql_host",
	CONF_OFFSET(pgsql_host),
	copy_string
	},
	{
	"pgsql_user",
	CONF_OFFSET(pgsql_user),
	copy_string
	},
	{
	"pgsql_password",
	CONF_OFFSET(pgsql_password),
	copy_string
	},
	{
	"pgsql_port",
	CONF_OFFSET(pgsql_port),
	copy_int
	},
#endif
#ifdef HAVE_FFMPEG
	{
	"ffmpeg_cap_new",
	CONF_OFFSET(ffmpeg_cap_new),
	copy_bool
	},
	{
	"ffmpeg_cap_motion",
	CONF_OFFSET(ffmpeg_cap_motion),
	copy_bool
	},
	{
	"ffmpeg_timelaps",
	CONF_OFFSET(timelaps),
	copy_bool
	},
	{
	"ffmpeg_bps",
	CONF_OFFSET(ffmpeg_bps),
	copy_int
	},
#endif
  { NULL, 0 , NULL }
};


static void conf_cmdline (struct context *cnt)
{
	struct config *conf=&cnt->conf;
	int c;

        while ((c=getopt(conf->argc, conf->argv, "a:BCc:d:DE:F:f:G:g:hi:lL:M:mNn:O:P:pQq:r:S:s:t:u:U:V:w"))!=EOF)
                switch (c) {
                        case 'a':                                               
                                sscanf (optarg, "%d", &cnt->alarmtime);              
                                break;
			case 'B':
				conf->mpeg=1;
				break;                                          
                        case 'C':                                               
                                conf->always_diff=1;                                  
                                break;                                          
                        case 'c':                                               
                                sscanf (optarg, "%d", &conf->max_changes);            
                                break;                                          
                        case 'd':                                               
                                conf->device=optarg;                                  
                                break;                                          
                        case 'E':                                               
                                conf->externcommand=optarg;                           
                                break;                                          
                        case 'D':                                               
                                cnt->daemon=1;
                                break;
			case 'F':
				conf->mask_file=optarg;
				break;                                          
                        case 'f':                                               
                                sscanf (optarg, "%d", &conf->frame_limit);            
                                break;
			case 'G':
				sscanf (optarg, "%d", &conf->mingap);
				break;                                          
                        case 'g':                                               
                                sscanf (optarg, "%d", &conf->gap);                    
                                break;                                          
                        case 'i':                                               
                                sscanf (optarg, "%d", &conf->input);                  
                                break;                                          
                        case 'l':                                               
                                conf->locate=0;                                       
                                break;
			case 'L':
				sscanf (optarg, "%d", &conf->noise);
				break;                                          
                        case 'M':                                               
                                conf->mail_address=optarg;                            
                                break;                                          
                        case 'm':                                               
                                conf->motion_img=1;                                   
                                break;                                          
                        case 'N':                                               
                                conf->new_img=0;                                      
                                break;                                          
                        case 'n':                                               
                                if (!strcmp(optarg, "pal")) conf->norm=NORM_PAL;
				if (!strcmp(optarg, "pal-nc")) conf->norm=NORM_PAL_NC;
                                if (!strcmp(optarg, "ntsc")) conf->norm=NORM_NTSC;
                                if (!strcmp(optarg, "secam")) conf->norm=NORM_SECAM; 
                                break;
			case 'O':
				conf->onsavecommand=optarg;
				break;
			case 'P':
				conf->vidpipe=optarg;
				break;
			case 'p':
				conf->ppm=1;
				break;                                          
                        case 'S':                                               
                                conf->sms_nr=optarg;                                  
                                break;                                          
                        case 'q':                                               
                                sscanf (optarg, "%d", &conf->quality);
                                break;
			case 'r':
				/* Should already be handled */
				break;
		        case 'Q':
			        conf->quiet=1;
			        break;
                        case 's':                                               
                                sscanf (optarg, "%dx%d", &conf->width, &conf->height);      
                                break;                                          
                        case 't':                                               
                                conf->filepath=optarg;                                
                                break;                                          
			case 'u':
				conf->netcam_userpass=optarg;
				break;
			case 'U':
				conf->netcam_url=optarg;
				break;
			case 'V':
				conf->motionvidpipe=optarg;
				break;
			case 'w':
				conf->lightswitch=1;
				break;
                        case 'h':                                               
                        /* Do the same as an invallid option, but without the   
                         * 'invalid option' error.... */                        
                        default:                                                
				usage();                                        
				exit(1);                                       
		}
	optind=1;
}

/*****************************************************************************/

struct context **conf_cmdparse(struct context **cnt, char *cmd, char *arg1)
{
	int i = 0;

	if(!cmd)
		return cnt;
  
	while( config_params[i].param_name != NULL ) {
		if(!strncasecmp(cmd, config_params[i].param_name , 255 + 50)) {
			if(config_params[i].conf_value && !arg1)
				return cnt;
			cnt=config_params[i].copy( cnt, arg1, config_params[i].conf_value );
			return cnt;
		}
		i++;
        }

	fprintf( stderr, "Unkown config option: \"%s\"\n", cmd ); 
        return cnt;
}

struct context **conf_process(struct context **cnt, FILE *fp)
{
	char line[256], *cmd = NULL, *arg1 = NULL;

	if (fp) {
		while (fgets(line, 256, fp)) {
			if(!(line[0]=='#' || strlen(line)< 2)) {/*skipcomment*/
				if (line[strlen(line)-1]=='\n')
					line[strlen(line) - 1] = 0; /*kill'\n'*/
				cmd = strtok( line, "\t =" );
				arg1 = strtok( NULL, "\t ");
				cnt=conf_cmdparse(cnt, cmd, arg1);
			}
		}
		fclose(fp);
	}
	return cnt;
}


/**************************************************************************
    Main function, called from motion.c
*/
struct context ** conf_load (struct context **cnt)
{
	FILE *fp;
	char filename[2048]; /* What is the maximum filelength anyways? */
	int i;
	char **argv=cnt[0]->conf.argv;
	int argc=cnt[0]->conf.argc;

	memcpy(&cnt[0]->conf, &conf_template, sizeof(struct config));
	memcpy(&cnt[0]->track, &track_template, sizeof(struct trackoptions));
	cnt[0]->conf.argv=argv;
	cnt[0]->conf.argc=argc;

	sprintf(filename, "%s/.motion/motion.conf", getenv("HOME"));
      	fp = fopen ("motion.conf", "r");
	if (!fp) {	/* specified file does not exist... try default file */
		fp = fopen (filename, "r");
		if (!fp) {
			fp = fopen (sysconfdir"/motion.conf", "r");
			if (!fp) {	/* there is no config file.... use defaults */
				perror("no config file");
			}
		}
	}
	cnt=conf_process(cnt, fp);

	i=-1;
	while(cnt[++i])
		conf_cmdline(cnt[i]);

//	/* Prevent any additional childs from being created on SIGHUP */
//	conf->nochild=1;
	return cnt;
}

/************************************************************************
 * copy functions 
 *
 *   copy_bool   - convert a bool representation to int
 *   copy_int    - convert a string to int
 *   copy_string - just a string copy
 *   
 * @param str -   A char *, pointing to a string representation of the
 *                value.
 * @param value - points to the place where to store the value. 
************************************************************************/
static struct context **copy_bool (struct context **cnt, char *str, int val)
{
	void *tmp;
	int i;
	
	i=-1;
	while(cnt[++i]) {
		tmp=(void *)cnt[i]+(int)val;
		if ( !strcmp(str, "1") || 
		     !strcasecmp(str, "yes") || 
		     !strcasecmp(str,"on")) 
		{
			*((int *)tmp) = 1;
		} else {
			*((int *)tmp) = 0;
		}
		if (cnt[0]->threadnr)
			return cnt;
	}
	return cnt;
}

static struct context ** copy_int(struct context **cnt, char *str, int val)
{
	void *tmp;
	int i;
	
	i=-1;
	while(cnt[++i]) {
		tmp=(void *)cnt[i]+val;
		*((int *)tmp) = atoi(str);
		if (cnt[0]->threadnr)
			return cnt;
	}
	return cnt;
}

static struct context **copy_string(struct context **cnt, char *str, int val)
{
	void *tmp;
	int i;
	
	i=-1;
	while(cnt[++i]) {
		tmp=(void *)cnt[i]+val;
		*(char **)tmp=malloc(255);
		strncpy(*(char **)tmp, str, 255);
		if (cnt[0]->threadnr)
			return cnt;
	}
	return cnt;
}

static struct context **config_thread(struct context **cnt, char *str, int val)
{
	int i;
	FILE *fp;

	printf("config_thread\n");
	if (cnt[0]->threadnr)
		return cnt;

	fp=fopen(str, "r");
	if (!fp) {
		perror("thread config file not found");
		return cnt;
	}

	i=-1;
	while (cnt[++i]);
	cnt=realloc(cnt, sizeof(struct context *)*(i+2));
	cnt[i]=malloc(sizeof(struct context));
	memcpy(cnt[i], cnt[0], sizeof(struct context));
	cnt[i]->threadnr=i;
	cnt[i+1]=NULL;
printf("%d\n", i);
	conf_process(cnt+i, fp);
	
	return cnt;
}

static void usage (void)
{
        printf("motion Version "VERSION", Copyright 2000 Jeroen Vreeken\n");
        printf("\nusage:\tmotion [options]\n");
        printf("\n");
        printf("Overall settings:\n");
        printf("-C\t\tOutput changes count for every frame, usable for tuning\n");
        printf("-D\t\tDeamonize\n");
        printf("-a seconds\ttime between two automated snapshots, default: 0\n");
        printf("-c changes\tthreshold for detecting a change, default: %d\n", DEF_CHANGES);
	printf("-F file\t\tppm mask file to use as a motion filter.\n");
        printf("-f nr\t\tMaximum nr of frames per second, default: none\n");
	printf("-G sec\t\tMinimum gap between two shots in seconds, default: 0\n");
	printf("-L noise\tNoise level, default: %d\n", DEF_NOISELEVEL);
        printf("-l\t\tDo not locate and mark motion on output pictures\n");
        printf("-m\t\tOutput 'motion' images\n");
        printf("-N\t\tDon't output normal images\n");
	printf("-p\t\tOutput ppm images instead of jpeg, uses a lot of diskspace\n");
        printf("-t target dir\tdestination for snapshots.\n");
	printf("-Q\t\tBe quiet, don't output beeps when detecting motion\n");
	printf("-w\t\tActivate light switch filter\n");
        printf("\nDevice settings:\n");
        printf("-d device\tvideo4linux capture device, default: "VIDEO_DEVICE"\n");
        printf("-i input\tinput channel to use, default: not set\n");
        printf("-n norm\t\tnorm to use (pal/ntsc/secam), default: pal\n");
	printf("-P device\tvideo4linux video loopback input device, default: not set\n");
        printf("-q quality\tJPEG image quality, default: %d\n", DEF_QUALITY);
        printf("-s widthxheight\tPicture size, default: %dx%d\n", DEF_WIDTH, DEF_HEIGHT);
#ifdef HAVE_CURL
	printf("-u user:pass\tNetwork camera username and password, default: none\n");
	printf("-U url\t\tNetwork camera url for download, default: none\n");
#endif
	printf("-V device\tvideo4linux video loopback input device for motion frames,\n");
	printf("\t\tdefault: not set.");
        printf("\nActions:\n");
        printf("-B\t\tEncode all jpg images to mpeg after an event using mpeg_encode\n");
        printf("-E command\tExecute 'command' when detecting motion.\n");
        printf("-M address\tSend a mail to 'address' when detecting motion.\n");
	printf("-O command\tExecute 'command' when saving an image.\n");
        printf("-S nr\t\tSend a SMS to nr using sms_client when detecting motion.\n");
        printf("-g seconds\tminimum gap between events, default %d\n", DEF_GAP);
        printf("\n");
}
                                                                                                                                                                                                                                                                                                                                                                                                                            
