/*	motion.c
 *
 *	Detect changes in a video stream.
 *	Copyright 2000 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	This software is distributed under the GNU public license version 2
 *	See also the file 'COPYING'.
 *
 */
#include "motion.h"
#include "video.h"
#include "conf.h"
#include "alg.h"
#include "track.h"
#include "event.h"
#include "picture.h"
#include "ffmpeg.h"

struct context **cnt=NULL;

int threads_running=0;
int restart=0;

void context_init (struct context *cnt)
{
	cnt->threadnr=0;

	cnt->alarmtime=0;
	cnt->snapshot=0;
	cnt->makemovie=0;
	cnt->finish=0;

	cnt->shots=0;
	cnt->event_nr=0;
	cnt->prev_event=0;
	cnt->currenttime=NULL;
	cnt->lasttime=0;
	cnt->eventtime=0;
	cnt->lastshot=0;
	cnt->lastrate=25;
	cnt->filesc=0;
	cnt->filesmc=0;
	cnt->files=NULL;
	cnt->filesm=NULL;
	cnt->filesrate=NULL;
	cnt->filesmrate=NULL;
	cnt->moved=0;
	cnt->switched=0;

	
#ifdef HAVE_MYSQL
	cnt->database=NULL;
#endif
#ifdef HAVE_PGSQL
	cnt->database_pg=NULL;
#endif
#ifdef HAVE_FFMPEG
	cnt->ffmpeg_new=NULL;
	cnt->ffmpeg_motion=NULL;
	cnt->ffmpeg_timelaps=NULL;
#endif

}


char *get_pgm (FILE *picture, int width, int height, int rgb)
{
	int x=0 ,y=0, maxval;
	char line[256];
	char *image;

	line[255]=0;
	fgets(line, 255, picture);
	if (strncmp(line, "P5", 2)) {
		fprintf(stderr, "This is not a ppm file, starts with: '%s'\n", line);
		return NULL;
	}
	/* skip comment */
	line[0]='#';
	while (line[0]=='#') fgets(line, 255, picture);
	/* check size */
	sscanf(line, "%d %d", &x, &y);
	if (x!=width || y!=height) {
		fprintf(stderr, "Wrong image size: %dx%d should be %dx%d\n", x, y, width, height);
		return NULL;
	}
	/* Maximum value */
	line[0]='#';
	while (line[0]=='#') fgets(line, 255, picture);
	sscanf(line, "%d", &maxval);
	/* read data */
	image=malloc(width*height*(rgb ? 3 : 1));
	for (y=0; y<height; y++) {
		for (x=0; x<width; x++) {
			fread(&image[y*width+x], 1, 1, picture);
			image[y*width+x]=(int)image[y*width+x]*255/maxval;
		}
	}
	/* Blow up to rgb size */
	if (rgb)
		for(x=width*height*3-1; x>=0; x--)
			image[x]=image[x/3];
	return image;
}

void cleanup (struct config *conf, char *filename)
{
	char tmp[255];

	remove(filename);
	if (!conf->oldlayout) {
		strncpy(tmp, filename, 255);
		tmp[strlen(conf->filepath)+1+16]=0;
		rmdir(tmp);
		tmp[strlen(conf->filepath)+1+13]=0;
		rmdir(tmp);
		tmp[strlen(conf->filepath)+1+10]=0;
		rmdir(tmp);
		tmp[strlen(conf->filepath)+1+7]=0;
		rmdir(tmp);
		tmp[strlen(conf->filepath)+1+4]=0;
		rmdir(tmp);
	}
}

void create_mpeg (struct context *cnt, int event_nr, int filesc, char **files, int *rate)
{
	int i, j;
	char params[255];
	char eventname[255];
	char mpegfilename[255];
	FILE *paramfile=NULL;
	char *eventtime;
	struct config *conf=&cnt->conf;

	if (!filesc) return;
	if (!fork()) {
		vid_close();
	
		if (conf->oldlayout) {
			eventtime=files[0]+strlen(cnt->conf.filepath)+1;
			while (*eventtime++!='-');
			strncpy(eventname, eventtime, 14);
			if (files[0][strlen(files[0])-5]=='m') {
				eventname[14]='m';
				eventname[15]=0;
			} else eventname[14]=0;
		} else {
			strncpy(eventname, files[0]+strlen(cnt->conf.filepath)+1, 11);/* daydir */
			strncpy(eventname+11, files[0]+11+strlen(cnt->conf.filepath)+1, 2);/* hour */
			strncpy(eventname+13, files[0]+14+strlen(cnt->conf.filepath)+1, 2);/* minute */
			strncpy(eventname+15, files[0]+17+strlen(cnt->conf.filepath)+1, 2);/* second */
			if (files[0][strlen(files[0])-5]=='m') {
				eventname[17]='m';
				eventname[18]=0;
			} else eventname[17]=0;
		}
		sprintf(params, "%s/%s.params", cnt->conf.filepath, eventname);
		sprintf(mpegfilename, "%s/%s.mpg", cnt->conf.filepath, eventname);
		paramfile=fopen(params, "w");
		if(paramfile==NULL) {
			printf("couldn't open %s\n", params);
			exit(0);
		}
		fprintf(paramfile,
			"PATTERN IBBPBBPBBPBBPBBP\n"
			"OUTPUT %s\n"
			"BASE_FILE_FORMAT ", mpegfilename);
		if (!strncmp("ppm", files[0]+strlen(files[0])-3, 3))
			fprintf(paramfile, "PPM\n");
		else 
			fprintf(paramfile, "JPEG\n");
		fprintf(paramfile,
			"INPUT_CONVERT *\n"
			"GOP_SIZE 30\n"
			"SLICES_PER_FRAME 1\n"
			"INPUT_DIR /\n"
			"INPUT\n");
		for (i=0; i<filesc; i++) {
			if (!conf->adjustrate || rate[i]>25) {
				fprintf(paramfile, "%s\n", files[i]);
			} else {
				fprintf(stderr, "%d %d\n", rate[i], 25/(rate[i] ? rate[i]: 1));
				for (j=0; j<25/(rate[i] ? rate[i]: 2); j++) {
					fprintf(paramfile, "%s\n", files[i]);
					fflush(paramfile);
				}
			}
		}
		fprintf(paramfile,
			"END_INPUT\n"
			"PIXEL HALF\n"
			"RANGE 10\n"
			"PSEARCH_ALG LOGARITHMIC\n"
			"BSEARCH_ALG CROSS2\n"
			"IQSCALE 4\n"
			"PQSCALE 5\n"
			"BQSCALE 13\n"
			"FORCE_ENCODE_LAST_FRAME\n"
			"REFERENCE_FRAME ORIGINAL\n"/*DECODED*/
			"FRAME_RATE 25\n");
		fclose(paramfile);

		if (!fork()) {
			nice(5);
			execlp(cnt->conf.mpeg_encode_bin, "mpeg_encode", params, 0);
			perror("Failed to start mpeg_encode");
			exit(1);
		}
		if (wait(0)==-1) {
			perror("mpeg creation unsuccesfull");
			exit(1);
		}
		event(EVENT_FILECREATE, cnt, mpegfilename, "video/mpeg", NULL);
		for (i=0; i<filesc; i++) {
			if (conf->jpg_cleanup) 
				cleanup(conf, files[i]);
			free(files[i]);
		}
		if (files)
			free(files);
		if (rate)
			free(rate);
		cleanup(conf, params);
		exit(0);
	}
	return;
}


/*	Our SIGNAL-Handler
 *	We need this to handle alarms and external signals
 */
static void sig_handler (int signo)
{
	int i;

	switch(signo) {
		case SIGHUP: {
			printf("Received SIGHUP, preparing restart\n");
			restart=1;
			if (cnt) {
				i=-1;
				while (cnt[++i]) {
					cnt[i]->makemovie=1;
					cnt[i]->finish=1;
				}
			}
			signal(SIGHUP, sig_handler);
			return;
		}
		case SIGALRM: {
			/* Somebody (maybe we ourself) wants us to make a snapshot */
			if (cnt) {
				i=-1;
				while (cnt[++i]) {
					cnt[i]->snapshot=1;
					if (cnt[i]->alarmtime) 
						alarm(cnt[i]->alarmtime);
				}
			}
			/* Set everything for the next snapshot */
			signal(SIGALRM, sig_handler);
			return;
		}
		case SIGUSR1: {
			/* Ouch! We have been hit from the outside! Someone wants us to
			   make a movie! */
			if (cnt) {
				i=-1;
				while (cnt[++i])
					cnt[i]->makemovie=1;
			}
			signal(SIGUSR1, sig_handler);
			return;
		}
		case SIGINT:
		case SIGTERM: {
			/* Somebody wants us to quit! We should better finish the actual
			    movie and end up! */
			signal(SIGTERM, sig_handler);
			if (cnt) {
				i=-1;
				while (cnt[++i]) {
					cnt[i]->makemovie=1;
					cnt[i]->finish=1;
				}
			}
			return;
		}
		default:
			printf("unknown signal: %d\n", signo);
			if (signo==11)
				exit(0);
	}

	return;
}


int motion_detected (struct context *cnt, int diffs, int dev, int devpipe, int devmpipe)
{
	struct config *conf=&cnt->conf;
	struct images *imgs=&cnt->imgs;
	struct coord location;

	if (conf->locate || cnt->track.port || conf->switchfilter)
		location=alg_locate_center(imgs, imgs->width, imgs->height);
	if (conf->locate)
		alg_locate(location, imgs, imgs->width, imgs->height);
	if (diffs && conf->switchfilter) {
		diffs=alg_switchfilter(cnt, diffs, &location);
		if (diffs <= cnt->conf.max_changes )
			return 0;
	}
	event(EVENT_MOTION, cnt, NULL, NULL, cnt->currenttime);
	cnt->lasttime=cnt->currenttimep;
	/* Take action if necesarry */
	if (cnt->event_nr!=cnt->prev_event) {
		event(EVENT_FIRSTMOTION, cnt, NULL, NULL, cnt->currenttime);
		cnt->prev_event=cnt->event_nr;
		cnt->eventtime=cnt->currenttimep;
	}

	if (cnt->shots<conf->frame_limit-1 && cnt->currenttimep-cnt->lastshot >= conf->mingap ) {
		cnt->lastshot=cnt->currenttimep;
		/* Output the latest picture 'image_new' or image_out for motion picture.
		 */
		event(EVENT_IMAGE_DETECTED, cnt, NULL, NULL, cnt->currenttime);
	}

	if (cnt->track.port) {
		if (track_move(cnt, dev, devpipe, devmpipe, location, imgs)) {
			cnt->moved=2;
		}
	}
	return diffs;
}


void *motion_loop(void *arg)
{
	struct context *cnt=arg;
	int dev, devpipe=-1, devmpipe=-1, i, j, diffs, detecting_motion;
	time_t lastframe=0;
	FILE *picture;
	int postcap=0;
	char tmp[20];

	int passflag=0;
	long int *rolling_average_data;
	long int rolling_average_limit, required_frame_time, frame_delay;
	int rolling_frame=0;
	struct timeval tv1, tv2;
	unsigned long int rolling_average, elapsedtime, splittime, timenow=0, timebefore=0;

	printf("thread pid: %d\n", getpid());

	if (!cnt->conf.filepath)
		cnt->conf.filepath=".";

	/* Work out expected frame rate based on config setting */
	if (cnt->conf.frame_limit)
		required_frame_time=1000000L/cnt->conf.frame_limit;
	else
		required_frame_time=0;
	frame_delay=required_frame_time;
	/* Reserve enough space for a 10 second timing history buffer */
	rolling_average_limit=10*cnt->conf.frame_limit;
	rolling_average_data=malloc(sizeof(long int)*rolling_average_limit);
	if (!rolling_average_data)
	{
		perror("Unable to allocate memory [rolling_average_data]");
		exit(1);
	}
	/* Preset history buffer with expected frame rate */
	for (j=0; j< rolling_average_limit; j++)
		rolling_average_data[j]=required_frame_time;

	/* set the device settings */
	dev=vid_start (cnt, cnt->conf.width, cnt->conf.height, cnt->conf.input, cnt->conf.norm);
	if (dev==-1) {
		perror("Capture error");
		exit(1);
	}
	cnt->imgs.new=malloc(cnt->imgs.size);
	cnt->imgs.ref=malloc(cnt->imgs.size);
	cnt->imgs.out=malloc(cnt->imgs.size);

	/* Allow videodevice to settle in */
	sleep(1);
	/* Capture first image, or we will get an alarm on start */
	if (!(cnt->imgs.new=vid_next(cnt, dev, cnt->imgs.new, cnt->conf.width, cnt->conf.height))) {
		perror("Capture error");
		exit(1);
	}
	/* create a reference frame */
	memcpy(cnt->imgs.ref, cnt->imgs.new, cnt->imgs.size);

	if (cnt->conf.vidpipe) {
		printf("Opening video loopback device for normal pictures\n");
		devpipe=vid_startpipe(cnt->conf.vidpipe, cnt->conf.width, cnt->conf.height, cnt->imgs.type);
		if (devpipe < 0) {
			printf ("Failed to open video loopback\n");
			exit(1);
		}
	}
	if (cnt->conf.motionvidpipe) {
		printf("Opening video loopback device for motion pictures\n");
		devmpipe=vid_startpipe(cnt->conf.motionvidpipe, cnt->conf.width, cnt->conf.height, cnt->imgs.type);
		if (devmpipe < 0) {
			printf ("Failed to open video loopback\n");
			exit(1);
		}
	}

#ifdef HAVE_MYSQL
	if(cnt->conf.mysql_db) {
		cnt->database=(MYSQL *) malloc(sizeof(MYSQL));
		mysql_init(cnt->database);
		if (!mysql_real_connect(cnt->database, cnt->conf.mysql_host, cnt->conf.mysql_user,
		    cnt->conf.mysql_password, cnt->conf.mysql_db, 0, NULL, 0)) {
			printf("Cannot connect to database\n");
			exit(1);
		}
	}
#endif	
#ifdef HAVE_PGSQL
	if (cnt->conf.pgsql_db) {
		char connstring[255];
	
		/* create the connection string. 
		   Quote the values so we can have null values (blank)*/
		snprintf(connstring, 255,
		    "dbname='%s' host='%s' user='%s' password='%s' port='%d'",
		    cnt->conf.pgsql_db, /* dbname */
		    (cnt->conf.pgsql_host ? cnt->conf.pgsql_host : ""), /* host (may be blank) */
		    (cnt->conf.pgsql_user ? cnt->conf.pgsql_user : ""), /* user (may be blank) */
		    (cnt->conf.pgsql_password ? cnt->conf.pgsql_password : ""), /* password (may be blank) */
		    cnt->conf.pgsql_port
		);

		cnt->database_pg = PQconnectdb(connstring);
		if (PQstatus(cnt->database_pg) == CONNECTION_BAD)
		{
			fprintf(stderr, "Connection to database '%s' failed.\n", cnt->conf.pgsql_db);
			fprintf(stderr, "%s", PQerrorMessage(cnt->database_pg));
			exit(1);
		}
	}
#endif

	if (cnt->conf.mask_file) {
		if ((picture=fopen(cnt->conf.mask_file, "r"))) {
			cnt->imgs.mask=get_pgm(picture, cnt->conf.width, cnt->conf.height, cnt->imgs.type==VIDEO_PALETTE_RGB24 ? 1 : 0);
			fclose(picture);
		} else {
			fprintf(stderr, "Error opening mask file\n");
		}
		if (!cnt->imgs.mask) {
			perror ("failed to read mask image");
			exit(1);
		}
	} else
		cnt->imgs.mask=NULL;

	/* Set signal handlers */
	signal(SIGALRM, sig_handler);
	signal(SIGUSR1, sig_handler);
	signal(SIGHUP, sig_handler);
	signal(SIGTERM, sig_handler);


	if (cnt->conf.webcam_port)
		webcam_init(cnt);

	/* Prevent first few frames from triggering motion... */
	cnt->moved=8;
	if (cnt->alarmtime) 
		alarm(cnt->alarmtime);
	/* Should go on forever... unless you bought vaporware :) */
	while (cnt->finish ? cnt->makemovie : (int)(cnt->imgs.new=vid_next(cnt, dev, cnt->imgs.new, cnt->conf.width, cnt->conf.height))) {
		/* Get current time and preserver last time for frame interval calc. */
		timebefore=timenow;
		gettimeofday(&tv1, NULL);
		timenow=tv1.tv_usec+1000000L*tv1.tv_sec;

		/* Make a differences picture in image_out */
		if (cnt->conf.max_changes)
			diffs=alg_diff(cnt);
		else
			diffs=0;
		if (cnt->conf.autobright)
			vid_autobright(&cnt->conf, dev, cnt->imgs.new, cnt->conf.width, cnt->conf.height);
		if (cnt->moved) {
			cnt->moved--;
			diffs=0;
		}
		cnt->currenttimep=time(NULL);
		cnt->currenttime=localtime(&cnt->currenttimep);
		if (lastframe!=cnt->currenttimep) {
			cnt->lastrate=cnt->shots;
			cnt->shots=-1;
			lastframe=cnt->currenttimep;
		}
		/* lights turned on ? */
		if (cnt->conf.lightswitch) {
			diffs=alg_lightswitch (cnt, diffs, dev, devpipe, devmpipe);
		}
		/* Is the mpeg movie to long? */
		if (cnt->conf.maxmpegtime && (
#ifdef HAVE_FFMPEG
		    (cnt->ffmpeg_new || cnt->ffmpeg_motion) ||
#endif /* HAVE_FFMPEG */
		    (cnt->files || cnt->filesm)))
			if (cnt->currenttimep-cnt->eventtime>= cnt->conf.maxmpegtime)
				cnt->makemovie=1;
		if ((cnt->currenttimep-cnt->lasttime>=cnt->conf.gap) || cnt->makemovie) {
			if (cnt->event_nr==cnt->prev_event || cnt->makemovie) {
				event(EVENT_ENDMOTION, cnt, NULL, NULL, cnt->currenttime);
				if (cnt->track.port) {
					printf("%d %d %d\n", cnt->prev_event, cnt->event_nr, cnt->makemovie);
					track_center(cnt);
					cnt->moved=3;
				}
				cnt->makemovie=0;
				cnt->event_nr++;
    				if (cnt->conf.mpeg) {
					if (cnt->filesc)
						create_mpeg(cnt, cnt->prev_event,
							    cnt->filesc,
							    cnt->files, 
							    cnt->filesrate
							    );
					if (cnt->filesmc)
						create_mpeg(cnt, cnt->prev_event,
							    cnt->filesmc, 
							    cnt->filesm, 
							    cnt->filesmrate
							    );
				}
				for (i=0; i<cnt->filesc; i++) free(cnt->files[i]);
				if (cnt->files)
					free(cnt->files);
				for (i=0; i<cnt->filesmc; i++) free(cnt->filesm[i]);
				if (cnt->filesm)
					free(cnt->filesm);
				if (cnt->filesrate)
					free(cnt->filesrate);
				if (cnt->filesmrate)
					free(cnt->filesmrate);
				cnt->filesc=0;
				cnt->files=NULL;
				cnt->filesmc=0;
				cnt->filesm=NULL;
				cnt->filesrate=NULL;
				cnt->filesmrate=NULL;
			}
		}

		/* Old image slowly decays, this will make it even harder on
		 * a slow moving object to stay undetected
		 */
		for (i=cnt->imgs.size-1; i>=0; i--) {
			cnt->imgs.ref[i]=(cnt->imgs.ref[i]+cnt->imgs.new[i])/2;
		}

		if (cnt->conf.drawtext_user)
			draw_text(cnt->imgs.new, 10, cnt->conf.height-10, cnt->conf.height, cnt->conf.width, cnt->conf.drawtext_user, cnt->imgs.type);

		cnt->shots++;
		sprintf(cnt->datestamp, "%04d-%02d-%02d",
		    cnt->currenttime->tm_year + 1900, cnt->currenttime->tm_mon + 1,
		    cnt->currenttime->tm_mday);
		
		if (cnt->conf.drawtext_shots) {
			sprintf(cnt->timestamp, "%02d:%02d:%02d-%02d",
			cnt->currenttime->tm_hour, cnt->currenttime->tm_min,
			cnt->currenttime->tm_sec, cnt->shots);
		} else {
			sprintf(cnt->timestamp, "%02d:%02d:%02d",
			cnt->currenttime->tm_hour, cnt->currenttime->tm_min,
			cnt->currenttime->tm_sec);
		}
		draw_text(cnt->imgs.new, cnt->conf.width-10, cnt->conf.height-20, cnt->conf.height, cnt->conf.width, cnt->datestamp, cnt->imgs.type);
		draw_text(cnt->imgs.new, cnt->conf.width-10, cnt->conf.height-10, cnt->conf.height, cnt->conf.width, cnt->timestamp, cnt->imgs.type);
		sprintf(tmp, "%d", diffs);
		if (cnt->conf.drawtext_changes)
			draw_text(cnt->imgs.new, cnt->conf.width-10, cnt->conf.height-30, cnt->conf.height, cnt->conf.width, tmp, cnt->imgs.type);
		
		/* The cat just walked in :) */
		detecting_motion=1;
		if (diffs > cnt->conf.max_changes) {
			if (motion_detected(cnt, diffs, dev, devpipe, devmpipe))
				postcap=cnt->conf.post_capture;
		} else if (postcap) {
			motion_detected(cnt, 0, dev, devpipe, devmpipe);
			postcap--;
		} else
			detecting_motion=0;

		if (cnt->conf.always_diff)
			printf("changes: %d\n", diffs);
		if (cnt->track.port && cnt->moved) {
			for (i=0; i<cnt->imgs.size; i++)
				cnt->imgs.ref[i]=cnt->imgs.new[i];
		}

		if (cnt->snapshot) {
			event(EVENT_IMAGE_SNAPSHOT, cnt, NULL, NULL, cnt->currenttime);
			cnt->snapshot=0;
		}
		if (cnt->conf.timelaps && cnt->currenttime->tm_sec==0 && cnt->shots==0) {
			if (cnt->currenttime->tm_hour==0 && cnt->currenttime->tm_min==0)
				event(EVENT_TIMELAPSEND, cnt, NULL, NULL, cnt->currenttime);
			event(EVENT_TIMELAPS, cnt, NULL, NULL, cnt->currenttime);
		}
		event(EVENT_IMAGE, cnt, cnt->imgs.new, &devpipe, cnt->currenttime);
		event(EVENT_IMAGEM, cnt, cnt->imgs.out, &devmpipe, cnt->currenttime);

		/* Get latest time to calculate time taken to process video data */
		gettimeofday(&tv2, NULL);
		splittime=tv2.tv_usec+1000000L*tv2.tv_sec;
		elapsedtime=splittime-timenow;
		/* Update history buffer but ignore first pass as timebofre will
		   variable will be inaccurate
		 */
		if (passflag)
			rolling_average_data[rolling_frame]=timenow-timebefore;
		rolling_frame++;
		passflag=1;
		if (rolling_frame>=rolling_average_limit)
			rolling_frame=0;
		rolling_average=0L;
		/* Calculate 10 second average and use deviation in delay calculation */
		for (j=0; j<rolling_average_limit; j++)
			rolling_average+=rolling_average_data[j];
		rolling_average/=rolling_average_limit;
		frame_delay=required_frame_time-elapsedtime-(rolling_average-required_frame_time);
		if (frame_delay>0) {
			/* Apply delay to meet frame time */
			if (frame_delay>required_frame_time)
				frame_delay=required_frame_time;
			if (DEBUG && !cnt->conf.quiet)
				printf("Split Time is %7lu. Applying %6lu delay. Avg Frame Time %7lu Frame Error %ld\n",
				    elapsedtime, frame_delay, rolling_average, rolling_average-required_frame_time);
			usleep(frame_delay);
		}

		/* This will limit the framerate to 1 frame while not detecting
		   motion. Using a different motion flag to allow for multiple frames per second
		 */
		if (cnt->conf.low_cpu && !detecting_motion) {
			/* Recalculate remaining time to delay for a total of 1s */
			if (required_frame_time+elapsedtime<1000000) {
				frame_delay=1000000-required_frame_time-elapsedtime;
				usleep(frame_delay);
			}
			/* Correct frame times to ensure required_frame_time is maintained */
			gettimeofday(&tv1, NULL);
			timenow=(tv1.tv_usec+1000000L*tv1.tv_sec)-required_frame_time;
		}
		fflush(0);
	}
	if (!cnt->finish)
		perror("Somebody stole the video device, lets hope we got his picture");
	event(EVENT_STOP, cnt, NULL, NULL, NULL);
	threads_running--;
	return NULL;
}

int childpid;

void sigquit_handler(int signo)
{
	printf("Killing childs...\n");
	kill(childpid, SIGKILL);
	exit(0);
}

int main (int argc, char **argv)
{
	int i;
	pthread_attr_t thread_attr;
	pthread_t thread_id;

	cnt=malloc(sizeof(struct context *)*2);
	cnt[0]=malloc(sizeof(struct context));
	context_init(cnt[0]);
	cnt[1]=NULL;

	signal(SIGCHLD, SIG_IGN);

	cnt[0]->conf.argv=argv;
	cnt[0]->conf.argc=argc;
	cnt=conf_load(cnt);

#ifdef HAVE_FFMPEG
	ffmpeg_init();
#endif


	if (cnt[0]->daemon) {
		if (fork()) {
			printf("Going to daemon mode\n");
			exit(0);
		}
		signal(SIGQUIT, sigquit_handler);
	
		setpgrp();
		if ((i=open("/dev/tty", O_RDWR)) >= 0) {
			ioctl(i, TIOCNOTTY, NULL);
			close(i);
		}
//		close(0);
		close(1);
		close(2);
//		open("/dev/null", O_RDWR));
		dup(open("/dev/zero", O_RDWR));
		signal(SIGTTOU, SIG_IGN);
		signal(SIGTTIN, SIG_IGN);
		signal(SIGTSTP, SIG_IGN);
	}

	pthread_attr_init(&thread_attr);
	pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);


	vid_init();
	do {
		if (restart) {
			i=-1;
			while (cnt[++i])
				free(cnt[i]);
			free(cnt);
			cnt=malloc(sizeof(struct context *)*2);
			cnt[0]=malloc(sizeof(struct context));
			context_init(cnt[0]);
			cnt[1]=NULL;

			signal(SIGCHLD, SIG_IGN);

			cnt[0]->conf.argv=argv;
			cnt[0]->conf.argc=argc;
			cnt=conf_load(cnt);
			restart=0;
		}
		i=-1;
		while (cnt[++i]) {
			printf("thread%d device: %s input: %d\n",
			    cnt[i]->threadnr,
			    cnt[i]->conf.netcam_url?cnt[i]->conf.netcam_url:cnt[i]->conf.device,
			    cnt[i]->conf.netcam_url?-1:cnt[i]->conf.input
			);
			threads_running++;
			pthread_create(&thread_id, &thread_attr, &motion_loop, cnt[i]);
		}
		printf("waiting for threads to finish, pid: %d\n", getpid());
		while(threads_running > 0) {
			sleep(1);
		}
		printf("threads finished\n");
	} while (restart);

	return 0;
}	
