/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mouse Gestures (Gesture drawing module).
 *
 * The Initial Developer of the Original Code is Jens Bannmann.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s) (alphabetical order):
 *  Alexis Saettler <asbin@asbin.org>
 *  Jens Bannmann <jens.b@web.de>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

// Configurable variables
var SIZE = 32, STROKE = 1, PADDING = 4, DOTSIZE = 5, COLOR = "#000000";

var strokeLength, centerX, centerY;

function calculateLayout(aGesture) {
  var minX = 0;
  var minY = 0;
  var maxX = 0;
  var maxY = 0;
  var x = 0;
  var y = 0;

  function mvUp() {
    y--;
    if (y < minY) minY = y;
  }

  function mvDown() {
    y++;
    if (y > maxY) maxY = y;
  }

  function mvLeft() {
    x--;
    if (x < minX) minX = x;
  }

  function mvRight() {
    x++;
    if (x > maxX) maxX = x;
  }

  var points = new Array();
  points.push({x: 0, y: 0, nextStroke: aGesture.charAt(0)});

  for (var i=0; i < aGesture.length; i++) {
    switch(aGesture.charAt(i)) {
      case '1' :
        mvDown();
      case 'L' :
        mvLeft();
        break;
      case '7' :
        mvLeft();
      case 'U' :
        mvUp();
        break;
      case '9' :
        mvUp();
      case 'R' :
        mvRight();
        break;
      case '3' :
        mvRight();
      case 'D' :
        mvDown();
        break;
    }
    var nextStroke = (i < aGesture.length - 1) ? aGesture.charAt(i+1) : "";
    points.push({idx: i, x: x, y: y, nextStroke: nextStroke});
  }

  return {minX: minX, minY: minY, maxX: maxX, maxY: maxY, points: points};
}

function draw(aCanvas, aGesture) {
  dummy(aCanvas);

  if (aGesture.charAt(0) == ":")
    drawRockerGesture(aCanvas, aGesture);
  else
    drawNormalGesture(aCanvas, aGesture);
}

//=== Normal gesture drawing ===

function drawNormalGesture(aCanvas, aGesture) {
  var layout = calculateLayout(aGesture);

  var xCount = layout.maxX - layout.minX;
  var yCount = layout.maxY - layout.minY;

  var strokeCount = Math.max(xCount, yCount);
  strokeLength = (SIZE - (PADDING * 2)) / strokeCount;

  centerX = Math.abs((xCount == 0 ? 0.5 : layout.minX) * strokeLength) + PADDING - (STROKE / 2);
  centerY = Math.abs((yCount == 0 ? 0.5 : layout.minY) * strokeLength) + PADDING - (STROKE / 2);

  // starting point
  var x = 0;
  var y = 0;

  // calculate physical positions from logical points
  var positions = new Array();
  var pos = getPos(layout.points[0], positions);
  positions.push(pos);

  var minXPos = Math.min(pos.x, SIZE);
  var minYPos = Math.min(pos.y, SIZE);
  var maxXPos = Math.max(pos.x, 0);
  var maxYPos = Math.max(pos.y, 0);

  for (var i=1; i < layout.points.length; i++) {
    var p = layout.points[i];

    pos = getPos(layout.points[i], positions);
    positions.push(pos);

    minXPos = Math.min(pos.x, minXPos);
    minYPos = Math.min(pos.y, minYPos);
    maxXPos = Math.max(pos.x, maxXPos);
    maxYPos = Math.max(pos.y, maxYPos);
  }

  // ensure everything is centered
  var xOffset = Math.round((SIZE - maxXPos - minXPos) / 2);
  var yOffset = Math.round((SIZE - maxYPos - minYPos) / 2);

  // let's paint the picture :
  aCanvas.setColor(COLOR);
  aCanvas.setStroke(STROKE);

  if (aGesture.charAt(0) == "*") {
    var pattern = [-1, -1, -1, 0, 1, 1, 1, 0];
    for (i = 0; i < 8; ++i) {
      var a = pattern[i] * 3;
      var b = pattern[(i + 2) % 8] * 3;
      aCanvas.drawLine(positions[0].x + xOffset + a,
                       positions[0].y + yOffset + b,
                       positions[0].x + xOffset - a,
                       positions[0].y + yOffset - b);
    }
  }
  else {
    aCanvas.fillCircle(positions[0].x + xOffset,
                       positions[0].y + yOffset,
                       DOTSIZE);
  }
  aCanvas.beginPath();

  for (var i=0; i < positions.length - 1; ++i) {
    aCanvas.drawLine(positions[i].x + xOffset,
                     positions[i].y + yOffset,
                     positions[i + 1].x + xOffset,
                     positions[i + 1].y + yOffset);
  }
}

function getPos(point, history) {
  var x = Math.ceil(centerX + (point.x * strokeLength));
  var y = Math.ceil(centerY + (point.y * strokeLength));
  var uncheckedPos = true;

  while (true) {
    for (var h = 0; h < history.length; ++h) {
      var hpos = history[h];
      if (hpos.x == x && hpos.y == y) {
        switch(hpos.nextStroke) {
          case "7":
          case "U":
          case "D":
          case "9":
            x += Math.ceil(strokeLength * 0.2);
            break;
          case "1":
          case "R":
          case "L":
          case "3":
            y += Math.ceil(strokeLength * 0.2);
            break;
        }
        // we just altered the position, so we need to check the history again
        continue;
      }
    }
    // we just completed the history check without matches, so we're done.
    break;
  }
  return {x: x, y: y, nextStroke: point.nextStroke};
}

//=== Rocker gesture drawing ===

function drawRockerGesture(aCanvas, aGesture) {
  var first = aGesture.charAt(1);
  var second = aGesture.charAt(2);

  aCanvas.setColor(COLOR);
  drawMouse(aCanvas, 1, 8, first);
  aCanvas.fillRect(14, 13, 1, 7);
  aCanvas.fillRect(15, 14, 1, 5);
  aCanvas.fillRect(16, 15, 1, 3);
  aCanvas.fillRect(17, 16, 1, 1);
  switch (second) {
    case "+":
    case "-":
      drawWheel(aCanvas, 19, 8, second);
      break;
    default:
      drawMouse(aCanvas, 19, 8, second);
  }
}

function drawWheel(aCanvas, x, y, button) {
  var f = (button == "-") ? -1 : 1;
  aCanvas.fillCircle(x + 4, y + 8, 5);
  aCanvas.fillRect(x + 8, y + 4, 1, 9);
  aCanvas.fillRect(x + 8, y + 8 + (6 * f), 1, 1);
  aCanvas.fillRect(x + 7, y + 8 + (5 * f), 3, 1);
  aCanvas.fillRect(x + 6, y + 8 + (4 * f), 5, 1);
}

function drawMouse(aCanvas, x, y, button) {
  aCanvas.setStroke(1);
  aCanvas.fillRect(x + 2, y + 0, 8, 1);
  aCanvas.fillRect(x + 0, y + 2, 1, 12);
  aCanvas.fillRect(x + 11, y + 2, 1, 12);

  aCanvas.drawLine(x + 0, y + 14, x + 2, y + 16);
  aCanvas.drawLine(x + 11, y + 14, x + 9, y + 16);

  aCanvas.fillRect(x + 3, y + 16, 6, 1);
  aCanvas.fillRect(x + 1, y + 5, 10, 1);
  aCanvas.fillRect(x + 4, y + 1, 1, 4);
  aCanvas.fillRect(x + 7, y + 1, 1, 4);

  if (button == 0)
    aCanvas.fillRect(x + 1, y + 1, 3, 4);
  else
    aCanvas.fillRect(x + 1, y + 1, 1, 1);

  if (button == 1)
    aCanvas.fillRect(x + 5, y + 1, 2, 4);

  if (button == 2)
    aCanvas.fillRect(x + 8, y + 1, 3, 4);
  else
    aCanvas.fillRect(x + 10, y + 1, 1, 1);
}

function dummy(aCanvas) {
  aCanvas.ctx = aCanvas.getContext("2d");

  aCanvas.drawLine = function(x1, y1, x2, y2) {
    aCanvas.ctx.moveTo(x1 + 0.5, y1 + 0.5);
    aCanvas.ctx.lineTo(x2 + 0.5, y2 + 0.5);
    aCanvas.ctx.stroke();
  };

  aCanvas.fillCircle = function(x, y, diameter) {
    aCanvas.ctx.arc(x + 0.5, y + 0.5, (diameter / 2), 0.1, 3, false);
    aCanvas.ctx.arc(x + 0.5, y + 0.5, (diameter / 2), 3, 0.1, false);
    aCanvas.ctx.fill();
  };

  aCanvas.fillRect = function(x, y, w, h) {
    aCanvas.ctx.lineJoin = "miter";
    aCanvas.ctx.fillRect(x, y, w, h);
  };

  aCanvas.setStroke = function(width) {
    aCanvas.ctx.lineWidth = width;
  };

  aCanvas.setColor = function(color) {
    aCanvas.ctx.strokeStyle = color;
    aCanvas.ctx.fillStyle = color;
  };

  aCanvas.beginPath = function() {
    aCanvas.ctx.beginPath();
  };

  return aCanvas.ctx;
}