/* files and filehandling

   Written by Matthias Hensler
   Copyright WSPse 1999-2001
   eMail: wsp@gmx.de

Created: 1999/06/12
Updated: 2001/06/27
*/

/* Copying:
   This program is free software; you can redistribute it and/or modify it under
   the terms of the GNU Gerneral Public License as published by the Free Soft-
   ware Foundation; either version 2 of License, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILTY or
   FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 675 Mass
   Ave, Cambridge, MA 02139, USA.
   */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <ncurses.h>
#include "mp3creat.h"

#ifdef HAVE_CTYPE_H
#include <ctype.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

/* Externals */
extern void wuuush(int);
extern void setup_stat_win(int max_length);
extern void print_stat_win(char *text);
extern void destroy_stat_win();
extern BOOL select_yesno_box(char *tx);
extern int  read_from_sock(char **line, int sock_fd, BOOL wait);
extern void popup_error_win(char *tx);
extern char *copy_char_str(char *old);
extern BOOL fn_toupper;
extern int  fn_mode;
extern char *fn_pattern;
extern char *fn_pattern_mix;
extern char *mp3_genre[];
extern char config_fancy_colors;
extern char *def_cddb_server;
extern char *def_cddb_bank;
extern char *def_cdrom_dev;
extern char *def_tmp_file;
extern char *def_mp3_dir;
extern char *def_mp3_info;
extern char *def_cd_rip_nf;
extern char *def_cd_rip_of;
extern char *def_mp3_enc_nf;
extern char *def_mp3_enc_of;
extern char *def_m3u_pattern;
extern char *def_m3u_patmix;
extern char *def_m3u_dir;
extern char *def_comment;
extern char *def_ill_chars;
extern char *def_exp_file;
extern char *def_version_str;
extern int  auto_save;
extern int  of_fifo_buf;
extern int  mp3_frame_mult;
extern char *external_cdrom_dev;
extern BOOL def_on_fly;
extern BOOL eased_char_hand;
extern char replace_slash_ch;
extern BOOL rip_enc_ordered;
extern char external_version_str[];
extern int  del_tmp_on_exit;
extern int  cache_remain;
extern BOOL clear_del_on_exp;
extern BOOL config_curs_dir;
extern int  def_m3u_rel_name;
extern BOOL config_cddb_enbl;
extern BOOL config_para_mp3c;
extern int  config_case_chg;
extern char *cddb_genres;
extern char *def_cddb_email;
extern char *def_smtp_server;
extern char *def_my_email;
extern BOOL config_open_tray;
extern BOOL config_ill_remove;
extern char *def_unknown_gen;
extern BOOL config_pat_prot;

extern struct {
  int min;
  int sec;
  int frame;
} cdtoc[100];

/* Globals */
char *extnl_MP3_name  = NULL;
char *extnl_TMP_name  = NULL;
char *extnl_M3U_name  = NULL;
char *extnl_PROC_mask = NULL;
const char *weekday[] = { "Sun", "Mon", "Tue","Wed","Thu", "Fri", "Sat" };
const char *month[]   = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul",
  "Aug", "Sep", "Oct", "Nov", "Dec" };
  
/* Internals */
char *kill_double_slashs(char *string);
int  create_sub_dirs(char *filename, BOOL mode);
char *return_track_tmpname(song_typ *track);
char *file_glob_in_name(char *name);
char *file_glob_out_name(char *name);
char *file_build_m3u_entry(song_typ *track);

/* short filename to format
   mode 0 = spaces allowed
   mode 1 = change spaces to underscores "_"
   mode 2 = kill spaces
   toupper(FALSE) = nothing
   toupper(TRUE)  = each char after a space or underscore is converted toupper */
char *filenm_short(char *file_name, int mode, BOOL toup, char *protect_mask)
{
  char *ret_name;
  int i,j;
  BOOL was_space;
  BOOL uscr_inserted;

/*  j = strlen(file_name);
  fprintf(stderr, "\n\r%s\n\r", file_name);
  if(protect_mask) {
    for(i=0;i<j;i++) {
      if(*(protect_mask+i)) {
	fprintf(stderr, "%c", *(protect_mask+i));
      } else {
	fprintf(stderr, "#");
      }
    }
  }
  fprintf(stderr, "\n\r");
  fflush(stderr);
  sleep(2);*/

  ret_name = (char *) malloc(sizeof(char) * (strlen(file_name)+1));
  if(ret_name == NULL) {
    perror("malloc");
    wuuush(1);
  }
  
  was_space     = TRUE;
  uscr_inserted = FALSE;
  j = 0;
  for(i=0;i<strlen(file_name);i++) {
    if(config_pat_prot && protect_mask && *(protect_mask + i)) {
      *(ret_name+j) = *(file_name+i);
      if(strchr(" -_/\"=?!.", *(ret_name + j))) was_space = TRUE;    /* new word starts */
      else uscr_inserted = FALSE;
      j++;
    } else {
      if(config_case_chg == 1) {
	*(file_name + i) = tolower(*(file_name + i));
      } else if(config_case_chg == 2) {
	*(file_name + i) = toupper(*(file_name + i));
      }
      if((*(file_name + i) == ' ')) {
	/* space detected, look what we must do now */
	was_space = TRUE;      /* next character can converted to upper (if flag is set) */
	if(mode == 0)      *(ret_name + (j++)) = ' ';   /* space allowed                 */
	else if((mode == 1) && (! uscr_inserted)) {     /* insert max one underscore     */
	  *(ret_name + (j++)) = '_';   /* convert to underscore */
	  uscr_inserted = TRUE;
	}
      } else if(strchr(def_ill_chars, *(file_name+i))) {
	/* illegal character detected, look what we must do */
	was_space = TRUE;      /* also convert to upper */
	if(! config_ill_remove) {   /* decide from mode what to do now */
	  if((mode == 1) && (! uscr_inserted)) {
	    *(ret_name + (j++)) = '_'; /* mode == 1: convert to underscore */
	    uscr_inserted = TRUE;
	  } else if(mode == 0) {
	  *(ret_name + (j++)) = ' '; /* mode == 0: insert space */
	  }
	}
      } else {
	/* normal character detected */
	if(toup && was_space) {
	  /* toupper flag is set, and space/illegal character was there */
	  *(ret_name + j) = toupper(*(file_name + i));
	  was_space = FALSE;
	} else {
	  /* just copy character */
	  *(ret_name + j) = *(file_name + i);
	}
	if(strchr(" -_/\"=?!.", *(ret_name + j))) was_space = TRUE;    /* new word starts */
	else uscr_inserted = FALSE;
	j++;
      }
    }
  }
 
  *(ret_name+j) = 0;
//  fprintf(stderr, "\n\r%s\n\r", ret_name);

  if(strlen(ret_name) != strlen(file_name)) {
    /* size has changed, free unused memory */
    ret_name = (char *) realloc(ret_name, sizeof(char) * (strlen(ret_name)+1));
    if(! ret_name) {
      perror("realloc");
      wuuush(1);
    }
  }
  
  return ret_name;
}

void correct_slashes(char *string)
{
  char *str_poi;

  if(replace_slash_ch == '0') return;
  
  str_poi = string;
  while(str_poi && *str_poi) {
    if(*str_poi == '/') *str_poi = replace_slash_ch;
    str_poi++;
  }
}

void filenm_generate(song_typ *track)
{
  char *fn_new;
  char *add_poi;
  char def_add[9];
  int i,j,k;
  BOOL pat;
  char *pattern;
  char *protect_mask;

  if(!(track && (track->fn_auto))) return;

  if(track->filename) free(track->filename);
  track->filename = NULL;

  if( ((strcmp(track->artist, "unknown")==0) ||
       (*(track->artist) == 0))              &&
      ((strcmp(track->title,  "unknown")==0) ||
       (*(track->title)  == 0))) {
    fn_new = (char *) malloc(sizeof(char) * 30);
    if(fn_new == NULL) {
      perror("malloc");
      wuuush(1);
    }
    snprintf(fn_new, 30, "unknown-%02d_%08lX.mp3", (track->toc)+1, track->cddb_id);
    track->filename = kill_double_slashs(fn_new);
    return;
  }

  fn_new = (char *) malloc(sizeof(char));
  if(fn_new == NULL) {
    perror("malloc");
    wuuush(1);
  }
  *fn_new = 0;

  protect_mask = (char *) malloc(sizeof(char));
  if(! protect_mask) {
    wuuush(1);
  }
  *protect_mask = 0;
  
  i   = 0;
  pat = FALSE;

  if(track->sampler) {
    pattern = fn_pattern_mix;
  } else {
    pattern = fn_pattern;
  }
  if(! pattern) return;

  for(j=0;j<strlen(pattern);j++) {
    if(pat) {
      pat = FALSE;
      k = j-1-i;
      if(k>0) {
	protect_mask = (char *) realloc(protect_mask, sizeof(char) *
					(strlen(fn_new)+k+1));
	if(! protect_mask) {
	  wuuush(1);
	}
	memcpy(protect_mask + strlen(fn_new), pattern+i, k);

	fn_new = (char *) realloc(fn_new, sizeof(char) * (strlen(fn_new)
							  + k + 1));
	if(fn_new == NULL) {
	  perror("realloc");
	  wuuush(1);
	}
	*(fn_new + strlen(fn_new) + k) = 0;
	memcpy((fn_new + strlen(fn_new)), (pattern + i), k);
      }
      i = j+1;
      switch(*(pattern + j)) {
	case '%':
	  def_add[0] = '%';
	  def_add[1] = 0;
	  add_poi = def_add;
	  break;
	case '1':
	  add_poi = track->artist;
	  break;
	case '2':
	  add_poi = track->title;
	  break;
	case '3':
	  add_poi = track->album;
	  break;
	case '4':
	  if(track->genre != TOT_GENRES) {
	    add_poi = mp3_genre[track->genre];
	  } else {
	    add_poi = def_unknown_gen;
	  }
	  break;
	case '5':
	  sprintf(def_add, "%d", track->year);
	  add_poi = def_add;
	  break;
	case '6':
	  sprintf(def_add, "%d", (track->toc)+1);
	  add_poi = def_add;
	  break;
	case '7':
	  sprintf(def_add, "%02d", (track->toc)+1);
	  add_poi = def_add;
	  break;
	case '8':
	  snprintf(def_add, 9, "%08lX", track->cddb_id);
	  add_poi = def_add;
	  break;
	default:
	  def_add[0] = 0;
	  add_poi    = def_add;
      }
      k = strlen(add_poi);
      if(k > 0) {
	fn_new = (char *) realloc(fn_new, sizeof(char) * (strlen(fn_new)
							  + k + 1));
	if(fn_new == NULL) {
	  perror("realloc");
	  wuuush(1);
	}
	
	protect_mask = (char *) realloc(protect_mask, sizeof(char) *
					(strlen(fn_new) + k + 1));
	if(! protect_mask) {
	  wuuush(1);
	}

	memset(protect_mask + strlen(fn_new), 0, k);
	
	strcat(fn_new, add_poi);
	correct_slashes(fn_new + (strlen(fn_new) - strlen(add_poi)));
      }
    } else if(*(pattern + j) == '%') pat = TRUE;
  }
  k = j-i;
  if(k>0) {
    protect_mask = (char *) realloc(protect_mask, sizeof(char) *
				    (strlen(fn_new)+k+1));
    if(! protect_mask) {
      wuuush(1);
    }
    memcpy(protect_mask + strlen(fn_new), pattern+i, k);
    
    fn_new = (char *) realloc(fn_new, sizeof(char) * (strlen(fn_new)
						      + k + 1));
    if(fn_new == NULL) {
      perror("realloc");
      wuuush(1);
    }
    *(fn_new + strlen(fn_new) + k) = 0;
    memcpy((fn_new + strlen(fn_new)), (pattern + i), k);
  }

  track->filename = kill_double_slashs(filenm_short(fn_new, fn_mode, fn_toupper, protect_mask));
  free(fn_new);
  free(protect_mask);
}

/* save config */
int save_config(char *fn)
{
  FILE *conf_fd;
  time_t lt;
  struct tm *ltm;
  char *glob_str;

  setup_stat_win(40);
  print_stat_win(_("saving configuration"));
  if(create_sub_dirs(fn, TRUE)) return -1;
  conf_fd = fopen(fn, "w");
  if(conf_fd == NULL) {
    print_stat_win(_("failed..."));
    sleep(1);
    destroy_stat_win();
    return -1;
  }

  fputs("# WSPse's MP3-Creator (Configfile)\n", conf_fd);
  fputs(_("# automaticly created, but you may edit this file manually.\n"), conf_fd);
  lt  = time(NULL);
  ltm = localtime(&lt);
  fprintf(conf_fd, "# [%02d.%02d.%02d %02d:%02d] - program version: "
	  PRG_VERSION "\n# {%s}\n",
	  ltm->tm_mday, ltm->tm_mon+1, ltm->tm_year + 1900,
	  ltm->tm_hour, ltm->tm_min, external_version_str);

  fputs("\n# version number (to detect updates and inform about news)\n", conf_fd);
  fprintf(conf_fd, "mp3c_version = \"" PRG_VERSION "\"\n");

  fputs("\n# cdrom-device\n", conf_fd);
  glob_str = file_glob_out_name(def_cdrom_dev);
  fprintf(conf_fd, "cd_dev = \"%s\"\n", glob_str);
  def_cdrom_dev = file_glob_in_name(glob_str);

  fputs("\n# CDDB-server [host:port] (more servers maybe seperated by commas)\n"
	"# use \"0\" to disable server-access\n"
	"# note: cddb.cddb.com:8880 is not longer usable\n" , conf_fd);
  fprintf(conf_fd, "cddb_serv = \"%s\"\n", def_cddb_server);

  fputs("\n# local CDDB database\n", conf_fd);
  glob_str = file_glob_out_name(def_cddb_bank);
  fprintf(conf_fd, "cddb_loc = \"%s\"\n", glob_str);
  def_cddb_bank = file_glob_in_name(glob_str);

  fputs("\n# if remote CDDB access is allowed\n", conf_fd);
  fprintf(conf_fd, "rem_cddb = %d\n", config_cddb_enbl);

  fputs("\n# mailadress to which CDDB should be sent (more addresses maybe seperated\n"
	"# by commas)\n", conf_fd);
  fprintf(conf_fd, "cddb_email = \"%s\"\n", def_cddb_email);

  fputs("\n# relaying smtp-server who handle my emails [host:port]\n", conf_fd);
  fprintf(conf_fd, "smtp_serv = \"%s\"\n", def_smtp_server);

  fputs("\n# my email adress for authorizing [user@host]\n", conf_fd);
  fprintf(conf_fd, "my_email = \"%s\"\n", def_my_email);

  fputs("\n# directory for mp3-files\n", conf_fd);
  glob_str = file_glob_out_name(def_mp3_dir);
  fprintf(conf_fd, "mp3_dir = \"%s\"\n", glob_str);
  def_mp3_dir = file_glob_in_name(glob_str);

  fputs("\n# program for ripping cd-tracks (to file)\n"
	"#  %1 = cdrom device\n"
	"#  %2 = track (numeric)\n"
	"#  %3 = outputfile\n"
	,conf_fd);
  glob_str = file_glob_out_name(def_cd_rip_nf);
  fprintf(conf_fd, "rip_nf_prg = \"%s\"\n", glob_str);
  def_cd_rip_nf = file_glob_in_name(glob_str);

  fputs("\n# program for ripping cd-tracks (to stdout)\n"
	"#  %1 = cdrom device\n"
	"#  %2 = track (numeric)\n"
	,conf_fd);
  glob_str = file_glob_out_name(def_cd_rip_of);
  fprintf(conf_fd, "rip_of_prg = \"%s\"\n", glob_str);
  def_cd_rip_of = file_glob_in_name(glob_str);

  fputs("\n# program for encoding wav->mp3 (from file to file)\n"
	"#   %1 = inputfile\n"
	"#   %2 = outputfile\n"
	"#   %3 = albumname\n"
	"#   %4 = MP3 genre by number\n"
	"#   %5 = year\n"
	"#   %6 = comment\n"
	"#   %7 = filename\n"
	"#   %8 = MP3 genre by name\n"
	"#   %a = tracknumber\n"
	"#   %b = tracknumber (with leading zeros)\n"
	"#   %c = artistname\n"
	"#   %d = title\n"
	,conf_fd);
  glob_str = file_glob_out_name(def_mp3_enc_nf);
  fprintf(conf_fd, "enc_nf_prg = \"%s\"\n", glob_str);
  def_mp3_enc_nf = file_glob_in_name(glob_str);

  fputs("\n# program for encoding wav->mp3 (from stdin to file)\n"
	"#  %1 = outputfile\n"
	"#  %3 = albumname\n"
	"#  %4 = MP3 genre by number\n"
	"#  %5 = year\n"
	"#  %6 = comment\n"
	"#  %7 = filename\n"
	"#  %8 = MP3 genre by name\n"
	"#  %a = tracknumber\n"
	"#  %b = tracknumber (with leading zeros)\n"
	"#  %c = artistname\n"
	"#  %d = title\n"
	,conf_fd);
  glob_str = file_glob_out_name(def_mp3_enc_of);
  fprintf(conf_fd, "enc_of_prg = \"%s\"\n", glob_str);
  def_mp3_enc_of = file_glob_in_name(glob_str);
	
  fputs("\n# program for creating mp3 info\n"
	"#  use mp3_info_prg = \"0\" to disable usage of tagprogram\n"
	"#  %1 = artistname\n"
	"#  %2 = title\n"
        "#  %3 = albumname\n"
	"#  %4 = genre (by number)\n"
	"#  %5 = year\n"
	"#  %6 = comment\n"
	"#  %7 = filename\n"
	"#  %8 = genre (by name)\n"
	"#  %a = tracknumber\n"
	"#  %b = tracknumber (with leading zeros)\n"
	,conf_fd);
  glob_str = file_glob_out_name(def_mp3_info);
  fprintf(conf_fd, "mp3_info_prg = \"%s\"\n", glob_str);
  def_mp3_info = file_glob_in_name(glob_str);

  fputs("\n# string which replaced %8 in mp3_info_prg, if genre is unknown\n"
	,conf_fd);
  fprintf(conf_fd, "unknown_genre = \"%s\"\n", def_unknown_gen);

  fputs("\n# size of fifo-buffer for on the fly encoding (KB)\n", conf_fd);
  fprintf(conf_fd, "of_fifo = %d\n", of_fifo_buf);
  
  fputs("\n# pattern for mp3-filename-creation\n"
	"#  %1 = artistname\n"
	"#  %2 = title\n"
	"#  %3 = albumname\n"
	"#  %4 = genrestring\n"
	"#  %5 = year\n"
	"#  %6 = tracknumber\n"
	"#  %7 = tracknumber (with leading zeros)\n"
	"#  %8 = cddb-id\n"
	,conf_fd);
  fprintf(conf_fd, "mp3_pattern = \"%s\"\n", fn_pattern);

  fputs("\n# pattern fro mp3-filename-creation for sampler cds\n"
	"#  same pattern like in mp3_pattern\n"
	,conf_fd);
  fprintf(conf_fd, "mp3_pattern_mix = \"%s\"\n", fn_pattern_mix);

  fputs("\n# mode for handling spaces in filenames\n"
	"#  0: spaces allowed, 1: spaces will be converted to underscores\n"
	"#  2: spaces will be killed\n", conf_fd);
  fprintf(conf_fd, "pat_mode = %d\n", fn_mode);

  fputs("\n# appereance of filename case\n"
	"#  0: as it is in CDDB entry\n"
	"#  1: convert to lowercase (first letter will be uppercase if pat_upc = 1\n"
	"#  2: convert to uppercase\n", conf_fd);
  fprintf(conf_fd, "case_chg = %d\n", config_case_chg);
  
  fputs("\n# convert first letter of filename to uppercase\n"
	"#  0: no, 1: yes\n", conf_fd);
  fprintf(conf_fd, "pat_upc = %d\n", fn_toupper);

  fputs("\n# illegal characters which aren't allowed in filenames\n"
	"# (converted to '_' if mode != 2, else killed\n", conf_fd);
  fprintf(conf_fd, "ill_chars = \"%s\"\n", def_ill_chars);

  fputs("\n# what to do with illegal characters, should they be removed?\n"
	"# (otherwise convert to '_' or space, depending on pat_mode)\n"
	"#  0: no, use pat_mode, 1: yes, remove\n", conf_fd);
  fprintf(conf_fd, "rem_ill_char = %d\n", config_ill_remove);

  fputs("\n# protect pattern from substitution operations?\n"
	"#  0: no, 1: yes\n", conf_fd);
  fprintf(conf_fd, "pattern_protect = %d\n", config_pat_prot);
  
  fputs("\n# character which should replace slashes in album, artist, title\n"
	"# (only one character allowed, use \"0\" to accept slashes in these\n"
	"   fields, which causes strange directory creation)\n", conf_fd);
  fprintf(conf_fd, "slash_rep_char = \"%c\"\n", replace_slash_ch);

  fputs("\n# non-strict character handling\n"
	"# 0: only printable chars allowed, 1: eased allowed chars\n", conf_fd);
  fprintf(conf_fd, "eased_char_hand = %d\n", eased_char_hand);
  
  fputs("\n# default comment for mp3-files\n"
	"#  %1 = artistname\n"
	"#  %2 = title\n"
	"#  %3 = albumname\n"
	"#  %4 = genrestring\n"
	"#  %5 = year\n"
	"#  %6 = tracknumber\n"
	"#  %7 = tracknumber (with leading zeros)\n"
	"#  %8 = version-string of MP3c\n"
	"#  %9 = cddb-id\n"
	"#  %a = actual day (is set when encoding starts, or batchfile created)\n"
	"#  %b = actual month\n"
	"#  %c = actual year (2 digits)\n"
	"#  %d = actual year (4 digits)\n"
	"#  %e = weekday (3 letters)\n"
	"#  %f = month (3 letters)\n"
	"#  %g = actual hour\n"
	"#  %h = actual minute\n"
	"#  %i = minute-part of track-length\n"
	"#  %j = second-part of track-length\n"
	, conf_fd);
  fprintf(conf_fd, "mp3_comment = \"%s\"\n", def_comment);
  
  fputs("\n# fancy color for windows (0: never, 1: sometimes, 2: ever)\n", conf_fd);
  fprintf(conf_fd, "fancy_color = %d\n", config_fancy_colors);

  fputs("\n# autosave configuration on exit (0: no, 1: yes)\n", conf_fd);
  fprintf(conf_fd, "auto_save = %d\n", auto_save);

  fputs("\n# default flag (0: default non-fly, 1: default on-fly)\n", conf_fd);
  fprintf(conf_fd, "def_on_fly = %d\n", def_on_fly);

  fputs("\n# rip-encode order (0: rip one track, then encode; 1: rip all tracks,\n"
	"#                   then encode)\n",  conf_fd);
  fprintf(conf_fd, "rip_enc_ord = %d\n", rip_enc_ordered);

  fputs("\n# open tray after encoding (0: no, 1:yes)\n", conf_fd);
  fprintf(conf_fd, "open_tray = %d\n", config_open_tray);
  
  fputs("\n# tmpfile (for non-on-the-fly convert)\n", conf_fd);
  glob_str = file_glob_out_name(def_tmp_file);
  fprintf(conf_fd, "tmp_file = \"%s\"\n", glob_str);
  def_tmp_file = file_glob_in_name(glob_str);

  fputs("\n# what to do with tempfiles on exits\n"
	"# 0: nothing, 1: delete marked, 2: delete all\n", conf_fd);
  fprintf(conf_fd, "del_tmp_on_exit = %d\n", del_tmp_on_exit);

  fputs("\n# default exportfile for ripped tracks\n", conf_fd);
  glob_str = file_glob_out_name(def_exp_file);
  fprintf(conf_fd, "def_exp_file = \"%s\"\n", glob_str);
  def_exp_file = file_glob_in_name(glob_str);

  fputs("\n# if deleteflag should cleared on export (0: no, 1: yes)\n", conf_fd);
  fprintf(conf_fd, "clear_del_on_exp = %d\n", clear_del_on_exp);

  fputs("\n# framemultiplikator (to calculate size)\n", conf_fd);
  fprintf(conf_fd, "frame_mult = %d\n", mp3_frame_mult);

  fputs("\n# pattern for m3u-playlist, use \"0\" to disable\n"
	"#  %1 = artistname\n"
	"#  %2 = title\n"
	"#  %3 = albumname\n"
	"#  %4 = genrestring\n"
	"#  %5 = year\n"
	"#  %6 = tracknumber\n"
	"#  %7 = tracknumber (with leading zeros)\n"
	"#  %8 = cddb-id\n"
	,conf_fd);
  fprintf(conf_fd, "m3u_pattern = \"%s\"\n", def_m3u_pattern);

  fputs("\n# pattern for m3u-playlist for sampler cds, use \"0\" to disable\n"
	"#  same pattern like in m3u_pattern\n"
	,conf_fd);
  fprintf(conf_fd, "m3u_patmix = \"%s\"\n", def_m3u_patmix);

  fputs("\n# directory for m3u-playlist, or \"0\" to use mp3-dir\n", conf_fd);
  glob_str = file_glob_out_name(def_m3u_dir);
  fprintf(conf_fd, "m3u_dir = \"%s\"\n", glob_str);
  def_m3u_dir = file_glob_in_name(glob_str);

  fputs("\n# how to build the filename for m3u-list\n"
	"#  0 = full path\n"
	"#  1 = relative to m3u-maindir\n"
	"#  2 = relative to m3u-file\n", conf_fd);
  fprintf(conf_fd, "m3u_rel_name_type = %d\n", def_m3u_rel_name);

  fputs("\n# existing CDDB genres, do not change manually\n", conf_fd);
  if(cddb_genres) {
    fprintf(conf_fd, "cddb_genre_list = \"%s\"\n", cddb_genres);
  } else {
    fputs("cddb_genre_list = 0\n", conf_fd);
  }

  fputs("\n# allow parallel running sessions of MP3c\n", conf_fd);
  fprintf(conf_fd, "para_sessions_allowed = %d\n", config_para_mp3c);

  fputs("\n# use internal directory requester\n", conf_fd);
  fprintf(conf_fd, "use_dir_request = %d\n", config_curs_dir);
  
  fclose(conf_fd);
  print_stat_win(_("configuration saved"));
  usleep(300000);
  destroy_stat_win();
  return 0;
}

/* load config */
int load_config(char *fn)
{
  int conf_fd;
  int lines;
  char *inp_line;
  int i,j,k,l;
  char *key, *arg;

  setup_stat_win(70);
  print_stat_win(_("loading configuration"));

  conf_fd = open(fn, O_RDONLY);
  if(conf_fd < 1) {
    print_stat_win(_("loading failed..."));
    sleep(1);
    destroy_stat_win();
    return -1;
  }

  lines        = 0;
  cache_remain = 0;
  while(read_from_sock(&inp_line, conf_fd, FALSE) == 0) {
    lines++;
    if(*inp_line != '#') {
      key = NULL;
      arg = NULL;
      for(i=0;i<strlen(inp_line)-1;i++) if(*(inp_line+i) == '=') break;
      j=i;
      for(i=0;i<j;i++) if(*(inp_line+i) != ' ') break;
      k=i;
      if(j) for(i=j-1;i>=0;i--) if(*(inp_line+i) != ' ') break;
      l=i;
      
      if((l-k)+1 > 0) {
	key = (char *) malloc(sizeof(char) * ((l-k)+2));
	if(key == NULL) {
	  perror("malloc");
	  wuuush(1);
	}
	memcpy(key, (inp_line + k), (l-k)+1);
	*(key + (l-k) + 1) = 0;
      }

      for(i=j+1;i<strlen(inp_line)-1;i++) if(*(inp_line+i) != ' ') break;
      k=i;
      for(i=strlen(inp_line);i>k;i--) if((*(inp_line+i) != 0) &&
					 (*(inp_line+i) != ' ') &&
					 (*(inp_line+i) != '\r') &&
					 (*(inp_line+i) != '\n')) break;
      l=i;

      if((l-k)+1 > 0) {
	arg = (char *) malloc(sizeof(char) * ((l-k)+2));
	if(arg == NULL) {
	  perror("malloc");
	  wuuush(1);
	}
	memcpy(arg, (inp_line + k), (l-k)+1);
	*(arg + (l-k) + 1) = 0;
	if((*arg == '\"') && (*(arg + (l-k)) == '\"')) {
	  *(arg + (l-k)) = 0;
	  if((l-k)-1 > 0) {
	    memmove(arg, (arg+1), (l-k));
	  }
	}
      }

      if(key && arg && *key && *arg) {
	if(strcmp(key, "cd_dev") == 0) {
	  if(! external_cdrom_dev) {
  	    if(def_cdrom_dev) free(def_cdrom_dev);
  	    def_cdrom_dev = file_glob_in_name(arg);
  	    arg = NULL;
	  }
	} else if(strcmp(key, "mp3c_version") == 0) {
	  if(def_version_str) free(def_version_str);
	  def_version_str = arg;
	  arg = NULL;
	} else if(strcmp(key, "cddb_serv") == 0) {
	  if(def_cddb_server) free(def_cddb_server);
	  def_cddb_server = arg;
	  arg = NULL;
	} else if(strcmp(key, "cddb_loc") == 0) {
	  if(def_cddb_bank) free(def_cddb_bank);
	  def_cddb_bank = file_glob_in_name(arg);
	  arg = NULL;
	} else if(strcmp(key, "mp3_dir") == 0) {
	  if(def_mp3_dir) free(def_mp3_dir);
	  def_mp3_dir = file_glob_in_name(arg);
	  arg = NULL;
	} else if(strcmp(key, "rip_nf_prg") == 0) {
	  if(def_cd_rip_nf) free(def_cd_rip_nf);
	  def_cd_rip_nf = file_glob_in_name(arg);
	  arg = NULL;
	} else if(strcmp(key, "rip_of_prg") == 0) {
	  if(def_cd_rip_of) free(def_cd_rip_of);
	  def_cd_rip_of = file_glob_in_name(arg);
	  arg = NULL;
	} else if(strcmp(key, "enc_nf_prg") == 0) {
	  if(def_mp3_enc_nf) free(def_mp3_enc_nf);
	  def_mp3_enc_nf = file_glob_in_name(arg);
	  arg = NULL;
	} else if(strcmp(key, "enc_of_prg") == 0) {
	  if(def_mp3_enc_of) free(def_mp3_enc_of);
	  def_mp3_enc_of = file_glob_in_name(arg);
	  arg = NULL;
	} else if(strcmp(key, "mp3_info_prg") == 0) {
	  if(def_mp3_info) free(def_mp3_info);
	  def_mp3_info = file_glob_in_name(arg);
	  arg = NULL;
	} else if(strcmp(key, "mp3_pattern") == 0) {
	  if(fn_pattern) free(fn_pattern);
	  fn_pattern = arg;
	  arg = NULL;
	} else if(strcmp(key, "mp3_pattern_mix") == 0) {
	  if(fn_pattern_mix) free(fn_pattern_mix);
	  fn_pattern_mix = arg;
	  arg = NULL;
	} else if(strcmp(key, "m3u_pattern") == 0) {
	  if(def_m3u_pattern) free(def_m3u_pattern);
	  def_m3u_pattern = arg;
	  arg = NULL;
	} else if(strcmp(key, "m3u_patmix") == 0) {
	  if(def_m3u_patmix) free(def_m3u_patmix);
	  def_m3u_patmix = arg;
	  arg = NULL;
	} else if(strcmp(key, "m3u_dir") == 0) {
	  if(def_m3u_dir) free(def_m3u_dir);
	  def_m3u_dir = file_glob_in_name(arg);
	  arg = NULL;
	} else if(strcmp(key, "mp3_comment") == 0) {
	  if(def_comment) free(def_comment);
	  def_comment = arg;
	  arg = NULL;
	} else if(strcmp(key, "ill_chars") == 0) {
	  if(def_ill_chars) free(def_ill_chars);
	  def_ill_chars = arg;
	  arg = NULL;
	} else if(strcmp(key, "def_exp_file") == 0) {
	  if(def_exp_file) free(def_exp_file);
	  def_exp_file = file_glob_in_name(arg);
	  arg = NULL;
	} else if(strcmp(key, "pat_mode") == 0) {
	  fn_mode = atoi(arg);
	  if((fn_mode < 0) || (fn_mode > 2)) {
	    print_stat_win(_("illegal value for pat_mode"));
	    fn_mode = 1;
	    usleep(500000);
	  }
	} else if(strcmp(key, "case_chg") == 0) {
	  config_case_chg = atoi(arg);
	  if(config_case_chg < 0 || config_case_chg > 2) {
	    config_case_chg = 0;
	  }
	} else if(strcmp(key, "del_tmp_on_exit") == 0) {
	  del_tmp_on_exit = atoi(arg);
	  if(del_tmp_on_exit < 0 || del_tmp_on_exit > 2) {
	    print_stat_win(_("illegal value for del_tmp_on_exit"));
	    del_tmp_on_exit = 1;
	    usleep(500000);
	  }
	} else if(strcmp(key, "of_fifo") == 0) {
	  of_fifo_buf = atoi(arg);
	  if((of_fifo_buf < 16) || (of_fifo_buf > 8192)) of_fifo_buf = 512;
	} else if(strcmp(key, "frame_mult") == 0) {
	  mp3_frame_mult = atoi(arg);
	  if((mp3_frame_mult < 1) || (mp3_frame_mult > 32000)) mp3_frame_mult = 214;
	} else if(strcmp(key, "pat_upc") == 0) {
	  fn_toupper = TRUE;
	  if(*arg == '0') fn_toupper = FALSE;
	} else if(strcmp(key, "def_on_fly") == 0) {
	  def_on_fly = FALSE;
	  if(*arg == '1') def_on_fly = TRUE;
	} else if(strcmp(key, "rip_enc_ord") == 0) {
	  rip_enc_ordered = FALSE;
	  if(*arg == '1') rip_enc_ordered = TRUE;
	} else if(strcmp(key, "pattern_protect") == 0) {
	  config_pat_prot = TRUE;
	  if(*arg == '0') config_pat_prot = FALSE;
	} else if(strcmp(key, "slash_rep_char") == 0) {
	  replace_slash_ch = *arg;
	  if(strlen(arg) == 0) replace_slash_ch = '-';
	  else if(strlen(arg) > 1) {
	    print_stat_win(_("more than one character in \"slash_rep_char\", skipping trailing garbage"));
	    usleep(500000);
	  }
	} else if(strcmp(key, "eased_char_hand") == 0) {
	  eased_char_hand = FALSE;
	  if(*arg == '1') eased_char_hand = TRUE;
	} else if(strcmp(key, "fancy_color") == 0) {
	  config_fancy_colors = atoi(arg);
	  if((config_fancy_colors < 0) || (config_fancy_colors > 2)) {
	    print_stat_win(_("illegal value for fancy_colors"));
	    config_fancy_colors = 1;
	    usleep(500000);
	  }
	} else if(strcmp(key, "m3u_rel_name_type") == 0) {
	  def_m3u_rel_name = atoi(arg);
	  if(def_m3u_rel_name < 0 || def_m3u_rel_name > 2) {
	    def_m3u_rel_name = 2;
	  }
	} else if(strcmp(key, "auto_save") == 0) {
	  auto_save = atoi(arg);
	  if(auto_save != 0) auto_save = 1;
	} else if(strcmp(key, "use_dir_request") == 0) {
	  config_curs_dir = atoi(arg);
	  if(config_curs_dir != FALSE) config_curs_dir = TRUE;
	} else if(strcmp(key, "rem_cddb") == 0) {
	  config_cddb_enbl = atoi(arg);
	  if(config_cddb_enbl != FALSE) config_cddb_enbl = TRUE;
	} else if(strcmp(key, "para_sessions_allowed") == 0) {
	  config_para_mp3c = atoi(arg);
	  if(config_para_mp3c != FALSE) config_para_mp3c = TRUE;
	} else if(strcmp(key, "open_tray") == 0) {
	  config_open_tray = atoi(arg);
	  if(config_open_tray != FALSE) config_open_tray = TRUE;
	} else if(strcmp(key, "rem_ill_char") == 0) {
	  config_ill_remove = atoi(arg);
	  if(config_ill_remove != FALSE) config_ill_remove = TRUE;
	} else if(strcmp(key, "clear_del_on_exp") == 0) {
	  clear_del_on_exp = atoi(arg);
	  if(clear_del_on_exp != 0) clear_del_on_exp = 1;
	} else if(strcmp(key, "tmp_file") == 0) {
	  if(def_tmp_file) free(def_tmp_file);
	  def_tmp_file = file_glob_in_name(arg);
	  arg = NULL;
	} else if(strcmp(key, "cddb_email") == 0) {
	  if(def_cddb_email) free(def_cddb_email);
	  def_cddb_email = arg;
	  arg = NULL;
	} else if(strcmp(key, "smtp_serv") == 0) {
	  if(def_smtp_server) free(def_smtp_server);
	  def_smtp_server = arg;
	  arg = NULL;
	} else if(strcmp(key, "my_email") == 0) {
	  if(def_my_email) free(def_my_email);
	  def_my_email = arg;
	  arg = NULL;
	} else if(strcmp(key, "unknown_genre") == 0) {
	  if(def_unknown_gen) free(def_unknown_gen);
	  def_unknown_gen = arg;
	  arg = NULL;
	} else if(strcmp(key, "cddb_genre_list") == 0) {
	  if(cddb_genres) free(cddb_genres);
	  cddb_genres = arg;
	  arg = NULL;
	  if(cddb_genres && strcmp(cddb_genres, "0") == 0) {
	    free(cddb_genres);
	    cddb_genres = NULL;
	  }
	} else {
	  if(arg) free(arg);
	  arg = (char *) malloc(sizeof(char) * (strlen(key)+70));
	  if(arg == NULL) {
	    perror("malloc");
	    wuuush(1);
	  }
	  sprintf(arg, _("unknown key \"%s\" in line %d"), key, lines);
	  popup_error_win(arg);
	}
      }
      
      if(key) free(key);
      if(arg) free(arg);
    }
    free(inp_line);
  }
      
  close(conf_fd);
  print_stat_win(_("configuration loaded"));
  usleep(300000);
  destroy_stat_win();
  return 0;
}

int get_config(char *file)
{
  char *pointer, *name;
 
  if(file) {
    if(load_config(file)) return -1;
    else                  return 0;
  }
  
  pointer = getenv("HOME");
  if(pointer) {
    name = (char *) malloc(sizeof(char) * (strlen(pointer) + strlen(MP3C_RCFILE) + 3));
    if(name == NULL) {
      perror("malloc");
      wuuush(1);
    }
    sprintf(name, "%s/.%s", pointer, MP3C_RCFILE);
    if(access(name, R_OK) == 0) {
      if(load_config(name) == 0) {
	free(name);
	return 0;
      }
    }
    free(name);
  }
  
  name = (char *) malloc(sizeof(char) * (strlen(MP3C_RCFILE) + strlen(DEF_ETC_PATH) + 2));
  if(name == NULL) {
    perror("malloc");
    wuuush(1);
  }
  sprintf(name, "%s/%s", DEF_ETC_PATH, MP3C_RCFILE);
  if(access(name, R_OK) == 0) {
    if(load_config(file) == 0) {
      free(name);
      return 0;
    }
    free(name);
    return -1;
  }
  free(name);

  setup_stat_win(55);
  print_stat_win(_("no configfile found, using defaults"));
  usleep(500000);
  destroy_stat_win();

  return -1;
}

int put_config(char *file, BOOL ask)
{
  char *pointer, *name;
  int status;

  /* alternative config given */
  if(file) {
    if(access(file, F_OK) != 0) {
      /* file does not exist */
      return save_config(file);
    }
    if(access(file, W_OK) != 0) {
      setup_stat_win(35);
      print_stat_win(_("permission denied to save config"));
      sleep(1);
      destroy_stat_win();
      return -1;
    }
    if(ask) {
      if(! select_yesno_box(_("overwrite config file?"))) {
	return 1;
      }
    }
    return (save_config(file));
  }

  pointer = getenv("HOME");
  if(pointer == NULL) return -1;

  name = (char *) malloc(sizeof(char) * (strlen(pointer) + strlen(MP3C_RCFILE)
					 + 3));
  if(name == NULL) {
    perror("malloc");
    wuuush(1);
  }

  sprintf(name, "%s/.%s", pointer, MP3C_RCFILE);
  
  if(access(name, F_OK) != 0) {
    /* file does not exist, can write */
    status = save_config(name);
  } else {
    if(access(name, W_OK) != 0) {
      /* file does exist, but have no permission to overwrite */
      setup_stat_win(55);
      print_stat_win(_("permission denied to save config"));
      sleep(1);
      status = -1;
      destroy_stat_win();
    } else {
      /* file exists, and is writeable */
      if(ask) {
	/* ask for overwriting */
	if(! select_yesno_box(_("overwrite config file?"))) {
	  status = 1;
	} else {
	  status = save_config(name);
	}
      } else {
	status = save_config(name);
      }
    }
  }

  free(name);
  return status;
}

char *build_mp3_filenm(song_typ *track)
{
  if(extnl_MP3_name) free(extnl_MP3_name);
  extnl_MP3_name = NULL;

  if((! track->filename) || (! track->dirname)) {
    extnl_MP3_name = (char *) malloc(sizeof(char) * 15);
    if(extnl_MP3_name == NULL) {
      perror("malloc");
      wuuush(1);
    }
    sprintf(extnl_MP3_name, "track-%d.mp3", (track->toc)+1);
    return (kill_double_slashs(extnl_MP3_name));
  }
  
  extnl_MP3_name = (char *) malloc(sizeof(char) * (strlen(track->filename) +
						   strlen(track->dirname) +2));
  if(extnl_MP3_name == NULL) {
    perror("malloc");
    wuuush(1);
  }
  sprintf(extnl_MP3_name, "%s/%s", track->dirname, track->filename);
  return (kill_double_slashs(extnl_MP3_name));
}

char *create_sub_string(song_typ *track, int mode)
{
  char *sub_str, *add_poi, *def_str;
  int i,j,k;
  BOOL pat;
  char def_add[15];
  time_t lt;
  struct tm *ltm;
  unsigned long int track_len;

  if(! track) return NULL;

  if(extnl_PROC_mask) {
    free(extnl_PROC_mask);
    extnl_PROC_mask = NULL;
  }

  lt  = time(NULL);
  ltm = localtime(&lt);

  track_len = ((cdtoc[(int) (track->toc)+1].frame + 75*
		((60*cdtoc[(int) (track->toc)+1].min) +
	 	 cdtoc[(int) (track->toc)+1].sec)) -
	       (cdtoc[(int) (track->toc)].frame + 75*
		((60*cdtoc[(int) (track->toc)].min)
		 + cdtoc[(int) (track->toc)].sec)));
  
  switch(mode) {
    case 1:
      def_str = def_mp3_enc_nf;
      break;
    case 2:
      def_str = def_mp3_enc_of;
      break;
    case 3:
      def_str = def_cd_rip_nf;
      break;
    case 4:
      def_str = def_cd_rip_of;
      break;
    case 5:
      def_str = def_mp3_info;
      break;
    case 6:
      if(track->sampler) {
	def_str = def_m3u_patmix;
      } else {
	def_str = def_m3u_pattern;
      }
      break;
    case 7:
      def_str = track->comment;
      break;
    default:
      return NULL;
      break;
  }
  
  sub_str = (char *) malloc(sizeof(char));
  if(sub_str == NULL) {
    perror("malloc");
    wuuush(1);
  }
  *sub_str = 0;

  extnl_PROC_mask = (char *) malloc(sizeof(char));
  if(! extnl_PROC_mask) {
    wuuush(1);
  }
  *extnl_PROC_mask = 0;

  i   = 0;
  pat = FALSE;
  for(j=0;j<strlen(def_str);j++) {
    if(pat) {
      pat = FALSE;
      k   = j-1-i;
      if(k > 0) {
	extnl_PROC_mask = (char *) realloc(extnl_PROC_mask, sizeof(char) *
					   (strlen(sub_str)+k+1));
	if(! extnl_PROC_mask) {
	  wuuush(1);
	}
	memcpy(extnl_PROC_mask+strlen(sub_str), def_str+i, k);
	
	sub_str = (char *) realloc(sub_str, sizeof(char) * (strlen(sub_str)+k+1));
	if(sub_str == NULL) {
	  perror("realloc");
	  wuuush(1);
	}
	*(sub_str + strlen(sub_str) + k) = 0;
	memcpy((sub_str + strlen(sub_str)), (def_str + i), k);
      }
      i       = j + 1;
      add_poi = NULL;
      if(*(def_str + j) == '%') {
	def_add[0] = '%';
	def_add[1] = 0;
	add_poi = def_add;
      } else {
	switch(mode) {
	  case 1:       /* encode non-fly */
	    switch(*(def_str + j)) {
	      case '1':
		add_poi = return_track_tmpname(track);
		break;
	      case '2':
		add_poi = build_mp3_filenm(track);
		break;
	      case '3':
		add_poi = track->album;
		break;
	      case '4':
		sprintf(def_add, "%d", track->genre);
		add_poi = def_add;
		break;
	      case '5':
		sprintf(def_add, "%d", track->year);
		add_poi = def_add;
		break;
	      case '6':
		if(extnl_MP3_name) free(extnl_MP3_name);
		extnl_MP3_name = create_sub_string(track, 7);
		add_poi = extnl_MP3_name;
		break;
	      case '7':
		add_poi = build_mp3_filenm(track);
		break;
	      case '8':
		if(track->genre != TOT_GENRES) {
		  add_poi = mp3_genre[track->genre];
		} else {
		  add_poi = def_unknown_gen;
		}
		break;
	      case 'a':
		sprintf(def_add, "%d", (track->toc)+1);
		add_poi = def_add;
		break;
	      case 'b':
		sprintf(def_add, "%02d", (track->toc)+1);
		add_poi = def_add;
		break;
	      case 'c':
		add_poi = track->artist;
		break;
	      case 'd':
		add_poi = track->title;
		break;
	      default:
		def_add[0] = 0;
	       	add_poi    = def_add;
		break;
	    }
	    break;

	  case 2:       /* encode on-fly */
	    switch(*(def_str + j)) {
	      case '1':
		add_poi = build_mp3_filenm(track);
		break;
	      case '3':
		add_poi = track->album;
		break;
	      case '4':
		sprintf(def_add, "%d", track->genre);
		add_poi = def_add;
		break;
	      case '5':
		sprintf(def_add, "%d", track->year);
		add_poi = def_add;
		break;
	      case '6':
		if(extnl_MP3_name) free(extnl_MP3_name);
		extnl_MP3_name = create_sub_string(track, 7);
		add_poi = extnl_MP3_name;
		break;
	      case '7':
		add_poi = build_mp3_filenm(track);
		break;
	      case '8':
		if(track->genre != TOT_GENRES) {
		  add_poi = mp3_genre[track->genre];
		} else {
		  add_poi = def_unknown_gen;
		}
		break;
	      case 'a':
		sprintf(def_add, "%d", (track->toc)+1);
		add_poi = def_add;
		break;
	      case 'b':
		sprintf(def_add, "%02d", (track->toc)+1);
		add_poi = def_add;
		break;
	      case 'c':
		add_poi = track->artist;
		break;
	      case 'd':
		add_poi = track->title;
		break;
	      default:
		def_add[0] = 0;
	       	add_poi    = def_add;
		break;
	    }
	    break;

	  case 3:       /* rip non-fly */
	  case 4:       /* rip on-fly  */
	    switch(*(def_str + j)) {
	      case '1':
		add_poi = def_cdrom_dev;
		break;
	      case '2':
		sprintf(def_add, "%d", (track->toc) + 1);
		add_poi = def_add;
		break;
	      case '3':
		add_poi = return_track_tmpname(track);
		if(mode == 4) {
		  def_add[0] = 0;
		  add_poi    = def_add;
		}
		break;
	      default:
		def_add[0] = 0;
	       	add_poi    = def_add;
		break;
	    }
	    break;

	  case 5:       /* mp3info */
	    switch(*(def_str + j)) {
	      case '1':
		add_poi = track->artist;
		break;
	      case '2':
		add_poi = track->title;
		break;
	      case '3':
		add_poi = track->album;
		break;
	      case '4':
		sprintf(def_add, "%d", track->genre);
		add_poi = def_add;
		break;
	      case '5':
		sprintf(def_add, "%d", track->year);
		add_poi = def_add;
		break;
	      case '6':
		if(extnl_MP3_name) free(extnl_MP3_name);
		extnl_MP3_name = create_sub_string(track, 7);
		add_poi = extnl_MP3_name;
		break;
	      case '7':
		add_poi = build_mp3_filenm(track);
		break;
	      case '8':
		if(track->genre != TOT_GENRES) {
		  add_poi = mp3_genre[track->genre];
		} else {
		  add_poi = def_unknown_gen;
		}
		break;
	      case 'a':
		sprintf(def_add, "%d", (track->toc)+1);
		add_poi = def_add;
		break;
	      case 'b':
		sprintf(def_add, "%02d", (track->toc)+1);
		add_poi = def_add;
		break;
	      default:
		def_add[0] = 0;
	       	add_poi    = def_add;
		break;
	    }
	    break;
		
	  case 6:     /* m3u-pattern */
    	    switch(*(def_str + j)) {
	      case '1':
		add_poi = track->artist;
		break;
	      case '2':
		add_poi = track->title;
		break;
	      case '3':
		add_poi = track->album;
		break;
	      case '4':
		if(track->genre != TOT_GENRES) {
		  add_poi = mp3_genre[track->genre];
		} else {
		  add_poi = def_unknown_gen;
		}
		break;
	      case '5':
		sprintf(def_add, "%d", track->year);
		add_poi = def_add;
		break;
	      case '6':
		sprintf(def_add, "%d", (track->toc)+1);
		add_poi = def_add;
		break;
	      case '7':
		sprintf(def_add, "%02d", (track->toc)+1);
		add_poi = def_add;
		break;
	      case '8':
		snprintf(def_add, 9, "%08lX", track->cddb_id);
		add_poi = def_add;
		break;
	      case '%':
		def_add[0] = '%';
		def_add[1] = 0;
		add_poi    = def_add;
		break;
	      default:
		def_add[0] = 0;
		add_poi    = def_add;
		break;
	    }
	    break;
	      
	  case 7:    /* comment-pattern */
	    switch(*(def_str + j)) {
	      case '%':
		def_add[0] = '%';
		def_add[1] = 0;
		add_poi    = def_add;
		break;
	      case '1':
		add_poi = track->artist;
		break;
	      case '2':
		add_poi = track->title;
		break;
	      case '3':
		add_poi = track->album;
		break;
	      case '4':
		if(track->genre != TOT_GENRES) {
		  add_poi = mp3_genre[track->genre];
		} else {
		  add_poi = def_unknown_gen;
		}
		break;
	      case '5':
		sprintf(def_add, "%04d", track->year);
		add_poi = def_add;
		break;
	      case '6':
		sprintf(def_add, "%d", (track->toc)+1);
		add_poi = def_add;
		break;
	      case '7':
		sprintf(def_add, "%02d", (track->toc)+1);
		add_poi = def_add;
		break;
	      case '8':
		strcpy(def_add, PRG_VERSION);
		add_poi = def_add;
		break;
	      case '9':
		snprintf(def_add, 9, "%08lX", track->cddb_id);
		add_poi = def_add;
		break;
	      case 'a':
		sprintf(def_add, "%02d", ltm->tm_mday);
		add_poi = def_add;
		break;
	      case 'b':
		sprintf(def_add, "%02d", (ltm->tm_mon)+1);
		add_poi = def_add;
		break;
	      case 'c':
		sprintf(def_add, "%02d", (ltm->tm_year) % 100);
		add_poi = def_add;
		break;
	      case 'd':
		if(ltm->tm_year < 1900) sprintf(def_add, "%04d", (ltm->tm_year) + 1900);
		else                    sprintf(def_add, "%04d", ltm->tm_year);
		add_poi = def_add;
		break;
	      case 'e':
		strcpy(def_add, weekday[ltm->tm_wday]);
		add_poi = def_add;
		break;
	      case 'f':
		strcpy(def_add, month[ltm->tm_mon]);
		add_poi = def_add;
		break;
	      case 'g':
		sprintf(def_add, "%02d", ltm->tm_hour);
		add_poi = def_add;
		break;
	      case 'h':
		sprintf(def_add, "%02d", ltm->tm_min);
		add_poi = def_add;
		break;
	      case 'i':
		sprintf(def_add, "%02ld", track_len / 4500);
		add_poi = def_add;
		break;
	      case 'j':
		sprintf(def_add, "%02ld", (track_len / 75) % 60);
		add_poi = def_add;
		break;
	      default:
		def_add[0] = 0;
		add_poi    = def_add;
		break;
	    }
	    break;
	   
	}
      }

      k = strlen(add_poi);
      if(k > 0) {
	extnl_PROC_mask = (char *) realloc(extnl_PROC_mask, sizeof(char) *
					   (strlen(sub_str)+k+1));
	if(! extnl_PROC_mask) {
	  wuuush(1);
	}
	memset(extnl_PROC_mask+strlen(sub_str), 0, k);
	
	sub_str = (char *) realloc(sub_str, sizeof(char) * (strlen(sub_str)
							    + k + 1));
	if(sub_str == NULL) {
	  perror("realloc");
	  wuuush(1);
	}
	strcat(sub_str, add_poi);
	for(k=strlen(sub_str) - strlen(add_poi);k<strlen(sub_str);k++) {
	  if(*(sub_str + k) == '\"') *(sub_str + k) = '\'';  /* 2000/08/28: Huch?! */
	}
      }
    } else if(*(def_str + j) == '%') pat = TRUE;
  }
  k = j - i;
  if(k > 0) {
    extnl_PROC_mask = (char *) realloc(extnl_PROC_mask, sizeof(char) *
				       (strlen(sub_str)+k+1));
    if(! extnl_PROC_mask) {
      wuuush(1);
    }
    memcpy(extnl_PROC_mask+strlen(sub_str), def_str+i, k);

    sub_str = (char *) realloc(sub_str, sizeof(char) * (strlen(sub_str)
							+ k + 1));
    if(sub_str == NULL) {
      perror("realloc");
      wuuush(1);
    }
    *(sub_str + strlen(sub_str) + k) = 0;
    memcpy((sub_str + strlen(sub_str)), (def_str + i), k);
  }
  
  return sub_str;
}

/* build up argument tree for execv */
char **build_arg_tree(const char *line)
{
  int i,j,k;
  BOOL quote;
  BOOL space;
  char **ret;

  quote = FALSE;   /* Check Quotes and number or args */
  space = TRUE;
  j     = 0;

  if(! line) return NULL;

  for(i=0;i<strlen(line);i++) {
    if(*(line + i) == '\"') {
      space = FALSE;
      if(! quote) quote=TRUE;
      else {
   	quote=FALSE;
	j++;
	space = TRUE;
      }
    } else if(*(line + i) == ' ') {
      if((! quote) && (!space)) {
	space = TRUE;
	j++;
      }
    } else {
      space = FALSE;
    }
  }

  if(quote) return NULL;
  if(space) j++;
  
  ret = (char **) malloc(sizeof(char *) * (j+1));
  if(ret == NULL) {
    perror("malloc");
    wuuush(1);
  }
  
  j = 0;
  k = 0;
  space = TRUE;
  quote = FALSE;
  for(i=0;i<strlen(line);i++) {
    if(*(line + i) == '\"') {
      if(! quote) {
	quote = TRUE;
	k = i+1;
      } else {
	quote = FALSE;
	ret[j] = (char *) malloc(sizeof(char) * (i-k+1));
	if(ret[j] == NULL) {
	  perror("malloc");
	  wuuush(1);
	}
	*(ret[j]+(i-k)) = 0;
	if(i-k > 0) {
	  memcpy((ret[j]), (line + k), i-k);
	}
	j++;
	space = TRUE;
      }
    } else if(*(line + i) == ' ') {
      if((! space) && (! quote)) {
	ret[j] = (char *) malloc(sizeof(char) * (i-k+1));
	if(ret[j] == NULL) {
	  perror("malloc");
	  wuuush(1);
	}
	*(ret[j]+(i-k)) = 0;
	if(i-k > 0) {
	  memcpy((ret[j]), (line + k), i-k);
	}
	j++;
	space = TRUE;
      }
    } else {
      if(space) k = i;
      space = FALSE;
    }
  }
 
  if(! space) {
    ret[j] = (char *) malloc(sizeof(char) * (i-k+1));
    if(ret[j] == NULL) {
      perror("malloc");
      wuuush(1);
    }
    *(ret[j]+(i-k)) = 0;
    if(i-k > 0) {
      memcpy((ret[j]), (line + k), i-k);
    }
    j++;
  }
  ret[j] = NULL;
  return ret;
}

/* build playlist filename (dir+m3u-file) */
char *build_m3u_fname(song_typ *song)
{
  char *ret_str;
  char *tmp_str;
  char *filenm;
  char *pattern;

  if(song->sampler) {
    pattern = def_m3u_patmix;
  } else {
    pattern = def_m3u_pattern;
  }
  
  if(! pattern)                 return NULL;
  if(strcmp(pattern, "0") == 0) return NULL;
  if(! def_m3u_dir)             return NULL;
  
  if(strcmp(def_m3u_dir, "0") == 0) tmp_str = def_mp3_dir;
  else                              tmp_str = def_m3u_dir;

  if(! tmp_str) return NULL;

  ret_str = create_sub_string(song, 6);
  if(! ret_str) return NULL;
  filenm = filenm_short(ret_str, fn_mode, fn_toupper, extnl_PROC_mask);
  free(ret_str);
  
  ret_str = (char *) malloc(sizeof(char) * (strlen(tmp_str) + strlen(filenm) + 2));
  if(! ret_str) {
    perror("malloc");
    wuuush(1);
  }
  sprintf(ret_str, "%s/%s", tmp_str, filenm);
  free(filenm);

  return (kill_double_slashs(ret_str));
}

/* add entry to playlist */
int add_to_m3u(song_typ *song)
{
  char *filenm;
  char *m3u_fm;
  FILE *m3u_fd;
  int i;
  char *tmp_str;

  filenm = file_build_m3u_entry(song);
  m3u_fm = build_m3u_fname(song);
  create_sub_dirs(m3u_fm, TRUE);
  if(! m3u_fm) return 1;
  if(access(m3u_fm, F_OK) == 0) {
    if(access(m3u_fm, W_OK) != 0) {
      popup_error_win(_("could not write to playlist!"));
      free(m3u_fm);
      return 1;
    }
    i = strlen(filenm);
    tmp_str = (char *) malloc(sizeof(char) * (i+3));
    if(! tmp_str) {
      perror("malloc");
      wuuush(1);
    }
    m3u_fd = fopen(m3u_fm, "r");
    if(m3u_fd) {
      while(1) {
	fgets(tmp_str, i+1, m3u_fd);
	if(feof(m3u_fd)) break;
	if(strcmp(tmp_str, filenm) == 0) {
	  fclose(m3u_fd);
	  free(m3u_fm);
	  free(tmp_str);
	  return 1;
	}
      }
      fclose(m3u_fd);
      free(tmp_str);
    }
  }

  m3u_fd = fopen(m3u_fm, "a");
  if(! m3u_fd) {
    popup_error_win(_("couldn't write to playlistfile"));
    free(m3u_fm);
    return 1;
  }
  fprintf(m3u_fd, "%s\n", filenm);
  fclose(m3u_fd);
  free(m3u_fm);

  return 0;
}

/* kills following slashs to one slashs: "///" -> "/" */
char *kill_double_slashs(char *string)
{
  int i;
  BOOL was_slash;

  if(! string) return NULL;

  was_slash = FALSE;
  i=0;
  while(1) {
    if(*(string+i) == 0) break;
    if(*(string+i) == '/') {
      if(! was_slash) was_slash = TRUE;
      else {
       	memmove(string+i, string+i+1, strlen(string)-i);
       	i--;
      }
    } else was_slash = FALSE;
    i++;
  }

  return string;
}

/* returns (new allocated) string with directory-path (without filename)
   "/dir1/dir2/dir3/file" -> "/dir1/dir2/dir3/" (with ending slash!) */
char *extract_dirname(char *file_str)
{
  int i;
  char *ret_str;

  if((! file_str) || (strlen(file_str) < 1)) return NULL;
  for(i=strlen(file_str)-1;i>=0;i--) if(*(file_str+i) == '/') break;
  
  /* only nessasary if more than one slash found (!= root-dir) */
  if(i < 1) return NULL;
  
  i += 2;
  ret_str = (char *) malloc(sizeof(char) * i);
  if(! ret_str) {
    perror("malloc");
    wuuush(1);
  }

  memcpy(ret_str, file_str, i-1);
  *(ret_str + (i-1)) = 0;
  return ret_str;
}

/* create needed subdirectorys for file, mode=TRUE: cut ending file */
int create_sub_dirs(char *filename, BOOL mode)
{
  char *dir_str, *error_str;
  int i;

  if(mode) {
    dir_str = extract_dirname(filename);
  } else {
    if(! filename) return 0;
    dir_str = (char *) malloc(sizeof(char) * (strlen(filename) +2));
    if(! dir_str) {
      perror("malloc");
      wuuush(1);
    }
    strcpy(dir_str, filename);
    if(*(dir_str + strlen(dir_str) -1) != '/') strcat(dir_str, "/");
  }
  if(! dir_str) return 0;

  i = 1;
  while(1) {
    if(*(dir_str + i) == 0) break;
    if(*(dir_str + i) == '/') {
      *(dir_str + i) = 0;
      if(access(dir_str, F_OK) != 0) {
	/* subdir doesn't exists, try to create */
	if(mkdir(dir_str, 0777)) {
	  /* failed... :-( */
	  error_str = (char *) malloc(sizeof(char) * (strlen(dir_str)+80));
	  if(! error_str) {
	    perror("malloc");
	    wuuush(1);
	  }
	  sprintf(error_str, _("creating of directory \"%s\" failed"), dir_str);
	  popup_error_win(error_str);
	  free(error_str);
	  free(dir_str);
	  return 1;
	}
      }
      *(dir_str + i) = '/';
    }
    i++;
  }

  free(dir_str);
  return 0;
}

/* generate a unique-tempfile for each track
 * -> insert tracknr before last dot
 * eg. "/tmp/mp3c-temp.wav" -> "/tmp/mp3c-temp00.wav"
 *
 * if no dot exists, insert behind string
 * eg. "/tmp/mp3c" -> "/tmp/mp3c00"
 *
 * return pointer to temp-file, must not be free-ed!
 * pointer changes when calling gen_track_tmp again
 */
char *return_track_tmpname(song_typ *track)
{
  int i;
  char track_nr[12];
  
  if(track->tmp_wav_file) return track->tmp_wav_file;

  if(! def_tmp_file) return NULL;     /* No default type -> couldn't generate */
  
  if(extnl_TMP_name) free(extnl_TMP_name);
  extnl_TMP_name = (char *) malloc(sizeof(char) * (strlen(def_tmp_file)+12));
  if(! extnl_TMP_name) {
    perror("malloc");
    wuuush(1);
  }

  snprintf(track_nr, 12, "%08lx_%02d", track->cddb_id, (track->toc)%100);
  for(i=strlen(def_tmp_file);i>=0;i--) if(*(def_tmp_file+i) == '.') break;
  if(i > 0 && strchr(def_tmp_file + i, '/')) i=0;
  if(i <= 0) {
    strcpy(extnl_TMP_name, def_tmp_file);
    strcat(extnl_TMP_name, track_nr);
  } else {
    memcpy(extnl_TMP_name, def_tmp_file, i);
    memcpy(extnl_TMP_name+i, track_nr, 12);
    strcat(extnl_TMP_name, def_tmp_file+i);
  }

  return extnl_TMP_name;
}

/* do unglobbing in filename, $HOME -> "~" */
char *file_glob_out_name(char *name)
{
  char *home;

  if(! name) return NULL;
  
  home = getenv("HOME");
  if(! home) return name;

  if(*(home + strlen(home)-1) != '/') {
    home = (char *) malloc(sizeof(char) * (strlen(home)+2));
    if(! home) {
      wuuush(1);
    }
    sprintf(home, "%s/", getenv("HOME"));
  } else {
    home = copy_char_str(home);
  }

  if(! strncmp(name, home, strlen(home))) {
    if(strlen(home) < 2) {
      name = (char *) realloc(name, sizeof(char) * (strlen(name)+3));
      if(! name) {
	wuuush(1);
      }
    }
    memmove(name+2, name+strlen(home), strlen(name + strlen(home))+1);
    *name     = '~';
    *(name+1) = '/';
  }
  
  free(home);

  return kill_double_slashs(name);
}

/* do globbing in filename "~" -> $HOME */
char *file_glob_in_name(char *name)
{
  char *home;

  if(! name) return NULL;

  home = NULL;

  if(! strncmp(name, "~/", 2)) {
    if(! home) {
      home = getenv("HOME");
      if(! home) {
	return name;
      }
    }

    name = (char *) realloc(name, sizeof(char) * (strlen(name) + strlen(home) + 1));
    if(! name) {
      wuuush(1);
    }

    memmove(name+strlen(home)+1, name+1, strlen(name));
    memmove(name, home, strlen(home));
    *(name + strlen(home)) = '/';
  }

  return kill_double_slashs(name);
}

/* build relative filename
   input1     = filename or path (with ending slash) which describes the starting
                directory.
   input2     = filename which should be relative to input1
   should_dir = if this is true, and input1 does not end with a slash, this
                slash is added automaticly (intern).
   output     = a relative string (new allocated, should be free()'ed

   example: input1 = "/home/mp3s/playlists/roxette/joyride.m3u"
                or = "/home/mp3s/playlists/roxette/"
	    input2 = "/home/mp3s/files/roxette/joyride/queen_of_rain.mp3"
	    
	    output will be:
	    output = "../../files/roxette/joyride/queen_of_rain.mp3"

	    -----
	    
	    input1 = "/home/mp3s/joyride.m3u"
	    input2 = "/home/mp3s/queen_of_rain.mp3"
	    output = "./queen_of_rain.mp3"
 */
char *file_build_relative_name(char *src_dir, char *dest_file, BOOL should_dir)
{
  int  beginning, pos;
  char *match;
  char *rel_file;

  if((!src_dir) || (!(*(src_dir))) || (!dest_file) || (!(*(src_dir)))) return NULL;
  
  beginning = 1;
  while((match = strchr(src_dir + beginning, '/'))) {
    pos = match - src_dir;
    if(strncmp(src_dir, dest_file, pos+1)) break;
    beginning = pos+1;
  }

  pos   = 0;
  match = src_dir + beginning;

  while((match = strchr(match, '/'))) {
    match++;
    pos++;
  }

  if(should_dir && (*(src_dir + strlen(src_dir) -1) != '/')) {
    pos++;
  }

  dest_file += beginning;
  rel_file = (char *) malloc(sizeof(char) * (strlen(dest_file) + (pos * 3) + 3));
  if(! rel_file) {
    wuuush(1);
  }

  beginning=0;
  if(! pos) {
    *(rel_file)     = '.';
    *(rel_file + 1) = '/';
    beginning = 2;
  } else {
    for(;pos>0;pos--) {
      *(rel_file + beginning)     = '.';
      *(rel_file + beginning + 1) = '.';
      *(rel_file + beginning + 2) = '/';
      beginning += 3;
    }
  }

  *(rel_file + beginning) = 0;
  strcat(rel_file, dest_file);

  return rel_file;
}

/* build an entry for a playlist */
char *file_build_m3u_entry(song_typ *track)
{
  char *full_m3u;
  
  if(extnl_M3U_name) free(extnl_M3U_name);
  extnl_M3U_name = NULL;
  
  switch(def_m3u_rel_name) {
    case 0:   /* full path */
      return build_mp3_filenm(track);
      break;
    case 1:   /* relative to m3udir */
      if(! def_m3u_dir) return NULL;
      if(strcmp(def_m3u_dir, "0") == 0) {
	full_m3u = kill_double_slashs(def_mp3_dir);
      } else {
	full_m3u = kill_double_slashs(def_m3u_dir);
      }
      extnl_M3U_name = file_build_relative_name(full_m3u, build_mp3_filenm(track), TRUE);
      return extnl_M3U_name;
      break;
    default:  /* relative to m3ufile */
      full_m3u = build_m3u_fname(track);
      extnl_M3U_name = file_build_relative_name(full_m3u, build_mp3_filenm(track), FALSE);
      if(full_m3u) free(full_m3u);
      return extnl_M3U_name;
  }
}

/* check for directory */
BOOL file_is_directory(char *path)
{
  struct stat st;
  char *error_string;

  if(! path) return FALSE;
  
  if(stat(path, &st) != -1) {
    if(S_ISDIR(st.st_mode)) return TRUE;
    error_string = (char *) malloc(sizeof(char) * (strlen(path) + 50));
    if(! error_string) {
      wuuush(1);
    }
    sprintf(error_string, _("\"%s\" is no directory"), path);
    popup_error_win(error_string);
    free(error_string);
  }

  return FALSE;
}

