/// McVerter.cpp
/** The command-line version of MRIConvert.
*/

#include "McVerter.h"

namespace po = boost::program_options;

// Macro in place of _() since std::cerr does not like utf16 or utf32.
#define _C(string) wxString(_(string)).utf8_str()

using namespace jcs;

// This permits cout to handle a vector.
template<class T>
std::ostream& operator<<(std::ostream& os, const std::vector<T>& v)
{
  copy(v.begin(), v.end(), std::ostream_iterator<T>(std::cerr, " ")); 
  return os;
}

int main(int argc, char** argv)
{
  // First make sure we can use wx stuff.
  wxInitializer initializer(0, (wxChar**) NULL);
  if ( !initializer ) {
    std::cerr << "Failed to initialize the wxWidgets library, aborting.";
    return EXIT_NO_WX;
  }

  // Localization steps
  /*
    wxLocale m_locale;
    if ( !m_locale.Init(wxLANGUAGE_DEFAULT, wxLOCALE_CONV_ENCODING) ) {
    wxLogError(_T("This language is not supported by the system."));
    }
    // Development only, distribution should install in standard location.
    wxLocale::AddCatalogLookupPathPrefix(_T("l10n"));
    // Initialize the catalogs we'll be using
    m_locale.AddCatalog(_T("mriconvert"));
  */

  // Variables to collect command-line arguments.
  std::string std_match;
  std::string std_outputDir;
  std::string std_format;
  long skip = 0;
  std::string std_fnformat;
  std::vector<std::string> std_input_files;

  po::options_description additional("Additional options");
  po::options_description required("Required options");
  po::options_description hidden;
  po::options_description documented;
  po::options_description all;
  po::positional_options_description positional;
  po::variables_map vm;

  required.add_options()
  ("output,o", po::value<std::string>(&std_outputDir)->required(), _C("Output directory for converted files"))
  ("format,f", po::value<std::string>(&std_format)->required(), _C("Output format: fsl, spm, meta, nifti, analyze, or bv"))
  ;
  additional.add_options()
  ("help,h", _C("Produce help message"))
  ("verbose,v", _C("Verbose mode"))
  ("quiet,q", _C("Quiet mode"))
  ("version,V", _C("Version info"))
  ("split_subj,j", _C("Save each subject to a separate directory"))
  ("split_dir,x", _C("Save each series to a separate directory"))
  ("fourd,d", _C("Save output volumes as 4D files"))
  ("ho,a", _C("Save header only (metaimage only)"))
  ("nii,n", _C("Save files as .nii (nifti/fsl only)"))
  ("v16,b", _C("Save .v16 files (BrainVoyager only)"))
  ("rescale,r", _C("Apply rescale slope and intercept to data"))
  ("patientid,u", _C("Use patient id instead of patient name for output file"))
  ("skip,s", po::value<long>(&skip)->default_value(0), _C("Skip first 'arg' volumes"))
  ("match,m", po::value<std::string>(&std_match), _C("Only convert files whose series descriptions include this string"))
  ("fnformat,F", po::value<std::string>(&std_fnformat), _C("Use this format for name of output file: +/- PatientName, \
PatientId, \
SeriesDate, \
SeriesTime, \
StudyId, \
StudyDescription, \
SeriesNumber, \
SequenceName, \
ProtocolName, \
SeriesDescription"))
  ;
  hidden.add_options()
  ("input-file", po::value<std::vector<std::string> >()->default_value(std::vector<std::string>(),"")->required(), _C("Input directory or file(s)"))
  ;
  positional.add("input-file", -1);
  documented.add(required).add(additional);
  all.add(required).add(additional).add(hidden);

  try {
    po::store(po::command_line_parser(argc, argv).options(all).positional(positional).run(), vm);

    // Known bail-out cases.
    if (vm.count("version")) {
      std::cout << "mcverter " << VERSION_STRING << std::endl;
      return EXIT_NO_PROCESSING;
    }
    if (vm.count("help")) {
      std::cout << "Usage: mcverter [options] <input directory or file(s)>" << std::endl;
      std::cout << documented;
      return EXIT_NO_PROCESSING;
    }
    std_input_files = vm["input-file"].as<std::vector<std::string> >();
    if (std_input_files.begin() == std_input_files.end()) {
      std::cerr << _C("No files or directory specified.") << std::endl;
      return EXIT_NO_INPUT_SPECIFIED;
    }
    po::notify(vm);
  }
  catch (po::invalid_syntax& e) {
    std::cerr << _C("Invalid syntax.") << e.kind() << " " << e.tokens() << std::endl;
    return EXIT_ARGUMENT_ERROR;
  }
  catch (po::unknown_option& e) {
    std::cerr << _C("Unrecognized option: ") << e.get_option_name() << std::endl;
    return EXIT_ARGUMENT_ERROR;
  }
  catch (po::ambiguous_option& e) {
    std::cerr << _C("Ambiguous option ") << e.get_option_name() << std::endl;
    return EXIT_ARGUMENT_ERROR;
  }
  catch (po::multiple_values& e) {
    std::cerr << _C("Multiple values invalid.") << std::endl;
    return EXIT_ARGUMENT_ERROR;
  }
  catch (po::multiple_occurrences& e) {
    std::cerr << _C("Multiple occurrences.") << std::endl;
    return EXIT_ARGUMENT_ERROR;
  }
  catch (po::validation_error& e) {
    std::cerr << _C("Validation error ") << e.what() << std::endl;
    return EXIT_ARGUMENT_ERROR;
  }
  catch (po::too_many_positional_options_error& e) {
    std::cerr << _C("General error.") << std::endl;
    return EXIT_ARGUMENT_ERROR;
  }
  catch (po::invalid_command_line_style& e) {
    std::cerr << _C("General error.") << std::endl;
    return EXIT_ARGUMENT_ERROR;
  }
  catch (po::reading_file& e) {
    std::cerr << _C("Error reading file.") << std::endl;
    return EXIT_ARGUMENT_ERROR;
  }
  catch (po::required_option& e) {
    std::cerr << _C("Option ") << e.get_option_name() << _C(" is required.") << std::endl;
    return EXIT_ARGUMENT_ERROR;
  }
  catch (po::error& e) {
    std::cerr << _C("General error.") << std::endl;
    return EXIT_ARGUMENT_ERROR;
  }

  // Transfer arguments to wxString variables.
  wxString match(std_match.c_str(), wxConvUTF8);
  wxString outputDir(std_outputDir.c_str(), wxConvUTF8);
  wxString format(std_format.c_str(), wxConvUTF8);
  wxString fnformat(std_fnformat.c_str(), wxConvUTF8);

  verbose = vm.count("verbose") ? true : false;
  quiet = vm.count("quiet") ? true : false;

  // Selector for output format.
  int found = wxNOT_FOUND;
  int format_type = 0;
  for (; format_type < OutputFactory::GetNumberOfTypes(); ++format_type) {
    found = format.Find(wxString(OutputFactory::GetShortDescription(format_type), wxConvLocal));
    if (found != wxNOT_FOUND) {
      break;
    }
  }
  if (found == wxNOT_FOUND) {
    // Get the list of permitted output formats.
    wxString allowed_formats_string;
    allowed_formats_string.append(_("Available formats: "));
    int type = 0;
    for (; type < (OutputFactory::GetNumberOfTypes() - 1); ++type) {
      allowed_formats_string.append(wxString(OutputFactory::GetShortDescription(type), wxConvLocal));
      allowed_formats_string.append(_T(", "));
    }
    allowed_formats_string.append(wxString(OutputFactory::GetShortDescription(type), wxConvLocal));
    std::cerr << _C("Invalid format type: ") << format.utf8_str() << std::endl;
    std::cerr << allowed_formats_string.utf8_str() << std::endl;
    return EXIT_ARGUMENT_ERROR;
  }
  
  // Selector for output directory.
  if (!wxDir::Exists(outputDir)) {
    if (!wxFileName::Mkdir(outputDir, 0777, wxPATH_MKDIR_FULL)) {
      std::cerr << _C("Unable to create directory: ") << outputDir << std::endl;
      return EXIT_PERMISSION_ERROR;
    }
    if (verbose && !quiet) {
      std::cout << _C("Created directory ") << outputDir << std::endl;
    }
  }
  
  // We know enough to create our converter and outputter.
  Converter* converter = new Converter(format_type);
  OutputterBase* outputter = converter->GetOutputter();
  outputter->SetOption("skip", static_cast<int> (skip));
  outputter->mOutputList.rootDir = outputDir;
  outputter->SetOption("split_subj", vm.count("split_subj") ? true : false);
  outputter->SetOption("split_dir", vm.count("split_dir") ? true : false);
  outputter->SetOption("dim", vm.count("fourd") ? 4 : 3);
  outputter->SetOption("ho", vm.count("ho") ? true : false);
  outputter->SetOption("nii", vm.count("nii") ? true : false);
  outputter->SetOption("v16", vm.count("v16") ? true : false);
  outputter->SetOption("rescale", vm.count("rescale") ? true : false);
  
  // Handle request for using PatientId rather than PatientName.
  if (vm.count("patientid")) {
    outputter->defaultNameFields[OutputterBase::PatientName].value = false;
    outputter->defaultNameFields[OutputterBase::PatientId].value = true;
  }
  
  // Handle request for alternate output filename format.
  std::string stdfnformat = std::string (fnformat.mb_str());
  OutputterBase::FieldMap::iterator fm_it = outputter->defaultNameFields.begin();
  OutputterBase::FieldMap::iterator fm_it_end = outputter->defaultNameFields.end();
  for (; fm_it != fm_it_end; ++fm_it) {
    if (stdfnformat.find("+" + fm_it->second.name) != -1) {
      fm_it->second.value = true;
    }
    if (stdfnformat.find("-" + fm_it->second.name) != -1) {
      fm_it->second.value = false;
    }
  }

  // We've processed all options and switches,
  // now add the requested files.
  if (verbose && !quiet) {
    std::cout << _C("Adding files...") << std::endl;
  }
  std::vector< std::string >::iterator it = std_input_files.begin();
  std::vector< std::string >::iterator it_end = std_input_files.end();
  for (; it != it_end; ++it) {
    wxFileName filename(wxString(std::string(*it).c_str(), wxConvUTF8));
    if (filename.FileExists()) {
      converter->AddFile(filename.GetFullPath(), match);
    }
    else if (filename.DirExists()) {
      wxArrayString files;
      wxDir::GetAllFiles(filename.GetFullPath(), &files);
      for (unsigned int i = 0; i < files.GetCount(); ++i) {
        converter->AddFile(files.Item(i), match);
      }
    }
    else {
      std::cerr << _C("File or directory ") << *it << _C(" not found. Add trailing directory separator and try again.") << std::endl;
      return EXIT_NO_INPUT_SPECIFIED;

    }
  }

  if (verbose && !quiet) {
    MessageList::iterator ml_it = converter->messages.begin();
    MessageList::iterator ml_it_end = converter->messages.end();
    for (; ml_it != ml_it_end; ++ml_it) {
      Message::iterator it = ml_it->begin();
      Message::iterator it_end = ml_it->end();
      for (; it != it_end; ++it) {
        std::cout << it->mb_str(wxConvLocal) << " ";
      }
      std::cout << std::endl;
    }
  }
  converter->messages.clear();
  converter->UpdateAllOutput();

  if (verbose && !quiet) std::cout << _C("Converting...") << std::endl;

  // Let the real work begin.
  converter->ConvertAll();

  if (verbose && !quiet) std::cout << _C("Finished") << std::endl;

  delete converter;

  return EXIT_SUCCESS;
}
