/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
/*---------------------------------------------------------------
    APPLICATION: Camera calibration
    FILE: CalibFilter.cpp
    AUTHOR: Vicente Arevalo Espejo <varevalo@ctima.uma.es>
             based on code from the OpenCV library.

	See README.txt for instructions.

   Part of the MRPT Library - http://babel.isa.uma.es/mrpt/
   ISA - Universidad de Malaga - http://www.isa.uma.es
  ---------------------------------------------------------------*/

#include <stdio.h>
#include <assert.h>
#include <cmath>


//#include <cv.h>
//#include <highgui.h>

#include <cv.h>
#include <highgui.h>

#include "CalibFilter.h"

#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/timeb.h>
#include <stdio.h>
#include <iostream>

#ifndef NOERROR
# define NOERROR 1
#endif

#ifndef E_INVALIDARG
#define E_INVALIDARG 0
#endif

#ifndef E_PENDING
#define E_PENDING 2
#endif

#ifndef E_FAIL
#define E_FAIL 10
#endif


extern int   CHECKBOARD_SIZE_X, CHECKBOARD_SIZE_Y;
extern float CHECKBOARD_DIMENSION_MM;


/* Constructor */
CCalibFilter::CCalibFilter()
{
    m_initial_params.etalon_type = CalibEtalon_ChessBoard;

    m_initial_params.etalon_params[0] = (float)CHECKBOARD_SIZE_X;
    m_initial_params.etalon_params[1] = (float)CHECKBOARD_SIZE_Y;
    m_initial_params.etalon_params[2] = CHECKBOARD_DIMENSION_MM; // mm
    m_initial_params.etalon_params_count = 3;

    m_initial_params.show_feature_points_flag = 1;
    m_initial_params.frame_interval = 500;  // ms
    m_initial_params.frames_to_collect = 25;
    m_initial_params.frames_collected = 0;
    m_initial_params.frames_passed = 0;
    m_initial_params.calib_state = CalibState_Initial;
    m_initial_params.last_frame_time = -1e6f;

    m_initial_params.enable_undistortion    = 1;
    m_initial_params.show_3d_window         = 0;

    m_params = m_initial_params;

    m_max_points = 0;
    m_imagePoints = 0;
    m_objectPoints = 0;
    m_transVects = 0;
    m_rotMatrs = 0;

    m_gray_img   = cvCreateImage( cvSize(1,1), IPL_DEPTH_8U, 1 );
    m_thresh_img = cvCreateImage( cvSize(1,1), IPL_DEPTH_8U, 1 );
	m_rgb_img    = 0;

    m_undist_img = cvCreateImage( cvSize(1,1), IPL_DEPTH_8U, 3 );

    memset( &m_undistort_params, 0, sizeof(m_undistort_params));
    m_undistort_data = 0;
}

CCalibFilter::~CCalibFilter()
{
}

CvSize  CCalibFilter::GetEtalonSize()
{
    return cvSize( cvRound(m_params.etalon_params[0]) - 1,
                      cvRound(m_params.etalon_params[1]) - 1 );
}

/* CheckReallocBuffers */
void  CCalibFilter::CheckReallocBuffers( IplImage* rgb_img )
{
    CvSize etalon_size = GetEtalonSize();
    int etalon_points = etalon_size.width*etalon_size.height;

    if( m_gray_img->imageData == 0 ||
        m_gray_img->width != rgb_img->width ||
        m_gray_img->height != rgb_img->height )
    {
        cvReleaseImage( &m_gray_img );
        m_gray_img = cvCreateImage(cvSize(rgb_img->width, rgb_img->height),IPL_DEPTH_8U, 1);

        cvReleaseImage( &m_thresh_img );
		m_thresh_img = cvCreateImage(cvSize(rgb_img->width, rgb_img->height),IPL_DEPTH_8U, 1 );

        cvReleaseImage( &m_undist_img );
        m_undist_img = cvCreateImage(cvSize(rgb_img->width, rgb_img->height),IPL_DEPTH_8U, 3 );
    }

    if( etalon_points * m_params.frames_to_collect > m_max_points )
    {
        int  new_max_points = etalon_points * (m_params.frames_to_collect + 1) + 128;
        CvPoint2D32f* imagePoints = (CvPoint2D32f*)malloc( new_max_points *
                                                           sizeof(CvPoint2D32f));

        memcpy( imagePoints, m_imagePoints, m_max_points * sizeof(CvPoint2D32f));

        free( m_imagePoints );
        free( m_objectPoints );
        free( m_transVects );
        free( m_rotMatrs );

        m_imagePoints  = imagePoints;
        m_objectPoints  = (CvPoint3D32f*)malloc( new_max_points * sizeof(CvPoint3D32f) );
        m_transVects = (CvPoint3D32f*)malloc( new_max_points * sizeof(CvPoint2D32f) );
        m_rotMatrs = (float*)malloc( new_max_points * 9 * sizeof(float));
        m_numsPoints = (int*)calloc( m_params.frames_to_collect,sizeof(int));

        m_max_points = new_max_points;
    }
}

void  CCalibFilter::DrawEtalon( IplImage* rgb_img, CvPoint2D32f* corners,
                                int corner_count, CvSize etalon_size,
                                int draw_ordered )
{
    const int r = 4;
    int i;

    if( corner_count == etalon_size.width * etalon_size.height && draw_ordered )
    {
        int x, y;
        CvPoint prev_pt = { 0, 0};
        const int line_max = 8;
        CvScalar line_colors[8];

		line_colors[0] = CV_RGB(255,0,0);
		line_colors[1] = CV_RGB(255,128,0);
		line_colors[2] = CV_RGB(255,128,0);
		line_colors[3] = CV_RGB(200,200,0);
		line_colors[4] = CV_RGB(0,255,0);
		line_colors[5] = CV_RGB(0,200,200);
		line_colors[6] = CV_RGB(0,0,255);
		line_colors[7] = CV_RGB(255,0,255);

        for( y = 0, i = 0; y < etalon_size.height; y++ )
        {
            CvScalar color = line_colors[y % line_max];
            for( x = 0; x < etalon_size.width; x++, i++ )
            {
                CvPoint pt;
                pt.x = cvRound(corners[i].x);
                pt.y = cvRound(corners[i].y);

                if( i != 0 ) cvLine( rgb_img, prev_pt, pt, color );

                cvLine( rgb_img,
                          cvPoint( pt.x - r, pt.y - r ),
                          cvPoint( pt.x + r, pt.y + r ), color );
                cvLine( rgb_img,
                          cvPoint( pt.x - r, pt.y + r),
                          cvPoint( pt.x + r, pt.y - r), color );
                cvCircle( rgb_img, pt, r+1, color );
                prev_pt = pt;
            }
        }
    }
    else
    {
        const CvScalar color = CV_RGB(255,0,0);
        for( i = 0; i < corner_count; i++ )
        {
            CvPoint pt;
            pt.x = cvRound(corners[i].x);
            pt.y = cvRound(corners[i].y);
            cvLine( rgb_img,
                      cvPoint( pt.x - r, pt.y - r ),
                      cvPoint( pt.x + r, pt.y + r ), color );
            cvLine( rgb_img,
                      cvPoint( pt.x - r, pt.y + r),
                      cvPoint( pt.x + r, pt.y - r), color );
            cvCircle( rgb_img, pt, r+1, color );
        }
    }
}

void  CCalibFilter::FillEtalonObjPoints( CvPoint3D32f* obj_points,
                                         CvSize etalon_size,
                                         float square_size )
{
    int x, y, i;

    for( y = 0, i = 0; y < etalon_size.height; y++ )
    {
        for( x = 0; x < etalon_size.width; x++, i++ )
        {
            obj_points[i].x = square_size * x;
            obj_points[i].y = square_size * y;
            obj_points[i].z = 0;
        }
    }
}

#ifdef MIRROR_POINTS
void MirrorPoints( CvPoint2D32f* points, int frames, CvSize etalon_size, CvSize imgSize )
{
    int i, j, k;
    for( i = 0; i < frames; i++ )
    {
        int start = i*etalon_size.width*etalon_size.height;

        for( j = 0; j < etalon_size.height; j++ )
        {
            for( k = 0; k < etalon_size.width/2; k++ )
            {
                CvPoint2D32f temp;
                CV_SWAP( points[start + j*etalon_size.width + k],
                         points[start + (j+1)*etalon_size.width - k - 1], temp );
            }

            for( k = 0; k < etalon_size.width; k++ )
            {
                points[start + j*etalon_size.width + k].y = imgSize.height -
                    points[start + j*etalon_size.width + k].y;
            }
        }
    }
}
#else
void MirrorPoints( CvPoint2D32f*, int, CvSize, CvSize )
{
}
#endif

long CCalibFilter::ProcessFrame(IplImage *frame)
{
	m_rgb_img = frame;
	return Transform();
}


double MOOSTime()
{
    double dfT=0.0;
    //grab the time..
#ifndef _WIN32

#if(1) //PMN May03 04
    struct timeb timebuffer;
    ftime( &timebuffer );
    dfT = timebuffer.time+ ((double)timebuffer.millitm)/1000;
#else

    struct timeval TimeVal;

    if(gettimeofday(&TimeVal,NULL)==0)
    {
        dfT = TimeVal.tv_sec+TimeVal.tv_usec/1000000.0;
    }
    else
    {
        dfT =-1;
    }
#endif



#else
    struct _timeb timebuffer;
    _ftime( &timebuffer );
    dfT = timebuffer.time+ ((double)timebuffer.millitm)/1000;



#endif
    return dfT;

}


long CCalibFilter::Transform()
{
    CalibState  state = m_params.calib_state;

    switch( state )
    {
		case CalibState_Initial:
        {
			m_params = m_initial_params;
            m_params.calib_state = CalibState_NotCalibrated;
			m_params.frames_collected = 0;
        }
		case CalibState_NotCalibrated:
		case CalibState_CalibrationProcess:
		case CalibState_Calibrated:
        {
            ProcessFrame(m_rgb_img, MOOSTime());
            ++m_params.frames_passed;
        }
        break;
		default:
        assert(0);
    }

    return NOERROR;
}

/* ProcessFrame */
void  CCalibFilter::ProcessFrame( IplImage* rgb_img, double frame_time )
{
    bool find_corners = m_initial_params.show_3d_window ||
                        m_params.calib_state == CalibState_CalibrationProcess;
    bool chess_found = false;
    CvSize etalon_size = GetEtalonSize();
    int   etalon_points = etalon_size.width * etalon_size.height;
    CvSize size;
    cvGetImageRawData(rgb_img, 0, 0, &size);
	CheckReallocBuffers( rgb_img );

    CvPoint2D32f* pt_ptr = m_imagePoints + m_params.frames_collected * etalon_points;

    if( find_corners )
    {
        /* Begin of find etalon points */
        int count = etalon_points;// + 10;

        cvCvtColor( rgb_img, m_gray_img, CV_BGR2GRAY );

        /*********************************************************/
        //////////////   FIND CHECKERBOARD CORNERS   //////////////
        ///////////////////////////////////////////////////////////

        chess_found = cvFindChessBoardCornerGuesses( m_gray_img, m_thresh_img, 0,
                                                     etalon_size, pt_ptr, &count ) != 0;
        if( count != 0 )
        {
            cvFindCornerSubPix(
                m_gray_img, pt_ptr, count, cvSize(5,5), cvSize(-1,-1),
                cvTermCriteria( CV_TERMCRIT_ITER|CV_TERMCRIT_EPS, 10, 0.01f ));
        }

        DrawEtalon( rgb_img, pt_ptr, count, etalon_size, chess_found );
    }


    if( m_params.calib_state == CalibState_Calibrated )
    {
        /* Calibration finished */
        if( m_initial_params.show_3d_window && chess_found )
        {/* We must show 3D etalon and compute extrinsic parameters */
            float  rotVect[3];
            //float  Jacobian[27];

            /* Collect object points */
            FillEtalonObjPoints( m_objectPoints, etalon_size,
                                 m_params.etalon_params[2] );

            MirrorPoints( pt_ptr, 1, etalon_size, size );

	    CvPoint2D32f    pts = m_camera.principalPoint;

            cvFindExtrinsicCameraParams( etalon_points,
                                         size,
                                         pt_ptr,
                                         m_objectPoints,
                                         m_camera.focalLength,
                                         pts,
                                         m_camera.distortion,
                                         rotVect,
                                         m_camera.transVect );

            {
                CvMat rmat = cvMat( 3, 3, CV_32FC1, m_camera.rotMatr );
                CvMat rvec = cvMat( 3, 1, CV_32FC1, rotVect );
                //CvMat jacob = cvMat( 3, 9, CV_32FC1, Jacobian );

                /* Calc rotation matrix by via Rodrigues Transform */
                cvRodrigues( &rmat, &rvec, 0, CV_RODRIGUES_V2M );
            }
        }

        if( m_initial_params.enable_undistortion )
        {
            /* Apply undistortion */
            if( memcmp( m_camera.matrix, m_undistort_params.matrix, sizeof(m_camera.matrix)) != 0 ||
                memcmp( m_camera.distortion, m_undistort_params.distortion, sizeof(m_camera.distortion)) != 0 )
            {
                memcpy( &m_undistort_params, &m_camera, sizeof(m_camera));

                if( !m_undistort_data || m_undistort_data->width != rgb_img->width ||
                    m_undistort_data->height != rgb_img->height )
                {
                    cvReleaseImage( &m_undistort_data );
                }
                m_undistort_data = cvCreateImage( cvSize( rgb_img->width, rgb_img->height ),
                                                  IPL_DEPTH_32S, 3 );
                cvUnDistortInit( rgb_img, m_undistort_data, m_undistort_params.matrix,
                                 m_undistort_params.distortion,1 );
            }

            cvUnDistort( rgb_img, m_undist_img, m_undistort_data,1 );
            cvCopyImage(m_undist_img, rgb_img);
        }
    } /* Check if Calibration not finished and the etalon is recognized */

    if( m_params.calib_state == CalibState_CalibrationProcess && chess_found &&
        frame_time >= m_params.last_frame_time + (m_params.frame_interval/1000.0f) )
    {
        m_params.last_frame_time = frame_time;
        m_params.frames_collected++;

        cvXorS( rgb_img, cvScalarAll( 255 ), rgb_img );

        if( m_params.frames_collected == m_params.frames_to_collect )
        {
            /* all frames are collected. Now will calibrate */
            CalculateCameraParams( size );
            m_params.calib_state = CalibState_Calibrated;
        }/* End calibration */
    } /* Else point accumulation */
}

void  CCalibFilter::CalculateCameraParams( CvSize size )
{
    int frame;
    CvSize etalon_size = GetEtalonSize();
    int   etalon_points = etalon_size.width * etalon_size.height;

    FillEtalonObjPoints( m_objectPoints, etalon_size,
                         m_params.etalon_params[2] );

    for( frame = 1; frame < m_params.frames_collected; frame++ )
    {
        memcpy( m_objectPoints + etalon_points*frame, m_objectPoints,
                etalon_points * sizeof(m_objectPoints[0]));
    }

    /* Set etalon points counters */
    for( frame = 0; frame < m_params.frames_collected; frame++ )
    {
        m_numsPoints[frame] = etalon_points;
    }

    MirrorPoints( m_imagePoints, m_params.frames_collected, etalon_size, size );

    /* Calirate camera */
    cvCalibrateCamera( m_params.frames_collected, m_numsPoints,
                       size, m_imagePoints, m_objectPoints,
                       m_camera.distortion, m_camera.matrix,
                       (float*)m_transVects, m_rotMatrs, 0 );

    /* Copy some camera parameters */
    m_camera.focalLength[0] = m_camera.matrix[0];
    m_camera.focalLength[1] = m_camera.matrix[4];

    m_camera.principalPoint.x = m_camera.matrix[2];
    m_camera.principalPoint.y = m_camera.matrix[5];
}

//////////////////////////////////////

long CCalibFilter::getEtalonParams(CalibEtalonType *etalon_type, float *etalon_params, long *etalon_params_count)
{
    int params_to_copy = m_initial_params.etalon_params_count;
    *etalon_type = m_initial_params.etalon_type;

    if( etalon_params )
    {
        if( params_to_copy > *etalon_params_count )
            params_to_copy = *etalon_params_count;

        memcpy( etalon_params, m_initial_params.etalon_params,
                params_to_copy*sizeof(float) );
        *etalon_params_count = params_to_copy;
    }

    return NOERROR;
}

long CCalibFilter::setEtalonParams(CalibEtalonType etalon_type, float *etalon_params, long etalon_params_count)
{
    if( etalon_type != CalibEtalon_ChessBoard ||
        etalon_params_count != 3 ) return E_INVALIDARG;

    m_initial_params.etalon_type = etalon_type;
    m_initial_params.etalon_params_count = etalon_params_count;
    memcpy( m_initial_params.etalon_params, etalon_params,
            etalon_params_count*sizeof(float) );

    return NOERROR;
}

long CCalibFilter::getFrameInterval(long *count)
{
    *count = m_params.frame_interval;

    return NOERROR;
}

long CCalibFilter::setFrameInterval(long count)
{
    if( count < 1 ) count = 1;
    m_initial_params.frame_interval = m_params.frame_interval = count;

    return NOERROR;
}

long CCalibFilter::getFramesToCollect(long *frames)
{
    *frames = m_params.frames_to_collect;

    return  NOERROR;
}

long CCalibFilter::setFramesToCollect(long frames)
{
    m_initial_params.frames_to_collect = m_params.frames_to_collect = frames;

    return  NOERROR;
}

long CCalibFilter::getEnableUndistortion(long *enable)
{
    *enable = m_params.enable_undistortion;

    return  NOERROR;
}

long CCalibFilter::setEnableUndistortion(long enable)
{
    m_initial_params.enable_undistortion = m_params.enable_undistortion = enable;

    return  NOERROR;
}

long CCalibFilter::getCameraParams( CvCameraParams* camera )
{
    if( m_params.calib_state != CalibState_Calibrated ) return E_PENDING;
    *camera = m_camera;

    return  NOERROR;
}

long CCalibFilter::getState(CalibState *calib_state, long *frames_collected, long *frames_passed, double *last_frame_time)
{
    *calib_state = m_params.calib_state;
    *frames_collected = m_params.frames_collected;
    *frames_passed = m_params.frames_passed;
	*last_frame_time = m_params.last_frame_time;

    return NOERROR;
}

//////////////////////////////////////

long CCalibFilter::StartCalibrate()
{
    m_params = m_initial_params;
    m_params.calib_state = CalibState_CalibrationProcess;
    m_params.frames_collected = 0;

    return  NOERROR;
}

long CCalibFilter::StopCalibrate()
{
    m_params.calib_state = CalibState_Initial;

    return  NOERROR;
}

//////////////////////////////////////

long CCalibFilter::SaveCameraParams(const char *fileName)
{
    if( m_params.calib_state == CalibState_Calibrated)
    {
        if( strlen(fileName) != 0)
        {
            FILE *file = fopen(fileName,"wt");
            if( file != 0)
            {
                int i, j;
                fprintf(file,"Camera Matrix:\n");
                for( i = 0; i < 3; i++ )
                {
                    for( j = 0; j < 3; j++ )
                    {
                        fprintf( file,"M[%d.%d]=%20.7f", i, j, m_camera.matrix[i*3+j]);
                    }
                    fprintf(file,"\n");
                }

                fprintf(file,"\n\nDistortion:\n");
                for( i = 0; i < 4; i++ )
                    fprintf(file,"D[%d]=%f\n", i, m_camera.distortion[i]);

                fclose(file);
            }
            else
            {
 //               MessageBox(0,"Can't open file","Save camera parameters",MB_OK|MB_ICONERROR);
            }
        }
    }
    else
    {
//        MessageBox(0,"Camera was not calibrated","Save camera parameters",MB_OK);
    }

    return  NOERROR;
}

long CCalibFilter::LoadCameraParams(const char *fileName)
{
#define BUF_SIZE 10000
    char buffer[BUF_SIZE + 100];

    if( strlen(fileName) != 0)
    {
        FILE *file = fopen(fileName,"rb");

        if( file != 0)
        {
            int i, j, k;
            float cameraMatrix[9];
            float distortion[4];

            int sz = (int) fread( buffer, 1, BUF_SIZE, file );
            char* ptr = buffer;
            buffer[sz] = '\0';

            /* read matrix */
            for( k = 0; k < 9; k++ )
            {
                ptr = strstr( ptr, "M[" );
                if( ptr )
                {
                    int s = 0;
                    ptr += 2;
                    if( sscanf( ptr, "%d%*[.,]%d%n", &i, &j, &s ) == 2 && i == k/3 && j == k%3 )
                    {
                        ptr += s;
                        ptr = strstr( ptr, "=" );
                        if( ptr )
                        {
                            s = 0;
                            ptr++;
                            if( sscanf( ptr, "%f%n", cameraMatrix + k, &s ) == 1 )
                            {
                                ptr += s;
                                continue;
                            }
                        }
                    }
                }

                /* else report a bug */
//                MessageBox(0,"Invalid file format","Load camera parameters",MB_OK|MB_ICONERROR);
                return E_FAIL;
            }

            /* read distortion */
            for( k = 0; k < 4; k++ )
            {
                ptr = strstr( ptr, "D[" );
                if( ptr )
                {
                    int s = 0;
                    ptr += 2;
                    if( sscanf( ptr, "%d%n", &i, &s ) == 1 && i == k )
                    {
                        ptr += s;
                        ptr = strstr( ptr, "=" );
                        if( ptr )
                        {
                            s = 0;
                            ptr++;
                            if( sscanf( ptr, "%f%n", distortion + k, &s ) == 1 )
                            {
                                ptr += s;
                                continue;
                            }
                        }
                    }
                }

                /* else report a bug */
//                MessageBox(0,"Invalid file format","Load camera parameters",MB_OK|MB_ICONERROR);
                return E_FAIL;
            }

            memcpy( m_camera.matrix, cameraMatrix, sizeof( cameraMatrix ));
            memcpy( m_camera.distortion, distortion, sizeof( distortion ));

            m_camera.focalLength[0] = m_camera.matrix[0];
            m_camera.focalLength[1] = m_camera.matrix[4];

            m_camera.principalPoint.x = m_camera.matrix[2];
            m_camera.principalPoint.y = m_camera.matrix[5];

            m_params.calib_state = CalibState_Calibrated;

            fclose(file);
        }
        else
        {
//            MessageBox(0,"Can't open file","Load camera parameters",MB_OK|MB_ICONERROR);
		return E_FAIL;
        }
    }

    return  NOERROR;
}

/* End of file. */
