/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

/*---------------------------------------------------------------
    APPLICATION: Kalman Filter-based SLAM implementation
    FILE: kf-slam_main.cpp
    AUTHOR: Jose Luis Blanco Claraco <jlblanco@ctima.uma.es>

	See README.txt for instructions.
 ---------------------------------------------------------------*/

#include <mrpt/core.h>

using namespace mrpt;
using namespace mrpt::slam;
using namespace mrpt::opengl;
using namespace mrpt::system;
using namespace mrpt::math;
using namespace mrpt::utils;
using namespace std;

std::string		configFile;

void Run_KF_SLAM();

// ------------------------------------------------------
//						MAIN
// ------------------------------------------------------
int main(int argc, char **argv)
{
	try
	{
		printf(" KF-SLAM version 0.2 - Part of the MRPT\n");
		printf(" MRPT C++ Library: %s - BUILD DATE %s\n", MRPT_getVersion().c_str(), MRPT_getCompilationDate().c_str());
		printf("-------------------------------------------------------------------\n");

		// Process arguments:
		if (argc<2)
		{
			printf("Use: kf-slam <config_file>\n\nPush any key to exit...\n");
			os::getch();
			return -1;
		}

		mrpt::registerAllClasses();

		configFile = std::string( argv[1] );

		Run_KF_SLAM();

		return 0;
	}
	catch (std::exception &e)
	{
		std::cerr << e.what() << std::endl << "Program finished for an exception!!" << std::endl;
		mrpt::system::pause();
		return -1;
	}
	catch (...)
	{
		std::cerr << "Untyped exception!!" << std::endl;
		mrpt::system::pause();
		return -1;
	}
}

/*

class CMyDebugStream : public CStream
{
protected:
	FILE	*f;
	size_t Read(void *Buffer, size_t Count) { return 0; }

	size_t Write(const void *Buffer, size_t Count) //{ return 0; }
	{
		cout << static_cast<const char*>(Buffer);
		if (f)
		{
			size_t w=fwrite(Buffer,1,Count,f);ASSERT_(w);
		}
		return Count;
	}
public:
	CMyDebugStream( const char *fileName )
	{
		f = os::fopen(fileName,"wt");
		if (!f) THROW_EXCEPTION("Cannot open log file!")
	}

	~CMyDebugStream()
	{
		if (f)
			os::fclose(f);
	}

	size_t Seek(long Offset, CStream::TSeekOrigin Origin) {return 0;}
	size_t getTotalBytesCount() { return 0;}
	size_t getPosition() { return 0;}

}  myDebugStream("./DEBUG_log_streaming.txt");
*/

// ------------------------------------------------------
//				TestMapping
// ------------------------------------------------------
void Run_KF_SLAM()
{
	CRangeBearingKFSLAM  mapping;
	CConfigFile			 cfgFile( configFile );
	std::string			 rawlogFileName;

	//mapping.debugOut = &myDebugStream;

	// The rawlog file:
	// ----------------------------------------
	rawlogFileName = cfgFile.read_string("MappingApplication","rawlog_file",std::string("log.rawlog"));
	unsigned int	rawlog_offset = cfgFile.read_int("MappingApplication","rawlog_offset",0);

	unsigned int SAVE_LOG_FREQUENCY= cfgFile.read_int("MappingApplication","SAVE_LOG_FREQUENCY",1);

	bool  SAVE_3D_SCENES = cfgFile.read_bool("MappingApplication","SAVE_3D_SCENES", true);
	bool  SAVE_MAP_REPRESENTATIONS = cfgFile.read_bool("MappingApplication","SAVE_MAP_REPRESENTATIONS", true);

	string OUT_DIR = cfgFile.read_string("MappingApplication","logOutput_dir","OUT_KF-SLAM");
	string ground_truth_file = cfgFile.read_string("MappingApplication","ground_truth_file","");

	cout << "RAWLOG FILE:" << endl << rawlogFileName << endl;
	ASSERT_( fileExists( rawlogFileName ) );
	CFileGZInputStream	rawlogFile( rawlogFileName );

	cout << "---------------------------------------------------" << endl << endl;

	createDirectory(OUT_DIR);
	deleteFilesInDirectory(OUT_DIR);

	// Load the config options for mapping:
	// ----------------------------------------
	mapping.loadOptions( CConfigFile(configFile) );
	mapping.KF_options.dumpToConsole();
	mapping.options.dumpToConsole();

	//			INITIALIZATION
	// ----------------------------------------
	//mapping.initializeEmptyMap();

	// The main loop:
	// ---------------------------------------
	CActionCollectionPtr	action;
	CSensoryFramePtr		observations;
	size_t			rawlogEntry = 0, step = 0;
	deque<CPose3D>          meanPath; // The estimated path

	CPose3DPDFGaussian		robotPose;
	std::vector<CPoint3D>	LMs;
	std::map<unsigned int,CLandmark::TLandmarkID> LM_IDs;
	CMatrixDouble	fullCov;
	CVectorDouble           fullState;

	for (;;)
	{
		if (os::kbhit())
        {
			char	pushKey = os::getch();
			if (27 == pushKey)
				break;
        }


		// Load action/observation pair from the rawlog:
		// --------------------------------------------------
		if (! CRawlog::readActionObservationPair( rawlogFile, action, observations, rawlogEntry) )
			break; // file EOF

		if (rawlogEntry>=rawlog_offset)
		{
			// Process the action and observations:
			// --------------------------------------------
			mapping.processActionObservation(*action,*observations);

			// Get current state:
			// -------------------------------
			mapping.getCurrentState( robotPose,LMs,LM_IDs,fullState,fullCov );
			cout << "Mean pose: " << endl << robotPose.mean << endl;
			cout << "# of landmarks in the map: " << LMs.size() << endl;

			// Build the path:
			meanPath.push_back( robotPose.mean );

			// Save mean pose:
			if (!(step % SAVE_LOG_FREQUENCY))
			{
				CPose3D  robPose3D(robotPose.mean);
				CMatrix  robotPose( robPose3D );
				robotPose.saveToTextFile(OUT_DIR+format("/robot_pose_%05u.txt",(unsigned int)step));
			}

			// Save full cov:
			if (!(step % SAVE_LOG_FREQUENCY))
			{
				fullCov.saveToTextFile(OUT_DIR+format("/full_cov_%05u.txt",(unsigned int)step));
			}

			// Save map to file representations?
			if (SAVE_MAP_REPRESENTATIONS  && !(step % SAVE_LOG_FREQUENCY))
			{
				mapping.saveMapAndPath2DRepresentationAsMATLABFile( OUT_DIR+format("/slam_state_%05u.m",(unsigned int)step) );
			}

			// Save 3D view of the filter state:
			if (SAVE_3D_SCENES && !(step % SAVE_LOG_FREQUENCY) )
			{
				COpenGLScene   scene3D;
				{
					opengl::CGridPlaneXYPtr grid = opengl::CGridPlaneXY::Create(-1000,1000,-1000,1000,0,5);
					grid->m_color_R =
					grid->m_color_G =
					grid->m_color_B = 0.4f;
					scene3D.insert( grid );
				}

				// Robot path:
				{
					opengl::CSetOfLinesPtr linesPath = opengl::CSetOfLines::Create();
					linesPath->m_color_R = 1;
					linesPath->m_color_G = 0;
					linesPath->m_color_B = 0;

					double x0=0,y0=0,z0=0;

					if (!meanPath.empty())
					{
						x0 = meanPath[0].x;
						y0 = meanPath[0].y;
						z0 = meanPath[0].z;
					}

					for (deque<CPose3D>::iterator it=meanPath.begin();it!=meanPath.end();it++)
					{
						linesPath->appendLine(
							x0,y0,z0,
							it->x, it->y, it->z );
						x0=it->x;
						y0=it->y;
						z0=it->z;
					}
					scene3D.insert( linesPath );
				}

				{
					opengl::CSetOfObjectsPtr  objs = opengl::CSetOfObjects::Create();
					mapping.getAs3DObject(objs);
					scene3D.insert( objs );
				}

				// Save to file:
				CFileGZOutputStream(OUT_DIR+format("/kf_state_%05u.3Dscene",(unsigned int)step)) << scene3D;
			}


			// Free rawlog items memory:
			// --------------------------------------------
			action.clear_unique();
			observations.clear_unique();

		} // (rawlogEntry>=rawlog_offset)

		cout << format("\nStep %u  - Rawlog entries processed: %i\n", (unsigned int)step, (unsigned int)rawlogEntry);

        step++;
	};	// end "while(1)"


    // Compute the "information" between partitions:
    if (mapping.options.doPartitioningExperiment)
    {
		// --------------------------------------------
		// PART I:
		//  Comparison to fixed partitioning every K obs.
		// --------------------------------------------

        // Compute the information matrix:
        size_t i;
        for (i=0;i<6;i++) fullCov(i,i) = max(fullCov(i,i), 1e-6);

        CMatrix		H( fullCov.inv() );
        H.saveToTextFile(OUT_DIR+string("/information_matrix_final.txt"));

        // Replace by absolute values:
        H.Abs();
        CMatrix H2(H); H2.adjustRange();
        CMRPTImageFloat   imgF(H2);
        imgF.saveToFile(OUT_DIR+string("/information_matrix_final.png"));


        // ----------------------------------------
        // Compute the "approximation error factor" E:
        //  E = SUM() / SUM(ALL ELEMENTS IN MATRIX)
        // ----------------------------------------
        vector<vector_uint>  landmarksMembership,partsInObsSpace;
        CMatrix  ERRS(50,3);

        for (i=0;i<ERRS.getRowCount();i++)
        {
            size_t K;

            if (i==0)
            {
                K=0;
                mapping.getLastPartitionLandmarks( landmarksMembership );
            }
            else
            {
                K=i+1;
                mapping.getLastPartitionLandmarksAsIfFixedSubmaps(i+1,landmarksMembership);
            }

            mapping.getLastPartition(partsInObsSpace);

            ERRS(i,0) = (float)K;
            ERRS(i,1) = (float)partsInObsSpace.size();
            ERRS(i,2) = mapping.computeOffDiagonalBlocksApproximationError(landmarksMembership);
        }


        ERRS.saveToTextFile( OUT_DIR+string("/ERRORS.txt" ));
        //printf("Approximation error from partition:\n"); cout << ERRS << endl;

		// --------------------------------------------
		// PART II:
		//  Sweep partitioning threshold:
		// --------------------------------------------
		size_t STEPS = 50;
        CVectorFloat	ERRS_SWEEP(STEPS),ERRS_SWEEP_THRESHOLD(STEPS);

		// Compute the error for each partitioning-threshold
        for (i=0;i<STEPS;i++)
        {
			float th = (1.0f*i)/(STEPS-1.0f);
			ERRS_SWEEP_THRESHOLD[i] = th;
			mapping.mapPartitionOptions()->partitionThreshold =  th;

			mapping.reconsiderPartitionsNow();

			mapping.getLastPartitionLandmarks( landmarksMembership );
            ERRS_SWEEP[i] = mapping.computeOffDiagonalBlocksApproximationError(landmarksMembership);
        }

        ERRS_SWEEP.saveToTextFile( OUT_DIR+string("/ERRORS_SWEEP.txt" ));
        ERRS_SWEEP_THRESHOLD.saveToTextFile( OUT_DIR+string("/ERRORS_SWEEP_THRESHOLD.txt" ));

    } // end if doPartitioningExperiment


    // Is there ground truth??
    if (ground_truth_file.size() && fileExists(ground_truth_file))
    {
        CMatrixFloat    GT(0,0);
        try
        {
            GT.loadFromTextFile(ground_truth_file);
        }
        catch(std::exception &e)
        {
            cerr << "Ignoring the following error loading ground truth file: " << e.what() << endl;
        }

        if (GT.getRowCount()>0 && LMs.size())
        {
            // Each row has:
            //   [0] [1] [2]  [6]
            //    x   y   z    ID
            vector_double ERRS(0);
            for (size_t i=0;i<LMs.size();i++)
            {
                // Find the entry in the GT for this mapped LM:
                bool found = false;
                for (size_t r=0;r<GT.getRowCount();r++)
                {
                    if ( LM_IDs[i] == GT(r,6) )
                    {
                        ERRS.push_back( LMs[i].distance3DTo( GT(r,0),GT(r,1),GT(r,2) ) );
                        found = true;
                        break;
                    }
                }
                if (!found)
                {
                    cerr << "Ground truth entry not found for landmark ID:" << LM_IDs[i] << endl;
                }
            }

            printf("ERRORS VS. GROUND TRUTH:\n");
            printf("Mean Error: %f meters\n", math::mean(ERRS) );
            printf("Minimum error: %f meters\n", math::minimum(ERRS) );
            printf("Maximum error: %f meters\n", math::maximum(ERRS) );
        }

    } // end if GT

	cout << "********* KF-SLAM finished! **********" << endl;
}






