/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Proradmming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#ifndef CPtuDPerception_H
#define CPtuDPerception_H

#include <mrpt/hwdrivers/CSerialPort.h>
#include <mrpt/hwdrivers/CPtuBase.h>

namespace mrpt
{
	namespace hwdrivers
	{
		/** This class implements initialization and comunication methods to
		  * control a Pan and Tilt Unit model PTU-46-17.5, working in radians .
		  */
		class HWDLLIMPEXP CPtuDPerception : public CPtuBase
		{

		public:

			/** Default constructor */

			CPtuDPerception() {};

			/** Destructor */

			~CPtuDPerception(){ close(); }

		/*************************** Commands ***************************/

		public:

			/** Specification of positions in absolute terms */

			bool moveToAbsPos(char axis,double nRad);

			/** Query position in absolute terms */

			bool absPosQ(char axis,double &nRad);

			/** Specify desired axis position as an offset from the current position. \n
			*	This method recives the number of radians to move.
			*	\code
			*	Example of use:
			*		TT-500 *
			*		A *
			*		TO * Current Tilt position is -500
			*		TO500 *
			*		A *
			*		TT * Current Pan position is 1000 
			*	\endcode
			*/

			bool moveToOffPos(char axis,double nRad);

			/** Query position in relative terms */

			bool offPosQ(char axis,double &nRad);

			/** Query max movement limit of a axis in absolute terms */

			bool maxPosQ(char axis,double &nRad);

			/** Query min movement limit of a axis in absolute terms */

			bool minPosQ(char axis,double &nRad);

			/** Query if exist movement limits */

			bool enableLimitsQ(bool &enable); // Query if exist some limit

			/** Enable/Disable movement limits */

			bool enableLimits(bool set);

			/** With I mode (default) instructs pan-tilt unit to immediately
			*	execute positional commands. \n
			*	In S mode instructs pan-tilt unit to execute positional commands
			*	only when an Await Position Command Completion command is executed
			*	or when put into Immediate Execution Mode. \n
			*	\code
			*	Example of use of S mode:
			*		DR *
			*		S *
			*		PP1500 *
			*		TP-900 *
			*		PP * Current Pan position is 0
			*		TP * Current Tilt position is 0
			*		A *
			*		PP * Current Pan position is 1500
			*		TP * Current Tilt position is -900
			*	\endcode
			*/

			bool inmediateExecution(bool set);

			/** Wait the finish of the last position command to
			*	continue accept commands
			*/

			bool aWait(void);

			/** Inmediately stop all */

			bool haltAll();

			/** Inmediately stop */

			bool halt(char axis);

			/** Specification of turn speed */

			bool  speed(char axis,double radSec);

			/** Query turn speed */

			bool  speedQ(char axis,double &radSec);

			/** Specification (de/a)celeration in turn */

			bool  aceleration(char axis,double radSec2);

			/** Query (de/a)celeration in turn */

			bool  acelerationQ(char axis,double &radSec2);

			/** Specification of velocity to which start and finish
			*	the (de/a)celeration
			*/

			bool  baseSpeed(char axis,double radSec);

			/** Query velocity to which start and finish
			*	the (de/a)celeration
			*/

			bool  baseSpeedQ(char axis,double &radSec);

			/** Specification of velocity upper limit */

			bool upperSpeed(char axis,double radSec);

			/** Query velocity upper limit */

			bool upperSpeedQ(char axis,double &radSec);

			/** Specification of velocity lower limit */

			bool lowerSpeed(char axis,double radSec);

			/** Query velocity lower limit */

			bool lowerSpeedQ(char axis,double &radSec);

			/** Reset PTU to initial state */

			bool reset(void);

			/** Save or restart default values */

			bool save(void);

			/** Restore default values */

			bool restoreDefaults(void);

			/** Restore factory default values */

			bool restoreFactoryDefaults(void);

			/** Version and CopyRights */

			bool version(char * nVersion);

			/** Query power mode */

			bool powerModeQ(bool transit,char &mode);

			/** Specification of power mode */

			bool powerMode(bool transit,char mode);


		/**************************** State Queries ********************/
			
			/** Check errors, returns 0 if there are not errors or error code in otherwise
			*	Error codes:
			*	\code
			*	1: Com error
			*	2: Time out error
			*	3: Init error
			*	4: Pan tilt hit error
			*	5: Pan hit error
			*	6: Tilt hit error
			*	7: Max limit error
			*	8: Min limit error
			*	9: Out of range
			*	10: Illegal command error
			*	11: Unexpected error
			**/

			int checkErrors();

			inline bool noError() { return nError==1; }
			inline bool comError() { return (nError % CPtuDPerception::ComError)==0; }
			inline bool timeoutError() { return (nError % CPtuDPerception::TimeoutError)==0; }
			inline bool initError() { return (nError % CPtuDPerception::InitError)==0; }
			inline bool panTiltHitError() { return (nError % CPtuDPerception::PanTiltHitError)==0; }
			inline bool panHitError() { return (nError % CPtuDPerception::PanHitError)==0; }
			inline bool tiltHitError() { return (nError % CPtuDPerception::TiltHitError)==0; }
			inline bool maxLimitError() { return (nError % CPtuDPerception::MaxLimitError)==0; }
			inline bool minLimitError () { return (nError % CPtuDPerception::MinLimitError)==0; }
			inline bool outOfRange() { return (nError % CPtuDPerception::OutOfRange)==0; }
			inline bool illegalCommandError() { return (nError % CPtuDPerception::IllegalCommandError)==0; }
			inline bool unExpectedError() { return (nError % CPtuDPerception::UnExpectedError)==0; }
			
			/** Clear errors **/

			inline void clearErrors() { nError=NoError; }


		/*************************** Other member methods *****************/

		public:

			/** PTU and serial port initialization */

			bool init(const std::string port);

			/** Close conection with serial port */

			void close();

			/** To obtains the mistake for use discrete values when the movement
			*	is expressed in radians. Parameters are the absolute position in
			*	radians and the axis desired
			*/

			double radError(char axis,double nRadMoved);

			/**  To obtain the discrete value for a number of radians */

			long radToPos(char axis,double nRad);

			/** To obtain the number of radians for a discrete value */

			double posToRad(char axis,long nPos);

			/** Performs a scan in the axis indicated and whit the precision desired. \n
			*		\param <axis> {Pan or Till} \n
			*		\param <tWait> {Wait time betwen commands} \n
			*		\param <initial> {initial position}
			*		\param <final> {final position}
			*		\param <radPre> {radians precision for the scan}
			*/

			bool scan(char axis, int wait, float initial, float final, double radPre);
						
			/** Query verbose mode */

			bool verboseQ(bool &modo);

			/** Set verbose. \n
			*	\conde
			*	Example of response with FV (verbose) active:
			*		FV *
			*		PP * Current pan position is 0
			*		Example of response with FT (terse) active:
			*		FT *
			*		PP * 0
			*	\endcode
			*/

			bool verbose(bool set);

			/** Query echo mode */

			bool echoModeQ(bool &mode);

			/** Enable/Disable echo response with command. \n
			*	\code
			*	Example of use (EE supposed):
			*		PP * 22
			*		ED *
			*		<pp entered again, but not echoed>* 22
			*	\endcode
			*/

			bool echoMode(bool mode);

			/** Query the pan and tilt resolution per position moved
			*	and initialize local atributes
			*/

			bool resolution(void);


		/*************************** Methods for internal use ****************/

		private:

			/** To transmition commands to the PTU */

			bool transmit(const char * command);

			/** To receive the responseof the PTU */

			bool receive(const char * command,char ** response);

			/** Used to obtains a number of radians */

			bool radQuerry(char axis,char command,double &nRad);

			/** Method used for asign a number of radians with a command */

			bool radAsign(char axis,char command,double nRad);

			/** Convert string to double */

			double convertToDouble(char *sDouble);

			/** Convert string to long */

			long convertToLong(char *sLong);

		/**************************** Atributes ********************/

		public:

			enum { NoError = 1, ComError = 2, TimeoutError = 3,
						InitError = 5,PanHitError = 7, TiltHitError = 11, PanTiltHitError=13,
						MaxLimitError = 17, MinLimitError = 19, OutOfRange = 23,
						IllegalCommandError = 29, UnExpectedError =31 };

			/** TimeoutError: Only occurs if the communication is cut with PTU
			*		so it is advisable to check the connection and initialize
			*		again the comunication.
			*/

			int nError;

			enum { Pan = 'P', Tilt = 'T' };
			enum { Regular = 'R', High = 'H', Low = 'L', Off = 'O' };
			enum { Com1 = 1, Com2 = 2, Com3 = 3, Com4 = 4 };


		};	// End of class

	} // End of namespace

} // End of namespace

#endif
