/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef opengl_CRenderizable_H
#define opengl_CRenderizable_H

#include <mrpt/utils/utils_defs.h>
#include <mrpt/utils/CSerializable.h>

#include <mrpt/synch/CCriticalSection.h>

namespace mrpt
{
	namespace poses { class CPose3D; }
	namespace utils { class CStringList; }

	namespace opengl
	{
		class COpenGLViewport;

		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE_PRE( CRenderizable )

		/** The base class of 3D objects that can be directly rendered through OpenGL.
		  *  In this class there are a set of common properties to all 3D objects, mainly:
		  *		- A name (m_name): A name that can be optionally asigned to objects for easing its reference.
		  *		- 6D coordinates (x,y,z,yaw,pitch,roll), relative to the "current" reference framework. By default, any object is referenced to global scene coordinates.
		  *		- A RGB color: This field will be used in simple elements (points, lines, text,...) but is ignored in more complex objects that carry their own color information (triangle sets,...)
		  *  See the main class opengl::COpenGLScene
		  *  \sa opengl::COpenGLScene, mrpt::opengl
		  */
		class MRPTDLLIMPEXP CRenderizable : public mrpt::utils::CSerializable
		{
			DEFINE_VIRTUAL_SERIALIZABLE( CRenderizable )

			friend class mrpt::opengl::COpenGLViewport;

 		public:
			std::string				m_name;
			double					m_color_R,m_color_G,m_color_B,m_color_A;    //!< Color components in the range [0,1]
			double					m_x,m_y,m_z;								//!< Translation relative to parent coordinate origin.
			double					m_yaw,m_pitch,m_roll;						//!< Rotation relative to parent coordinate origin, in **DEGREES**.
			bool					m_show_name;

			static void	renderTextBitmap( const char *str, void *fontStyle );

			/** Default constructor:
			  */
			CRenderizable() :
				m_name(),
				m_color_R(1),m_color_G(1),m_color_B(1),m_color_A(1),
				m_x(0),m_y(0),m_z(0),
				m_yaw(0),m_pitch(0),m_roll(0),
				m_show_name(false)
			{
			}

			virtual ~CRenderizable() { }

			/** Interface for the stlplus smart pointer class. */
			CRenderizable * clone() const
			{
				return static_cast<CRenderizable*>( this->duplicate() );
			}

			/** This virtual method in the base class performs common tasks like coordinates transformation,color,...
			  */
			virtual void  render() = 0;

			/** Set the 3D pose from a mrpt::poses::CPose3D object */
			void setPose( const mrpt::poses::CPose3D &o );

		protected:
			/** Checks glGetError and throws an exception if an error situation is found */
			static void checkOpenGLError();

			void  writeToStreamRender(utils::CStream &out) const;
			void  readFromStreamRender(utils::CStream &in);

			/** Returns the lowest, free texture name.  */
			static unsigned int getNewTextureNumber();
			static void releaseTextureName(unsigned int i);

		};



	} // end namespace

} // End of namespace


#endif
