/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CPOINT3D_H
#define CPOINT3D_H

#include <mrpt/poses/CPoint.h>

namespace mrpt
{
namespace poses
{
    class CPose3D;

	DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE( CPoint3D, CPoint )

	/** A class used to store a 3D point.
	 *
	 *  For a complete description of Points/Poses, see mrpt::poses::CPoseOrPoint, or refer
	 *    to the <a href="http://babel.isa.uma.es/mrpt/index.php/2D_3D_Geometry">2D/3D Geometry tutorial</a> in the wiki.
	 *

<div align=center>

<table class=MsoTableGrid border=1 cellspacing=0 cellpadding=0
 style='border-collapse:collapse;border:none'>
 <tr>
  <td width=576 colspan=2 style='width:432.2pt;border:solid windowtext 1.0pt;
  background:#E6E6E6;padding:0cm 5.4pt 0cm 5.4pt'>
  <p   align=center style='text-align:center'>poses::CPoint3D</p>
  </td>
 </tr>
 <tr>
  <td width=288 style='width:216.1pt;border:solid windowtext 1.0pt;border-top:
  none;padding:0cm 5.4pt 0cm 5.4pt'>
  <p   align=center style='text-align:center'>Homogeneous
  transfomation matrix</p>
  </td>
  <td width=288 style='width:216.1pt;border-top:none;border-left:none;
  border-bottom:solid windowtext 1.0pt;border-right:solid windowtext 1.0pt;
  padding:0cm 5.4pt 0cm 5.4pt'>
  <p   align=center style='text-align:center'>Spatial
  representation</p>
  </td>
 </tr>
 <tr style='height:108.3pt'>
  <td width=288 style='width:216.1pt;border:solid windowtext 1.0pt;border-top:
  none;padding:0cm 5.4pt 0cm 5.4pt;height:108.3pt'>
  <div align=center>
  <table  Table border=0 cellspacing=0 cellpadding=0 width="46%"
   style='width:46.84%;border-collapse:collapse'>
   <tr style='height:16.5pt'>
    <td width=32 style='width:24.0pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>1</p>
    </td>
    <td width=32 style='width:24.05pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>0</p>
    </td>
    <td width=32 style='width:24.05pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>0</p>
    </td>
    <td width=32 style='width:24.05pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>x</p>
    </td>
   </tr>
   <tr style='height:16.5pt'>
    <td width=32 style='width:24.0pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>0</p>
    </td>
    <td width=32 style='width:24.05pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>1</p>
    </td>
    <td width=32 style='width:24.05pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>0</p>
    </td>
    <td width=32 style='width:24.05pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>y</p>
    </td>
   </tr>
   <tr style='height:16.5pt'>
    <td width=32 style='width:24.0pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>0</p>
    </td>
    <td width=32 style='width:24.05pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>0</p>
    </td>
    <td width=32 style='width:24.05pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>1</p>
    </td>
    <td width=32 style='width:24.05pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>z</p>
    </td>
   </tr>
   <tr style='height:16.5pt'>
    <td width=32 style='width:24.0pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>0</p>
    </td>
    <td width=32 style='width:24.05pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>0</p>
    </td>
    <td width=32 style='width:24.05pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>0</p>
    </td>
    <td width=32 style='width:24.05pt;padding:0cm 5.4pt 0cm 5.4pt;height:16.5pt'>
    <p   align=center style='text-align:center'>1</p>
    </td>
   </tr>
  </table>
  </div>
  <p   align=center style='text-align:center'></p>
  </td>
  <td width=288 style='width:216.1pt;border-top:none;border-left:none;
  border-bottom:solid windowtext 1.0pt;border-right:solid windowtext 1.0pt;
  padding:0cm 5.4pt 0cm 5.4pt;height:108.3pt'>
  <p   align=center style='text-align:center'><img src="CPoint3D.gif"></p>
  </td>
 </tr>
</table>

</div>


	  * \sa CPoseOrPoint,CPose, CPoint
	 */
	class MRPTDLLIMPEXP  CPoint3D : public CPoint
	{
		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CPoint3D )
	 protected:
		mutable math::CMatrixDouble	aux_HM;
	 public:
		 /** Constructor for initializing point coordinates.
		  */
		 CPoint3D(const double & x=0,const double & y=0,const double & z=0);

		 /** Constructor from an CPoint2D object.
		  */
		 CPoint3D( const CPoint2D &);

		 /** Constructor from an CPose3D object.
		  */
		 CPoint3D( const CPose3D &);

		 /** Constructor from an CPose2D object.
		  */
		 CPoint3D( const CPose2D &);

		 /** Returns this point as seen from "b", i.e. result = this - b
		   */
		 CPoint3D  operator - (const CPose3D& b) const;

		 /** Returns this point minus point "b", i.e. result = this - b
		   */
		 CPoint3D  operator - (const CPoint3D& b) const;

		 /** Returns this point plus point "b", i.e. result = this + b
		   */
		 CPoint3D  operator + (const CPoint3D& b) const;

		 /** Returns this point plus pose "b", i.e. result = this + b
		   */
		 CPose3D	operator + (const CPose3D& b) const;

		 /** Returns the corresponding 4x4 homogeneous
		   *  transformation matrix for the point(translation),
		   *  or pose (translation+orientation).
		   * \sa getInverseHomogeneousMatrix
		   */
		 const math::CMatrixDouble& getHomogeneousMatrix() const;

		 /** Returns the corresponding 4x4 homogeneous
		   *  transformation matrix for the point(translation),
		   *  or pose (translation+orientation).
		   * \sa getInverseHomogeneousMatrix
		   */
		 void  getHomogeneousMatrix(const math::CMatrixDouble *& ptrHM ) const;

	}; // End of class def.


	std::ostream MRPTDLLIMPEXP & operator << (std::ostream& o, const CPoint3D& p);

	} // End of namespace
} // End of namespace

#endif
