/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.



#include <mrpt/config.h>

#include <mrpt/gui/CDisplayWindow.h>
#include <mrpt/gui/CDisplayWindow3D.h>
#include <mrpt/gui/CDisplayWindowPlots.h>

#include <mrpt/system/os.h>


#include <mrpt/gui/WxSubsystem.h>

#if MRPT_HAS_WXWIDGETS


using namespace mrpt;
using namespace mrpt::gui;
using namespace mrpt::utils;
using namespace mrpt::synch;
using namespace std;


//std::map<wxFrame*,CDisplayWindow*>		WxSubsystem::m_windowsList;
//synch::CCriticalSection				    WxSubsystem::m_csWindowsList;

//std::map<wxFrame*,CDisplayWindow3D*>	WxSubsystem::m_3DwindowsList;
//synch::CCriticalSection				    WxSubsystem::m_cs3DWindowsList;

synch::CCriticalSection  	WxSubsystem::CWXMainFrame::cs_windowCount;
int                       	WxSubsystem::CWXMainFrame::m_windowCount = 0;

std::queue<WxSubsystem::TRequestToWxMainThread*>  WxSubsystem::listPendingWxRequests;
synch::CCriticalSection              WxSubsystem::cs_listPendingWxRequests;


WxSubsystem::CWXMainFrame* WxSubsystem::CWXMainFrame::oneInstance = NULL;
bool WxSubsystem::isConsoleApp = true;

// ---------------------------------------------------------------------------------------
// The wx dummy frame:
// ---------------------------------------------------------------------------------------
BEGIN_EVENT_TABLE(WxSubsystem::CWXMainFrame,wxFrame)

END_EVENT_TABLE()


const long ID_TIMER_WX_PROCESS_REQUESTS = wxNewId();


WxSubsystem::CWXMainFrame::CWXMainFrame(wxWindow* parent,wxWindowID id)
{
    Create(
        parent,
        id,
        _("MRPT-dummy frame window"),
        wxDefaultPosition,
        wxSize(1,1),
        0, //wxDEFAULT_FRAME_STYLE,
        _T("id"));

	if (oneInstance)
	{
		cerr << "[CWXMainFrame] More than one instance running!" << endl;
	}
	oneInstance = this;

    // ------------------------------------------------------------------------------------------
    // Create a timer so requests from the main application thread can be processed regularly:
    // ------------------------------------------------------------------------------------------
    Connect(ID_TIMER_WX_PROCESS_REQUESTS, wxEVT_TIMER ,(wxObjectEventFunction)&CWXMainFrame::OnTimerProcessRequests);
    m_theTimer =  new wxTimer(this, ID_TIMER_WX_PROCESS_REQUESTS );

    m_theTimer->Start( 10, true );  // One-shot
}

WxSubsystem::CWXMainFrame::~CWXMainFrame()
{
//	cout << "[CWXMainFrame] Destructor." << endl;
	delete m_theTimer;
	oneInstance=NULL;

}

int WxSubsystem::CWXMainFrame::notifyWindowCreation()
{
	CCriticalSectionLocker	lock(&cs_windowCount);
    return ++m_windowCount;
}

int WxSubsystem::CWXMainFrame::notifyWindowDestruction()
{
	CCriticalSectionLocker	lock(&cs_windowCount);
    return --m_windowCount;
}

/** Thread-safe method to return the next pending request, or NULL if there is none (After usage, FREE the memory!)
  */
WxSubsystem::TRequestToWxMainThread * WxSubsystem::popPendingWxRequest()
{
	synch::CCriticalSectionLocker	locker( &cs_listPendingWxRequests );

    // Is empty?
    if (!listPendingWxRequests.size())
        return NULL;

    TRequestToWxMainThread *ret=listPendingWxRequests.front();
    listPendingWxRequests.pop(); // Remove from the queue

    return ret;
}

/** Thread-safe method to insert a new pending request (The memory must be dinamically allocated with "new T[1]", will be freed by receiver.)
  */
void WxSubsystem::pushPendingWxRequest( WxSubsystem::TRequestToWxMainThread *data )
{
    CCriticalSectionLocker locker(&cs_listPendingWxRequests);
    listPendingWxRequests.push( data );
}


/** This method processes the pending requests from the main MRPT application thread.
  *  The requests may be to create a new window, close another one, change title, etc...
  */
void WxSubsystem::CWXMainFrame::OnTimerProcessRequests(wxTimerEvent& event)
{
    try
    {
        TRequestToWxMainThread *msg;

		//cout << "[CWXMainFrame] Entering" << endl;

        // For each pending request:
        while ( NULL != (msg = popPendingWxRequest() ) )
        {
            //cout << "[CWXMainFrame] OPCODE: " << msg->OPCODE << endl;
            // Process it:
            if (msg->source2D )
            {
                // Is a 2D window request:
                switch (msg->OPCODE)
                {
                    // CREATE NEW WINDOW
                    case 200:
                    {
                        CWindowDialog  *wnd = new CWindowDialog(msg->source2D, this, (wxWindowID) -1, msg->str );

                        // Set the "m_hwnd" member of the window:
                        * ((void**)msg->voidPtr) = (void*)wnd;

                        // Signal to the constructor (still waiting) that the window is now ready so it can continue:
                        msg->source2D->m_semThreadReady.release();

                        wnd->Show();
                    }
                    break;
                    // UPDATE IMAGE
                    case 201:
                    {
                        CWindowDialog  *wnd = (CWindowDialog*) msg->voidPtr; //msg->source2D->getWxObject();
                        wxImage        *img = (wxImage*) msg->voidPtr2;

                        wnd->m_image->SetBitmap( wxBitmap( *img ) );
                        wnd->Fit();

                        wnd->m_image->Refresh();

                        delete img;
                    }
                    break;
                    // Set position
                    case 202:
                    {
                        CWindowDialog  *wnd = (CWindowDialog*) msg->source2D->getWxObject();
                        if (wnd)
                            wnd->SetSize( msg->x, msg->y, wxDefaultCoord, wxDefaultCoord );
                    }
                    break;
                    // Set size
                    case 203:
                    {
                        CWindowDialog  *wnd = (CWindowDialog*) msg->source2D->getWxObject();
                        if (wnd)
                            wnd->SetClientSize( msg->x, msg->y );
                    }
                    break;
                    // Set window's title:
                    case 204:
                    {
                        CWindowDialog  *wnd = (CWindowDialog*) msg->source2D->getWxObject();
                        if (wnd)
							wnd->SetLabel( _U(msg->str.c_str()) );
                    }
                    break;
                    // DESTROY EXISTING WINDOW:
                    case 299:
                    {
                        CWindowDialog  *wnd = (CWindowDialog*) msg->source2D->getWxObject();
                        if (wnd)
                        {
                            //delete wnd;
                            wnd->Close();
                        }
                    }
                    break;
                }

            }
            else
            if (msg->source3D )
            {
                // Is a 3D window request:
                switch (msg->OPCODE)
                {
                    // CREATE NEW WINDOW
                    case 300:
                    {
                        C3DWindowDialog  *wnd = new C3DWindowDialog(msg->source3D, this, (wxWindowID) -1, msg->str );

                        // Set the "m_hwnd" member of the window:
                        * ((void**)msg->voidPtr) = (void*)wnd;

                        // Signal to the constructor (still waiting) that the window is now ready so it can continue:
                        msg->source3D->m_semThreadReady.release();

                        wnd->Show();
                    }
                    break;
                    // Set position
                    case 302:
                    {
                        C3DWindowDialog  *wnd = (C3DWindowDialog*) msg->source3D->getWxObject();
                        if (wnd)
                            wnd->SetSize( msg->x, msg->y, wxDefaultCoord, wxDefaultCoord );
                    }
                    break;
                    // Set size
                    case 303:
                    {
                        C3DWindowDialog  *wnd = (C3DWindowDialog*) msg->source3D->getWxObject();
                        if (wnd)
                            wnd->SetClientSize( msg->x, msg->y );
                    }
                    break;
                    // Set window's title:
                    case 304:
                    {
                        C3DWindowDialog  *wnd = (C3DWindowDialog*) msg->source3D->getWxObject();
                        if (wnd)
							wnd->SetLabel( _U(msg->str.c_str()) );
                    }
                    break;
                    // FORCE REPAINT
                    case 350:
                    {
                        C3DWindowDialog  *wnd = (C3DWindowDialog*) msg->source3D->getWxObject();
                        if (wnd)
                        {
                            wnd->Refresh(false);
                        }
                    }
                    break;

                    // DESTROY EXISTING WINDOW:
                    case 399:
                    {
                        C3DWindowDialog  *wnd = (C3DWindowDialog*) msg->source3D->getWxObject();
                        if (wnd)
                        {
                            //delete wnd;
                            wnd->Close();
                        }
                    }
                    break;
                }

            }
            else
            if (msg->sourcePlots )
            {
                // Is a Plots window request:
                switch (msg->OPCODE)
                {
                    // CREATE NEW WINDOW
                    case 400:
                    {
                        CWindowDialogPlots  *wnd = new CWindowDialogPlots(msg->sourcePlots, this, (wxWindowID) -1, msg->str, wxSize(msg->x,msg->y) );

                        // Set the "m_hwnd" member of the window:
                        * ((void**)msg->voidPtr) = (void*)wnd;

                        // Signal to the constructor (still waiting) that the window is now ready so it can continue:
                        msg->sourcePlots->m_semThreadReady.release();

                        wnd->Show();
                    }
                    break;
                    // Set position
                    case 402:
                    {
                    	CWindowDialogPlots *wnd = (CWindowDialogPlots*) msg->sourcePlots->getWxObject();
                        if (wnd)
                            wnd->SetSize( msg->x, msg->y, wxDefaultCoord, wxDefaultCoord );
                    }
                    break;
                    // Set size
                    case 403:
                    {
                    	CWindowDialogPlots *wnd = (CWindowDialogPlots*) msg->sourcePlots->getWxObject();
                        if (wnd)
                            wnd->SetClientSize( msg->x, msg->y );
                    }
                    break;
                    // Set window's title:
                    case 404:
                    {
                    	CWindowDialogPlots *wnd = (CWindowDialogPlots*) msg->sourcePlots->getWxObject();
                        if (wnd)
							wnd->SetLabel( _U(msg->str.c_str()) );
                    }
                    break;
                    // Mouse pan
                    case 410:
                    {
                    	CWindowDialogPlots *wnd = (CWindowDialogPlots*) msg->sourcePlots->getWxObject();
                        if (wnd)
							wnd->m_plot->EnableMousePanZoom(msg->boolVal);
                    }
                    break;
                    // Aspect ratio
                    case 411:
                    {
                    	CWindowDialogPlots *wnd = (CWindowDialogPlots*) msg->sourcePlots->getWxObject();
                        if (wnd)
							wnd->m_plot->LockAspect(msg->boolVal);
                    }
                    break;

                    // Zoom over a rectangle vectorx[0-1] & vectory[0-1]
                    case 412:
                    {
                    	CWindowDialogPlots *wnd = (CWindowDialogPlots*) msg->sourcePlots->getWxObject();
                        if (wnd)
                        {
                        	if (msg->vector_x.size()==2 &&  msg->vector_y.size()==2)
                        	{
								wnd->m_plot->Fit( msg->vector_x[0],msg->vector_x[1],msg->vector_y[0],msg->vector_y[1] );
								wnd->m_plot->LockAspect(msg->boolVal);
                        	}
                        }
                    }
                    break;
                    // Axis fit, with aspect ratio fix to boolVal.
                    case 413:
                    {
                    	CWindowDialogPlots *wnd = (CWindowDialogPlots*) msg->sourcePlots->getWxObject();
                        if (wnd)
						{
							wnd->m_plot->LockAspect(msg->boolVal);
							wnd->m_plot->Fit();
						}
                    }
                    break;

                    // Create/modify 2D plot
                    case 420:
                    {
                    	CWindowDialogPlots *wnd = (CWindowDialogPlots*) msg->sourcePlots->getWxObject();
                        if (wnd)
							wnd->plot( msg->vector_x, msg->vector_y, msg->str, msg->plotName );
                    }
                    break;

                    // Create/modify 2D ellipse
                    case 421:
                    {
                    	CWindowDialogPlots *wnd = (CWindowDialogPlots*) msg->sourcePlots->getWxObject();
                        if (wnd)
							wnd->plotEllipse( msg->vector_x, msg->vector_y, msg->str, msg->plotName );
                    }
                    break;

                    // Create/modify bitmap image
                    case 422:
                    {
                    	CWindowDialogPlots *wnd = (CWindowDialogPlots*) msg->sourcePlots->getWxObject();
                        if (wnd)
							wnd->image(
								msg->voidPtr2,
								msg->vector_x[0],
								msg->vector_x[1],
								msg->vector_x[2],
								msg->vector_x[3],
								msg->plotName );
                    }
                    break;

                    // DESTROY EXISTING WINDOW:
                    case 499:
                    {
                    	CWindowDialogPlots *wnd = (CWindowDialogPlots*) msg->sourcePlots->getWxObject();
                        if (wnd)
                        {
                            //delete wnd;
                            wnd->Close();
                        }
                    }
                    break;
                }

            }

            // Free the memory:
            delete[] msg;
        }
    }
    catch(...)
    {
    }

    m_theTimer->Start( 10, true );  // One-shot
}


// ---------------------------------------------------------------------------------------
// The wx app:
// ---------------------------------------------------------------------------------------

// Init. of static member must be here:
synch::CSemaphore       WxSubsystem::m_semWxMainThreadReady(0,1);
TThreadHandle            WxSubsystem::m_wxMainThreadId;
synch::CCriticalSection WxSubsystem::m_csWxMainThreadId;


class CDisplayWindow_WXAPP : public wxApp
{
    public:
        virtual bool OnInit();
		virtual int OnExit();
};

bool CDisplayWindow_WXAPP::OnInit()
{
    wxInitAllImageHandlers();

    //cout << "[wxApp::OnInit] wxApplication OnInit called." << endl;

    // Create a dummy frame:
    WxSubsystem::CWXMainFrame* Frame = new WxSubsystem::CWXMainFrame(0);
    //Frame->Show();
    //SetTopWindow(Frame);
    Frame->Hide();

    // We are ready!!
    //cout << "[wxMainThread] Signaling semaphore." << endl;
    WxSubsystem::m_semWxMainThreadReady.release();

    return true;
}

// This will be called when all the windows / frames are closed.
int CDisplayWindow_WXAPP::OnExit()
{
	CCriticalSectionLocker	lock(& WxSubsystem::m_csWxMainThreadId );

    //cout << "[wxApp::OnExit] wxApplication OnExit called." << endl;

    //cout << "[wxApp::OnExit] Calling wx cleanup." << endl;
    wxApp::OnExit();
    CleanUp();

	WxSubsystem::m_wxMainThreadId.clear();

    return 0;
}


/** This method must be called in the destructor of the user class FROM THE MAIN THREAD, in order to wait for the shutdown of the wx thread if this was the last open window.
  */
void WxSubsystem::waitWxShutdownsIfNoWindows()
{
	// Any open windows?
	int nOpenWnds;
	{
		CCriticalSectionLocker	locker(&CWXMainFrame::cs_windowCount);
		nOpenWnds = CWXMainFrame::m_windowCount;
	}

    if (!nOpenWnds && WxSubsystem::isConsoleApp)
    {
    	// Then we must be shutting down in the wx thread (we are in the main MRPT application thread)...
    	// Wait until wx is safely shut down:
    	bool done=false;
    	int timeout = 5000;
    	while (!done && timeout-->0)
    	{
    		system::sleep(10);
			WxSubsystem::m_csWxMainThreadId.enter();
			done = WxSubsystem::m_wxMainThreadId.isClear();
			WxSubsystem::m_csWxMainThreadId.leave();
    	}

    	if (!timeout)
    	{
    		cerr << "[WxSubsystem::waitWxShutdownsIfNoWindows] Timeout waiting for WxWidgets thread to shutdown!" << endl;
    	}
    }
}

//#define WXSUBSYSTEM_APP_IS_CONSOLE

//IMPLEMENT_APP_NO_MAIN(CDisplayWindow_WXAPP)
//IMPLEMENT_WXWIN_MAIN   <-- Call ::wxEntry manually instead!!

wxAppConsole *mrpt_wxCreateApp()
{
	wxAppConsole::CheckBuildOptions(WX_BUILD_OPTIONS_SIGNATURE, "your program");
	return new CDisplayWindow_WXAPP;
}
//wxAppInitializer mrpt_wxTheAppInitializer((wxAppInitializerFunction) mrpt_wxCreateApp);


//DECLARE_APP(CDisplayWindow_WXAPP)
extern CDisplayWindow_WXAPP& wxGetApp();

//IMPLEMENT_WX_THEME_SUPPORT


// Aux. funcs used in WxSubsystem::wxMainThread
// --------------------------------------------------
//int mrpt_wxEntryReal(int& argc, wxChar **argv)
int mrpt_wxEntryReal(int& argc, char **argv)
{
    // library initialization
    if ( !wxEntryStart(argc, argv) )
    {
#if wxUSE_LOG
        // flush any log messages explaining why we failed
        delete wxLog::SetActiveTarget(NULL);
#endif
        return -1;
    }

    // if wxEntryStart succeeded, we must call wxEntryCleanup even if the code
    // below returns or throws
    //wxCleanupOnExit cleanupOnExit;
    //WX_SUPPRESS_UNUSED_WARN(cleanupOnExit);

    try
    {

        // app initialization
        if ( ! wxTheApp->CallOnInit() )
        {
            // don't call OnExit() if OnInit() failed
            return -1;
        }

        // ensure that OnExit() is called if OnInit() had succeeded
        /*class CallOnExit
        {
        public:
            ~CallOnExit() { wxTheApp->OnExit(); }
        } callOnExit;
        */
        //WX_SUPPRESS_UNUSED_WARN(callOnExit);

        // app execution
        int ret = wxTheApp->OnRun();
        wxTheApp->OnExit();  // This replaces the above callOnExit class
        return ret;
    }
    catch(...)
	{
		wxTheApp->OnUnhandledException();
		wxEntryCleanup();
		return -1;
	}
	wxEntryCleanup();
}


//int mrpt_wxEntry(int& argc, wxChar **argv)
int mrpt_wxEntry(int& argc, char **argv)
{
    //DisableAutomaticSETranslator();

    try
    {
        return mrpt_wxEntryReal(argc, argv);
    }
	catch(...)
	{
		return -1;
	}
}

/*---------------------------------------------------------------
					wxMainThread
 This will be the "MAIN" of wxWidgets: It starts an application
   object and does not end until all the windows are closed.
 Only for console apps, not for user GUI apps already with wx.
 ---------------------------------------------------------------*/
void WxSubsystem::wxMainThread(void *)
{
	MRPT_TRY_START;

    // Prepare wxWidgets:
    int argc=1;
    char **argv = new char* [2];
    argv[0]=strdup("./MRPT");
    argv[1]=NULL;

    //cout << "[wxMainThread] Starting..." << endl;

	// Are we in a console or wxGUI application????
	wxAppConsole *app_con = wxApp::GetInstance();
	if (!app_con)
	{
		//cout << "[wxMainThread] I am in a console app" << endl;
		// We are NOT in a wx application (console):
		//  Start a new wx application object:

		//::wxEntry(argc, argv);
		// JLBC OCT2008: wxWidgets little hack to enable console/gui mixed applications:
		wxApp::SetInitializerFunction( (wxAppInitializerFunction) mrpt_wxCreateApp );
		mrpt_wxEntry(argc,argv);
	}
	else
	{
		// We are ALREADY in a wx application:
		//cout << "[wxMainThread] I am in a GUI app" << endl;

		wxWindow *topWin = static_cast<wxApp*>(app_con)->GetTopWindow();

		WxSubsystem::CWXMainFrame* Frame = new WxSubsystem::CWXMainFrame( topWin );
		//Frame->Show();
		//SetTopWindow(Frame);
		Frame->Hide();

		// We are ready!!
		//cout << "[wxMainThread] Signaling semaphore." << endl;
		WxSubsystem::m_semWxMainThreadReady.release();
	}

    //cout << "[wxMainThread] Finished" << endl;

    free(argv[0]);
    delete[]argv;

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
					createOneInstanceMainThread
 ---------------------------------------------------------------*/
bool WxSubsystem::createOneInstanceMainThread()
{
	CCriticalSectionLocker	lock(&m_csWxMainThreadId);

	wxAppConsole *app_con = wxApp::GetInstance();
	if (app_con && m_wxMainThreadId.isClear())
	{
		// We are NOT in a console application: There is already a wxApp instance running and it's not us.
		WxSubsystem::isConsoleApp = false;
		//cout << "[createOneInstanceMainThread] Mode: User GUI." << endl;
		if (!WxSubsystem::CWXMainFrame::oneInstance)
		{
			// We are ALREADY in a wx application:
			wxWindow *topWin = static_cast<wxApp*>(app_con)->GetTopWindow();

			WxSubsystem::CWXMainFrame* Frame = new WxSubsystem::CWXMainFrame( topWin );
			//Frame->Show();
			//SetTopWindow(Frame);
			Frame->Hide();

			// We are ready!!
		}
	}
	else
	{
		//cout << "[createOneInstanceMainThread] Mode: Console." << endl;
		WxSubsystem::isConsoleApp = true;
		if (m_wxMainThreadId.isClear())
		{
			// Create a thread for message processing there:
			m_wxMainThreadId = system::createThread( wxMainThread, NULL );

			if(! m_semWxMainThreadReady.waitForSignal(5000) )  // 2 secs should be enough...
			{
				cerr << "[WxSubsystem::createOneInstanceMainThread] Timeout waiting wxApplication to start up!" << endl;
				return false;
			}
		}
	}

	return true; // OK
}


#endif // MRPT_HAS_WXWIDGETS
