/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.



#include <mrpt/slam/CMetricMapBuilderRBPF.h>
#include <mrpt/slam/CObservationStereoImages.h>
#include <mrpt/utils/CEnhancedMetaFile.h>
#include <mrpt/math/utils.h>

using namespace mrpt;
using namespace mrpt::slam;
using namespace mrpt::utils;
using namespace mrpt::poses;
using namespace mrpt::bayes;

/*---------------------------------------------------------------
						Constructor
  ---------------------------------------------------------------*/
CMetricMapBuilderRBPF::CMetricMapBuilderRBPF(  const TConstructionOptions &initializationOptions ) :
	mapPDF(
		initializationOptions.PF_options,
		&initializationOptions.mapsInitializers,
		&initializationOptions.predictionOptions ),
    m_PF_options( initializationOptions.PF_options ),
	insertionLinDistance(initializationOptions.insertionLinDistance),
	insertionAngDistance(initializationOptions.insertionAngDistance),
	linDistSinceLast(0),
	angDistSinceLast(0),
	currentMetricMapEstimation(NULL)
{
	// Reset:
	clear();
}

/*---------------------------------------------------------------
						Destructor
  ---------------------------------------------------------------*/
CMetricMapBuilderRBPF::~CMetricMapBuilderRBPF()
{

}

/*---------------------------------------------------------------
						clear
  ---------------------------------------------------------------*/
void  CMetricMapBuilderRBPF::clear()
{
	static CPose2D		nullPose(0,0,0);

	// Reset traveled distances counters:
	linDistSinceLast=angDistSinceLast=0;

	// Clear maps for each particle:
	mapPDF.clear( nullPose );
}

/*---------------------------------------------------------------
					processActionObservation
  ---------------------------------------------------------------*/
void  CMetricMapBuilderRBPF::processActionObservation(
					CActionCollection	&action,
					CSensoryFrame		&observations )
{
	MRPT_TRY_START;

	// Enter critical section (updating map)
	enterCriticalSection();

	CParticleFilter	pf;
	pf.m_options = m_PF_options;

	CPose3DPDFParticles					poseEstimation;

	// Process all particle-filters for the robot pose
	//   using each particle-asociated gridmap:
	// -----------------------------------------------------

	// Get previous pose estimation:
	CPose3D				lastMeanPose;
	{
		CPose3DPDFParticles	lastPoseEstimation;
		mapPDF.getEstimatedPosePDF(lastPoseEstimation);
		lastMeanPose = lastPoseEstimation.getEstimatedPose( );
	}

	// Execute particle filter:
	// -------------------------------
	pf.executeOn( mapPDF, &action, &observations );

	// Get current pose estimation:
	mapPDF.getEstimatedPosePDF(poseEstimation);

	// Update the traveled distance estimations:
	//CActionRobotMovement2D	*robotMovement = action.getBestMovementEstimation();
	CPose3D		meanPose = poseEstimation.getEstimatedPose();

	linDistSinceLast += lastMeanPose.distanceTo(meanPose);
	angDistSinceLast += fabs( mrpt::math::wrapToPi( lastMeanPose.yaw - meanPose.yaw) );

	if (options.verbose)
	{
		CPose3D		estPos = poseEstimation.getEstimatedPose();
		CMatrixD	cov    = poseEstimation.getEstimatedCovariance();

		std::cout << " New pose=" << estPos << "\n New ESS:"<< mapPDF.ESS() << "\n";
		std::cout << format("   STDs: x=%2.3f y=%2.3f z=%.03f yaw=%2.3fdeg\n", sqrt(cov(0,0)),sqrt(cov(1,1)),sqrt(cov(2,2)),RAD2DEG(sqrt(cov(3,3))));
	}

	// Include the observations into the map PDF??
	// --------------------------------------------------------

	bool SFhasImages =	observations.getObservationByClass<CObservationBearingRange>().present() ||
						observations.getObservationByClass<CObservationStereoImages>().present();

	if ( !mapPDF.SFs.size() ||	// This is the first observation!
			options.debugForceInsertion ||
			//mapPDF.newInfoIndex >0.10f )
			linDistSinceLast>insertionLinDistance ||
			angDistSinceLast>insertionAngDistance ||
			(options.insertImagesAlways && SFhasImages ))
	{
		// Reset distance counters:
		linDistSinceLast = angDistSinceLast = 0;

		if (options.verbose)
		{
			printf(" 3) New observation inserted into the map!\n");
		}

		// Add current observation to the map:
		// -----------------------------------------------------
		mapPDF.insertObservation( observations, false);

		m_statsLastIteration.observationsInserted = true;
	}
	else
	{
		m_statsLastIteration.observationsInserted = false;
	}

	// Added 29/JUN/2007 JLBC: Say to all the maps that they can free aux. variables
	//  (if any) since one PF cycle is over:
	for (CMultiMetricMapPDF::CParticleList::iterator	it = mapPDF.m_particles.begin(); it!=mapPDF.m_particles.end();it++)
		it->d->mapTillNow.auxParticleFilterCleanUp();


	leaveCriticalSection(); /* Leaving critical section (updating map) */

	MRPT_TRY_END_WITH_CLEAN_UP( leaveCriticalSection(); /* Leaving critical section (updating map) */ );
}

/*---------------------------------------------------------------
					initialize
  ---------------------------------------------------------------*/
void  CMetricMapBuilderRBPF::initialize(
		CSensFrameProbSequence		&initialMap,
		CPosePDF					*x0 )
{
	// Enter critical section (updating map)
	enterCriticalSection();

	if (options.verbose)
	{
		printf("[initialize] Called with %u nodes in fixed map, and x0=", static_cast<unsigned>(initialMap.size()) );
		if (x0)
			std::cout << x0->getEstimatedPose() << "\n";
		else
			printf("(Not supplied)\n");
	}

	// Leaving critical section (updating map)
	leaveCriticalSection();
}


/*---------------------------------------------------------------
						getMarkovLocalization
  ---------------------------------------------------------------*/
CPose3DPDFPtr CMetricMapBuilderRBPF::getCurrentPoseEstimation() const
{
	CPose3DPDFParticlesPtr posePDF = CPose3DPDFParticlesPtr( new CPose3DPDFParticles());
	mapPDF.getEstimatedPosePDF(*posePDF);
	return posePDF;
}

/*---------------------------------------------------------------
						getCurrentMostLikelyPath
  ---------------------------------------------------------------*/
void  CMetricMapBuilderRBPF::getCurrentMostLikelyPath( std::deque<CPose2D> &outPath ) const
{
	std::deque<CPose3D> aux_path;
	getCurrentMostLikelyPath(aux_path);

	outPath.resize( aux_path.size() );

	std::deque<CPose3D>::iterator i;
	std::deque<CPose2D>::iterator j;

	for(i=aux_path.begin(),j=outPath.begin();i!=aux_path.end();i++,j++)
		*j = *i;
}

/*---------------------------------------------------------------
						getCurrentMostLikelyPath
  ---------------------------------------------------------------*/
void  CMetricMapBuilderRBPF::getCurrentMostLikelyPath( std::deque<CPose3D> &outPath ) const
{
	double maxW = -1, w;
	size_t mostLik=0;
	for (size_t i=0;i<mapPDF.particlesCount();i++)
	{
		w = mapPDF.getW(i);
		if (w>maxW)
		{
			maxW = w;
			mostLik = i;
		}
	}

	mapPDF.getPath( mostLik, outPath );
}

/*---------------------------------------------------------------
						getCurrentlyBuiltMap
  ---------------------------------------------------------------*/
void  CMetricMapBuilderRBPF::getCurrentlyBuiltMap(
			CSensFrameProbSequence		&out_map)
{
	mapPDF.updateSensoryFrameSequence();
	out_map = mapPDF.SFs;
}

/*---------------------------------------------------------------
						getCurrentlyBuiltMetricMap
  ---------------------------------------------------------------*/
CMultiMetricMap*   CMetricMapBuilderRBPF::getCurrentlyBuiltMetricMap()
{
	currentMetricMapEstimation = mapPDF.getCurrentMetricMapEstimation(  );

	return currentMetricMapEstimation.get();
}

/*---------------------------------------------------------------
			getCurrentlyBuiltMapSize
  ---------------------------------------------------------------*/
unsigned int  CMetricMapBuilderRBPF::getCurrentlyBuiltMapSize()
{
	return mapPDF.SFs.size();
}


/*---------------------------------------------------------------
					drawCurrentEstimationToImage
  ---------------------------------------------------------------*/
void  CMetricMapBuilderRBPF::drawCurrentEstimationToImage( utils::CMRPTCanvas *img )
{
	unsigned int			i, M = mapPDF.particlesCount();
	std::deque<CPose3D>		path;
	unsigned int			imgHeight=0;

	MRPT_TRY_START;

	// Estimated map:
//		currentMetricMapEstimation = mapPDF.getCurrentMetricMapEstimation( );
	currentMetricMapEstimation = mapPDF.getCurrentMostLikelyMetricMap( );

	ASSERT_( currentMetricMapEstimation->m_gridMaps.size()>0 );


	// Find which is the most likely path index:
	unsigned int	bestPath = 0;
	double			bestPathLik = -1;
	for (i=0;i<M;i++)
	{
		if (mapPDF.getW(i)>bestPathLik)
		{
			bestPathLik = mapPDF.getW(i);
			bestPath = i;
		}
	}

	// Compute the length of the paths:
	mapPDF.getPath(0, path);

	// Adapt the canvas size:
	bool   alreadyCopiedImage = false;
	{
		CMRPTImage *obj = dynamic_cast<CMRPTImage*>( img );
		if (obj)
			obj->resize(
				currentMetricMapEstimation->m_gridMaps[0]->getSizeX(),
				currentMetricMapEstimation->m_gridMaps[0]->getSizeY(),
				1,
				true);
	}
	{
		CMRPTImageFloat *obj = dynamic_cast<CMRPTImageFloat*>( img );
		if (obj)
		{
			currentMetricMapEstimation->m_gridMaps[0]->getAsImage( *obj, true );
			alreadyCopiedImage = true;
			imgHeight = obj->getHeight();
		}
	}

	if (!alreadyCopiedImage)
	{
		CMRPTImageFloat			imgFl;

		// grid map as bitmap:
		// ----------------------------------
		currentMetricMapEstimation->m_gridMaps[0]->getAsImage( imgFl );

		img->drawImage( 0,0, imgFl );
		imgHeight = imgFl.getHeight();
	}

	int		x1=0,x2=0,y1=0,y2=0;
	float	x_min = currentMetricMapEstimation->m_gridMaps[0]->getXMin();
	float	y_min = currentMetricMapEstimation->m_gridMaps[0]->getYMin();
	float   resolution = currentMetricMapEstimation->m_gridMaps[0]->getResolution();

	// Paths hypothesis:
	// ----------------------------------
	/***/
	for (i=0;i<=M;i++)
	{
		if (i!=bestPath || i==M)
		{
			mapPDF.getPath(i==M ? bestPath:i, path);

			size_t	nPoses = path.size();

			// First point: (0,0)
			x2 = round( ( path[0].x - x_min) / resolution);
			y2 = round( ( path[0].y - y_min) / resolution );

			// Draw path in the bitmap:
			for (size_t j=0;j<nPoses;j++)
			{
				// For next segment
				x1 = x2;
				y1 = y2;

				// Coordinates -> pixels
				x2 = round( ( path[j].x - x_min) / resolution );
				y2 = round( ( path[j].y - y_min) / resolution );

				// Draw line:
				img->line(
					x1, round( (imgHeight-1)-y1 ),
					x2, round( (imgHeight-1)-y2 ),
					i==M ? 0x000000 : 0x505050,	 // Color, gray levels,
					i==M ? 70 : 30				 // Line width
					);
			}
		}
	}

	/** /
	const float						MIN_DISTANCE_BETWEEN_ELLIPSES = 2.5f;
	float							stdCount	= 3.0f;
	CMatrixD						cov(2,2); //,eigVal,eigVec,M;
	CPosePDFParticles				parts;
	CPose2D							mean;

	CPose2D		lastMeanPose;
	bool		drawEllipse;

	for (i=0;i<pathsLen;i++)
	{
		mapPDF.getEstimatedPosePDFAtTime(i,parts);

		mean = parts.getEstimatedPose();
		cov = parts.getEstimatedCovariance();
		cov.setSize(2,2);

		// Draw the ellipse??
		drawEllipse = (i==0) || (i==pathsLen-1);
		if (i>1)
			drawEllipse |= (mean.distanceTo( lastMeanPose ) > MIN_DISTANCE_BETWEEN_ELLIPSES);
		if ( drawEllipse )
		{
			img->ellipseGaussian( &cov,(double)mean.x,(double)mean.y, stdCount, 0x000000,1, 50 );

			lastMeanPose = mean;

		} // end of "if (drawEllipse)..."


	} // end for i
	/ **/

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					saveCurrentEstimationToImage
  ---------------------------------------------------------------*/
void  CMetricMapBuilderRBPF::saveCurrentEstimationToImage(const std::string &file, bool formatEMF_BMP )
{
	MRPT_TRY_START;

	if (formatEMF_BMP)
	{
		// Draw paths (using vectorial plots!) over the EMF file:
		// --------------------------------------------------------
		CEnhancedMetaFile		EMF( file,  100 /* Scale */ );
		drawCurrentEstimationToImage( &EMF );
	}
	else
	{
		CMRPTImageFloat		img(1,1);
		drawCurrentEstimationToImage( &img );
		img.saveToFile(file);
	}

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
					getCurrentJointEntropy
  ---------------------------------------------------------------*/
double  CMetricMapBuilderRBPF::getCurrentJointEntropy()
{
	return	mapPDF.getCurrentJointEntropy();
}

/*---------------------------------------------------------------
					saveCurrentPathEstimationToTextFile
  ---------------------------------------------------------------*/
void  CMetricMapBuilderRBPF::saveCurrentPathEstimationToTextFile( std::string  fil )
{
	mapPDF.saveCurrentPathEstimationToTextFile( fil );
}

/*---------------------------------------------------------------
						TConstructionOptions
  ---------------------------------------------------------------*/
CMetricMapBuilderRBPF::TConstructionOptions::TConstructionOptions() :
	insertionLinDistance		( 1.0f ),
	insertionAngDistance		( DEG2RAD(30) ),
	PF_options(),
	mapsInitializers(),
	predictionOptions()
{
}

/*---------------------------------------------------------------
					dumpToTextStream
  ---------------------------------------------------------------*/
void  CMetricMapBuilderRBPF::TConstructionOptions::dumpToTextStream(CStream	&out)
{
	out.printf("\n----------- [CMetricMapBuilderRBPF::TConstructionOptions] ------------ \n\n");

	out.printf("insertionLinDistance                    = %f m\n", insertionLinDistance );
	out.printf("insertionAngDistance                    = %f deg\n", RAD2DEG(insertionAngDistance) );

	PF_options.dumpToTextStream(out);

	out.printf("  Now showing 'mapsInitializers' and 'predictionOptions':\n");
	out.printf("\n");

	mapsInitializers.dumpToTextStream(out);
	predictionOptions.dumpToTextStream(out);

}

/*---------------------------------------------------------------
					loadFromConfigFile
  ---------------------------------------------------------------*/
void  CMetricMapBuilderRBPF::TConstructionOptions::loadFromConfigFile(
	const mrpt::utils::CConfigFileBase  &iniFile,
	const std::string &section)
{
	MRPT_TRY_START;

	PF_options.loadFromConfigFile(iniFile,section);

	insertionLinDistance		= iniFile.read_float(section.c_str(),"insertionLinDistance",insertionLinDistance);
	insertionAngDistance		= DEG2RAD(iniFile.read_float(section.c_str(),"insertionAngDistance_deg",RAD2DEG(insertionAngDistance)));

	mapsInitializers.loadFromConfigFile(iniFile,section);
	predictionOptions.loadFromConfigFile(iniFile,section);

	MRPT_TRY_END;
}
