/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/random.h>
#include <mrpt/math/utils.h>
#include <mrpt/utils/CTicTac.h>
#include <mrpt/utils/CFileStream.h>

#include <mrpt/slam/CMultiMetricMapPDF.h>
#include <mrpt/slam/CActionRobotMovement2D.h>
#include <mrpt/slam/CActionRobotMovement3D.h>
#include <mrpt/slam/CActionCollection.h>
#include <mrpt/poses/CPosePDFGaussian.h>
#include <mrpt/poses/CPosePDFGrid.h>
#include <mrpt/slam/CObservationBeaconRanges.h>
#include <mrpt/slam/CICP.h>
#include <mrpt/slam/CSimplePointsMap.h>
#include <mrpt/slam/CLandmarksMap.h>
#include <mrpt/math_mrpt.h>

using namespace mrpt;
using namespace mrpt::math;
using namespace mrpt::slam;
using namespace mrpt::poses;
using namespace mrpt::utils;
using namespace std;

IMPLEMENTS_SERIALIZABLE( CMultiMetricMapPDF, CSerializable, mrpt::slam )
IMPLEMENTS_SERIALIZABLE( CRBPFParticleData,  CSerializable, mrpt::slam )

/** Auxiliary for optimal sampling in RO-SLAM
  */
struct TAuxRangeMeasInfo
{
	TAuxRangeMeasInfo() :
		sensorLocationOnRobot(),
		sensedDistance(0),
		beaconID(INVALID_BEACON_ID),
		nGaussiansInMap(0)
	{}

	CPoint3D		sensorLocationOnRobot;
	float			sensedDistance;
	int64_t			beaconID;
	size_t			nGaussiansInMap; // Number of Gaussian modes in the map representation
};

/** Auxiliary for optimal sampling in RO-SLAM
  */
bool cmpBeaconIdRangePair_Asc(const TAuxRangeMeasInfo &a, const TAuxRangeMeasInfo &b)
{
    return a.nGaussiansInMap < b.nGaussiansInMap;
}


#ifdef MRPT_HAS_SSE
	/*---------------------------------------------------------------
	Auxiliary function: Detect the processor support of SSE.
		Only executed the first time.
	---------------------------------------------------------------*/
	bool checkSuportSSE()
	{
		return false;

/*		static	bool	firstTime = true;
		static	bool	SSE_Supported;

		// Already checked:
		if (!firstTime)
			return SSE_Supported;

		printf("[MRPT] Probing processor support for SSE...");

		firstTime = false;

		try
		{
			__asm
				{
					xorps xmm0, xmm0
				}
		}
		catch(...)
		{
			// Error: Not supported:
			printf("*NOT PASS*\n");
			SSE_Supported = false;
			return SSE_Supported;
		}

		// Ok, SSE suported:
		printf("PASS!\n");
		SSE_Supported = true;
		return SSE_Supported;
*/
	}
#endif


/*---------------------------------------------------------------
				Constructor
  ---------------------------------------------------------------*/
CMultiMetricMapPDF::CMultiMetricMapPDF(
	const bayes::CParticleFilter::TParticleFilterOptions    &opts,
	const  mrpt::slam::TSetOfMetricMapInitializers		        *mapsInitializers,
	const  TPredictionParams						        *predictionOptions) :
		averageMap( mapsInitializers ),
		averageMapIsUpdated(false),
		SFs(),
		SF2robotPath(),
		options(),
		newInfoIndex(0),
		m_pfAuxiliaryPFOptimal_estimatedProb(),
		m_maxLikelihood(),
		m_movementDrawer(),
		m_movementDrawMaximumLikelihood()
{
	m_particles.resize( opts.sampleSize );
	for (CParticleList::iterator it=m_particles.begin();it!=m_particles.end();it++)
	{
		it->log_w = 0;
		it->d = new CRBPFParticleData( mapsInitializers );
	}

	// Initialize:
	static CPose3D		nullPose(0,0,0);
	clear(nullPose);

	// If provided, copy the whole set of params now:
	if (predictionOptions!=NULL)
		options = *predictionOptions;
}

/*---------------------------------------------------------------
						clear
  ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::clear( const CPose2D &initialPose )
{
	CPose3D	p(initialPose);
	clear(p);
}

/*---------------------------------------------------------------
						clear
  ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::clear( const CPose3D &initialPose )
{
	size_t	i,M = m_particles.size();

	for (i=0;i<M;i++)
	{
		m_particles[i].log_w = 0;

		m_particles[i].d->mapTillNow.clear();

		m_particles[i].d->robotPath.resize(1);
		m_particles[i].d->robotPath[0]=initialPose;
	}

	SFs.clear();
	SF2robotPath.clear();

	averageMapIsUpdated = false;
}


/*---------------------------------------------------------------
						operator =
  ---------------------------------------------------------------*/
/*
CMultiMetricMapPDF & CMultiMetricMapPDF::operator = (const CMultiMetricMapPDF &o)
{
	CParticleList::iterator			it;
	CParticleList::const_iterator	it2;

	if (this == &o) return *this;		// It may be used sometimes

	// Copy options, easy data...
	options					= o.options;
	averageMap				= o.averageMap;
	SFs						= o.SFs;
	SF2robotPath			= o.SF2robotPath;

	// Copy the m_particles
	if (m_particles.size()==o.m_particles.size())
	{
		for (it=m_particles.begin(),it2=o.m_particles.begin();it!=m_particles.end();it++)
		{
			*(it->d) = *(it2->d);
			it->log_w = it2->log_w;
		}
	}
	else
	{
		clearParticles();
		m_particles.resize( o.m_particles.size() );

		for (it=m_particles.begin(), it2=o.m_particles.begin();it!=m_particles.end();it++,it2++)
		{
			it->d = new CRBPFParticleData( * it2->d );
			it->log_w = it2->log_w;
		}
	}

	return *this;
}
*/

/*---------------------------------------------------------------
	Destructor
  ---------------------------------------------------------------*/
CMultiMetricMapPDF::~CMultiMetricMapPDF( )
{
	clearParticles();
	SFs.clear();
	SF2robotPath.clear();
}

/*---------------------------------------------------------------
						getEstimatedPosePDF
  Returns an estimate of the pose, (the mean, or mathematical expectation of the PDF), computed
		as a weighted average over all m_particles.
 ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::getEstimatedPosePDF( CPose3DPDFParticles	&out_estimation ) const
{
	ASSERT_(m_particles[0].d->robotPath.size()>0);
	getEstimatedPosePDFAtTime( m_particles[0].d->robotPath.size()-1, out_estimation );
}

/*---------------------------------------------------------------
						getEstimatedPosePDFAtTime
 ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::getEstimatedPosePDFAtTime(
			size_t				timeStep,
			CPose3DPDFParticles	&out_estimation ) const
{
	//CPose3D	p;
	size_t	i,n = m_particles.size();

	// Delete current content of "out_estimation":
	out_estimation.clearParticles();

	// Create new m_particles:
	out_estimation.m_particles.resize(n);
	for (i=0;i<n;i++)
	{
		out_estimation.m_particles[i].d = new CPose3D( m_particles[i].d->robotPath[ timeStep ] );
		out_estimation.m_particles[i].log_w = m_particles[i].log_w;
	}

}


/*---------------------------------------------------------------
						writeToStream
  ---------------------------------------------------------------*/
void  CRBPFParticleData::writeToStream(CStream &out,int *version) const
{
	THROW_EXCEPTION("Shouldn't arrive here")
}

/*---------------------------------------------------------------
						readFromStream
  ---------------------------------------------------------------*/
void  CRBPFParticleData::readFromStream(CStream &in, int version)
{
	THROW_EXCEPTION("Shouldn't arrive here")
}

/*---------------------------------------------------------------
						writeToStream
  ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::writeToStream(CStream &out,int *version) const
{
	if (version)
		*version = 0;
	else
	{
		uint32_t	i,n,j,m;

		// The data
		n = static_cast<uint32_t>(m_particles.size());
		out << n;
		for (i=0;i<n;i++)
		{
			out << m_particles[i].log_w << m_particles[i].d->mapTillNow;
			m = static_cast<uint32_t>(m_particles[i].d->robotPath.size());
			out << m;
			for (j=0;j<m;j++)
				out << m_particles[i].d->robotPath[j];
		}
		out << SFs << SF2robotPath;
	}
}

/*---------------------------------------------------------------
						readFromStream
  ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::readFromStream(CStream &in, int version)
{
	switch(version)
	{
	case 0:
		{
			uint32_t	i,n,j,m;

			// Delete current contents:
			// --------------------------
			clearParticles();
			SFs.clear();
			SF2robotPath.clear();

			averageMapIsUpdated = false;

			// Load the new data:
			// --------------------
			in >> n;

			m_particles.resize(n);
			for (i=0;i<n;i++)
			{
				m_particles[i].d = new CRBPFParticleData();

				// Load
				in >> m_particles[i].log_w >> m_particles[i].d->mapTillNow;

				in >> m;
				m_particles[i].d->robotPath.resize(m);
				for (j=0;j<m;j++)
					in >> m_particles[i].d->robotPath[j];
			}

			in >> SFs >> SF2robotPath;

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}

/*---------------------------------------------------------------
			prediction_and_update_pfOptimalProposal
 ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::prediction_and_update_pfOptimalProposal(
	const mrpt::slam::CActionCollection	* actions,
	const mrpt::slam::CSensoryFrame		* sf,
	const bayes::CParticleFilter::TParticleFilterOptions &PF_options )
{
	MRPT_TRY_START;

	// ----------------------------------------------------------------------
	//						PREDICTION STAGE
	// ----------------------------------------------------------------------
	vector_float				rndSamples;
	size_t						M = m_particles.size();
	bool						updateStageAlreadyDone = false;
	CPose3D						initialPose,incrPose, finalPose;

	// ICP used if "pfOptimalProposal_mapSelection" = 0 or 1
	static CICP					icp;
	static CICP::TReturnInfo	icpInfo;
	static bool icpInit = false;
	CParticleList::iterator		partIt;

	ASSERT_(sf!=NULL)

	// Find a robot movement estimation:
	CPose3D						motionModelMeanIncr;	// The mean motion increment:
	CPoseRandomSampler			robotActionSampler;
	{
		CActionRobotMovement2DPtr	robotMovement2D = actions->getBestMovementEstimation();
		
		// If there is no 2D action, look for a 3D action:
		if (robotMovement2D.present())
		{
			robotActionSampler.setPosePDF( robotMovement2D->poseChange );
			motionModelMeanIncr = robotMovement2D->poseChange->getEstimatedPose();
		}
		else
		{
			CActionRobotMovement3DPtr	robotMovement3D = actions->getActionByClass<CActionRobotMovement3D>();
			if (robotMovement3D)
			{
				robotActionSampler.setPosePDF( robotMovement3D->poseChange );
				motionModelMeanIncr = robotMovement3D->poseChange.getEstimatedPose();
			}
			else
			{
				motionModelMeanIncr.setFromValues(0,0,0);
			}
		}
	}

	// Average map will need to be updated after this:
	averageMapIsUpdated = false;

	// --------------------------------------------------------------------------------------
	//  Prediction:
	//
	//  Compute a new mean and covariance by sampling around the mean of the input "action"
	//
	// See: "Improved Grid-based SLAM with Rao-Blackwellized PF by Adaptive Proposals
	//         and Selective Resampling" (G. Grisetti, C. Stachniss, W. Burgard)
	// --------------------------------------------------------------------------------------
	printf(" 1) Prediction...");
	M = m_particles.size();

	if ( !icpInit && (options.pfOptimalProposal_mapSelection==0 || options.pfOptimalProposal_mapSelection==1))
	{
		icpInit=true;

		// ICP options
		// ------------------------------
		icp.options.maxIterations = 80;
		icp.options.thresholdDist = 0.50f;
		icp.options.thresholdAng = DEG2RAD( 20 );
		icp.options.smallestThresholdDist = 0.20f;
		icp.options.ALFA		  = 0.5f;
		icp.options.onlyClosestCorrespondences = true;
		icp.options.doRANSAC = false;
	}

	// To be computed as an average from all m_particles:
	size_t particleWithHighestW = 0;
	for (size_t i=0;i<M;i++)
		if (getW(i)>getW(particleWithHighestW))
			particleWithHighestW = i;


	//   The paths MUST already contain the starting location for each particle:
	ASSERT_( !m_particles[0].d->robotPath.empty() )

	// Update particle poses:
	size_t i;
	for (i=0,partIt = m_particles.begin(); partIt!=m_particles.end(); partIt++,i++)
	{
		// Set initial robot pose estimation for this particle:
		const CPose3D *ith_last_pose = &(*partIt->d->robotPath.rbegin()); // The last robot pose in the path

		CPose3D		initialPoseEstimation = *ith_last_pose + motionModelMeanIncr;

		// Use ICP with the map associated to particle?
		if ( options.pfOptimalProposal_mapSelection==0 || options.pfOptimalProposal_mapSelection==1)
		{
			CPosePDFGaussian			icpEstimation;

			// Configure the matchings into the ICP process:
			if (partIt->d->mapTillNow.m_pointsMaps.size())
			{
				ASSERT_(partIt->d->mapTillNow.m_pointsMaps.size()==1);
				partIt->d->mapTillNow.m_pointsMaps[0]->insertionOptions.matchStaticPointsOnly = false;
			}

			if (options.pfOptimalProposal_mapSelection==0)
			{
				CSimplePointsMap	localMapPoints;

				ASSERT_( m_particles[0].d->mapTillNow.m_gridMaps.size() > 0);
				float	minDistBetweenPointsInLocalMaps = 0.02f; //3.0f * m_particles[0].d->mapTillNow.m_gridMaps[0]->getResolution();

				// Build local map:
				localMapPoints.clear();
				localMapPoints.insertionOptions.minDistBetweenLaserPoints =  minDistBetweenPointsInLocalMaps;
				localMapPoints.insertionOptions.also_interpolate = false;
				localMapPoints.insertionOptions.isPlanarMap = true;
				sf->insertObservationsInto( &localMapPoints );

				ASSERT_( partIt->d->mapTillNow.m_gridMaps.size() > 0 );

				// Use ICP to align to each particle's map:
				CPosePDFPtr alignEst =
				icp.Align(
					partIt->d->mapTillNow.m_gridMaps[0].pointer(),
					&localMapPoints,
					initialPoseEstimation,
					NULL,
					&icpInfo);
                icpEstimation.copyFrom( *alignEst );
			}
			else
			{
				static CLandmarksMap	localMapLandmarks;
			// Build local map:
				localMapLandmarks.clear();
				sf->insertObservationsInto( &localMapLandmarks );

				ASSERT_( partIt->d->mapTillNow.m_landmarksMap.present() );

				// Use ICP to align to each particle's map:
				CPosePDFPtr alignEst =
				icp.Align(
					partIt->d->mapTillNow.m_landmarksMap.pointer(),
					&localMapLandmarks,
					initialPoseEstimation,
					NULL,
					&icpInfo);
                icpEstimation.copyFrom( *alignEst );
			}

			if (i==particleWithHighestW)
			{
				// Check if update is needed:
//					newStaticPointsRatio = partIt->d->mapTillNow.getNewStaticPointsRatio(
//								&localMap,
//								icpEstimation->mean );
				newInfoIndex = 1 - icpInfo.goodness; //newStaticPointsRatio; //* icpInfo.goodness;
				//printf("\ninfo index=%f  --- staticRatio=%f Goodness=%f\n",newInfoIndex,newStaticPointsRatio,icpInfo.goodness);
			}

			// Set the gaussian pose:
			CPose3DPDFGaussian finalEstimatedPoseGauss( icpEstimation );

			//printf("[rbpf-slam] gridICP[%u]: %.02f%%\n", i, 100*icpInfo.goodness);
			if (icpInfo.goodness<options.ICPGlobalAlign_MinQuality && SFs.size())
			{
				printf("[rbpf-slam] Warning: gridICP[%u]: %.02f%% -> Using odometry instead!\n", (unsigned int)i, 100*icpInfo.goodness);
				icpEstimation.mean = initialPoseEstimation;
			}

//				printf("gICP: %.02f%%, Iters=%u\n",icpInfo.goodness,icpInfo.nIterations);

			// HACK:
			CPose3D Ap = finalEstimatedPoseGauss.mean - *ith_last_pose;
			double  Ap_dist = Ap.norm();

			finalEstimatedPoseGauss.cov.zeros();
			finalEstimatedPoseGauss.cov(0,0) = square( fabs(Ap_dist)*0.01 );
			finalEstimatedPoseGauss.cov(1,1) = square( fabs(Ap_dist)*0.01 );
			finalEstimatedPoseGauss.cov(2,2) = square( fabs(Ap.yaw)*0.02 );

			// Generate gaussian-distributed 2D-pose increments according to "finalEstimatedPoseGauss":
			// -------------------------------------------------------------------------------------------
			finalPose = finalEstimatedPoseGauss.mean;					// Add to the new robot pose:
			mrpt::random::randomNormalMultiDimensional( finalEstimatedPoseGauss.cov, rndSamples );
			finalPose.x += rndSamples[0];		// Add noise:
			finalPose.y += rndSamples[1];
			finalPose.yaw += rndSamples[2];
		}
		else
		if ( options.pfOptimalProposal_mapSelection==2)
		{
			// --------------------------------------------------------
			//     Perform optimal sampling with the beacon map:
			//  Described in paper...
			// --------------------------------------------------------
			/** \todo Add paper ref!
			  */
			ASSERT_( partIt->d->mapTillNow.m_beaconMap.present() );
			CBeaconMapPtr beacMap = partIt->d->mapTillNow.m_beaconMap;

			updateStageAlreadyDone = true; // We'll also update the weight of the particle here

			// =====================================================================
			// SUMMARY:
			// For each beacon measurement in the SF, stored in "lstObservedRanges",
			//  compute the SOG of the observation model, and multiply all of them
			//  (fuse) in "fusedObsModels". The result will hopefully be a very small
			//  PDF where to draw a sample from for the new robot pose.
			// =====================================================================
			bool 		methodSOGorGrid = false; // TRUE=SOG
			CPoint3D  	newDrawnPosition;
			float  		firstEstimateRobotHeading=0;

			// The parameters to discard too far gaussians:
			CPoint3D  	centerPositionPrior( *ith_last_pose );
			float		centerPositionPriorRadius=2.0f;

			if ( !robotActionSampler.isPrepared() )
			{
				firstEstimateRobotHeading = ith_last_pose->yaw;
				// If the map is empty: There is no solution!:
				// THROW_EXCEPTION("There is no odometry & the initial beacon map is empty: RO-SLAM has no solution -> ABORTED!!!");

				if (!beacMap->m_beacons.size())
				{
					// First iteration only...
					cerr << "[RO-SLAM] Optimal filtering without map & odometry...this message should appear only the first iteration!!" << endl;
				}
				else
				{
					// To make RO-SLAM to have a solution, arbitrarily fix one of the beacons so
					//  unbiguity dissapears.
					if ( beacMap->m_beacons[0].m_typePDF==CBeacon::pdfSOG)
					{
						cerr << "[RO-SLAM] Optimal filtering without map & odometry->FIXING ONE BEACON!" << endl;
						ASSERT_(beacMap->m_beacons[0].m_locationSOG.m_modes.size()>0)

						CPoint3D 	fixedBeacon( beacMap->m_beacons[0].m_locationSOG.m_modes[0].val.mean );

						// Pass to gaussian without uncertainty:
						beacMap->m_beacons[0].m_typePDF=CBeacon::pdfGauss;
						beacMap->m_beacons[0].m_locationSOG.m_modes.clear();
						beacMap->m_beacons[0].m_locationGauss.mean = fixedBeacon;
						beacMap->m_beacons[0].m_locationGauss.cov.unit(3);
						beacMap->m_beacons[0].m_locationGauss.cov *= 1e-6f;
					}
				}
			} // end if there is no odometry


			// 1. Make the list of beacon IDs-ranges:
			// -------------------------------------------
			deque<TAuxRangeMeasInfo>	lstObservedRanges;

			for (CSensoryFrame::const_iterator itObs = sf->begin();itObs!=sf->end();itObs++)
			{
				if ( IS_CLASS( (*itObs),CObservationBeaconRanges) )
				{
					const CObservationBeaconRanges *obs = static_cast<const CObservationBeaconRanges*> (itObs->pointer());
					deque<CObservationBeaconRanges::TMeasurement>::const_iterator itRanges;
					for (itRanges=obs->sensedData.begin();itRanges!=obs->sensedData.end();itRanges++)
					{
						ASSERT_( itRanges->beaconID != INVALID_BEACON_ID )
						// only add those in the map:
						for (TSequenceBeacons::iterator itBeacs=beacMap->m_beacons.begin();itBeacs!=beacMap->m_beacons.end();itBeacs++)
						{
							if ( itBeacs->m_ID == itRanges->beaconID)
							{
								TAuxRangeMeasInfo	newMeas;
								newMeas.beaconID = itRanges->beaconID;
								newMeas.sensedDistance = itRanges->sensedDistance;
								newMeas.sensorLocationOnRobot = itRanges->sensorLocationOnRobot;

								ASSERT_( itBeacs->m_typePDF==CBeacon::pdfGauss || itBeacs->m_typePDF==CBeacon::pdfSOG )
								newMeas.nGaussiansInMap = itBeacs->m_typePDF==CBeacon::pdfSOG ? itBeacs->m_locationSOG.m_modes.size() : 1 /*pdfGauss*/;

								lstObservedRanges.push_back( newMeas );
								break; // Next observation
							}
						}
					}
				}
			}

			//ASSERT_( lstObservedRanges.size()>=2 );

			// Sort ascending ranges: Smallest ranges first -> the problem is easiest!
			sort( lstObservedRanges.begin(),lstObservedRanges.end(), cmpBeaconIdRangePair_Asc );


			if ( methodSOGorGrid )
			{
				CPointPDFSOG	fusedObsModels; //<- p(z_t|x_t) for all the range measurements (fused)
				fusedObsModels.m_modes.clear();

				// 0. OPTIONAL: Create a "prior" as a first mode in "fusedObsModels"
				//       using odometry. If there is no odometry, we *absolutely* need
				//       at least one well-localized beacon at the beginning, or the symmetry cannot be broken!
				// -----------------------------------------------------------------------------------------------
				float  firstEstimateRobotHeading;
				if ( robotActionSampler.isPrepared() )
				{
					CPointPDFSOG::TGaussianMode	newMode;
					newMode.log_w = 0;

					CPose3D auxPose(*ith_last_pose + motionModelMeanIncr); // CPose3D(robotMovement->poseChange->getEstimatedPose()));
					firstEstimateRobotHeading = auxPose.yaw;

					newMode.val.mean = auxPose;

					// Uncertainty in z is null:
					//CMatrix poseCOV =  robotMovement->poseChange->getEstimatedCovariance();
					CMatrixD poseCOV;
					robotActionSampler.getOriginalPDFCov2D( poseCOV );

					poseCOV.setSize(2,2);
					poseCOV.setSize(3,3);
					newMode.val.cov =poseCOV;
					fusedObsModels.m_modes.push_back(newMode);	// Add it:
				}


				// 2. Generate the optimal proposal by fusing obs models
				// -------------------------------------------------------------
				for (TSequenceBeacons::iterator itBeacs=beacMap->m_beacons.begin();itBeacs!=beacMap->m_beacons.end();itBeacs++)
				{
					// for each observed beacon (by its ID), generate observation model:
					for (deque<TAuxRangeMeasInfo>::iterator itObs=lstObservedRanges.begin();itObs!=lstObservedRanges.end();itObs++)
					{
						if (itBeacs->m_ID==itObs->beaconID)
						{
							// Match:
							float sensedRange = itObs->sensedDistance;

							CPointPDFSOG	newObsModel;
							itBeacs->generateObservationModelDistribution(
								sensedRange,
								newObsModel,
								beacMap.pointer(),        			 // The beacon map, for options
								itObs->sensorLocationOnRobot,// Sensor location on robot
								centerPositionPrior,
								centerPositionPriorRadius );

							if (! fusedObsModels.m_modes.size() )
							{
								// This is the first one: Just copy the obs. model here:
								fusedObsModels = newObsModel;
							}
							else
							{
								// Fuse with existing:
								CPointPDFSOG	tempFused(0);
								tempFused.bayesianFusion(
									fusedObsModels,
									newObsModel,
									3   // minMahalanobisDistToDrop
									);
								fusedObsModels = tempFused;
							}

							// Remove modes with negligible weights:
							// -----------------------------------------------------------

							{
								cout << "#modes bef: " << fusedObsModels.m_modes.size() << " ESS: " << fusedObsModels.ESS() << endl;
								double max_w=-1e100;
								//int idx;

								CPointPDFSOG::CListGaussianModes::iterator it;

								for (it=fusedObsModels.m_modes.begin();it!=fusedObsModels.m_modes.end();it++)
									max_w = max(max_w,it->log_w);	// keep the maximum mode weight

								for (it=fusedObsModels.m_modes.begin();it!=fusedObsModels.m_modes.end(); )
								{
									if (max_w - it->log_w > 20 ) // Remove the mode:
										 it = fusedObsModels.m_modes.erase( it  );
									else  it++;
								}

								cout << "#modes after: " << fusedObsModels.m_modes.size() << endl;
							}

							// Shall we simplify the PDF?
							// -----------------------------------------------------------
							CMatrix currentCov( fusedObsModels.getEstimatedCovariance() );
							ASSERT_(currentCov(0,0)>0 && currentCov(1,1)>0)
							if ( sqrt(currentCov(0,0))< 0.10f &&
								 sqrt(currentCov(1,1))< 0.10f &&
								 sqrt(currentCov(2,2))< 0.10f )
							{
								// Approximate by a single gaussian!
								CPoint3D currentMean( fusedObsModels.getEstimatedPoint() );
								fusedObsModels.m_modes[0].log_w = 0;
								fusedObsModels.m_modes[0].val.mean = currentMean;
								fusedObsModels.m_modes[0].val.cov = currentCov;

								fusedObsModels.m_modes.resize(1);
							}


							{/*
								CMatrixD  evalGrid;
								fusedObsModels.evaluatePDFInArea(-3,3,-3,3,0.1,0,evalGrid, true);
								evalGrid *= 1.0/evalGrid.maximum();
								CMRPTImageFloat imgF(evalGrid);
								static int autoCount=0;
								imgF.saveToFile(format("debug_%04i.png",autoCount++));*/
							}
						}
					} // end for itObs (in lstObservedRanges)

				} // end for itBeacs


				/** /
				COpenGLScene	scene;
				opengl::CSetOfObjects *obj = new opengl::CSetOfObjects();
				fusedObsModels.getAs3DObject( *obj );
				scene.insert( obj );
				CFileStream("debug.3Dscene",fomWrite) << scene;
				cout << "fusedObsModels # of modes: " << fusedObsModels.m_modes.size() << endl;
				printf("ESS: %f\n",fusedObsModels.ESS() );
				cout << fusedObsModels.getEstimatedCovariance() << endl;
				system::pause(); / **/

				if (beacMap->m_beacons.size())
					fusedObsModels.drawSingleSample( newDrawnPosition );
			}
			else
			{
				// =============================
				//         GRID METHOD
				// =============================

				float   grid_min_x = ith_last_pose->x - 0.5f;
				float   grid_max_x = ith_last_pose->x + 0.5f;
				float   grid_min_y = ith_last_pose->y - 0.5f;
				float   grid_max_y = ith_last_pose->y + 0.5f;
				float   grid_resXY = 0.02f;

				bool repeatGridCalculation=false;

				do
				{
					CPosePDFGrid	*pdfGrid = new CPosePDFGrid(
						grid_min_x,grid_max_x,
						grid_min_y,grid_max_y,
						grid_resXY, DEG2RAD(180), 0, 0 );

					pdfGrid->uniformDistribution();

					// Fuse all the observation models in the grid:
					// -----------------------------------------------------
					for (TSequenceBeacons::iterator itBeacs=beacMap->m_beacons.begin();itBeacs!=beacMap->m_beacons.end();itBeacs++)
					{
						// for each observed beacon (by its ID), generate observation model:
						for (deque<TAuxRangeMeasInfo>::iterator itObs=lstObservedRanges.begin();itObs!=lstObservedRanges.end();itObs++)
						{
							if (itBeacs->m_ID==itObs->beaconID)
							{
								// Match:
								float sensedRange = itObs->sensedDistance;

/** /
								CPointPDFSOG	newObsModel;
								itBeacs->generateObservationModelDistribution(
									sensedRange,
									newObsModel,
									beacMap,        			 // The beacon map, for options
									itObs->sensorLocationOnRobot,// Sensor location on robot
									centerPositionPrior,
									centerPositionPriorRadius );
/ **/
								for (size_t idxX=0;idxX<pdfGrid->getSizeX();idxX++)
								{
									float grid_x = pdfGrid->idx2x(idxX);
									for (size_t idxY=0;idxY<pdfGrid->getSizeY();idxY++)
									{
										double grid_y = pdfGrid->idx2y(idxY);

										// Evaluate obs model:
										double *cell = pdfGrid->getByIndex(idxX,idxY,0);

										//
										double lik = 1; //newObsModel.evaluatePDF(CPoint3D(grid_x,grid_y,0),true);
										switch (itBeacs->m_typePDF)
										{
											case CBeacon::pdfSOG:
											{
												CPointPDFSOG	*sog = & itBeacs->m_locationSOG;
												double sensorSTD2 = square(beacMap->likelihoodOptions.rangeStd);

												CPointPDFSOG::CListGaussianModes::iterator	it;
												for (it = sog->m_modes.begin();it!=sog->m_modes.end();it++)
												{
													lik *= exp( -0.5*square( sensedRange -
														it->val.mean.distance2DTo( grid_x+itObs->sensorLocationOnRobot.x, grid_y+itObs->sensorLocationOnRobot.y))/sensorSTD2 );
												}
											}
											break;
											default: break; //THROW_EXCEPTION("NO")

										}

										(*cell) *= lik;

									} // for idxY
								} // for idxX
								pdfGrid->normalize();
							} // end if match
						} // end for itObs
					} // end for beacons in map

					// Draw the single pose from the grid:
					if (beacMap->m_beacons.size())
					{
						// Take the most likely cell:
						float maxW = -1e10f;
						float maxX=0,maxY=0;
						for (size_t idxX=0;idxX<pdfGrid->getSizeX();idxX++)
						{
							//float grid_x = pdfGrid->idx2x(idxX);
							for (size_t idxY=0;idxY<pdfGrid->getSizeY();idxY++)
							{
								//float grid_y = pdfGrid->idx2y(idxY);

								// Evaluate obs model:
								float c = *pdfGrid->getByIndex(idxX,idxY,0);
								if (c>maxW)
								{
									maxW=c;
									maxX=pdfGrid->idx2x(idxX);
									maxY=pdfGrid->idx2y(idxY);
								}
							}
						}
						newDrawnPosition.x = maxX;
						newDrawnPosition.y = maxY;

						/**/
						{
							//cout << "Grid: " << pdfGaussApprox << endl;
							//pdfGrid->saveToTextFile("debug.txt");
							CMatrixDouble outMat;
							pdfGrid->getAsMatrix(0, outMat );
							outMat *= 1.0f/outMat.maximum();
							CMRPTImageFloat imgF(outMat);
							static int autocount=0;
							imgF.saveToFile(format("debug_grid_%f_%05i.png",grid_resXY,autocount++));
							printf("grid res: %f   MAX: %f,%f\n",grid_resXY,maxX,maxY);
							//system::pause();
						}
						/**/

						if (grid_resXY>0.01f)
						{
							grid_min_x = maxX - 0.03f;
							grid_max_x = maxX + 0.03f;
							grid_min_y = maxY - 0.03f;
							grid_max_y = maxY + 0.03f;
							grid_resXY=0.002f;
							repeatGridCalculation = true;
						}
						else
							repeatGridCalculation = false;


						/*

						// Approximate by a Gaussian:
						CPosePDFGaussian pdfGaussApprox(
							pdfGrid->getEstimatedPose(),
							pdfGrid->getEstimatedCovariance() );

						CPose2D newDrawnPose;
						//pdfGrid->drawSingleSample( newDrawnPose );
						pdfGaussApprox.drawSingleSample( newDrawnPose );
						newDrawnPosition = newDrawnPose;

						*/
					}

					delete pdfGrid; pdfGrid = NULL;

				} while (repeatGridCalculation);

				// newDrawnPosition has the pose:
			}

			// 3. Get the new "finalPose" of this particle as a random sample from the
			//  optimal proposal:
			// ---------------------------------------------------------------------------
			if (!beacMap->m_beacons.size())
			{
				// If we are in the first iteration (no map yet!) just stay still:
				finalPose = *ith_last_pose;
			}
			else
			{


				cout << "Drawn: " << newDrawnPosition << endl;
				//cout << "Final cov was:\n" << fusedObsModels.getEstimatedCovariance() << endl << endl;

				finalPose.x = newDrawnPosition.x;
				finalPose.y = newDrawnPosition.y;
				finalPose.z = newDrawnPosition.z;
				finalPose.yaw = firstEstimateRobotHeading;	// !!!
			}
			/** \todo If there are 2+ sensors on the robot, compute phi?
			  */

			// 4. Update the weight:
			//     In optimal sampling this is the expectation of the
			//     observation likelihood: This is the observation likelihood averaged
			//     over the whole optimal proposal.
			// ---------------------------------------------------------------------------
			double weightUpdate=0;
			partIt->log_w += PF_options.powFactor * weightUpdate;

		}
		else
		{
			// By default:
			// Generate gaussian-distributed 2D-pose increments according to mean-cov:
			if ( !robotActionSampler.isPrepared() )
				THROW_EXCEPTION("Action list does not contain any CActionRobotMovement2D or CActionRobotMovement3D object!");

			robotActionSampler.drawSample( incrPose );

			finalPose = *ith_last_pose + incrPose;
		}

		// Insert as the new pose in the path:
		partIt->d->robotPath.push_back( finalPose );

	} // end of for each particle "i" & "partIt"

	// ----------------------------------------------------------------------
	//						UPDATE STAGE
	// ----------------------------------------------------------------------
	printf("Ok\n 2) Update...");
	if (!updateStageAlreadyDone)
	{
		// Compute all the likelihood values:
		for (i=0,partIt = m_particles.begin(); partIt!=m_particles.end(); partIt++,i++)
		{
			const CPose3D	*partPose = getLastPose(i);		// Take the particle data:
			partIt->log_w += PF_options.powFactor * auxiliarComputeObservationLikelihood(PF_options,this,i,sf,partPose);
		} // for each particle "i"
	} // if update not already done...

	printf("Ok\n");

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
			prediction_and_update_pfAuxiliaryPFOptimal
 ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::prediction_and_update_pfAuxiliaryPFOptimal(
	const mrpt::slam::CActionCollection	* actions,
	const mrpt::slam::CSensoryFrame		* sf,
	const bayes::CParticleFilter::TParticleFilterOptions &PF_options )
{
	MRPT_TRY_START;

	size_t				i,k,N,M = m_particles.size();
	std::vector<vector_float>	rndSamples;

	// ----------------------------------------------------------------------
	//	  We can execute optimal PF only when we have both, an action, and
	//     a valid observation from which to compute the likelihood:
	//   Accumulate odometry/actions until we have a valid observation, then
	//    process them simultaneously.
	// ----------------------------------------------------------------------
	static CActionRobotMovement2D	accumRobotMovement;
	static bool						accumRobotMovementIsValid = false;
	bool  SFhasValidObservations = false;
	// A valid action?
	if (actions!=NULL)
	{
		CActionRobotMovement2DPtr act = actions->getBestMovementEstimation();	// Find a robot movement estimation:
		if (!act) THROW_EXCEPTION("Action list does not contain any CActionRobotMovement2D derived object!");

		if (!accumRobotMovementIsValid) // Reset accum.
		{
				accumRobotMovement.rawOdometryIncrementReading = act->poseChange->getEstimatedPose();
				accumRobotMovement.motionModelConfiguration = act->motionModelConfiguration;
		}
		else 	accumRobotMovement.rawOdometryIncrementReading = accumRobotMovement.rawOdometryIncrementReading + act->poseChange->getEstimatedPose();

		accumRobotMovementIsValid = true;
	}

	if (sf!=NULL)
	{
		ASSERT_(m_particles.size()>0);
		SFhasValidObservations = (*m_particles.begin()).d->mapTillNow.canComputeObservationsLikelihood( *sf );
	}

	// All the needed things?
	if (!accumRobotMovementIsValid || !SFhasValidObservations)
		return; // Nothing we can do here...

	// OK, we have all we need, let's start!

	// Take the accum. actions as input:
	CActionRobotMovement2D		theResultingRobotMov;
	theResultingRobotMov.computeFromOdometry( accumRobotMovement.rawOdometryIncrementReading, accumRobotMovement.motionModelConfiguration );

	const CActionRobotMovement2D		*robotMovement = &theResultingRobotMov;

	accumRobotMovementIsValid = false; // To reset odometry at next iteration!


	// ----------------------------------------------------------------------
	//		0) Common part:  Prepare m_particles "draw" and compute
	// ----------------------------------------------------------------------
	m_movementDrawer.setPosePDF( robotMovement->poseChange );

	m_pfAuxiliaryPFOptimal_estimatedProb.resize(M);
	m_maxLikelihood.clear();
	m_maxLikelihood.resize(M,0);
	m_movementDrawMaximumLikelihood.resize(M);

	// Prepare data for executing "fastDrawSample"
	CTicTac		tictac;
	printf("[prepareFastDrawSample] Computing...");
	tictac.Tic();
	prepareFastDrawSample(
		PF_options,
		particlesEvaluator_AuxPFOptimal,
		NULL, // Not used // robotMovement,
		sf );
	printf("Done! in %.06f ms\n",tictac.Tac()*1e3f);

#if 1  	/** DEBUG **/
	printf("[prepareFastDrawSample] max      (log) = %10.06f\n",  math::maximum(m_pfAuxiliaryPFOptimal_estimatedProb) );
	printf("[prepareFastDrawSample] max-mean (log) = %10.06f\n", -math::mean(m_pfAuxiliaryPFOptimal_estimatedProb) + math::maximum(m_pfAuxiliaryPFOptimal_estimatedProb) );
	printf("[prepareFastDrawSample] max-min  (log) = %10.06f\n", -math::minimum(m_pfAuxiliaryPFOptimal_estimatedProb) + math::maximum(m_pfAuxiliaryPFOptimal_estimatedProb) );
#endif	/*****/

	// Now we have the vector "m_fastDrawProbability" filled out with:
	//     w[i]·p(zt|z^{t-1},x^{[i],t-1},X)
	//  where X is the robot pose prior (as implemented in
	//  the aux. function "particlesEvaluator_AuxPFOptimal"),
	//  and also the "m_maxLikelihood" filled with the maximum lik. values.

	vector<CPose3D>			newParticles;
	vector_double			newParticlesWeight;
	vector<size_t>			newParticlesDerivedFromIdx;

	// We need the (aproximate) maximum likelihood value for each
	//  previous particle [i]:
	//
	//     max{ p( z^t | data^[i], x_(t-1)^[i], u_(t) ) }
	//
	//vector_double					maxLikelihood(M, -1 );

	float							MIN_ACCEPT_UNIF_DISTRIB = 0.00f;

	CPose3D							movementDraw,newPose,oldPose;
	double							acceptanceProb,newPoseLikelihood,ratioLikLik;
	unsigned int					statsTrialsCount = 0, statsTrialsSuccess = 0;
	std::vector<bool>				maxLikMovementDrawHasBeenUsed(M, false);
	unsigned int					statsWarningsAccProbAboveOne = 0;
	double							maxMeanLik = math::maximum( m_pfAuxiliaryPFOptimal_estimatedProb ); // For normalization purposes only

	if ( !PF_options.adaptiveSampleSize )
	{
		// ----------------------------------------------------------------------
		//						1) FIXED SAMPLE SIZE VERSION
		// ----------------------------------------------------------------------
		newParticles.resize(M);
		newParticlesWeight.resize(M);
		newParticlesDerivedFromIdx.resize(M);

		bool	doResample = ESS() < 0.5;

		for (i=0;i<M;i++)
		{
			// Generate a new particle:
			//   (a) Draw a "t-1" m_particles' index:
			// ----------------------------------------------------------------
			if (doResample)
					k = fastDrawSample(PF_options);		// Based on weights of last step only!
			else	k = i;

			oldPose = *getLastPose(k);

            //   (b) Rejection-sampling: Draw a new robot pose from x[k],
			//       and accept it with probability p(zk|x) / maxLikelihood:
			// ----------------------------------------------------------------
			if ( !SFs.size() )
			{
				// The first robot pose in the SLAM execution: All m_particles start
				// at the same point (this is the lowest bound of subsequent uncertainty):
				movementDraw = CPose3D(0,0,0);
				newPose = oldPose + movementDraw;
			}
			else
			{
				// Rejection-sampling:
				do
				{
					// Draw new robot pose:
					if (!maxLikMovementDrawHasBeenUsed[k])
					{
						// No! first take advantage of a good drawn value, but only once!!
						maxLikMovementDrawHasBeenUsed[k] = true;
						movementDraw = m_movementDrawMaximumLikelihood[k];
#if 0
						cout << "Drawn pose (max. lik): " << movementDraw << endl;
#endif
					}
					else
					{
						// Draw new robot pose:
						m_movementDrawer.drawSample( movementDraw );
					}

					newPose = oldPose + movementDraw;

					// Compute acceptance probability:
					newPoseLikelihood = auxiliarComputeObservationLikelihood(PF_options, this,k,sf,&newPose);
					ratioLikLik = exp( newPoseLikelihood - m_maxLikelihood[k] );
					acceptanceProb = min( 1.0, ratioLikLik );

					if ( ratioLikLik > 1)
					{
						if (ratioLikLik>1.5)
						{
							statsWarningsAccProbAboveOne++;
							// DEBUG
							//printf("[pfAuxiliaryPFOptimal] Warning!! p(z|x)/p(z|x*)=%f\n",ratioLikLik);
						}
						m_maxLikelihood[k] = newPoseLikelihood; //  :'-( !!!
						acceptanceProb = 0;		// Keep searching!!
					}

					statsTrialsCount++;		// Stats

				} while ( acceptanceProb < mrpt::random::RandomUni(MIN_ACCEPT_UNIF_DISTRIB,0.999f) );

				statsTrialsSuccess++;		// Stats:

			}

			// Insert the new particle!:
			newParticles[i] = newPose;

			// And its weight:
			double	weightFact = m_pfAuxiliaryPFOptimal_estimatedProb[k] * PF_options.powFactor;

			if (doResample)
					newParticlesWeight[i] = 0;
			else	newParticlesWeight[i] = m_particles[k].log_w + weightFact;

			// and its heritance:
			newParticlesDerivedFromIdx[i] = k;

		} // for i

	} // end fixed sample size
	else
	{
		// -------------------------------------------------------------------------------------------------
		//				 				2) ADAPTIVE SAMPLE SIZE VERSION
		//
		//	Implementation of Dieter Fox's KLD algorithm
		//		JLBC (3/OCT/2006)
		// TODO: Optimize the management of bins (tree?)
		// -------------------------------------------------------------------------------------------------
        // Create a (x,y,phi) bins storage:
		std::deque<TPathBin>					pathBins;
		std::deque<TPathBin>::iterator			it;
		TPathBin								p;

		// The new particle set:
		newParticles.clear();
		newParticlesWeight.clear();
		newParticlesDerivedFromIdx.clear();

		// ------------------------------------------------------------------------------
		// 2.1) PRELIMINAR STAGE: Build a list of pairs<TPathBin,vector_uint> with the
		//      indexes of m_particles that fall into each multi-dimensional-path bins
		//      //The bins will be saved into "pathBinsLastTimestep", and the list
		//      //of corresponding m_particles (in the last timestep), in "pathBinsLastTimestepParticles"
		//  - Added JLBC (01/DEC/2006)
		// ------------------------------------------------------------------------------
/**/
		std::deque<TPathBin>					pathBinsLastTimestep;
		std::deque<vector_uint>					pathBinsLastTimestepParticles;
		CParticleList::iterator		partIt;
		unsigned int							partIndex;
		unsigned int							minPartsPerBin = 1;		// The minimum amount of m_particles per path-bin

		printf( "[FIXED_SAMPLING] Computing...\n");
		for (partIt = m_particles.begin(),partIndex=0; partIt!=m_particles.end(); partIt++,partIndex++)
		{
			// Load the bin from the path data:
			loadTPathBinFromPath( p, &partIt->d->robotPath );

			// Is a new bin?
			int		posFound = findTPathBinIntoSet( p, pathBinsLastTimestep );
			if ( -1 == posFound )
			{
				// Yes, create a new pair <bin,index_list> in the list:
				pathBinsLastTimestep.push_back( p );
				pathBinsLastTimestepParticles.push_back( vector_uint(1,partIndex) );
			}
			else
			{
				// No, add the particle's index to the existing entry:
				pathBinsLastTimestepParticles[posFound].push_back( partIndex );
			}
		}
/**/

		// ------------------------------------------------------------------------------
		// 2.2)    THE MAIN KLD-BASED DRAW SAMPLING LOOP
		// ------------------------------------------------------------------------------
		double									delta_1 = 1.0 - options.KLD_params.KLD_delta;
		double									epsilon_1 = 0.5 / options.KLD_params.KLD_epsilon;
		bool									doResample = ESS() < 0.5;
		//double									maxLik = math::maximum(m_maxLikelihood); // For normalization purposes only
		k = 0;
		N = 0;

		// The desired dynamic number of m_particles (to be modified dynamically below):
		unsigned int							Nx = options.KLD_params.KLD_minSampleSize;
		Nx = max((size_t)options.KLD_params.KLD_minSampleSize, minPartsPerBin*pathBinsLastTimestep.size() );

		vector_size_t permutationAuxVector( m_particles.size() );
		for (k=0;k<m_particles.size();k++)
			permutationAuxVector[k]=k;

		vector_size_t permutationPathsAuxVector( pathBinsLastTimestepParticles.size() );
		for (k=0;k<pathBinsLastTimestepParticles.size();k++)
			permutationPathsAuxVector[k]=k;

		do // "N" is the index of the current "new particle":
		{
			// Generate a new particle:
			//
			//   (a) Propagate the last set of m_particles, and only if the
			//       desired number of m_particles in this step is larger,
			//       perform a UNIFORM sampling from the last set. In this way
			//       the new weights can be computed in the same way for all m_particles.
			// ---------------------------------------------------------------------------
			if (doResample)
			{
				k = fastDrawSample(PF_options);		// Based on weights of last step only!
			}
			else
			{
				// Assure that at least one particle per "discrete-path" is taken (if the
				//  number of samples allows it)
				if (permutationPathsAuxVector.size())
				{
					size_t idx = (size_t)floor(mrpt::random::RandomUni(0, permutationPathsAuxVector.size()-0.05f ));
					size_t nPath = permutationPathsAuxVector[idx];
					permutationPathsAuxVector.erase( permutationPathsAuxVector.begin() + idx );

					size_t idx2 = (size_t)floor(mrpt::random::RandomUni(0, pathBinsLastTimestepParticles[nPath].size()-0.05f ));
					k = pathBinsLastTimestepParticles[nPath][idx2];
					ASSERT_(k<m_particles.size());

					// Also erase from the other permutation vector list:
					bool		erased = false;
					for (vector_size_t::iterator it=permutationAuxVector.begin();!erased && it!=permutationAuxVector.end();it++)
						if (*it==k)
						{
							permutationAuxVector.erase( it );
							break;
						}

					// ADD-ON: If the 'm_pfAuxiliaryPFOptimal_estimatedProb[k]' is **extremelly** low relative to the other m_particles,
					//  resample only this particle with a copy of another one, uniformly:
					if ( (m_pfAuxiliaryPFOptimal_estimatedProb[k]-maxMeanLik) < -20 )
					{
						// Select another 'k' uniformly:
						k = (size_t)floor(mrpt::random::RandomUni(0, m_particles.size()-0.05f ));
					}
				}
				else
				{
					// Select a particle from the previous set with a UNIFORM distribution,
					// in such a way we will assign each particle the updated weight accounting
					// for its last weight.
					// The first "old_N" m_particles will be drawn using a uniform random selection
					// without repetitions:
					//
					// Select a index from "permutationAuxVector" and remove it from the list:
					if (permutationAuxVector.size())
					{
						unsigned int idx = (unsigned int)floor(mrpt::random::RandomUni(0, permutationAuxVector.size()-0.05f ));
						k = permutationAuxVector[idx];
						permutationAuxVector.erase( permutationAuxVector.begin() + idx );
					}
					else
					{
						// N>N_old -> Uniformly draw index:
						k = (size_t)floor(mrpt::random::RandomUni(0, m_particles.size()-0.05f ));
						ASSERT_(k<m_particles.size());
					}
				}
			}


			// Get the last robot pose for particle [k]:
			oldPose = *getLastPose(k);

            //   (b) Rejection-sampling: Draw a new robot pose from x[k],
			//       and accept it with probability p(zk|x) / maxLikelihood:
			// ----------------------------------------------------------------
			if ( !SFs.size() )
			{
				// The first robot pose in the SLAM execution: All m_particles start
				// at the same point (this is the lowest bound of subsequent uncertainty):
				movementDraw = CPose3D(0,0,0);
				newPose = oldPose; // + movementDraw;
			}
			else
			{
				// Rejection-sampling:
				do
				{
					// Draw new robot pose:
					if (!maxLikMovementDrawHasBeenUsed[k])
					{
						// No! first take advantage of a good drawn value, but only once!!
						maxLikMovementDrawHasBeenUsed[k] = true;
						movementDraw = m_movementDrawMaximumLikelihood[k];
					}
					else
					{
						// Draw new robot pose:
						m_movementDrawer.drawSample( movementDraw );
					}

					newPose = oldPose + movementDraw;

					// Compute acceptance probability:
					newPoseLikelihood = auxiliarComputeObservationLikelihood(PF_options,this,k,sf,&newPose);
					ratioLikLik = exp( newPoseLikelihood - m_maxLikelihood[k] );
					acceptanceProb = min( 1.0, ratioLikLik );

					if ( ratioLikLik > 1)
					{
						if (ratioLikLik>1.5)
						{
							statsWarningsAccProbAboveOne++;
							// DEBUG
							//printf("[pfAuxiliaryPFOptimal] Warning!! p(z|x)/p(z|x*)=%f\n",ratioLikLik);
						}
						m_maxLikelihood[k] = newPoseLikelihood; //  :'-( !!!
						acceptanceProb = 0;		// Keep searching!!
					}

					statsTrialsCount++;		// Stats:

				} while ( acceptanceProb < mrpt::random::RandomUni(MIN_ACCEPT_UNIF_DISTRIB,0.999f) );
			}

			statsTrialsSuccess++;		// Stats:

			/*** DEBUG ****/
			printf(".");
			/**************/

			// Insert the new particle!:
			newParticles.push_back( newPose );

			// And its weight:
			double	weightFact = m_pfAuxiliaryPFOptimal_estimatedProb[k] * PF_options.powFactor;

/*			if (doResample)
					newParticlesWeight.push_back( weightFact  );
			else	newParticlesWeight.push_back( m_particles[k].w * weightFact  );
*/
			if (doResample)
					newParticlesWeight.push_back( 0 );
			else	newParticlesWeight.push_back( m_particles[k].log_w + weightFact  );

			// and its heritance:
			newParticlesDerivedFromIdx.push_back( k );

			// Now, look if the particle's PATH falls into a new bin or not:
			// ----------------------------------------------------------------
			//loadTPathBinFromPath( p, &m_particles[k].d->robotPath, &newPose );
			loadTPathBinFromPath( p, NULL, &newPose );

			// -----------------------------------------------------------------------------
			// Look for the bin "p" into "pathBins": If it is not yet into the set,
			//  then we may increase the desired particle number:
			// -----------------------------------------------------------------------------

			// Found?
			if ( findTPathBinIntoSet( p, pathBins ) == -1 )
			{
				// It falls into a new bin:
				// Add to the poseBins:
				pathBins.push_back( p );

				// K = K + 1
				int K = pathBins.size();
				if ( K>1) //&& newParticles.size() > options.KLD_params.KLD_minSampleSize )
				{
					// Update the number of m_particles!!
					Nx = (unsigned int) (epsilon_1 * math::chiSquareQuantile(delta_1,K-1));
				}
			}

			N = newParticles.size();

		} while ((  N < options.KLD_params.KLD_maxSampleSize &&
					N < max(Nx,options.KLD_params.KLD_minSampleSize)) ||
					(permutationPathsAuxVector.size() && !doResample) );

		printf("\n[ADAPTIVE SAMPLE SIZE]  #Bins: %u \t #Particles: %u \t Nx=%u\n", static_cast<unsigned>(pathBins.size()),static_cast<unsigned>(N), Nx);
	} // end adaptive sample size


	// ---------------------------------------------------------------------------------
	// Substitute old by new particle set:
	//   Old are in "m_particles"
	//   New are in "newParticles", "newParticlesWeight","newParticlesDerivedFromIdx"
	// ---------------------------------------------------------------------------------
	N = newParticles.size();
	CParticleList				newParticlesArray(N);
	CParticleList::iterator		newPartIt,trgPartIt;

	// For efficiency, just copy the "CRBPFParticleData" from the old particle into the
	//  new one, but this can be done only once:
	std::vector<bool>						oldParticleAlreadyCopied(m_particles.size(),false);
	CRBPFParticleData						*newPartData;

	for (newPartIt=newParticlesArray.begin(),i=0;newPartIt!=newParticlesArray.end();newPartIt++,i++)
	{
		// The weight:
		newPartIt->log_w = newParticlesWeight[i];

		// The data (CRBPFParticleData):
		if (!oldParticleAlreadyCopied[newParticlesDerivedFromIdx[i]])
		{
			// The first copy of this old particle:
			newPartData = m_particles[ newParticlesDerivedFromIdx[i] ].d;
            oldParticleAlreadyCopied[newParticlesDerivedFromIdx[i]] = true;
		}
		else
		{
			// Make a copy:
			newPartData = new CRBPFParticleData( *m_particles[ newParticlesDerivedFromIdx[i] ].d );
		}

		newPartIt->d = newPartData;
	} // end for "newPartIt"


	// Now add the new robot pose to the paths: (this MUST be done after the above loop, separately):
	for (newPartIt=newParticlesArray.begin(),i=0;newPartIt!=newParticlesArray.end();newPartIt++,i++)
		newPartIt->d->robotPath.push_back( newParticles[i] );


	// Free those old m_particles not being copied into the new ones:
	for (i=0;i<m_particles.size();i++)
	{
		if (!oldParticleAlreadyCopied[i])
		{
			delete m_particles[ i ].d;
			m_particles[ i ].d = NULL;
		}
	}

	// Copy into "m_particles"
	// Copy into "m_particles":
	m_particles.resize( newParticlesArray.size() );
	for (newPartIt=newParticlesArray.begin(),trgPartIt=m_particles.begin(); newPartIt!=newParticlesArray.end(); newPartIt++, trgPartIt++ )
	{
		trgPartIt->log_w = newPartIt->log_w;
		trgPartIt->d = newPartIt->d;
		newPartIt->d = NULL;
	}
	//m_particles = newParticlesArray;

	// Free buffers:
	newParticles.clear();
	newParticlesArray.clear();
	newParticlesWeight.clear();
	newParticlesDerivedFromIdx.clear();

	normalizeWeights();			// Normalize weights:

	/** DEBUG **/
	printf("[REJ-SAMP.RATIO: \t%.03f%% \t %u out of %u with P(acep)>1]\n\n",
		100.0f*statsTrialsSuccess / static_cast<float>(max(1u,statsTrialsCount)),
		statsWarningsAccProbAboveOne,
		statsTrialsCount
		);
	/****/

	MRPT_TRY_END;
}


/*---------------------------------------------------------------
						getLastPose
 ---------------------------------------------------------------*/
const CPose3D* CMultiMetricMapPDF::getLastPose(const size_t &i) const
{
	if (i>=m_particles.size()) THROW_EXCEPTION("Particle index out of bounds!");

	size_t	n = m_particles[i].d->robotPath.size();

	if (n)
			return &m_particles[i].d->robotPath[n-1];
	else	return NULL;
}

/*---------------------------------------------------------------
						getCurrentMetricMapEstimation
 ---------------------------------------------------------------*/
CMultiMetricMap *  CMultiMetricMapPDF::getCurrentMetricMapEstimation( )
{
	rebuildAverageMap();

	return &averageMap;
}


/*---------------------------------------------------------------
						getWeightedAveragedMap
 ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::rebuildAverageMap()
{
	//size_t				M = particlesCount();
	float				min_x = 1e6, max_x=-1e6, min_y = 1e6, max_y = -1e6;
	CParticleList::iterator	part;

	if (averageMapIsUpdated)
		return;

//	CTicTac		tictac;
//	tictac.Tic();
//	printf("[EM...");

	// ---------------------------------------------------------
	//					GRID
	// ---------------------------------------------------------
	for (part=m_particles.begin();part!=m_particles.end();part++)
	{
		ASSERT_( part->d->mapTillNow.m_gridMaps.size()>0 );

		min_x = min( min_x, part->d->mapTillNow.m_gridMaps[0]->getXMin() );
		max_x = max( max_x, part->d->mapTillNow.m_gridMaps[0]->getXMax() );
		min_y = min( min_y, part->d->mapTillNow.m_gridMaps[0]->getYMin() );
		max_y = max( max_y, part->d->mapTillNow.m_gridMaps[0]->getYMax() );
	}

	// Asure all maps have the same dimensions:
	for (part=m_particles.begin();part!=m_particles.end();part++)
		part->d->mapTillNow.m_gridMaps[0]->resizeGrid(min_x,max_x,min_y,max_y,0.5f,false);

	for (part=m_particles.begin();part!=m_particles.end();part++)
	{
		min_x = min( min_x, part->d->mapTillNow.m_gridMaps[0]->getXMin() );
		max_x = max( max_x, part->d->mapTillNow.m_gridMaps[0]->getXMax() );
		min_y = min( min_y, part->d->mapTillNow.m_gridMaps[0]->getYMin() );
		max_y = max( max_y, part->d->mapTillNow.m_gridMaps[0]->getYMax() );
	}

	// Prepare target map:
	ASSERT_( averageMap.m_gridMaps.size()>0 );
	averageMap.m_gridMaps[0]->setSize(
		min_x,
		max_x,
		min_y,
		max_y,
		m_particles[0].d->mapTillNow.m_gridMaps[0]->getResolution(),
		0);

	// Compute the sum of weights:
	double		sumLinearWeights = 0;
	for (part=m_particles.begin();part!=m_particles.end();part++)
		sumLinearWeights += exp( part->log_w );

	// CHECK:
	for (part=m_particles.begin();part!=m_particles.end();part++)
	{
		ASSERT_(part->d->mapTillNow.m_gridMaps[0]->getSizeX() == averageMap.m_gridMaps[0]->getSizeX());
		ASSERT_(part->d->mapTillNow.m_gridMaps[0]->getSizeY() == averageMap.m_gridMaps[0]->getSizeY());
	}


#ifdef MRPT_HAS_SSE
	// Autodetect SSE support:
	bool	SSE_SUPORTED = checkSuportSSE();
	if (SSE_SUPORTED)
	{
        MRPT_TRY_START;

		// ******************************************************
		//     Implementation with the SSE Instructions Set
		//  http://www.rz.uni-karlsruhe.de/rz/docs/VTune/reference/
		//  JLBC - 16/MAR/2006
		// ******************************************************

		// For each particle in the RBPF:
		part=m_particles.begin();
		do
		{
			// The cells in the source map:
			unsigned short*		srcCell;
			unsigned short*		firstSrcCell = (unsigned short*)&(*part->d->mapTillNow.m_gridMaps[0]->map.begin());
			unsigned short*		lastSrcCell = (unsigned short*)&(*part->d->mapTillNow.m_gridMaps[0]->map.end());

			// Assure sizes:
			ASSERT_(0==(part->d->mapTillNow.m_gridMaps[0]->size_x % 8));
			ASSERT_(0==(averageMap.m_gridMaps[0]->size_x % 8));
			ASSERT_(averageMap.m_gridMaps[0]->map.size()==part->d->mapTillNow.m_gridMaps[0]->map.size());

			// The destination cells:
			unsigned short*		destCell;
			unsigned short*		firstDestCell = (unsigned short*)&averageMap.m_gridMaps[0]->map[0];

			// The weight of particle:
			__declspec(align(16)) unsigned short weights_array[8];
			weights_array[0] = weights_array[1] = weights_array[2] = weights_array[3] =
			weights_array[4] = weights_array[5] = weights_array[6] = weights_array[7] = (unsigned short)(exp(part->log_w) * 65535 / sumLinearWeights);
			unsigned short*		weights_8 = weights_array;

			srcCell = firstSrcCell;
			destCell = firstDestCell;
			//size_t	cellsCount = averageMap.gridMap->map.size();

			// For each cell in individual maps:
			__asm
			{
				push	eax
				push	edx
				push	ecx

				mov		eax, dword ptr[srcCell]
				mov		edx, dword ptr[destCell]
				mov     ecx, dword ptr[weights_8]

				_loop:

				// Load 8 cells from the particle's map:
				movdqu	xmm0, [eax]

				// Move 8 copies of the weight to XMM1
				movdqu	xmm1, [ecx]

				// Packed multiply, Unsigned, 16bits
				//  Multiply the 8 cells in XMM1 with the associated weight in XMM0, and save it in XMM1
				pmulhuw	xmm1, xmm0

				// Packed ADD with saturation, Unsigned, 16bits
				//  Accumulate the Expected Map in XMM3:
				paddusw	xmm1, [edx]

				// Store again in the expected map:
				movdqu	[edx],xmm1

				// Increment pointers:
				add		eax, 16
				add		edx, 16

				// End of loop:
				cmp		eax, dword ptr[lastSrcCell]
				jnz		_loop

				// Clear MMX flags
				emms

				pop		ecx
				pop		edx
				pop		eax
			}

			// Next particle:
			part++;
		}  while (part!=m_particles.end());

		MRPT_TRY_END_WITH_CLEAN_UP( printf("Exception in Expected Map computation with SSE!! Dumping variables:\n EM size in cells: %ux%u\n",averageMap.m_gridMaps[0]->size_x,averageMap.m_gridMaps[0]->size_y); printf("Particle map: %ux%u\n",part->d->mapTillNow.m_gridMaps[0]->size_x,part->d->mapTillNow.m_gridMaps[0]->size_y); );

	}	// End of SSE supported
	else
#endif
	{
		// ******************************************************
		//     Implementation WITHOUT the SSE Instructions Set
		// ******************************************************

		MRPT_TRY_START;

		// Reserve a float grid-map, add weight all maps
		// -------------------------------------------------------------------------------------------
		std::vector<float>	floatMap;
		floatMap.resize(averageMap.m_gridMaps[0]->map.size(),0);

		// For each particle in the RBPF:
		double		sumW = 0;
		for (part=m_particles.begin();part!=m_particles.end();part++)
			sumW+=exp(part->log_w);

		if (sumW==0) sumW=1;

		for (part=m_particles.begin();part!=m_particles.end();part++)
		{
			// Variables:
			std::vector<COccupancyGridMap2D::cellType>::iterator	srcCell;
			std::vector<COccupancyGridMap2D::cellType>::iterator	firstSrcCell = part->d->mapTillNow.m_gridMaps[0]->map.begin();
			std::vector<COccupancyGridMap2D::cellType>::iterator	lastSrcCell = part->d->mapTillNow.m_gridMaps[0]->map.end();
			std::vector<float>::iterator							destCell;

			// The weight of particle:
			float		w =  exp(part->log_w) / sumW;

			ASSERT_( part->d->mapTillNow.m_gridMaps[0]->map.size() == floatMap.size() );

			// For each cell in individual maps:
			for (srcCell = firstSrcCell, destCell = floatMap.begin();srcCell!=lastSrcCell;srcCell++,destCell++)
				(*destCell) += w * (*srcCell);

		}

		// Copy to fixed point map:
		std::vector<float>::iterator							srcCell;
		std::vector<COccupancyGridMap2D::cellType>::iterator	destCell = averageMap.m_gridMaps[0]->map.begin();

		ASSERT_( averageMap.m_gridMaps[0]->map.size() == floatMap.size() );

		for (srcCell=floatMap.begin();srcCell!=floatMap.end();srcCell++,destCell++)
			*destCell = static_cast<COccupancyGridMap2D::cellType>( *srcCell );

		MRPT_TRY_END;
	}	// End of SSE not supported

	// ---------------------------------------------------------
	//					POINTS
	// ---------------------------------------------------------
/*	averageMap.pointsMap->clear();
	averageMap.pointsMap->insertionOptions.fuseWithExisting = false;
	averageMap.pointsMap->insertionOptions.matchStaticPointsOnly = false;

	for (i=0;i<M;i++)
	{
		averageMap.pointsMap->fuseWith( m_particles[i].d->mapTillNow.pointsMap );
	}
*/

//	printf("%.03fms]",1000*tictac.Tac());

	// Don't calculate again until really necesary.
	averageMapIsUpdated = true;
}

/*---------------------------------------------------------------
						insertObservation
 ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::insertObservation(CSensoryFrame	&sf,bool forceInsertion)
{
	MRPT_UNUSED_PARAM(forceInsertion);

	int			i,M = particlesCount();

	// Insert into SFs:
	CPose3DPDFParticlesPtr posePDF = CPose3DPDFParticlesPtr( new CPose3DPDFParticles() );
	getEstimatedPosePDF(*posePDF);

	// Insert it into the SFs and the SF2robotPath list:
	SFs.insert(
		posePDF,
		CSensoryFramePtr( new CSensoryFrame(sf) ) );
	SF2robotPath.push_back( m_particles[0].d->robotPath.size()-1 );

	for (i=0;i<M;i++)
	{
		CPose3D		robotPose(*getLastPose(i));

		// Set options:
		// --------------
		//m_particles[i].d->mapTillNow.gridMap->insertionOptions.isStaticGridMap = false; //true;
		/*if (m_particles[i].d->mapTillNow.m_pointsMap)
		{
			m_particles[i].d->mapTillNow.m_pointsMap->insertionOptions.minDistBetweenLaserPoints	= 0.04f;
			m_particles[i].d->mapTillNow.m_pointsMap->insertionOptions.also_interpolate				= false;
			m_particles[i].d->mapTillNow.m_pointsMap->insertionOptions.matchStaticPointsOnly		= false;
			m_particles[i].d->mapTillNow.m_pointsMap->insertionOptions.fuseWithExisting				= true;
			m_particles[i].d->mapTillNow.m_pointsMap->insertionOptions.disableDeletion				= true;
			m_particles[i].d->mapTillNow.m_pointsMap->insertionOptions.isPlanarMap					= true;
		}*/

		// Insert to desired maps only:
		//m_particles[i].d->mapTillNow.mapsUsage = options.mapsUsage;

		sf.insertObservationsInto( &m_particles[i].d->mapTillNow, &robotPose );

	}

	averageMapIsUpdated = false;
}

/*---------------------------------------------------------------
						getPath
 ---------------------------------------------------------------*/
void	 CMultiMetricMapPDF::getPath(size_t i, std::deque<CPose3D> &out_path) const
{
	if (i>=m_particles.size())
		THROW_EXCEPTION("Index out of bounds");
	out_path = m_particles[i].d->robotPath;
}

/*---------------------------------------------------------------
					getCurrentEntropyOfPaths
  ---------------------------------------------------------------*/
double  CMultiMetricMapPDF::getCurrentEntropyOfPaths()
{
	size_t				i;
	size_t				N=m_particles[0].d->robotPath.size();			// The poses count along the paths
	

	// Compute paths entropy:
	// ---------------------------
	double	H_paths = 0;

	if (N)
	{
		// For each pose along the path:
		for (i=0;i<N;i++)
		{
			// Get pose est. as m_particles:
			CPose3DPDFParticles	posePDFParts;
			getEstimatedPosePDFAtTime(i,posePDFParts);

			// Approximate to gaussian and compute entropy of covariance:
			H_paths+= posePDFParts.getEstimatedCovarianceEntropy();
		}
		H_paths /= N;
	}
	return H_paths;
}

/*---------------------------------------------------------------
					getCurrentJointEntropy
  ---------------------------------------------------------------*/
double  CMultiMetricMapPDF::getCurrentJointEntropy()
{
	double								H_joint,H_paths,H_maps;
	size_t								i,M = m_particles.size();
	COccupancyGridMap2D::TEntropyInfo	entropy;

	// Entropy of the paths:
	H_paths = getCurrentEntropyOfPaths();


	float				min_x = 1e6, max_x=-1e6, min_y = 1e6, max_y = -1e6;
	CParticleList::iterator	part;

	// ---------------------------------------------------------
	//			ASSURE ALL THE GRIDS ARE THE SAME SIZE!
	// ---------------------------------------------------------
	for (part=m_particles.begin();part!=m_particles.end();part++)
	{
		ASSERT_( part->d->mapTillNow.m_gridMaps.size()>0 );

		min_x = min( min_x, part->d->mapTillNow.m_gridMaps[0]->getXMin() );
		max_x = max( max_x, part->d->mapTillNow.m_gridMaps[0]->getXMax() );
		min_y = min( min_y, part->d->mapTillNow.m_gridMaps[0]->getYMin() );
		max_y = max( max_y, part->d->mapTillNow.m_gridMaps[0]->getYMax() );
	}

	// Asure all maps have the same dimensions:
	for (part=m_particles.begin();part!=m_particles.end();part++)
		part->d->mapTillNow.m_gridMaps[0]->resizeGrid(min_x,max_x,min_y,max_y,0.5f,false);


	// Sum of linear weights:
	double	sumLinearWeights = 0;
	for (i=0;i<M;i++)
		sumLinearWeights += exp(m_particles[i].log_w);

	// Compute weighted maps entropy:
	// --------------------------------
	H_maps = 0;
	for (i=0;i<M;i++)
	{
		ASSERT_( m_particles[i].d->mapTillNow.m_gridMaps.size()>0 );

		m_particles[i].d->mapTillNow.m_gridMaps[0]->computeEntropy( entropy );
		H_maps += exp(m_particles[i].log_w) * entropy.H / sumLinearWeights;
	}

	printf("H_paths=%e\n",H_paths);
	printf("H_maps=%e\n",H_maps);

	H_joint = H_paths + H_maps;
	return	H_joint;
}

/*---------------------------------------------------------------
					Entropy aux. function
 ---------------------------------------------------------------*/
float  CMultiMetricMapPDF::H(float p)
{
	if (p==0 || p==1)	return 0;
	else				return -p*log(p);
}


/*---------------------------------------------------------------
					getCurrentMostLikelyMetricMap
  ---------------------------------------------------------------*/
CMultiMetricMap  * CMultiMetricMapPDF::getCurrentMostLikelyMetricMap( )
{
	size_t		i,max_i=0, n = m_particles.size();
	double		max_w = m_particles[0].log_w;

	for (i=0;i<n;i++)
	{
		if ( m_particles[i].log_w > max_w )
		{
			max_w = m_particles[i].log_w;
			max_i = i;
		}
	}

	// Return its map:
	return &m_particles[max_i].d->mapTillNow;
}

/*---------------------------------------------------------------
				updateSensoryFrameSequence
  ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::updateSensoryFrameSequence()
{
	MRPT_TRY_START
	CPose3DPDFParticles	posePartsPDF;
	CPose3DPDFPtr		previousPosePDF;
	CSensoryFramePtr	dummy;

	for (size_t i=0;i<SFs.size();i++)
	{
		// Get last estimation:
		SFs.get(i,previousPosePDF,dummy);

		// Compute the new one:
		getEstimatedPosePDFAtTime(SF2robotPath[i], posePartsPDF);

		// Copy into SFs:
		previousPosePDF->copyFrom( posePartsPDF );
	}

	MRPT_TRY_END
}

/*---------------------------------------------------------------
				saveCurrentPathEstimationToTextFile
  ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::saveCurrentPathEstimationToTextFile( const std::string  &fil )
{
	FILE	*f=os::fopen( fil.c_str(), "wt");
	if (!f) return;

	for (CParticleList::iterator it=m_particles.begin();it!=m_particles.end();it++)
	{
		for (size_t i=0;i<it->d->robotPath.size();i++)
		{
			CPose3D  &p = it->d->robotPath[i];

			os::fprintf(f,"%.04f %.04f %.04f %.04f %.04f %.04f ", 
				p.x,p.y,p.z, 
				p.yaw, p.pitch, p.roll );
		}
		os::fprintf(f," %e\n", it->log_w );
	}

	os::fclose(f);
}

/*---------------------------------------------------------------
			particlesEvaluator_AuxPFOptimal
 ---------------------------------------------------------------*/
double  CMultiMetricMapPDF::particlesEvaluator_AuxPFOptimal(
	const bayes::CParticleFilter::TParticleFilterOptions &PF_options,
	const CParticleFilterCapable	*obj,
	size_t					index,
	const void				*action,
	const void				*observation )
{
	MRPT_UNUSED_PARAM(action);

	MRPT_TRY_START;

	const CMultiMetricMapPDF	*myObj = static_cast<const CMultiMetricMapPDF*>( obj );
//	CActionRobotMovement2D	*act   = (CActionRobotMovement2D*) action;  // Not needed, since we use the "movementDrawer"

	// Compute the quantity:
	//     w[i]·p(zt|z^{t-1},x^{[i],t-1})
	// As the Monte-Carlo approximation of the integral over all posible $x_t$.
	// --------------------------------------------
	double  indivLik, maxLik= -1e300;
	CPose3D maxLikDraw;
	size_t  N = PF_options.pfAuxFilterOptimal_MaximumSearchSamples;

	ASSERT_( N > 1 );

	CPose3D			oldPose( *( myObj->getLastPose(index)) );
	CPose2D			drawnSample;
	vector_double   vectLiks(N,0);		// The vector with the individual log-likelihoods.

	for (size_t q=0;q<N;q++)
	{
		myObj->m_movementDrawer.drawSample(drawnSample);
		CPose3D		x_predict = oldPose + CPose3D(drawnSample);

		// Estimate the mean...
		indivLik = auxiliarComputeObservationLikelihood(
			PF_options,
			obj,
			index,
			static_cast<const CSensoryFrame*>(observation),
			&x_predict );

		MRPT_CHECK_NORMAL_NUMBER(indivLik);
		vectLiks[q] = indivLik;
		//accum+= exp(indivLik);

		// and the maximum value:
		if ( indivLik > maxLik )
		{
			maxLikDraw	= drawnSample;
			maxLik		= indivLik;
		}
	}

	// This is done to avoid floating point overflow!!
	//      average_lik    =      \sum(e^liks)   * e^maxLik  /     N
	// log( average_lik  ) = log( \sum(e^liks) ) + maxLik   - log( N )
	double avrgLogLik = math::averageLogLikelihood( vectLiks );

	// Save into the object:
	myObj->m_pfAuxiliaryPFOptimal_estimatedProb[index] = avrgLogLik; // log( accum / N );
	myObj->m_maxLikelihood[index] = maxLik;
	myObj->m_movementDrawMaximumLikelihood[index] = maxLikDraw;

	// and compute the resulting probability of this particle:
	// ------------------------------------------------------------
	return myObj->m_particles[index].log_w + myObj->m_pfAuxiliaryPFOptimal_estimatedProb[index];

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
				auxiliarComputeObservationLikelihood
 ---------------------------------------------------------------*/
double  CMultiMetricMapPDF::auxiliarComputeObservationLikelihood(
	const bayes::CParticleFilter::TParticleFilterOptions &PF_options,
	const CParticleFilterCapable	*obj,
	size_t			particleIndexForMap,
	const CSensoryFrame	*observation,
	const CPose3D			*x )
{
	const CMultiMetricMapPDF	*theObj = static_cast<const CMultiMetricMapPDF*>( obj );
	CMultiMetricMap			*map = &theObj->m_particles[particleIndexForMap].d->mapTillNow;
	double					ret = 0;

	for (CSensoryFrame::const_iterator it=observation->begin();it!=observation->end();it++)
		ret += map->computeObservationLikelihood( (CObservation*)it->pointer(), *x );

	// Done!
	return ret;
}

/*---------------------------------------------------------------
				auxiliarComputeObservationLikelihood
 ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::TPathBin::dumpToStdOut() const
{
	vector_int::const_iterator	it ;

	std::cout << "x   = [";
	for (it= x.begin(); it!=x.end(); it++) std::cout << *it << " ";
	std::cout << "]" << std::endl;

	std::cout << "y   = [";
	for (it= y.begin(); it!=y.end(); it++) std::cout << *it << " ";
	std::cout << "]" << std::endl;

	std::cout << "Phi = [";
	for (it= phi.begin(); it!=phi.end(); it++) std::cout << *it << " ";
	std::cout << "]" << std::endl;

}

/*---------------------------------------------------------------
					loadTPathBinFromPath
 ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::loadTPathBinFromPath(
	CMultiMetricMapPDF::TPathBin	&outBin,
	std::deque<CPose3D>				*path,
	CPose3D							*newPose )
{
	size_t	lenBinPath;

	if (path!=NULL)
			lenBinPath = path->size();
	else	lenBinPath = 0;

	std::deque<CPose3D>::const_iterator	itSrc;
	vector_int::iterator				itX,itY,itPHI;

	// Set the output bin dimensionality:
	outBin.x.resize(lenBinPath + (newPose!=NULL ? 1:0) );
	outBin.y.resize(lenBinPath + (newPose!=NULL ? 1:0));
	outBin.phi.resize(lenBinPath + (newPose!=NULL ? 1:0));

	// Is a path provided??
	if (path!=NULL)
	{
		// Fill the bin data:
		for (itSrc = path->begin(), itX=outBin.x.begin(),itY=outBin.y.begin(),itPHI=outBin.phi.begin();
				itSrc != path->end();
				itSrc++, itX++, itY++, itPHI++ )
		{
			*itX	= round( itSrc->x / options.KLD_params.KLD_binSize_XY );
			*itY	= round( itSrc->y / options.KLD_params.KLD_binSize_XY );
			*itPHI	= round( itSrc->yaw / options.KLD_params.KLD_binSize_PHI );
		} // end-for build path bin
	}

	// Is a newPose provided??
	if (newPose!=NULL)
	{
		// And append the last pose: the new one:
		outBin.x[lenBinPath]   = round( newPose->x / options.KLD_params.KLD_binSize_XY );
		outBin.y[lenBinPath]   = round( newPose->y / options.KLD_params.KLD_binSize_XY );
		outBin.phi[lenBinPath] = round( newPose->yaw / options.KLD_params.KLD_binSize_PHI );
	}
}

/*---------------------------------------------------------------
					findTPathBinIntoSet
 ---------------------------------------------------------------*/
int  CMultiMetricMapPDF::findTPathBinIntoSet(
	CMultiMetricMapPDF::TPathBin						&desiredBin,
	std::deque<CMultiMetricMapPDF::TPathBin>			&theSet
	)
{
	// it = pathBins.find( p ); <---- This doesn't work!!!
	//  TODO: A more efficient search algorithm here!!
	std::deque<CMultiMetricMapPDF::TPathBin>::iterator		it;
	int														ret;

	for (it=theSet.begin(),ret=0;it!=theSet.end();it++,ret++)
		if ( (it->x==desiredBin.x) && (it->y==desiredBin.y) && (it->phi==desiredBin.phi)  )
			return ret;

	// Not found!
	return -1;
}


/*---------------------------------------------------------------
			prediction_and_update_pfStandardProposal
 ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::prediction_and_update_pfStandardProposal(
	const mrpt::slam::CActionCollection	* actions,
	const mrpt::slam::CSensoryFrame		* sf,
	const bayes::CParticleFilter::TParticleFilterOptions &PF_options )
{
	MRPT_TRY_START;

	// ----------------------------------------------------------------------
	//						PREDICTION STAGE
	// ----------------------------------------------------------------------
	CActionRobotMovement2DPtr	robotMovement2D;
	//vector_float				rndSamples;
	size_t 						i,M = m_particles.size();
	CPose3D						initialPose,finalPose;

	// Find a robot movement estimation:
	CPoseRandomSampler			robotActionSampler;
	CPose3D						motionModelMeanIncr, incrPose;
	{
		CActionRobotMovement2DPtr	robotMovement2D = actions->getBestMovementEstimation();
		
		// If there is no 2D action, look for a 3D action:
		if (robotMovement2D.present())
		{
			robotActionSampler.setPosePDF( robotMovement2D->poseChange );
			motionModelMeanIncr = robotMovement2D->poseChange->getEstimatedPose();
		}
		else
		{
			CActionRobotMovement3DPtr	robotMovement3D = actions->getActionByClass<CActionRobotMovement3D>();
			if (robotMovement3D)
			{
				robotActionSampler.setPosePDF( robotMovement3D->poseChange );
				motionModelMeanIncr = robotMovement3D->poseChange.getEstimatedPose();
			}
			else
			{
				THROW_EXCEPTION("Action list does not contain any CActionRobotMovement2D or CActionRobotMovement3D object!");
			}
		}
	}		

	// Average map will need to be updated after this:
	averageMapIsUpdated = false;

	// --------------------------------------------------------------------------------------
	//  Prediction: Simply draw samples from the motion model
	// --------------------------------------------------------------------------------------
	printf(" 1) Prediction...");
	M = m_particles.size();

	// Update particle poses:
	for (i=0;i<M;i++)
	{
		// Generate gaussian-distributed 2D-pose increments according to mean-cov:
		robotActionSampler.drawSample( incrPose );
		finalPose = (*getLastPose(i)) + incrPose;

		// Insert as the new pose in the path:
		m_particles[i].d->robotPath.push_back( finalPose );
	} // end of for each particle

	// ----------------------------------------------------------------------
	//						UPDATE STAGE
	// ----------------------------------------------------------------------
	printf("Ok\n 2) Update...");
	ASSERT_(IS_CLASS(sf,CSensoryFrame));

	vector_double	likelihoodEvals(M);

	// Compute all the likelihood values:
	for (i=0;i<M;i++)
	{
		const CPose3D		*partPose = getLastPose(i);		// Take the particle data:
		//printf("Part %d: (%.3f, %.3f, %.3f, %.3f, %.3f, %.3f)\n", i, partPose->x, partPose->y, partPose->z, partPose->yaw, partPose->pitch, partPose->roll );
		//mrpt::system::pause();
		likelihoodEvals[i] = auxiliarComputeObservationLikelihood(PF_options,this,i,sf,partPose);
	} // for each particle "i"

//	double maxLik = math::maximum( likelihoodEvals);

	// Update m_particles' weights:
	for (i=0;i<M;i++)
		m_particles[i].log_w += likelihoodEvals[i] * PF_options.powFactor;

	printf("Ok\n");

	MRPT_TRY_END;
}



/*---------------------------------------------------------------
				TPredictionParams
  ---------------------------------------------------------------*/
CMultiMetricMapPDF::TPredictionParams::TPredictionParams() :
	pfOptimalProposal_mapSelection(0),
	ICPGlobalAlign_MinQuality(0.70f),
	update_gridMapLikelihoodOptions(),
	KLD_params()
{
}

/*---------------------------------------------------------------
					dumpToTextStream
  ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::TPredictionParams::dumpToTextStream(CStream	&out)
{
	out.printf("\n----------- [CMultiMetricMapPDF::TPredictionParams] ------------ \n\n");

	out.printf("pfOptimalProposal_mapSelection          = %i\n", pfOptimalProposal_mapSelection );
	out.printf("ICPGlobalAlign_MinQuality               = %f\n", ICPGlobalAlign_MinQuality );

	KLD_params.dumpToTextStream(out);
	out.printf("\n");
}

/*---------------------------------------------------------------
					loadFromConfigFile
  ---------------------------------------------------------------*/
void  CMultiMetricMapPDF::TPredictionParams::loadFromConfigFile(
	const mrpt::utils::CConfigFileBase  &iniFile,
	const std::string &section)
{
	pfOptimalProposal_mapSelection = iniFile.read_int(section,"pfOptimalProposal_mapSelection",pfOptimalProposal_mapSelection, true);

	MRPT_LOAD_CONFIG_VAR( ICPGlobalAlign_MinQuality, float,   iniFile,section );

	KLD_params.loadFromConfigFile(iniFile, section);
}

