/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 



#include <mrpt/vision/CStereoServerBumblebee.h>
#include <mrpt/slam/CObservationStereoImages.h>
#include <mrpt/scan_matching/scan_matching.h>
#include <mrpt/utils/CTicTac.h>

using namespace mrpt::vision;
using namespace mrpt::slam;
using namespace mrpt::utils;

// TODO : Docs

/***** Constructor ******/
CStereoServerBumblebee::CStereoServerBumblebee( int cameraIndex, unsigned int resolutionX, unsigned int resolutionY ) :
			grabber(NULL),
			preImgs(NULL),
			curImgs(NULL),
			pvX(), pvY(), pvZ(),
			cvX(), cvY(), cvZ(),
			preGrid(),curGrid(),tmpGrid(),KLTListL(), KLTListR(),
			nStereoImages(0),nIter(0),lastTS(),safetyZone(),
			alert(false),lastTSGrid(),safetyRange(),
			csImgs(),csGrid(),csList(),csGrabber(),csVec()
{
	grabber = new CStereoGrabber_Bumblebee( cameraIndex, resolutionX, resolutionY );
	nStereoImages = 0;
	nIter = 0;

	preImgs = NULL;
	curImgs = NULL;

	// SAFETY ZONE:
	// TO DO: LOAD FROM AN INI FILE ACCORDING TO THE PROPER ROBOT
	// BY NOW: FIXED VALUES (4x4x2 CUBE)
	safetyZone.xMin = -2;
	safetyZone.xMax = 2;
	safetyZone.yMin = -2;
	safetyZone.yMax = 2;
	safetyZone.zMin = 0;
	safetyZone.zMax = 1;

	safetyRange = 2;

	alert = false;

} // end CStereoServerBumblebee()

CStereoServerBumblebee::~CStereoServerBumblebee()
{
	delete grabber;
	// BORRAR PUNTEROS

} // end CStereoServerBumblebee()

// ********************************************************
//						MONITOR CAPTURES
// ********************************************************
/******************************************************************
 						monitorCaptureImgs
 ******************************************************************/

void  CStereoServerBumblebee::monitorCaptureImgs()
{
	// Main function of the class
	// Capture images and disparity in a temp variable
	MRPT_TRY_START;

	csImgs.enter();
	if (preImgs != NULL)
		delete preImgs;

	preImgs = curImgs;
	curImgs = new CObservationStereoImages();

	if ( grabber->getImagesPairObservation( *curImgs ) == false )
		THROW_EXCEPTION("Exception in get Images: MRPT_HAS_BUMBLEBEE NOT ACTIVE!");

	csImgs.leave();

	if( nStereoImages != 2 )
		nStereoImages++;

	MRPT_TRY_END;

} // end monitorCaptureImgs

/******************************************************************
 						monitorCaptureGrid
 ******************************************************************/
void  CStereoServerBumblebee::monitorCaptureGrid()
{
	// Main function of the class
	// Capture images and disparity in a temp variable
	MRPT_TRY_START;

	CObservationVisualLandmarks	tmpObs;

	if( grabber->getObservation( safetyZone, tmpObs ) == false )
		THROW_EXCEPTION("Exception in get Images: MRPT_HAS_BUMBLEBEE NOT ACTIVE!");

	csGrid.enter();
	preGrid = curGrid;
	curGrid = tmpObs;
	csGrid.leave();

	MRPT_TRY_END;

} // end monitorCaptureGrid

/******************************************************************
 					monitorCaptureImgsAndGrid
 ******************************************************************/
void  CStereoServerBumblebee::monitorCaptureImgsAndGrid()
{
	// Main function of the class
	// Capture images and disparity in a temp variable
	MRPT_TRY_START;

	CObservationVisualLandmarks	tmpObs;
	vector_float myvX, myvY, myvZ;

	csImgs.enter();

	if (preImgs != NULL)
		delete preImgs;

	preImgs = curImgs;
	curImgs = new CObservationStereoImages();

	if( grabber->getBothObservation( myvX, myvY, myvZ, *curImgs ) == false )
		THROW_EXCEPTION("Exception in get Images: MRPT_HAS_BUMBLEBEE NOT ACTIVE!");

	csImgs.leave();

	csVec.enter();

	pvX = cvX;
	pvY = cvY;
	pvZ = cvZ;
	cvX = myvX;
	cvY = myvY;
	cvZ = myvZ;

	curGrid = tmpObs;
	csVec.leave();

	if( nStereoImages != 2 )
		nStereoImages++;

	MRPT_TRY_END;


} // end monitorCaptureImgsAndGrid

// ********************************************************
//						MONITOR PROCESSING
// ********************************************************
/******************************************************************
						monitorProcessTracking
 ******************************************************************/
/** /
void  CStereoServerBumblebee::monitorProcessTracking()
{
	//CKanadeLucasTomasi::TKLTFeatureList				KLTList1, KLTList2;
	CFeatureList									KLTList1, KLTList2;
	TSIFTFeatureList								SIFToutList1, SIFToutList2, SIFTmatchL, SIFTmatchR;
	CObservationStereoImages						pImgs, cImgs;

	//CMRPTImageFloat									imgL1, imgR1;		// Previous images
	//CMRPTImageFloat									imgL2, imgR2;		// Current images
	CMRPTImage										imgL1, imgR1;		// Previous images
	CMRPTImage										imgL2, imgR2;		// Current images

	CObservationVisualLandmarks						cloud1, cloud2;
	CLandmark										lm;
	CKanadeLucasTomasi								trackerL, trackerR;
	CTicTac											tictac; // For timing comparative

	// ITERATORS
	CFeatureList::iterator							itKLT;
	//CKanadeLucasTomasi::TKLTFeatureList::iterator	itKLT; //, itKLT2;
	TSIFTFeatureList::iterator						itSIFT;

	unsigned int									mynStereoImages = nStereoImages;		// Temp variable

	if( mynStereoImages == 0 ) // If there is not images to process --> End
		return;

		// Local copies of the shared resources
	csImgs.enter();
	pImgs = preImgs;
	//cImgs = curImgs;

	if( curImgs->timestamp == lastTS ) // If the images have been already processed --> End.
		return;

	// If there are new data --> Process

	// Stereo system parameters -------------
	TStereoSystemParams param;
	param.baseline = curImgs->rightCameraPose.x;
	param.K = curImgs->intrinsicParams;
	param.stdDisp = 1;
	param.stdPixel = 1;
	//---------------------------------------

	MRPT_TRY_START;
	if( mynStereoImages == 1 )	// There are only ONE visual observation (stored in 'curImgs')
	{

		// Set the last timestamp in the Server
		// lastTS = curImgs->timestamp;

		// -----------------------------------------------------------------------------------
		//									COMPUTE SIFTS
		// -----------------------------------------------------------------------------------
		// *************** LEFT IMAGE ***************
		// Extract as many features as possible
		imgL1 = curImgs->imageLeft;
		//tictac.Tic();
		trackerL.selectGoodFeatures( imgL1, KLTList1 );
		//std::cout << "Select good features : " << tictac.Tac()*1000.0f << " ms" << std::endl;

		// Copy detected points into a temporal text file: "temp_KLT_feats.txt"
		//tictac.Tic();
		FILE *fout = os::fopen("temp_KLT_feats.txt","wb");

		for(itKLT = KLTList1.begin(); itKLT != KLTList1.end(); itKLT++)
			os::fprintf(fout, "%.6f %.6f\n", itKLT->x, itKLT->y);
		os::fclose(fout);
		//std::cout << "Copy to file : " << tictac.Tac()*1000.0f << " ms" << std::endl;

		// EXTRACT SIFT
		//tictac.Tic();
		vision::computeSiftFeatures( curImgs->imageLeft, SIFToutList1 );
		//std::cout << "Compute SIFT : " << tictac.Tac()*1000.0f << " ms" << std::endl;

		// *************** RIGHT IMAGE ***************
		// Extract as many features as possible
		imgR1 = curImgs->imageRight;
		//tictac.Tic();
		trackerR.selectGoodFeatures( imgR1, KLTList2 );
		//std::cout << "Select good features : " << tictac.Tac()*1000.0f << " ms" << std::endl;

		// Copy detected points into a temporal text file: "temp_KLT_feats.txt"
		//tictac.Tic();
		fout = os::fopen("temp_KLT_feats.txt","wb");
		for(itKLT = KLTList2.begin(); itKLT != KLTList2.end(); itKLT++)
			os::fprintf(fout, "%.6f %.6f\n", itKLT->x, itKLT->y);
		os::fclose(fout);
		//std::cout << "Copy to file : " << tictac.Tac()*1000.0f << " ms" << std::endl;

		//tictac.Tic();
		vision::computeSiftFeatures( curImgs->imageRight, SIFToutList2 );
		//std::cout << "Compute SIFT : " << tictac.Tac()*1000.0f << " ms" << std::endl;

		// -----------------------------------------------------------------------------------
		//									MATCH SIFTS
		// -----------------------------------------------------------------------------------
		//tictac.Tic();
		vision::matchSIFT( SIFToutList1, SIFToutList2, SIFTmatchL, SIFTmatchR );
		//std::cout << "Match SIFT : " << tictac.Tac()*1000.0f << " ms" << std::endl;
		//std::cout << "Left Matched : " << SIFTmatchL.size() << std::endl;
		//std::cout << "Right Matched : " << SIFTmatchR.size() << std::endl;

		// -----------------------------------------------------------------------------------
		//					PROJECT TO 3D AND STORE THE RESULTING CLOUD
		// -----------------------------------------------------------------------------------
		//tictac.Tic();
		vision::projectMatchedSIFT( SIFTmatchL, SIFTmatchR, param, cloud1.landmarks );
		//std::cout << "Projection : " << tictac.Tac()*1000.0f << " ms" << std::endl;
		//std::cout << "Projected Landmarks : " << cloud1.landmarks.landmarks.size() << std::endl;

		// Convert matched SIFT list to matched KLT list
		KLTList1.resize( SIFTmatchL.size() );
		for( itSIFT = SIFTmatchL.begin(), itKLT = KLTList1.begin(); itSIFT != SIFTmatchL.end(); itSIFT++, itKLT++ )
		{
			itKLT->x = itSIFT->x;
			itKLT->y = itSIFT->y;
			itKLT->ID = itSIFT->ID;
		}


		// Convert matched SIFT list to matched KLT list
		KLTList2.resize( SIFTmatchR.size() );
		for( itSIFT = SIFTmatchR.begin(), itKLT = KLTList2.begin(); itSIFT != SIFTmatchR.end(); itSIFT++, itKLT++ )
		{
			itKLT->x = itSIFT->x;
			itKLT->y = itSIFT->y;
			itKLT->ID = itSIFT->ID;
		}

		// Assign the same timestamp than the images
		cloud1.timestamp = lastTS = curImgs->timestamp;
		cloud1.landmarks.changeCoordinatesReference( (curImgs->cameraPose) );

		// Leave critical section
		csImgs.leave();

		csList.enter();
		KLTListL = KLTList1;
		KLTListR = KLTList2;
		csList.leave();

		csGrid.enter();
		curGrid = cloud1;
		csGrid.leave();

		//cloud1.landmarks.saveToMATLABScript3D("myout\\cloud1.m");

	}
	else if( mynStereoImages == 2 )	// New images in current observation
	{

		// Set the last timestamp in the Server
		// lastTS = curImgs->timestamp;

		// Temporal copy of the KLTLists
		csList.enter();
		KLTList1 = KLTListL;
		KLTList2 = KLTListR;
		csList.leave();

		// Temporal copy of the first cloud
		csGrid.enter();
		cloud1 = curGrid;
		csGrid.leave();

		// -----------------------------------------------------------------------------------
		//									TRACKING
		// -----------------------------------------------------------------------------------
		// Enter critical section
		csImgs.enter();

		// Convert to floatimage
		imgL1 = preImgs->imageLeft;
		imgR1 = preImgs->imageRight;
		imgL2 = curImgs->imageLeft;
		imgR2 = curImgs->imageRight;

		//tictac.Tic();
		trackerL.trackFeatures( imgL1, imgL2, KLTList1 );
		//std::cout << "Track Left: " << tictac.Tac()*1000.0f << " ms" << std::endl;
		//std::cout << KLTList1.size() << " Features tracked in left" << std::endl;

		//tictac.Tic();
		trackerR.trackFeatures( imgR1, imgR2, KLTList2 );
		//std::cout << "Track Right: " << tictac.Tac()*1000.0f << " ms" << std::endl;
		//std::cout << KLTList2.size() << " Features tracked in right" << std::endl;

		// -----------------------------------------------------------------------------------
		//								MATCHES CHECKING
		// -----------------------------------------------------------------------------------
		// By now: row checking -> Delete bad correspondences
		//for( int idx = (int)KLTList1.size()-1; idx >= 0; idx-- )
		CKanadeLucasTomasi::TKLTFeatureList::iterator it1;
		CKanadeLucasTomasi::TKLTFeatureList::iterator it2;
		for (it1=KLTList1.begin(),it2=KLTList2.begin();it1!=KLTList1.end(); )
		{
			if( fabs( it1->y - it2->y ) > 2.0f )
			{
				it1 = KLTList1.erase( it1 );
				it2 = KLTList2.erase( it2 );
			}
			else
			{
				it1++;
				it2++;
			}
		}

		// ARE THERE ENOUGH FEATURES?
		if( KLTList1.size() < trackerL.m_options.min_num_features )
		{
			std::cout << "Not enough features!!!" << std::endl;
			CKanadeLucasTomasi::TKLTFeatureList	TmpKLTList1, TmpKLTList2;

			trackerL.findMoreFeatures( imgL2, KLTList1, TmpKLTList1, 0 );
			// vision::computeSiftFeatures( curImgs->imageLeft, SIFToutList1 );

			//// Process:
			//// 1. Mask current features (and the surrondings 9x9 pixels)
			//// 2. Select new features
			//// 3. Compute SIFTs
			//// 4. Match SIFTs
			//// 5. Add to the KLTLists

			//CMatrix *mask1 = new CMatrix( imgL2.getHeight(), imgL2.getWidth() );
			//CMatrix *mask2 = new CMatrix( imgR2.getHeight(), imgR2.getWidth() );
			//mask1->fill(false);
			//mask2->fill(false);

			//for( itKLT = KLTList1.begin(); itKLT != KLTList1.end(); itKLT++ )
			//{
			//	unsigned int cx = (unsigned int)itKLT->x;
			//	unsigned int cy = (unsigned int)itKLT->y;
			//
			//	// Mask surronding pixels
			//	int xxI = max( 0, cx - 9 );
			//	int xxE = min( cx + 9, mask1->getColCount()-1 );
			//	int	yyI = max( 0, cy - 9 );
			//	int yyE = min( cy + 9, mask1->getRowCount()-1 );

			//	for(int yy = yyI; yy < yyE; yy++)
			//		for(int xx = xxI; xx < xxE; xx++)
			//			(*mask1)( yy, xx ) = true;
			//}

			//for( itKLT = KLTList2.begin(); itKLT != KLTList2.end(); itKLT++ )
			//{
			//	unsigned int cx = (unsigned int)itKLT->x;
			//	unsigned int cy = (unsigned int)itKLT->y;
			//
			//	// Mask surronding pixels
			//	int xxI = max( 0, cx - 9 );
			//	int xxE = min( cx + 9, mask1->getColCount()-1 );
			//	int	yyI = max( 0, cy - 9 );
			//	int yyE = min( cy + 9, mask1->getRowCount()-1 );

			//	for(int yy = yyI; yy < yyE; yy++)
			//		for(int xx = xxI; xx < xxE; xx++)
			//			(*mask2)( yy, xx ) = true;
			//}

			//trackerL.selectGoodFeatures( imgL2, TmpKLTList1, 0, mask1);

			// COMPUTE SIFT FEATURES
			FILE *fout = os::fopen("temp_KLT_feats.txt","wb");

			for(itKLT = TmpKLTList1.begin(); itKLT != TmpKLTList1.end(); itKLT++)
				os::fprintf(fout, "%.6f %.6f\n", itKLT->x, itKLT->y);
			os::fclose(fout);

			// Compute SIFT
			vision::computeSiftFeatures( curImgs->imageLeft, SIFToutList1 );


			//trackerR.selectGoodFeatures( imgR2, TmpKLTList2, 0, mask2);
			trackerL.findMoreFeatures( imgR2, KLTList2, TmpKLTList2, 0 );

			// COMPUTE SIFT FEATURES
			fout = os::fopen("temp_KLT_feats.txt","wb");

			for(itKLT = TmpKLTList2.begin(); itKLT != TmpKLTList2.end(); itKLT++)
				os::fprintf(fout, "%.6f %.6f\n", itKLT->x, itKLT->y);
			os::fclose(fout);

			// Compute SIFT
			vision::computeSiftFeatures( curImgs->imageRight, SIFToutList2 );


			// Match SIFT
			vision::matchSIFT( SIFToutList1, SIFToutList1, SIFTmatchL, SIFTmatchR );

			// Add to the previous KLTList!!
			TSIFTFeatureList::iterator	itSIFT1, itSIFT2;
			for( itSIFT1 = SIFTmatchL.begin(), itSIFT2 = SIFTmatchR.begin(); itSIFT1 != SIFTmatchL.end(); itSIFT1++, itSIFT2++ )
			{
				CKanadeLucasTomasi::TKLTFeature KLTf;

				KLTf.x = itSIFT1->x;
				KLTf.y = itSIFT1->y;
				KLTf.ID = itSIFT1->ID;

				KLTList1.push_back( KLTf );

				KLTf.x = itSIFT2->x;
				KLTf.y = itSIFT2->y;
				KLTf.ID = itSIFT2->ID;

				KLTList2.push_back( KLTf );
			}

		} // end if

		// -----------------------------------------------------------------------------------
		//					PROJECT TO 3D AND STORE THE RESULTING CLOUD
		// -----------------------------------------------------------------------------------
		//std::cout << KLTList1.size() << " Features tracked in left" << std::endl;
		//std::cout << KLTList2.size() << " Features tracked in right" << std::endl;
		//tictac.Tic();
		vision::vision::projectMatchedKLT( KLTList1, KLTList2, param, cloud2.landmarks );
		//std::cout << "Projection KLT: " << tictac.Tac()*1000.0f << " ms" << std::endl;

		// Assign the same timestamp than the images
		cloud2.timestamp = lastTS = curImgs->timestamp;
		cloud2.landmarks.changeCoordinatesReference( curImgs->cameraPose );

		// Leave critical section
		csImgs.leave();

		//cloud2.landmarks.saveToMATLABScript3D("myout\\cloud2.m");

		// Store in class members
		csGrid.enter();
		preGrid = cloud1;
		curGrid = cloud2;
		csGrid.leave();

		// -----------------------------------------------------------------------------------
		//									ICP 6D
		// -----------------------------------------------------------------------------------
		// WE HAVE TWO CLOUDS OF 3D POINTS FOR MAKING 6D ICP!!
		// Transform the clouds of points into a CMetricMap::TMatchedPairList

		CMetricMap::TMatchingPair							pair;
		CMetricMap::TMatchingPairList						list;
		CLandmarksMap::TCustomSequenceLandmarks::iterator	itLand1, itLand2;

		CPose3D												outRt;

		//std::cout << "Cloud1 size: " << cloud1.landmarks.landmarks.size() << std::endl;
		//std::cout << "Cloud2 size: " << cloud2.landmarks.landmarks.size() << std::endl;

		for(itLand1 = cloud1.landmarks.landmarks.begin(); itLand1 != cloud1.landmarks.landmarks.end(); itLand1++)
		{
			for(itLand2 = cloud2.landmarks.landmarks.begin(); itLand2 != cloud2.landmarks.landmarks.end(); itLand2++)
			{
				//std::cout << "ID1 = " << itLand1->ID << " - ID2 = " << itLand2->ID << std::endl;
				if( itLand1->ID == itLand2->ID )
				{
					// Match found!
					pair.this_idx = pair.other_idx = itLand1->ID;
					pair.this_x = itLand1->pose_mean_x;
					pair.this_y = itLand1->pose_mean_y;
					pair.this_z = itLand1->pose_mean_z;

					pair.other_x = itLand2->pose_mean_x;
					pair.other_y = itLand2->pose_mean_y;
					pair.other_z = itLand2->pose_mean_z;

					list.push_back( pair );
				} // end if
			} // end for
		} // end for

		mrpt::scan_matching::leastSquareErrorRigidTransformation6D( list, outRt );

		CMatrix mat = outRt.getHomogeneousMatrix();

		char	filename[50];
		os::sprintf( filename, 50, "myout\\mat%d.txt", nIter );
		mat.saveToTextFile( filename );

		std::cout << "Computed transformation matrix: " << std::endl << mat << std::endl;
	}
	nIter++;

	//// Local copies of the shared resources
	//csImgs.enter();
	//pImgs = preImgs;
	//cImgs = curImgs;
	//csImgs.leave();

	//if( cImgs.timestamp == lastTS ) // If the images have been already processed --> End.
	//	return;

	//// If there are new data --> Process

	//// Stereo system parameters -------------
	//TStereoSystemParams param;
	//param.baseline = cImgs.rightCameraPose.x;
	//param.K = cImgs.intrinsicParams;
	//param.stdDisp = 1;
	//param.stdPixel = 1;
	////---------------------------------------

	//MRPT_TRY_START;
	//if( mynStereoImages == 1 )	// There are only ONE visual observation (stored in 'curImgs')
	//{

	//	// Set the last timestamp in the Server
	//	lastTS = cImgs.timestamp;

	//	// -----------------------------------------------------------------------------------
	//	//									COMPUTE SIFTS
	//	// -----------------------------------------------------------------------------------
	//	// *************** LEFT IMAGE ***************
	//	// Extract as many features as possible
	//	imgL1 = cImgs.imageLeft;
	//	//tictac.Tic();
	//	trackerL.selectGoodFeatures( imgL1, KLTList1 );
	//	//std::cout << "Select good features : " << tictac.Tac()*1000.0f << " ms" << std::endl;

	//	// Copy detected points into a temporal text file: "temp_KLT_feats.txt"
	//	//tictac.Tic();
	//	FILE *fout = os::fopen("temp_KLT_feats.txt","wb");

	//	for(itKLT = KLTList1.begin(); itKLT != KLTList1.end(); itKLT++)
	//		os::fprintf(fout, "%.6f %.6f\n", itKLT->x, itKLT->y);
	//	os::fclose(fout);
	//	//std::cout << "Copy to file : " << tictac.Tac()*1000.0f << " ms" << std::endl;

	//	// EXTRACT SIFT
	//	//tictac.Tic();
	//	vision::computeSiftFeatures( cImgs.imageLeft, SIFToutList1 );
	//	//std::cout << "Compute SIFT : " << tictac.Tac()*1000.0f << " ms" << std::endl;

	//	// *************** RIGHT IMAGE ***************
	//	// Extract as many features as possible
	//	imgR1 = cImgs.imageRight;
	//	//tictac.Tic();
	//	trackerR.selectGoodFeatures( imgR1, KLTList2 );
	//	//std::cout << "Select good features : " << tictac.Tac()*1000.0f << " ms" << std::endl;

	//	// Copy detected points into a temporal text file: "temp_KLT_feats.txt"
	//	//tictac.Tic();
	//	fout = os::fopen("temp_KLT_feats.txt","wb");
	//	for(itKLT = KLTList2.begin(); itKLT != KLTList2.end(); itKLT++)
	//		os::fprintf(fout, "%.6f %.6f\n", itKLT->x, itKLT->y);
	//	os::fclose(fout);
	//	//std::cout << "Copy to file : " << tictac.Tac()*1000.0f << " ms" << std::endl;

	//	//tictac.Tic();
	//	vision::computeSiftFeatures( cImgs.imageRight, SIFToutList2 );
	//	//std::cout << "Compute SIFT : " << tictac.Tac()*1000.0f << " ms" << std::endl;

	//	// -----------------------------------------------------------------------------------
	//	//									MATCH SIFTS
	//	// -----------------------------------------------------------------------------------
	//	//tictac.Tic();
	//	vision::matchSIFT( SIFToutList1, SIFToutList2, SIFTmatchL, SIFTmatchR );
	//	//std::cout << "Match SIFT : " << tictac.Tac()*1000.0f << " ms" << std::endl;
	//	//std::cout << "Left Matched : " << SIFTmatchL.size() << std::endl;
	//	//std::cout << "Right Matched : " << SIFTmatchR.size() << std::endl;

	//	// -----------------------------------------------------------------------------------
	//	//					PROJECT TO 3D AND STORE THE RESULTING CLOUD
	//	// -----------------------------------------------------------------------------------
	//	//tictac.Tic();
	//	vision::projectMatchedSIFT( SIFTmatchL, SIFTmatchR, param, cloud1.landmarks );
	//	//std::cout << "Projection : " << tictac.Tac()*1000.0f << " ms" << std::endl;
	//	//std::cout << "Projected Landmarks : " << cloud1.landmarks.landmarks.size() << std::endl;

	//	// Convert matched SIFT list to matched KLT list
	//	KLTList1.resize( SIFTmatchL.size() );
	//	for( itSIFT = SIFTmatchL.begin(), itKLT = KLTList1.begin(); itSIFT != SIFTmatchL.end(); itSIFT++, itKLT++ )
	//	{
	//		itKLT->x = itSIFT->x;
	//		itKLT->y = itSIFT->y;
	//		itKLT->ID = itSIFT->ID;
	//	}


	//	// Convert matched SIFT list to matched KLT list
	//	KLTList2.resize( SIFTmatchR.size() );
	//	for( itSIFT = SIFTmatchR.begin(), itKLT = KLTList2.begin(); itSIFT != SIFTmatchR.end(); itSIFT++, itKLT++ )
	//	{
	//		itKLT->x = itSIFT->x;
	//		itKLT->y = itSIFT->y;
	//		itKLT->ID = itSIFT->ID;
	//	}

	//	// Assign the same timestamp than the images
	//	cloud1.timestamp = lastTS = pImgs.timestamp;

	//	csList.enter();
	//	KLTListL = KLTList1;
	//	KLTListR = KLTList2;
	//	csList.leave();

	//	csGrid.enter();
	//	curGrid = cloud1;
	//	csGrid.leave();

	//	//cloud1.landmarks.saveToMATLABScript3D("myout\\cloud1.m");

	//}
	//else if( mynStereoImages == 2 )	// New images in current observation
	//{

	//	// Set the last timestamp in the Server
	//	lastTS = cImgs.timestamp;

	//	// Temporal copy of the KLTLists
	//	csList.enter();
	//	KLTList1 = KLTListL;
	//	KLTList2 = KLTListR;
	//	csList.leave();

	//	// Temporal copy of the first cloud
	//	csGrid.enter();
	//	cloud1 = curGrid;
	//	csGrid.leave();

	//	// -----------------------------------------------------------------------------------
	//	//									TRACKING
	//	// -----------------------------------------------------------------------------------
	//	// Convert to floatimage
	//	imgL1 = pImgs.imageLeft;
	//	imgR1 = pImgs.imageRight;
	//	imgL2 = cImgs.imageLeft;
	//	imgR2 = cImgs.imageRight;

	//	//tictac.Tic();
	//	trackerL.trackFeatures( imgL1, imgL2, KLTList1 );
	//	//std::cout << "Track Left: " << tictac.Tac()*1000.0f << " ms" << std::endl;
	//	//std::cout << KLTList1.size() << " Features tracked in left" << std::endl;
	//
	//	//tictac.Tic();
	//	trackerR.trackFeatures( imgR1, imgR2, KLTList2 );
	//	//std::cout << "Track Right: " << tictac.Tac()*1000.0f << " ms" << std::endl;
	//	//std::cout << KLTList2.size() << " Features tracked in right" << std::endl;

	//	// -----------------------------------------------------------------------------------
	//	//								MATCHES CHECKING
	//	// -----------------------------------------------------------------------------------
	//	// By now: row checking -> Delete bad correspondences
	//	for( int idx = (int)KLTList1.size()-1; idx >= 0; idx-- )
	//	{
	//		if( abs( KLTList1[idx].y - KLTList2[idx].y ) > 2.0f )
	//		{
	//			KLTList1.erase( KLTList1.begin() + idx );
	//			KLTList2.erase( KLTList2.begin() + idx );
	//		}
	//	}

	//	//for( itKLT = KLTList1.begin(), itKLT2 = KLTList2.begin(); itKLT != KLTList1.end(); itKLT++, itKLT2++)
	//	//{
	//	//	if( abs(itKLT->y - itKLT2->y) > 2.0f )
	//	//	{
	//	//		KLTList1.erase( itKLT );
	//	//		KLTList2.erase( itKLT2 );
	//	//	}
	//	//}

	//	// ARE THERE ENOUGH FEATURES?
	//	if( KLTList1.size() < trackerL.m_options.min_num_features )
	//	{
	//		std::cout << "Not enough features!!!" << std::endl;
	//		// Process:
	//		// 1. Mask current features (and the surrondings 9x9 pixels)
	//		// 2. Select new features
	//		// 3. Compute SIFTs
	//		// 4. Match SIFTs
	//		// 5. Add to the KLTLists

	//		CMatrix *mask1 = new CMatrix( imgL2.getHeight(), imgL2.getWidth() );
	//		CMatrix *mask2 = new CMatrix( imgR2.getHeight(), imgR2.getWidth() );
	//		mask1->fill(false);
	//		mask2->fill(false);

	//		for( itKLT = KLTList1.begin(); itKLT != KLTList1.end(); itKLT++ )
	//		{
	//			unsigned int cx = (unsigned int)itKLT->x;
	//			unsigned int cy = (unsigned int)itKLT->y;
	//
	//			// Mask surronding pixels
	//			int xxI = max( 0, cx - 9 );
	//			int xxE = min( cx + 9, mask1->getColCount()-1 );
	//			int	yyI = max( 0, cy - 9 );
	//			int yyE = min( cy + 9, mask1->getRowCount()-1 );

	//			for(int yy = yyI; yy < yyE; yy++)
	//				for(int xx = xxI; xx < xxE; xx++)
	//					(*mask1)( yy, xx ) = true;
	//		}

	//		for( itKLT = KLTList2.begin(); itKLT != KLTList2.end(); itKLT++ )
	//		{
	//			unsigned int cx = (unsigned int)itKLT->x;
	//			unsigned int cy = (unsigned int)itKLT->y;
	//
	//			// Mask surronding pixels
	//			int xxI = max( 0, cx - 9 );
	//			int xxE = min( cx + 9, mask1->getColCount()-1 );
	//			int	yyI = max( 0, cy - 9 );
	//			int yyE = min( cy + 9, mask1->getRowCount()-1 );

	//			for(int yy = yyI; yy < yyE; yy++)
	//				for(int xx = xxI; xx < xxE; xx++)
	//					(*mask2)( yy, xx ) = true;
	//		}

	//		CKanadeLucasTomasi::TKLTFeatureList	TmpKLTList1, TmpKLTList2;

	//		trackerL.selectGoodFeatures( imgL2, TmpKLTList1, 0, mask1);

	//		// COMPUTE SIFT FEATURES
	//		FILE *fout = os::fopen("temp_KLT_feats.txt","wb");

	//		for(itKLT = TmpKLTList1.begin(); itKLT != TmpKLTList1.end(); itKLT++)
	//			os::fprintf(fout, "%.6f %.6f\n", itKLT->x, itKLT->y);
	//		os::fclose(fout);

	//		// Compute SIFT
	//		vision::computeSiftFeatures( cImgs.imageLeft, SIFToutList1 );


	//		trackerR.selectGoodFeatures( imgR2, TmpKLTList2, 0, mask2);
	//
	//		// COMPUTE SIFT FEATURES
	//		fout = os::fopen("temp_KLT_feats.txt","wb");

	//		for(itKLT = TmpKLTList2.begin(); itKLT != TmpKLTList2.end(); itKLT++)
	//			os::fprintf(fout, "%.6f %.6f\n", itKLT->x, itKLT->y);
	//		os::fclose(fout);

	//		// Compute SIFT
	//		vision::computeSiftFeatures( cImgs.imageRight, SIFToutList2 );

	//		// Match SIFT
	//		vision::matchSIFT( SIFToutList1, SIFToutList1, SIFTmatchL, SIFTmatchR );

	//		// Add to the previous KLTList!!
	//		TSIFTFeatureList::iterator	itSIFT1, itSIFT2;
	//		for( itSIFT1 = SIFTmatchL.begin(), itSIFT2 = SIFTmatchR.begin(); itSIFT1 != SIFTmatchL.end(); itSIFT1++, itSIFT2++ )
	//		{
	//			CKanadeLucasTomasi::TKLTFeature KLTf;

	//			KLTf.x = itSIFT1->x;
	//			KLTf.y = itSIFT1->y;
	//			KLTf.ID = itSIFT1->ID;

	//			KLTList1.push_back( KLTf );

	//			KLTf.x = itSIFT2->x;
	//			KLTf.y = itSIFT2->y;
	//			KLTf.ID = itSIFT2->ID;

	//			KLTList2.push_back( KLTf );
	//		}

	//	} // end if

	//	// -----------------------------------------------------------------------------------
	//	//					PROJECT TO 3D AND STORE THE RESULTING CLOUD
	//	// -----------------------------------------------------------------------------------
	//	//std::cout << KLTList1.size() << " Features tracked in left" << std::endl;
	//	//std::cout << KLTList2.size() << " Features tracked in right" << std::endl;
	//	//tictac.Tic();
	//	vision::vision::projectMatchedKLT( KLTList1, KLTList2, param, cloud2.landmarks );
	//	//std::cout << "Projection KLT: " << tictac.Tac()*1000.0f << " ms" << std::endl;
	//
	//	// Assign the same timestamp than the images
	//	cloud2.timestamp = lastTS = cImgs.timestamp;

	//	//cloud2.landmarks.saveToMATLABScript3D("myout\\cloud2.m");

	//	// Store in class members
	//	csGrid.enter();
	//	preGrid = cloud1;
	//	curGrid = cloud2;
	//	csGrid.leave();

	//	// -----------------------------------------------------------------------------------
	//	//									ICP 6D
	//	// -----------------------------------------------------------------------------------
	//	// WE HAVE TWO CLOUDS OF 3D POINTS FOR MAKING 6D ICP!!
	//	// Transform the clouds of points into a CMetricMap::TMatchedPairList

	//	CMetricMap::TMatchingPair							pair;
	//	CMetricMap::TMatchingPairList						list;
	//	CLandmarksMap::TCustomSequenceLandmarks::iterator	itLand1, itLand2;
	//
	//	CPose3D												outRt;

	//	//std::cout << "Cloud1 size: " << cloud1.landmarks.landmarks.size() << std::endl;
	//	//std::cout << "Cloud2 size: " << cloud2.landmarks.landmarks.size() << std::endl;

	//	for(itLand1 = cloud1.landmarks.landmarks.begin(); itLand1 != cloud1.landmarks.landmarks.end(); itLand1++)
	//	{
	//		for(itLand2 = cloud2.landmarks.landmarks.begin(); itLand2 != cloud2.landmarks.landmarks.end(); itLand2++)
	//		{
	//			//std::cout << "ID1 = " << itLand1->ID << " - ID2 = " << itLand2->ID << std::endl;
	//			if( itLand1->ID == itLand2->ID )
	//			{
	//				// Match found!
	//				pair.this_idx = pair.other_idx = (unsigned int)itLand1->ID;
	//				pair.this_x = itLand1->pose_mean_x;
	//				pair.this_y = itLand1->pose_mean_y;
	//				pair.this_z = itLand1->pose_mean_z;

	//				pair.other_x = itLand2->pose_mean_x;
	//				pair.other_y = itLand2->pose_mean_y;
	//				pair.other_z = itLand2->pose_mean_z;

	//				list.push_back( pair );
	//			} // end if
	//		} // end for
	//	} // end for
	//
	//	mrpt::scan_matching::ICP6D( list, outRt );

	//	CMatrix mat = outRt.getHomogeneousMatrix();

	//	char	filename[50];
	//	os::sprintf( filename, 50, "myout\\mat%d.txt", nIter );
	//	mat.saveToTextFile( filename );

	//	std::cout << "Computed transformation matrix: " << std::endl << mat << std::endl;
	//}
	//nIter++;
	MRPT_TRY_END;
}
/ ** /
/ ******************************************************************
					monitorObstacleAvoidance
 ******************************************************************/

void  CStereoServerBumblebee::monitorObstacleAvoidance()
{
	MRPT_TRY_START;

	CTicTac			tictac;
	vector_float	myvX, myvY, myvZ;
	float			absMin;

	std::cout << "monitorObstacleAvoidance started" << std::endl;

	//// ******************************************
	//// OPTION 1 -> FILTERING AND POSTPROCESSING
	//// ******************************************
	//csGrid.enter();
	//myObs = curGrid;
	//csGrid.leave();

	//// Not processing the same observation
	//if( myObs.timestamp == lastTSGrid )
	//	return;

	//// Reset ALERT flag
	//alert = false;
	//// Set the timestamp for this observation
	//lastTSGrid = myObs.timestamp;

	//// Process observation
	//// TO DO: Spurius filtering
	//std::cout << "SIZE: " << myObs.landmarks.size() << std::endl;
	//tictac.Tic();
	//mrpt::slam::CLandmarksMap::TCustomSequenceLandmarks::iterator	itLand;
	//for( itLand = myObs.landmarks.landmarks.begin(); itLand != myObs.landmarks.landmarks.end(); itLand++ )
	//{
	//	if( itLand->pose_mean_x > safetyZone.xMin && itLand->pose_mean_x < safetyZone.xMax &&
	//		itLand->pose_mean_y > safetyZone.yMin && itLand->pose_mean_y < safetyZone.yMax &&
	//		itLand->pose_mean_z > safetyZone.zMin && itLand->pose_mean_z < safetyZone.zMax )
	//	{
	//		// ALERT!! -> OBSTACLE DETECTED
	//		std::cout << "ALERT! OBSTACLE DETECTED IN SAFETY ZONE!" << std::endl;
	//		alert = true;
	//	} // end if

	//} // end for
	//std::cout << "Elapsed time:" << 1000.0f * tictac.Tac() << std::endl;

	//if( alert == false )
	//	std::cout << "OK" << std::endl;

	// *********************************************
	// OPTION 2 -> DETECT ONLY IN THE SAFETY AREA
	// *********************************************
	csVec.enter();
	myvX = cvX;
	myvY = cvY;
	myvZ = cvZ;
	csVec.leave();

	//// Not processing the same observation
	//if( myObs.timestamp == lastTSGrid ){
	//	std::cout << "Same observation" << std::endl;
	//	return;
	//}

	// CObservation2DRangeScan
	CObservation2DRangeScan	scan;
	//scan.aperture = (float)DEG2RAD( 66.71 );
	scan.aperture = M_PIf;
	scan.rightToLeft = false;
	scan.maxRange = 14.9f;

	int nBins = 360;
	float step = scan.aperture/nBins;

	// Vector of mins
	vector_float vMin;
	vMin.resize( nBins, scan.maxRange );

	scan.validRange.resize( nBins, 0 );

	absMin = scan.maxRange;

	vector_float::iterator	itX, itY, itZ;
	for( itX = myvX.begin(), itY = myvY.begin(), itZ = myvZ.begin();
		itX != myvX.end(), itY != myvY.end(), itZ != myvZ.end();
		itX++, itY++, itZ++ )
	{
		// Compute 'r' and '\theta' of every pixel
		float r = sqrt( (*itX)*(*itX) + (*itZ)*(*itZ) );
		float theta = atan2( *itX, *itZ );

		// Insert into the corresponding bin
		int bin = round(theta/step);

		if( r < vMin[bin] )
		{
			scan.validRange[bin] = 1;
			vMin[bin] = r;
			if( r < absMin )
				absMin = r;

		}

	} // end for

	// Reset the ALERT flag
	if ( absMin < safetyRange )
	 alert = false;

	// Set the timestamp for this observation
	//lastTSGrid = myObs.timestamp;

	//std::cout << "TEST: " << myObs.landmarks.landmarks.size() << std::endl;
	//if( myObs.landmarks.landmarks.size() != 0 )
	//{
	//		// ALERT!! -> OBSTACLE DETECTED
	//		std::cout << "ALERT! OBSTACLE DETECTED IN SAFETY ZONE!" << std::endl;
	//		alert = true;
	//} // end if

	//if( alert == false )
	//	std::cout << "OK" << std::endl;

	MRPT_TRY_END;

} // end monitorObstacleAvoidance

/******************************************************************
					captureImgsAndObs
 ******************************************************************/

void  CStereoServerBumblebee::captureImgsAndObs()
{
	// Main function of the class
	// Capture images and disparity in a temp variable

	//csTmpImgs.enter();
	//csTmpGrid.enter();
	//if ( grabber->getBothObservation( tmpGrid, tmpImgs ) == false )
	//	THROW_EXCEPTION("Exception in get Images: MRPT_HAS_BUMBLEBEE NOT ACTIVE!");
	//csTmpGrid.leave();
	//csTmpImgs.leave();
	//
	//// Copy to the proper member
	//csImgs.enter();
	//curImgs = tmpImgs;
	//
	//csImgs.leave();

	//csGrid.enter();
	//curGrid = tmpGrid;
	//csGrid.leave();
}

/******************************************************************
							getImgs
 ******************************************************************/
void  CStereoServerBumblebee::getImgs(
	mrpt::slam::CObservationStereoImages &outImgs)
{
	csImgs.enter();
	outImgs = *curImgs;
	csImgs.leave();
} // end getImages

/******************************************************************
 							getObs
 ******************************************************************/
void  CStereoServerBumblebee::getObs(
	mrpt::slam::CObservationVisualLandmarks &outObs)
{
	csGrid.enter();
	outObs = curGrid;
	csGrid.leave();
} // end getImages

/******************************************************************
 							getImgsAndObs
 ******************************************************************/
void  CStereoServerBumblebee::getImgsAndObs(
	mrpt::slam::CObservationStereoImages &outImgs,
	mrpt::slam::CObservationVisualLandmarks &outGrid)
{
	csImgs.enter();
	outImgs = *curImgs;
	csImgs.leave();

	csGrid.enter();
	outGrid = curGrid;
	csGrid.leave();
} // end getImages


//void  CStereoServerBumblebee::getImages(
//	mrpt::slam::CObservationStereoImages &outImgs)
//{
//
//	cSecImgs.enter();
//	if ( grabber->getImagesPairObservation( curImgs ) == false )
//		THROW_EXCEPTION("Exception in get Images: MRPT_HAS_BUMBLEBEE NOT ACTIVE!");
//	outImgs = curImgs;
//	cSecImgs.leave();
//} // end getImages()
//
//
//void  CStereoServerBumblebee::getGrid3D(
//			mrpt::slam::CObservationVisualLandmarks &outObs)
//{
//	// There is no ROI
//	cSecGrid.enter();
//	if ( grabber->getObservation( curGrid ) == false )
//		THROW_EXCEPTION("Exception in GRID 3D Computation with Bumblebee: MRPT_HAS_BUMBLEBEE NOT ACTIVE!");
//	outObs = curGrid;
//	cSecGrid.leave();
//}
//
//void  CStereoServerBumblebee::getGrid3D(
//			TROI ROI,
//			mrpt::slam::CObservationVisualLandmarks &outObs)
//{
//	// Insert ROI into getObservation
//	cSecGrid.enter();
//	if ( grabber->getObservation( ROI, curGrid ) == false )
//		THROW_EXCEPTION("Exception in GRID 3D Computation with Bumblebee (+ROI): MRPT_HAS_BUMBLEBEE NOT ACTIVE!");
//	outObs = curGrid;
//	cSecGrid.leave();
//}
//void  CStereoServerBumblebee::getBothImagesAndGrid(
//			mrpt::slam::CObservationStereoImages &outImgs,
//			mrpt::slam::CObservationVisualLandmarks &outObs )
//{
//	cSecGrid.enter();
//	if ( grabber->getBothObservation( curGrid, curImgs )== false )
//		THROW_EXCEPTION("Exception in both images and grid extraction: MRPT_HAS_BUMBLEBEE NOT ACTIVE!");
//	outObs = curGrid;
//	outImgs = curImgs;
//	cSecGrid.leave();
//}
//
//void  CStereoServerBumblebee::getBothImagesAndGrid(
//			TROI ROI,
//			mrpt::slam::CObservationStereoImages &outImgs,
//			mrpt::slam::CObservationVisualLandmarks &outObs )
//{
//	// Insert ROI into getObservation
//	cSecGrid.enter();
//	if ( grabber->getBothObservation( ROI, curGrid, curImgs ) == false )
//		THROW_EXCEPTION("Exception in GRID 3D Computation with Bumblebee (+ROI): MRPT_HAS_BUMBLEBEE NOT ACTIVE!");
//	outObs = curGrid;
//	outImgs = curImgs;
//	cSecGrid.leave();
//}
//

