/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2008  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/system/os.h>
#include <mrpt/hwdrivers/CCameraSensor.h>
#include <mrpt/vision/CImageGrabber_OpenCV.h>
#include <mrpt/utils/CTypeSelector.h>
#include <mrpt/slam/CObservationImage.h>

using namespace mrpt::hwdrivers;
using namespace mrpt::utils;
using namespace mrpt::slam;
using namespace mrpt::system;
using namespace mrpt::vision;
using namespace std;

IMPLEMENTS_GENERIC_SENSOR(CCameraSensor,mrpt::hwdrivers)

/* -----------------------------------------------------
                Constructor
   ----------------------------------------------------- */
CCameraSensor::CCameraSensor() :
	m_sensorPose			(),
	m_sensorLabel			("CAMERA"),
	m_grabber_type			("opencv"),
	m_cv_camera_index		(0),
	m_cv_camera_type		("CAMERA_CV_AUTODETECT"),
	m_cv_options			(),
	m_dc1394_camera_guid	(0),
	m_dc1394_camera_unit	(0),
	m_dc1394_options		(),
	m_preview_decimation	(0),
	m_path_for_external_images	(),
	m_external_images_format	("jpg"),
	m_cap_cv			(NULL),
	m_cap_dc1394		(NULL),
	m_preview_counter	(0),
	m_preview_win		(NULL)
{
	m_state = CGenericSensor::ssInitializing;
}

/* -----------------------------------------------------
                initialize
   ----------------------------------------------------- */
void CCameraSensor::initialize()
{
	cout << "[CCameraSensor::initialize] Opening camera..." << endl;
	close();

	// Select type of device
	m_grabber_type = trim( toLowerCase( m_grabber_type ) );
	m_cv_camera_type= trim( toUpperCase(m_cv_camera_type) );

	if (m_grabber_type=="opencv")
	{
		// OpenCV driver:
		mrpt::utils::CTypeSelector	camera_type(
		"CAMERA_CV_AUTODETECT,CAMERA_CV_DC1394,CAMERA_CV_VFL,CAMERA_CV_VFW,CAMERA_CV_MIL",
		"CAMERA_CV_AUTODETECT");

		int idx = camera_type.checkTypeIndex(m_cv_camera_type);
		if (idx<0)
		{
			m_state = CGenericSensor::ssError;
			THROW_EXCEPTION_CUSTOM_MSG1("Invalid value of :'%s'",m_cv_camera_type.c_str())
		}

		cout << format("[CCameraSensor::initialize] opencv camera, index: %i type: %i...\n", int(m_cv_camera_index),idx);
		m_cap_cv = new CImageGrabber_OpenCV( m_cv_camera_index, TCameraType(idx), m_cv_options );

		if (!m_cap_cv->isOpen())
		{
			m_state = CGenericSensor::ssError;
			THROW_EXCEPTION("[CCameraSensor::initialize] ERROR: Couldn't open OpenCV camera.")
		}

		cout << "[CCameraSensor::initialize] Done!" << endl;
		m_state = CGenericSensor::ssWorking;
	}
	else if (m_grabber_type=="dc1394")
	{
		//m_cap_dc1394
		cout << format("[CCameraSensor::initialize] dc1394 camera, GUID: 0x%lX  UNIT:%d...\n", long(m_dc1394_camera_guid),m_dc1394_camera_unit);
		m_cap_dc1394 = new CImageGrabber_dc1394( m_dc1394_camera_guid, m_dc1394_camera_unit, m_dc1394_options, true /* verbose */ );

		if (!m_cap_dc1394->isOpen())
		{
			m_state 	= CGenericSensor::ssError;
			THROW_EXCEPTION("[CCameraSensor::initialize] ERROR: Couldn't open dc1394 camera.")
		}

		cout << "[CCameraSensor::initialize] Done!" << endl;
		m_state = CGenericSensor::ssWorking;
	}
	else
	THROW_EXCEPTION_CUSTOM_MSG1("Unknown 'grabber_type' found: %s", m_grabber_type.c_str() )
}

/* -----------------------------------------------------
                close
   ----------------------------------------------------- */
void CCameraSensor::close()
{
	if (m_cap_cv)
	{
		delete m_cap_cv;
		m_cap_cv = NULL;
	}
	if (m_cap_dc1394)
	{
		delete m_cap_dc1394;
		m_cap_dc1394 = NULL;
	}
	m_state = CGenericSensor::ssInitializing;
}

/* -----------------------------------------------------
                Destructor
   ----------------------------------------------------- */
void  CCameraSensor::loadConfig(
	const mrpt::utils::CConfigFileBase &configSource,
	const std::string	  &iniSection )
{
	m_sensorLabel	= configSource.read_string( iniSection, "sensorLabel", m_sensorLabel );

	m_grabber_type	= configSource.read_string_first_word( iniSection, "grabber_type", m_grabber_type );
	MRPT_LOAD_HERE_CONFIG_VAR( preview_decimation, int, m_preview_decimation , configSource, iniSection )

	// OpenCV options:
	m_cv_camera_type= configSource.read_string_first_word( iniSection, "cv_camera_type", m_cv_camera_type );
	m_cv_camera_index = configSource.read_int( iniSection, "cv_camera_index", m_cv_camera_index);

	m_cv_options.frame_width 	= configSource.read_int( iniSection, "cv_frame_width", m_cv_options.frame_width );
	m_cv_options.frame_height	= configSource.read_int( iniSection, "cv_frame_height", m_cv_options.frame_height );
	m_cv_options.gain			= configSource.read_double( iniSection, "cv_gain", m_cv_options.gain );
	m_cv_options.ieee1394_fps	= configSource.read_double( iniSection, "cv_fps", m_cv_options.ieee1394_fps );

	// dc1394 options:
	MRPT_LOAD_HERE_CONFIG_VAR( dc1394_camera_guid, uint64_t, m_dc1394_camera_guid, configSource, iniSection )
	MRPT_LOAD_HERE_CONFIG_VAR( dc1394_camera_unit, int, m_dc1394_camera_unit, configSource, iniSection )

	MRPT_LOAD_HERE_CONFIG_VAR( dc1394_frame_width, int, m_dc1394_options.frame_width, configSource, iniSection )
	MRPT_LOAD_HERE_CONFIG_VAR( dc1394_frame_height, int, m_dc1394_options.frame_height, configSource, iniSection )

	MRPT_LOAD_HERE_CONFIG_VAR( dc1394_mode7, int, m_dc1394_options.mode7, configSource, iniSection )

	MRPT_LOAD_HERE_CONFIG_VAR( dc1394_shutter, int, m_dc1394_options.shutter, configSource, iniSection )
	MRPT_LOAD_HERE_CONFIG_VAR( dc1394_gain, int, m_dc1394_options.gain, configSource, iniSection )
	MRPT_LOAD_HERE_CONFIG_VAR( dc1394_gamma, int, m_dc1394_options.gamma, configSource, iniSection )
	MRPT_LOAD_HERE_CONFIG_VAR( dc1394_brightness, int, m_dc1394_options.brightness, configSource, iniSection )
	MRPT_LOAD_HERE_CONFIG_VAR( dc1394_exposure, int, m_dc1394_options.exposure, configSource, iniSection )
	MRPT_LOAD_HERE_CONFIG_VAR( dc1394_sharpness, int, m_dc1394_options.sharpness, configSource, iniSection )
	MRPT_LOAD_HERE_CONFIG_VAR( dc1394_white_balance, int, m_dc1394_options.white_balance, configSource, iniSection )

	// Special stuff: FPS
	map<double,grabber_dc1394_framerate_t>	map_fps;
	map<double,grabber_dc1394_framerate_t>::iterator it_fps;
	map_fps[1.875] = FRAMERATE_1_875;
	map_fps[3.75] = FRAMERATE_3_75;
	map_fps[7.5] = FRAMERATE_7_5;
	map_fps[15] = FRAMERATE_15;
	map_fps[30] = FRAMERATE_30;
	map_fps[60] = FRAMERATE_60;
	map_fps[120] = FRAMERATE_120;
	map_fps[240] = FRAMERATE_240;

	double the_fps = configSource.read_double( iniSection, "dc1394_framerate", 15.0 );
	it_fps = map_fps.find(the_fps);
	if (it_fps == map_fps.end())
		THROW_EXCEPTION_CUSTOM_MSG1("ERROR: Framerate seems to be not a valid number: %f",the_fps);

	m_dc1394_options.framerate =  it_fps->second;

	// Special stuff: color encoding:
	map<string,grabber_dc1394_color_coding_t>			map_color;
	map<string,grabber_dc1394_color_coding_t>::iterator it_color;
#define ADD_COLOR_MAP(c)  map_color[#c] = c;
	ADD_COLOR_MAP(COLOR_CODING_MONO8)
	ADD_COLOR_MAP(COLOR_CODING_YUV411)
	ADD_COLOR_MAP(COLOR_CODING_YUV422)
	ADD_COLOR_MAP(COLOR_CODING_YUV444)
	ADD_COLOR_MAP(COLOR_CODING_RGB8)
	ADD_COLOR_MAP(COLOR_CODING_MONO16)

	string the_color_coding = mrpt::system::toUpperCase( configSource.read_string_first_word( iniSection, "dc1394_color_coding", "COLOR_CODING_YUV422" ) );
	it_color = map_color.find(the_color_coding );
	if (it_color == map_color.end())
		THROW_EXCEPTION_CUSTOM_MSG1("ERROR: Color coding seems not to be valid : '%s'",the_color_coding.c_str() );
	m_dc1394_options.color_coding = it_color->second;


	// Sensor pose:
	m_sensorPose.setFromValues(
		configSource.read_float(iniSection,"pose_x",0),
		configSource.read_float(iniSection,"pose_y",0),
		configSource.read_float(iniSection,"pose_z",0),
		DEG2RAD( configSource.read_float(iniSection,"pose_yaw",0) ),
		DEG2RAD( configSource.read_float(iniSection,"pose_pitch",0) ),
		DEG2RAD( configSource.read_float(iniSection,"pose_roll",0) )
		);
}

/* -----------------------------------------------------
                Destructor
   ----------------------------------------------------- */
CCameraSensor::~CCameraSensor()
{
	close();

	if (m_preview_win)
	{
		delete m_preview_win;
		m_preview_win=NULL;
	}
}


/* -----------------------------------------------------
				doProcess
----------------------------------------------------- */
void  CCameraSensor::doProcess()
{
	CObservationImagePtr obs = CObservationImagePtr( new CObservationImage() );

	bool  capture_ok = false;

	if (m_cap_cv)
	{
		if (!m_cap_cv->getObservation( *obs ))
		{	// Error
			m_state = CGenericSensor::ssError;
			THROW_EXCEPTION("Error grabbing image");
		}
		else capture_ok = true;
	}
	else if (m_cap_dc1394)
	{
		if (!m_cap_dc1394->getObservation( *obs ))
		{	// Error
			m_state = CGenericSensor::ssError;
			THROW_EXCEPTION("Error grabbing image");
		}
		else capture_ok = true;
	}

	// If we grabbed an image: prepare it and add it to the internal queue:
	if (capture_ok)
	{	// OK:
		obs->sensorLabel = m_sensorLabel;
		obs->cameraPose  = m_sensorPose;

		// External storage?
		if (!m_path_for_external_images.empty())
		{
			// CMRPTImage::IMAGES_PATH_BASE must be set in the calling application.
			string filName = fileNameStripInvalidChars( trim(m_sensorLabel) ) + format( "_%f.%s", (double)timestampTotime_t( obs->timestamp ), m_external_images_format.c_str() );
			//cout << "[CCameraSensor] Saving " << filName << endl;
			obs->image.saveToFile( CMRPTImage::IMAGES_PATH_BASE + string("/") +filName );
			obs->image.setExternalStorage( filName );
		}
		appendObservation(obs);

		// Show preview??
		if (m_preview_decimation>0)
		{	// Yes
			if (++m_preview_counter > m_preview_decimation)
			{
				m_preview_counter = 0;

				// Create the first time:
				if (!m_preview_win)
				{
					string caption = string("Preview of ")+m_sensorLabel;
					if (m_preview_decimation>1)
						caption += format(" (decimation: %i)",m_preview_decimation);

					m_preview_win = new mrpt::gui::CDisplayWindow(caption);
				}
				m_preview_win->showImage(obs->image);
			}

		} // end show preview


	}
}

/* -----------------------------------------------------
				setPathForExternalImages
----------------------------------------------------- */
void CCameraSensor::setPathForExternalImages( const std::string &directory )
{
	if (!mrpt::system::createDirectory( directory ))
	{
		THROW_EXCEPTION_CUSTOM_MSG1("Error: Cannot create the directory for externally saved images: %s",directory.c_str() )
	}

	m_path_for_external_images = directory;
}
