/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#ifndef CRoboticHeadInterface_H
#define CRoboticHeadInterface_H

#include <mrpt/hwdrivers/CInterfaceFTDI.h>
#include <mrpt/hwdrivers/CInterfaceFTDIMessages.h>
#include <mrpt/utils/CMRPTMessage.h>
#include <mrpt/synch.h>
#include <mrpt/math/CMatrixTemplate.h>
#include <mrpt/utils/CDebugOutputCapable.h>
#include <mrpt/utils/CConfigFileBase.h>
#include <mrpt/slam/CObservationRange.h>


namespace mrpt
{
	namespace hwdrivers
	{
		using namespace mrpt::math;

		/**   This "software driver" implements the communication protocol for interfacing a Robotic Head Board through a custom
		  *   USB RS-422 interface board.
		  *   In this class the "bind" is ignored since it is designed for USB connections only, thus it internally generate the required object for simplicity of use.
		  *   The default serial number is "OREJA001"
		  *
		  * Warning: Avoid defining an object of this class in a global scope if you want to catch all potential
		  *      exceptions during the constructors (like USB interface DLL not found, etc...)
		  *
		  *  \code
		  *  PARAMETERS IN THE ".INI"-LIKE CONFIGURATION STRINGS:
		  * -------------------------------------------------------
		  *   [supplied_section_name]
		  *   HEAD_serialNumber=OREJA001
		  *   HEAD_gain=127,127,127
		  *   HEAD_yaw=0		//orientación inicial
		  *   HEAD_pitch=0		//elevación inicial
		  *  \endcode
		  *
		  */
		class HWDLLIMPEXP CRoboticHeadInterface : public mrpt::utils::CDebugOutputCapable
		{
			private:
				CInterfaceFTDIMessages	m_usbConnection;
				utils::CMRPTMessage		msg;
				std::string				m_serialNumber;
				std::vector<int32_t>	gain;
				int						head_yaw, head_pitch;

				bool 		checkControllerIsConnected();

			public:
				/** Constructor
				  */
				CRoboticHeadInterface();

				/** Destructor
				  */
				~CRoboticHeadInterface(){}

				/** Loads specific configuration for the device from a given source of configuration parameters, for example, an ".ini" file,
				  *  loading from the section "[iniSection]" (see utils::CConfigFileBase and derived classes)
				  *  See hwdrivers::CSonarSRF10 for the possible parameters
				  */
				void  loadConfig(	const mrpt::utils::CConfigFileBase *configSource,
									const std::string	  &iniSection );

				/** Changes the serial number of the device to open
				  */
				void  setDeviceSerialNumber(const std::string &deviceSerialNumber)
				{
					m_serialNumber = deviceSerialNumber;
				}

				/** Read the gain for the amplifier of the ear "channel", where channel is 0, 1 or 2.
				  */
				void GetGain(int &_gain,int &channel);

				/** Set the gain for the amplifier each ear. The value range is [0x00(min) .. 0x7F(max)]. The value 0x80 set the resistor
				  * in high impedance state, ¡¡¡DON'T USE IT!!!
				  */
				bool SetGain(int &new_gain,int &channel);

				/** This function return the angle where last sound where detected. This angle is related to the robot pose, NOT head pose.
				  * \code
				  * angle > 0deg --> Sound detected in the left
				  * angle = 0deg --> Sound detected in front of the head
				  * angle < 0deg --> Sound detected in the right
				  * \endcode
				  */
				void GetSoundLocation(int &ang);

				/** Debug only!!! This function return the last 500 acquired samples for each sound channel.
				  *
				  */
				void Get3SoundBuffer(CMatrixTemplate<int>	&buf);

				/** Move the head in:
				  * \code
				  * elevation = 'yaw' degrees
				  * orientation = 'pitch' degrees
				  * \endcode
				  */
				void SetHeadPose(int &yaw, int &pitch);

				//TO DO --> Funciones del acelerómetro
				//			Movimientos faciales

		};	// End of class

	} // End of namespace
} // end of namespace

#endif

