/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef opengl_CPolyhedron_H
#define opengl_CPolyhedron_H

#include <mrpt/opengl/CRenderizable.h>
#include <mrpt/poses/CPoint3D.h>
#include <mrpt/utils/stl_extensions.h>

namespace mrpt	{
namespace opengl	{
	using namespace mrpt::utils;
	using namespace mrpt::poses;
	using namespace std;
	using mrpt::poses::CPoint3D;

	class MRPTDLLIMPEXP CPolyhedron;
	//class MRPTDLLIMPEXP TPolyhedronEdge;
	//class MRPTDLLIMPEXP TPolyhedronFace;
	// This must be added to any CSerializable derived class:

	DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE(CPolyhedron,CRenderizable)
	/** An arbitrary polyhedron.
	  * \sa opengl::COpenGLScene
	  */
	class MRPTDLLIMPEXP CPolyhedron:public CRenderizable	{
		DEFINE_SERIALIZABLE(CPolyhedron)
	public:
		struct TPolyhedronEdge	{
			uint32_t v1;
			uint32_t v2;
			TPolyhedronEdge():v1(0),v2(0)	{}
			bool operator==(const TPolyhedronEdge &e) const	{
				if (e.v1==v1&&e.v2==v2) return true;
				else return e.v1==v2&&e.v2==v1;
			}
			float length(const vector_serializable<CPoint3D> &vertices) const;
			~TPolyhedronEdge()	{}
		};
		struct TPolyhedronFace	{
			vector_serializable<uint32_t> vertices;
			float normal[3];
			TPolyhedronFace():vertices()	{}
			~TPolyhedronFace()	{}
			float area(const vector_serializable<CPoint3D> &vertices) const;
		};
	protected:
		vector_serializable<CPoint3D> mVertices;
		vector_serializable<TPolyhedronEdge> mEdges;
		vector_serializable<TPolyhedronFace> mFaces;
		bool mWireframe;
		float mLineWidth;
	public:
		static CPolyhedronPtr Create(const vector<CPoint3D> &vertices,const vector<vector<uint32_t> > &faces)	{
			vector<TPolyhedronFace> aux;
			for (vector<vector<uint32_t> >::const_iterator it=faces.begin();it!=faces.end();it++)	{
				TPolyhedronFace f;
				f.vertices=*it;
				aux.push_back(f);
			}
			return Create(vertices,aux);
		}
		static CPolyhedronPtr Create(const vector<CPoint3D> &vertices,const vector<TPolyhedronFace> &faces)	{
			return CPolyhedronPtr(new CPolyhedron(vertices,faces,true));
		}
		/** Static methods to create frequent polyhedrons. More bizarre polyhedrons are intended to be added in a near future.
		  */

		//Regular polyhedrons
		static CPolyhedronPtr CreateTetrahedron(const float radius);
		static CPolyhedronPtr CreateHexahedron(const float radius);
		static CPolyhedronPtr CreateOctahedron(const float radius);
		static CPolyhedronPtr CreateDodecahedron(const float radius);
		static CPolyhedronPtr CreateIcosahedron(const float radius);

		//A few archimedean solids
		static CPolyhedronPtr CreateTruncatedTetrahedron(const float radius);
		static CPolyhedronPtr CreateCuboctahedron(const float radius);
		static CPolyhedronPtr CreateTruncatedHexahedron(const float radius);
		static CPolyhedronPtr CreateTruncatedOctahedron(const float radius);
		static CPolyhedronPtr CreateRhombicuboctahedron(const float radius);
		static CPolyhedronPtr CreateIcosidodecahedron(const float radius);
		//static CPolyhedronPtr CreateTruncatedDodecahedron(const float radius);
		//static CPolyhedronPtr CreateTruncatedIcosahedron(const float radius);

		//Other common polyhedrons
		static CPolyhedronPtr CreateCubicPrism(const float x1,const float x2,const float y1,const float y2,const float z1,const float z2);
		static CPolyhedronPtr CreatePyramid(const vector<CPoint2D> &baseVertices,const float height);
		static CPolyhedronPtr CreateDoublePyramid(const vector<CPoint2D> &baseVertices,const float height1,const float height2);
		static CPolyhedronPtr CreateTruncatedPyramid(const vector<CPoint2D> &baseVertices,const float height,const float ratio);
		static CPolyhedronPtr CreateCustomPrism(const vector<CPoint2D> &baseVertices,const float height);
		static CPolyhedronPtr CreateRegularAntiprism(const uint32_t numBaseEdges,const float baseRadius,const float height);
		static CPolyhedronPtr CreateRegularPrism(const uint32_t numbaseEdges,const float baseRadius,const float height);
		static CPolyhedronPtr CreateCustomAntiprism(const vector<CPoint2D> &bottomBase,const vector<CPoint2D> &topBase,const float height);
		static CPolyhedronPtr CreateParallelepiped(const CPoint3D &base,const CPoint3D &v1,const CPoint3D &v2,const CPoint3D &v3);
		static CPolyhedronPtr CreateRegularPyramid(const uint32_t numBaseEdges,const float baseRadius,const float height);
		static CPolyhedronPtr CreateRegularDoublePyramid(const uint32_t numBaseEdges,const float baseRadius,const float height1,const float height2);
		static CPolyhedronPtr CreateArchimedeanRegularPrism(const uint32_t numBaseEdges,const float baseRadius);
		static CPolyhedronPtr CreateArchimedeanRegularAntiprism(const uint32_t numBaseEdges,const float baseRadius);
		static CPolyhedronPtr CreateRegularTruncatedPyramid(const uint32_t numBaseEdges,const float baseRadius,const float height,const float ratio);

		/** Render
		  */
		void render() const;
		/** Ray trace
		  */
		virtual bool traceRay(const mrpt::poses::CPose3D &o,float &dist) const;
		void getVertices(vector<CPoint3D> &vertices) const	{
			vertices=mVertices;
		}
		void getEdges(vector<TPolyhedronEdge> &edges) const	{
			edges=mEdges;
		}
		void getFaces(vector<TPolyhedronFace> &faces) const	{
			faces=mFaces;
		}
		uint32_t getNumberOfVertices() const	{
			return mVertices.size();
		}
		uint32_t getNumberOfEdges() const	{
			return mEdges.size();
		}
		uint32_t getNumberOfFaces() const	{
			return mFaces.size();
		}
		void getEdgesLength(vector<float> &lengths) const;
		void getFacesArea(vector<float> &areas) const;
		float getVolume() const;
		inline bool isWireframe() const	{
			return mWireframe;
		}
		inline void setWireframe(bool enabled=true)	{
			mWireframe=enabled;
		}
		inline float getLineWidth() const	{
			return mLineWidth;
		}
		inline void setLineWidth(float lineWidth)	{
			mLineWidth=lineWidth;
		}
	private:
		bool setNormal(TPolyhedronFace &f,bool doCheck=true);
		void addEdges(const TPolyhedronFace &e);
		static bool checkConsistence(const vector<CPoint3D> &vertices,const vector<TPolyhedronFace> &faces);
		CPolyhedron():mVertices(),mEdges(),mFaces(),mWireframe(false),mLineWidth(1)	{}
		CPolyhedron(const vector<CPoint3D> &vertices,const vector<TPolyhedronFace> &faces,bool doCheck=true):mVertices(vertices),mEdges(),mFaces(faces),mWireframe(false),mLineWidth(1)	{
			if (doCheck) if (!checkConsistence(vertices,faces)) throw std::logic_error("Face list accesses a vertex out of range");
			for (vector<TPolyhedronFace>::iterator it=mFaces.begin();it!=mFaces.end();it++)	{
				if (!setNormal(*it,doCheck)) throw std::logic_error("Bad face specification");
				addEdges(*it);
			}
		}
		static CPolyhedronPtr CreateNoCheck(const vector<CPoint3D> &vertices,const vector<TPolyhedronFace> &faces)	{
			return CPolyhedronPtr(new CPolyhedron(vertices,faces,false));
		}
		static CPolyhedronPtr CreateEmpty()	{
			return CPolyhedronPtr(new CPolyhedron());
		}
		virtual ~CPolyhedron()	{}
	};
	MRPTDLLIMPEXP mrpt::utils::CStream& operator>>(mrpt::utils::CStream& in,CPolyhedron::TPolyhedronEdge &o);
	MRPTDLLIMPEXP mrpt::utils::CStream& operator<<(mrpt::utils::CStream& out,const CPolyhedron::TPolyhedronEdge &o);
	MRPTDLLIMPEXP mrpt::utils::CStream& operator>>(mrpt::utils::CStream& in,CPolyhedron::TPolyhedronFace &o);
	MRPTDLLIMPEXP mrpt::utils::CStream& operator<<(mrpt::utils::CStream& out,const CPolyhedron::TPolyhedronFace &o);
}
}
#endif
